// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_2.x.x_GSC_DN/samples/sio4flash/perform.c $
// $Rev: 54418 $
// $Date: 2024-05-15 14:14:30 -0500 (Wed, 15 May 2024) $

// SIO4: Sample Application: source file

#include "main.h"




//*****************************************************************************
static void _display_number(long num)
{
	const char*	factor	= "";

	if ((num % 1024) == 0)
	{
		factor	= "K";
		num		/= 1024;
	}

	if ((num % 1024) == 0)
	{
		factor	= "M";
		num		/= 1024;
	}

	printf("%ld%s", num, factor);
}



//*****************************************************************************
// returned: 0 = PASS, <0 = error code.
static int _buffer_allocate(flash_t* flash, const char* name, void** ptr)
{
	int	ret	= 0;

	gsc_label(name);

	ptr[0]	= malloc(flash->dev_bytes);

	if (ptr[0])
	{
		printf("PASS  (");
		_display_number(flash->dev_bytes);
		printf("B, ");
		gsc_label_long_comma(flash->dev_bytes);
		printf(" Bytes, ");
		memset(ptr[0], 0, flash->dev_bytes);
		printf("cleared)\n");
	}
	else
	{
		ret	= errno ? errno : -ENOMEM;
		printf("FAIL <---  (");
		_display_number(flash->dev_bytes);
		printf("B, ");
		gsc_label_long_comma(flash->dev_bytes);
		printf(" Bytes: %d)\n", ret);
	}

	return(ret);
}



//*****************************************************************************
// returned: 0 = PASS, <0 = error code.
static int _buffers_allocate(flash_t* flash)
{
	int	ret;
	int	tmp;

	gsc_label("Allocate Buffers");
	printf("\n");
	gsc_label_level_inc();

	ret	= _buffer_allocate(flash, "Flash Buffer", (void**) &flash->dev_buffer);
	tmp	= _buffer_allocate(flash, "File Buffer", (void**) &flash->file_buffer);
	ret	= ret ? ret : tmp;

	gsc_label_level_dec();
	return(ret);
}



//*****************************************************************************
// returned: 0 = PASS, <0 = error code.
static int _buffers_free(flash_t* flash)
{
	gsc_label("Free Buffers");

	if (flash->dev_buffer)
	{
		free(flash->dev_buffer);
		flash->dev_buffer	= NULL;
	}

	if (flash->file_buffer)
	{
		free(flash->file_buffer);
		flash->file_buffer	= NULL;
	}

	printf("Done\n");
	return(0);
}



//*****************************************************************************
// returned: 0 = PASS, <0 = error code.
static int _check_access(const args_t* args, flash_t* flash)
{
	int			access;
	gsc_reg_t	reg;
	int			ret;

	for (;;)	// A convenience loop.
	{
		gsc_label("Register Access");

		reg.reg	= FLASH_REG_RFR;
		ret		= sio4_ioctl(args->fd, SIO4_IOCTL_REG_READ, &reg);

		if (ret)
		{
			flash->supported	= 0;
			printf("FAIL <---  (RFR read failed: %d)\n", ret);
			break;
		}

		flash->rfr	= reg.value;
		access		= flash->rfr & 0xFFFF;

		switch (access)
		{
			default:

				ret					= EINVAL;
				flash->supported	= 0;
				printf(	"FAIL <---  (option not recognized: 0x%08X, RFR 0x%08lX)\n",
						access,
						(long) flash->rfr);
				break;

			case 0:

				printf("PASS  (32-bit registers, 32-bit boundaries)\n");
				break;

			case 1:

				flash->supported	= 0;
				printf(	"FAIL <---  (Not Supported: "
						"16-bit registers, 32-bit boundaries)\n");
				break;

			case 2:

				flash->supported	= 0;
				printf(	"FAIL <---  (Not Supported: "
						"16-bit registers, 16-bit boundaries)\n");
				break;
		}

		break;
	}

	return(ret);
}



//*****************************************************************************
static void _dev_id_set(flash_t* flash, const char* model, int bits)
{
	flash->dev_model	= model;
	flash->dev_bits		= bits;
	flash->dev_bytes	= flash->dev_bits / 8;
	flash->dev_pages	= flash->dev_bytes / PAGE_SIZE;

	printf("%s  (", flash->dev_model);
	_display_number(flash->dev_bits);
	printf("b, ");
	_display_number(flash->dev_bytes);
	printf("B, ");
	_display_number(flash->dev_pages);
	printf(" Pages @ %d Bytes Each)\n", PAGE_SIZE);
}



//*****************************************************************************
// returned: 0 = PASS, <0 = error code.
static int _get_dev_id(const args_t* args, flash_t* flash)
{
	gsc_reg_t	reg;
	int			ret;

	for (;;)	// A convenience loop.
	{
		gsc_label("Reprogramming ID");

		reg.reg		= FLASH_REG_RIR;
		ret			= sio4_ioctl(args->fd, SIO4_IOCTL_REG_READ, &reg);
		flash->rir	= reg.value;

		if (ret)
		{
			flash->supported	= 0;
			printf("FAIL <---  (RIR read error: %d)\n", ret);
			break;
		}

		printf("\n");
		gsc_label_level_inc();

		gsc_label("Board ID");
		flash->board_id	= (flash->rir & 0xFFFF) >> 4;
		printf("0x%04X\n", flash->board_id);

		gsc_label("Reprogramming Rev.");
		flash->reprogram_rev	= flash->rir & 0xF;
		printf("%d  (", flash->reprogram_rev);

		if (flash->reprogram_rev == 0)
			printf("NR)\n");
		else
			printf("%c)\n", flash->reprogram_rev + 'A' - 1);

		gsc_label_level_dec();
		break;
	}

	return(ret);
}



//*****************************************************************************
// returned: 0 = PASS, <0 = error code.
static int _id_flash_device(const args_t* args, flash_t* flash)
{
	gsc_reg_t	reg;
	int			ret;

	for (;;)	// A convenience loop.
	{
		gsc_label("Flash Device");

		ret	= flash_command(args, FLASH_CMD_RX_SIL_ID, 1);

		if (ret)
		{
			flash->supported	= 0;
			printf("FAIL <---  (Read Silicon ID failed: %d)\n", ret);
			break;
		}

		reg.reg	= FLASH_REG_RCSR;
		ret		= sio4_ioctl(args->fd, SIO4_IOCTL_REG_READ, &reg);

		if (ret)
		{
			flash->supported	= 0;
			printf("FAIL <---  (RCSR read failed: %d)\n", ret);
			break;
		}

		flash->dev_id	= reg.value >> 24;

		switch (flash->dev_id)
		{
			default:

				ret					= -EINVAL;
				flash->supported	= 0;
				printf(	"FAIL <---  (Unrecognized: 0x%02X, RCSR 0x%08lX)\n",
						flash->dev_id,
						(long) reg.value);
				break;

			case 0x10:	_dev_id_set(flash, "EPCS1",   1 * 1024 * 1024);	break;
			case 0x12:	_dev_id_set(flash, "EPCS4",   4 * 1024 * 1024);	break;
			case 0x14:	_dev_id_set(flash, "EPCS16", 16 * 1024 * 1024);	break;
			case 0x16:	_dev_id_set(flash, "EPCS64", 64 * 1024 * 1024);	break;
		}

		break;
	}

	return(ret);
}



//*****************************************************************************
// returned: 0 = PASS, <0 = error code.
static int _check_implimentation(const args_t* args, flash_t* flash)
{
	gsc_reg_t	reg;
	int			ret;

	for (;;)	// A convenience loop.
	{
		gsc_label("Implimentation ID");

		reg.reg		= FLASH_REG_RCR;
		ret			= sio4_ioctl(args->fd, SIO4_IOCTL_REG_READ, &reg);
		flash->rcr	= reg.value;

		if (ret)
		{
			flash->supported	= 0;
			printf("FAIL <---  (RCR read failed: %d)\n", ret);
			break;
		}

		if (flash->rcr == 0xA54812A5)
		{
			printf("PASS  (RCR = 0x%08lX)\n", (long) flash->rcr);
		}
		else
		{
			flash->supported	= 0;
			printf(	"FAIL <---  (RCR not recognized: 0x%08lX)\n",
					(long) flash->rcr);
		}

		break;
	}

	return(ret);
}



//*****************************************************************************
// returned: 0 = PASS, <0 = error code.
static int _check_support(const args_t* args, flash_t* flash)
{
	s32			arg;
	gsc_reg_t	reg;
	int			ret;
	u32			value;

	for (;;)	// Convenience loop.
	{
		gsc_label("Flash Reprogramming");

		arg	= SIO4_QUERY_REG_FR;
		ret	= sio4_ioctl(args->fd, SIO4_IOCTL_QUERY, &arg);

		if (ret)
		{
			flash->supported	= 0;
			printf("FAIL <---  (FR query error: %d)\n", ret);
			break;
		}

		if (arg == 0)
		{
			flash->supported	= 0;
			printf("Not Supported  (FR not supported.)\n");
			break;
		}

		reg.reg	= SIO4_GSC_FR;
		ret		= sio4_ioctl(args->fd, SIO4_IOCTL_REG_READ, &reg);

		if (ret)
		{
			flash->supported	= 0;
			printf("FAIL <---  (FR read error: %d)\n", ret);
			break;
		}

		value	= GSC_FIELD_DECODE(reg.value, 17, 16);

		if (value != 1)
		{
			flash->supported	= 0;
			printf("Not Supported  (Feature not supported.)\n");
			break;
		}

		printf("Supported\n");
		flash->supported	= 1;
		break;
	}

	return(ret);
}



//*****************************************************************************
// returned: 0 = PASS, >0 = number of errors
int get_model_number(const args_t* args, const char** model)
{
	static char	buf[128];
	int			errs	= 0;
	const char*	psz		= buf;

	errs	+= sio4_id_form_factor(args->fd, &psz, 1);
	strcpy(buf, psz);

	errs	+= sio4_id_model(args->fd, &psz);
	strcat(buf, psz);

	errs	+= sio4_id_fifo_size_total(args->fd, 1, &psz);
	strcat(buf, psz);

	if (model)
		model[0]	= buf;

	return(errs);
}



//*****************************************************************************
// returned: 0 = PASS, >0 = number of errors
int perform_tests(const args_t* args)
{
	int		errs;
	flash_t	flash;
	int		ret;

	memset(&flash, 0, sizeof(flash));

	ret	= _check_support(args, &flash);

	if ((ret == 0) && (flash.supported))
		ret = _check_access(args, &flash);

	if ((ret == 0) && (flash.supported))
		ret = _check_implimentation(args, &flash);

	if ((ret == 0) && (flash.supported))
		ret = _get_dev_id(args, &flash);

	if ((ret == 0) && (flash.supported))
		ret = _id_flash_device(args, &flash);

	if ((ret == 0) && (flash.supported))
		ret = _buffers_allocate(&flash);

	if ((ret == 0) && (flash.supported))
	{
		switch (args->action)
		{
			default:

				ret	= -EINVAL;
				gsc_label("INTERNAL ERROR");
				printf(	"FAIL <---  (Unrecognized action: %d)\n",
						(int) args->action);
				break;

			case ACT_NONE:	break;
			case ACT_ERASE:	ret	= flash_erase(args);			break;
			case ACT_READ:	ret	= flash_read(args, &flash);		break;
			case ACT_WRITE:	ret	= flash_write(args, &flash);	break;
		}

	}

	if ((flash.dev_buffer) || (flash.file_buffer))
		_buffers_free(&flash);

	errs	= ret ? 1 : 0;
	return(errs);
}


