// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_2.x.x_GSC_DN/samples/sio4flash/flash.c $
// $Rev: 54418 $
// $Date: 2024-05-15 14:14:30 -0500 (Wed, 15 May 2024) $

// SIO4: Sample Application: source file

#include "main.h"



// prototypes *****************************************************************

static int	_read_flash_page(const args_t* args, int adrs);
static void	_reverse_dev_buffer_page(flash_t* flash, int page);
static u8	_reverse_u8_bits(u32 value);
static void	_time_stamp(char* dst);
static int	_write_flash_page(const args_t* args, int adrs);
static int	_xfer_page_to_dev_buffer(const args_t* args, flash_t* flash, int adrs);
static int	_xfer_page_to_dev_ram(const args_t* args, flash_t* flash, int adrs);



//*****************************************************************************
static void _page_label(const args_t* args, flash_t* flash, int page)
{
	static char	buf[80]	= "";
	int			end;
	int			i;
	int			len;

	if (args->verbose)
	{
		if ((page % 64) == 0)
		{
			if (page)
				printf("\n");

			end	= page + 63;

			if (end > flash->dev_pages)
				end	= flash->dev_pages;

			strcpy(buf, "Pages ");
			gsc_label_long_comma_buf(page, buf + strlen(buf));
			strcat(buf, " to ");
			gsc_label_long_comma_buf(end, buf + strlen(buf));
			gsc_label(buf);
		}

		printf(".");
	}
	else
	{
		len	= (int) strlen(buf);

		if (page)
		{
			for (i = 0; i < len; i++)
				printf("\b");
		}

		if (page >= 0)
		{
			strcpy(buf, "Page ");
			gsc_label_long_comma_buf(page, buf + 5);
			printf("%s", buf);
		}
		else
		{
			for (i = 0; i < len; i++)
				printf(" ");

			for (i = 0; i < len; i++)
				printf("\b");
		}
	}

	fflush(stdout);
}



//*****************************************************************************
// returned: 0 = PASS, <0 = error code.
static int _read_flash(const args_t* args, flash_t* flash)
{
	int	adrs;
	int	page;
	int	ret		= 0;

	flash->dev_bytes	= flash->dev_pages * PAGE_SIZE;

	gsc_label("Reading");

	if (args->verbose)
	{
		printf("\n");
		gsc_label_level_inc();
	}

	// Read flash =========================================

	for (page = 0; (ret == 0) && (page < flash->dev_pages); page++)
	{
		adrs	= page * PAGE_SIZE;
		_page_label(args, flash, page);
		ret		= _read_flash_page(args, adrs);

		if (ret == 0)
			ret	= _xfer_page_to_dev_buffer(args, flash, adrs);

		if (ret == 0)
			_reverse_dev_buffer_page(flash, page);
	}

	if (args->verbose)
	{
		printf("\n");
		gsc_label_level_dec();
	}
	else
	{
		_page_label(args, flash, -1);

		if (ret == 0)
		{
			printf("PASS  (");
			gsc_label_long_comma(flash->dev_pages);
			printf(" Pages)\n");
		}
	}

	return(ret);
}



//*****************************************************************************
// returned: 0 = PASS, <0 = error code.
static int _read_flash_page(const args_t* args, int adrs)
{
	gsc_reg_t	reg;
	int			ret;

	for (;;)	// A convenience loop.
	{
		reg.reg		= FLASH_REG_RSAR;
		reg.value	= adrs;
		ret			= sio4_ioctl(args->fd, SIO4_IOCTL_REG_WRITE, &reg);

		if (ret)
		{
			printf(" FAIL <---  (RSAR write failed: %d)\n", ret);
			break;
		}

		ret	= flash_command(args, FLASH_CMD_RX_BLOCK, 1);

		if (ret)
		{
			printf(" FAIL <---  (Read Block failed: %d)\n", ret);
			break;
		}

		break;
	}

	return(ret);
}



//*****************************************************************************
// returned: 0 = PASS, <0 = error code.
static int _read_rpd_file(const args_t* args, flash_t* flash)
{
	FILE*	file	= NULL;
	long	got;
	int		ret		= 0;
	long	size;

	for (;;)	// A convenience loop.
	{
		gsc_label("Read .rpd File");
		file	= file_open(args->src_file);

		if (file == NULL)
		{
			ret	= -errno;
			printf("FAIL <---  (File open failure: %s)\n", args->src_file);
			break;
		}

		size	= file_size(file);

		if (size > flash->dev_bytes)
		{
			ret	= -EINVAL;
			printf("FAIL <---  (Source file too large: ");
			gsc_label_long_comma(size);
			printf(" Bytes)\n");
			break;
		}

		got	= file_read_bin(file, (void*) flash->file_buffer, size);

		if (got != size)
		{
			ret	= -EIO;
			printf("FAIL <---  (Failed to read file: ");
			printf("requested ");
			gsc_label_long_comma(size);
			printf(" Bytes, got ");
			gsc_label_long_comma(got);
			printf(" Bytes)\n");
			break;
		}

		printf("PASS\n");
		break;
	}

	if (file)
		file_close(file);

	return(ret);
}



//*****************************************************************************
static void _reverse_dev_buffer_page(flash_t* flash, int page)
{
	int	i;
	int	index;
	u8	tmp;

	// Reverse u8 bits ====================================
	index	= page * PAGE_SIZE;

	for (i = 0; i < PAGE_SIZE; i++, index++)
		flash->dev_buffer[index]	= _reverse_u8_bits(flash->dev_buffer[index]);

	// Reverse u32 bytes ==================================

	index	= page * PAGE_SIZE;

	for (i = 0; i < PAGE_SIZE; i += 4, index += 4)
	{
		// bytes 0 and 3
		tmp								= flash->dev_buffer[index];
		flash->dev_buffer[index]		= flash->dev_buffer[index + 3];
		flash->dev_buffer[index + 3]	= tmp;

		// bytes 1 and 2
		tmp								= flash->dev_buffer[index + 1];
		flash->dev_buffer[index + 1]	= flash->dev_buffer[index + 2];
		flash->dev_buffer[index + 2]	= tmp;
	}
}



//*****************************************************************************
static u8 _reverse_u8_bits(u32 value)
{
	u8	dst		= 0;
	u8	dst_b	= 0x80;
	int	i;
	u8	src_b	= 0x01;

	for (i = 0; i < 8; i++, src_b <<= 1, dst_b >>= 1)
	{
		if (value & src_b)
			dst	|= dst_b;
	}

	return(dst);
}



//*****************************************************************************
// returned: 0 = PASS, <0 = error code.
static int _save_rpd_file(const args_t* args, flash_t* flash)
{
	long		bytes;
	int			errs;
	FILE*		file;
	const char*	model	= "";
	char		name[256];
	int			ret		= 0;

	for (;;)	// A convenience loop.
	{
		gsc_label("Save .rpd File");

		errs	= get_model_number(args, &model);

		if (errs)
		{
			ret	= -1;
			printf("FAIL <---  (Failed to acquire model number.)\n");
			break;
		}

		sprintf(name, "%s_", model);

		_time_stamp(name + strlen(name));
		strcat(name, ".rpd");
		file	= file_create(name);

		if (file == NULL)
		{
			ret	= -1;
			printf("FAIL <---  (File creation failed: %s)\n", name);
			break;
		}

		bytes	= file_write_bin(file, (void*) flash->dev_buffer, flash->dev_bytes);

		if (bytes < 0)
		{
			ret	= -1;
			printf("FAIL <---  (Write failure: %s)\n", name);
		}
		else if (bytes != (long) flash->dev_bytes)
		{
			ret	= -1;
			printf("FAIL <---  (Write failure: only ");
			gsc_label_long_comma(bytes);
			printf(" Bytes, ");
			gsc_label_long_comma(flash->dev_bytes);
			printf(" requested, %s)\n", name);
		}

		file_close(file);

		if (ret == 0)
		{
			printf("PASS  (");
			gsc_label_long_comma(bytes);
			printf(" Bytes, ");
			printf("%s)\n", name);
		}

		break;
	}

	return(ret);
}



//*****************************************************************************
static void _time_stamp(char* dst)
{
	struct tm	st;
	 time_t		tt;

	time(&tt);
	st	= *localtime(&tt);
	sprintf(dst,
			"%04d%02d%02d_%02d%02d%02d",
			st.tm_year + 1900,
			st.tm_mon + 1,
			st.tm_mday,
			st.tm_hour,
			st.tm_min,
			st.tm_sec);
}



//*****************************************************************************
// returned: 0 = PASS, <0 = error code.
static int _wait_for_done(const args_t* args)
{
	gsc_reg_t	reg;
	int			ret;

	do
	{
		reg.reg	= FLASH_REG_RCSR;
		ret		= sio4_ioctl(args->fd, SIO4_IOCTL_REG_READ, &reg);
	} while ((ret == 0) && (reg.value & D8));

	return(ret);
}



//*****************************************************************************
// returned: 0 = PASS, <0 = error code.
static int _wait_for_done_seconds(const args_t* args, int* duration)
{
	char			buf[64]	= "";
	int				i;
	int				len;
	os_time_ns_t	now;
	gsc_reg_t		reg;
	int				ret;
	int				seconds	= -1;
	os_time_ns_t	t0		= { 0, 0 };

	for (;;)	// A convenience loop.
	{
		os_sleep_ms(1);
		os_time_get_ns(&now);

		if ((now.tv_sec > t0.tv_sec) && (now.tv_nsec > t0.tv_nsec))
		{
			t0	= now;
			len	= strlen(buf);

			for (i = 0; i < len; i++)
				printf("\b");

			seconds++;
			gsc_label_long_comma_buf(seconds, buf);
			printf("%s", buf);
			fflush(stdout);
		}

		reg.reg	= FLASH_REG_RCSR;
		ret		= sio4_ioctl(args->fd, SIO4_IOCTL_REG_READ, &reg);

		if (ret)
			break;

		if (reg.value & D8)
			continue;

		break;
	}

	len	= strlen(buf);

	for (i = 0; i < len; i++)
		printf("\b \b");

	if (duration)
		duration[0]	= seconds;

	return(ret);
}



//*****************************************************************************
// returned: 0 = PASS, <0 = error code.
static int _write_flash(const args_t* args, flash_t* flash)
{
	int	adrs;
	int	page;
	int	ret		= 0;

	flash->dev_bytes	= flash->dev_pages * PAGE_SIZE;

	gsc_label("Writing");

	if (args->verbose)
	{
		printf("\n");
		gsc_label_level_inc();
	}

	// Copy to dev buffer =================================
	memcpy(flash->dev_buffer, flash->file_buffer, flash->dev_bytes);

	// Write to flash =====================================

	for (page = 0; (ret == 0) && (page < flash->dev_pages); page++)
	{
		adrs	= page * PAGE_SIZE;
		_page_label(args, flash, page);

		_reverse_dev_buffer_page(flash, page);

		ret	= _xfer_page_to_dev_ram(args, flash, adrs);

		if (ret == 0)
			ret	= _write_flash_page(args, adrs);
	}

	if (args->verbose)
	{
		printf("\n");
		gsc_label_level_dec();
	}
	else
	{
		_page_label(args, flash, -1);

		if (ret == 0)
		{
			printf("PASS  (");
			gsc_label_long_comma(flash->dev_pages);
			printf(" Pages)\n");
		}
	}

	return(ret);
}



//*****************************************************************************
// returned: 0 = PASS, <0 = error code.
static int _write_flash_page(const args_t* args, int adrs)
{
	gsc_reg_t	reg;
	int			ret;

	for (;;)	// A convenience loop.
	{
		reg.reg		= FLASH_REG_RSAR;
		reg.value	= adrs;
		ret			= sio4_ioctl(args->fd, SIO4_IOCTL_REG_WRITE, &reg);

		if (ret)
		{
			printf(" FAIL <---  (RSAR write failed: %d)\n", ret);
			break;
		}

		ret	= flash_command(args, FLASH_CMD_TX_ENABLE, 1);

		if (ret)
		{
			printf(" FAIL <---  (Write Enable failed: %d)\n", ret);
			break;
		}

		ret	= flash_command(args, FLASH_CMD_TX_BLOCK, 1);

		if (ret)
		{
			printf(" FAIL <---  (Write Block failed: %d)\n", ret);
			break;
		}

		break;
	}

	return(ret);
}



//*****************************************************************************
// returned: 0 = PASS, <0 = error code.
static int _xfer_page_to_dev_buffer(const args_t* args, flash_t* flash, int adrs)
{
	int			i;
	int			offset;
	u32*		ptr		= (void*) (flash->dev_buffer + adrs);
	gsc_reg_t	reg;
	int			ret;

	offset	= GSC_REG_OFFSET_DECODE(FLASH_REG_RAM);

	if (adrs & D8)
		offset	+= 256;	// It is placed in the 2nd RAM image area.

	for (i = 0; i < PAGE_SIZE; i += 4, offset += 4, ptr++)
	{
		reg.reg	= SIO4_GSC_ENCODE(GSC_REG_TYPE_BAR2, 0, 0, 4, offset);
		ret		= sio4_ioctl(args->fd, SIO4_IOCTL_REG_READ, &reg);
		ptr[0]	= reg.value;

		if (ret)
			break;
	}

	return(ret);
}



//*****************************************************************************
// returned: 0 = PASS, <0 = error code.
static int _xfer_page_to_dev_ram(const args_t* args, flash_t* flash, int adrs)
{
	int			i;
	int			offset;
	u32*		ptr		= (void*) (flash->dev_buffer + adrs);
	gsc_reg_t	reg;
	int			ret;

	offset	= GSC_REG_OFFSET_DECODE(FLASH_REG_RAM);

	if (adrs & D8)
		offset	+= 256;	// It is placed in the 2nd RAM image area.

	for (i = 0; i < PAGE_SIZE; i += 4, offset += 4, ptr++)
	{
		reg.reg		= SIO4_GSC_ENCODE(GSC_REG_TYPE_BAR2, 0, 0, 4, offset);
		reg.value	= ptr[0];
		ret			= sio4_ioctl(args->fd, SIO4_IOCTL_REG_WRITE, &reg);

		if (ret)
			break;
	}

	return(ret);
}



//*****************************************************************************
// returned: 0 = PASS, <0 = error code.
int flash_command(const args_t* args, int cmd, int wait)
{
	int	ret;

	ret	= sio4_reg_write(args->fd, -1, 0, FLASH_REG_RCSR, cmd);

	if ((ret == 0) && (wait))
		ret	= _wait_for_done(args);

	return(ret);
}



//*****************************************************************************
// returned: 0 = PASS, <0 = error code.
int flash_erase(const args_t* args)
{
	int	ret;
	int	seconds;

	for (;;)	// A convenience loop.
	{
		gsc_label("Erase Flash");

		ret	= flash_command(args, FLASH_CMD_TX_ENABLE, 1);

		if (ret)
		{
			printf(" FAIL <---  (Write Enable failed: %d)\n", ret);
			break;
		}

		ret	= flash_command(args, FLASH_CMD_ERASE_BULK, 0);

		if (ret)
		{
			printf(" FAIL <---  (Write Enable failed: %d)\n", ret);
			break;
		}

		// Wait while ticking off seconds.
		ret	= _wait_for_done_seconds(args, &seconds);

		if (ret == 0)
		{
			printf("PASS  (~");
			gsc_label_long_comma(seconds);
			printf(" Seconds)\n");
		}

		break;
	}

	return(ret);
}



//*****************************************************************************
// returned: 0 = PASS, <0 = error code.
int flash_read(const args_t* args, flash_t* flash)
{
	int	ret;

	gsc_label("Read Flash Image");
	printf("\n");
	gsc_label_level_inc();

	// Clear Buffer =======================================
	gsc_label("Clear Buffer");
	memset(flash->dev_buffer, 0, flash->dev_bytes);
	printf("Done\n");

	// Read from Flash ====================================
	ret	= _read_flash(args, flash);

	// Save to file =======================================

	if ((ret == 0) && (args->save))
		ret	= _save_rpd_file(args, flash);

	gsc_label_level_dec();
	return(ret);
}



//*****************************************************************************
// returned: 0 = PASS, <0 = error code.
int flash_write(const args_t* args, flash_t* flash)
{
	int	ret;

	gsc_label("Write Flash Image");
	printf("\n");
	gsc_label_level_inc();

	for (;;)	// A convenience loop.
	{
		// Load File ======================================
		ret	= _read_rpd_file(args, flash);

		if (ret)
			break;

		// Erase Flash ====================================
		ret	= flash_erase(args);

		if (ret)
			break;

		// Write to Flash =================================
		ret	= _write_flash(args, flash);

		if (ret)
			break;

		// Clear Buffer ===================================
		gsc_label("Clear Buffer");
		memset(flash->dev_buffer, 0, flash->dev_bytes);
		printf("Done\n");

		// Read from Flash ================================
		ret	= _read_flash(args, flash);

		if (ret)
			break;

		// Save to file ===================================

		if (args->save)
			ret	= _save_rpd_file(args, flash);

		if (ret)
			break;

		// Compare ========================================
		gsc_label("Compareison");
		ret	= memcmp(flash->dev_buffer, flash->file_buffer, flash->dev_bytes);

		if (ret)
		{
			ret	= -EINVAL;
			printf("FAIL <---  (Validation failed.)\n");
		}
		else
		{
			printf("PASS\n");
		}

		break;
	}

	gsc_label_level_dec();
	return(ret);
}


