// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_2.x.x_GSC_DN/samples/sio4flash/main.c $
// $Rev: 54418 $
// $Date: 2024-05-15 14:14:30 -0500 (Wed, 15 May 2024) $

// SIO4: Sample Application: source file

#include "main.h"



//*****************************************************************************
static int _parse_args(int argc, char** argv, args_t* args)
{
	const gsc_arg_item_t	list[]	=
	{
		//	type				var							arg			values			desc
		{	GSC_ARG_S32_FLAG,	&args->action,				"-erase",	{ ACT_ERASE },	"Erase the flash device."						},
		{	GSC_ARG_S32_FLAG,	&args->action,				"-rx",		{ ACT_READ },	"Read the flash image into memory."				},
		{	GSC_ARG_S32_FLAG,	&args->save,				"-save",	{ 1 },			"Save the -rx and post -tx image to disk."		},
		{	GSC_ARG_STR_PTR,	(void*) &args->src_file,	"-src",		{ 0 },			"Use this as the -tx source file."				},
		{	GSC_ARG_S32_FLAG,	&args->action,				"-tx",		{ ACT_WRITE },	"Write the -src=str file to flash: xxx.rpd"		},
		{	GSC_ARG_S32_FLAG,	&args->verbose,				"-v",		{ 1 },			"Generate verbose output."						},
		{	GSC_ARG_DEV_INDEX,	&args->index,				"index",	{ 1 },			"The zero based index of the device to access."	}
	};

	const gsc_arg_set_t	set	=
	{
		/* name	*/	"sio4flash",
		/* desc	*/	"Update and save device firmware.",
		/* qty	*/	SIZEOF_ARRAY(list),
		/* list	*/	list
	};

	int	errs;
	int	len;

	memset(args, 0, sizeof(args_t));

	errs	= gsc_args_parse(argc, argv, &set);

	for (;;)	// A convenience loop.
	{
		if (args->action != ACT_WRITE)
			break;

		if (args->src_file == NULL)
		{
			errs++;
			gsc_label("MISSING ARGUMENT");
			printf("FAIL <---  (Using -tx requires a -src=XXX argument.)\n");
			break;
		}

		len	= strlen(args->src_file);

		if (len <= 4)
		{
			errs++;
			gsc_label("INVALID ARGUMENT");
			printf("FAIL <---  (The -src=str argument requires a .rpd extension.)\n");
			break;
		}

		if ( (args->src_file[len - 4] != '.') ||
			((args->src_file[len - 3] != 'r') && (args->src_file[len - 3] != 'R')) ||
			((args->src_file[len - 2] != 'p') && (args->src_file[len - 2] != 'P')) ||
			((args->src_file[len - 1] != 'd') && (args->src_file[len - 1] != 'D')))
		{
			errs++;
			gsc_label("INVALID ARGUMENT");
			printf("FAIL <---  (The -src=str argument requires a .rpd extension.)\n");
			break;
		}

		break;
	}

	return(errs);
}



//*****************************************************************************
static void _show_access_index(const args_t* args)
{
	gsc_label("Accessing Device");
	printf("%d\n", args->index);
}



//*****************************************************************************
static void _show_time_stamp(void)
{
	const char*	psz;
	struct tm*	stm;
	time_t		tt;

	time(&tt);
	stm	= localtime(&tt);
	psz	= (char*) asctime(stm);
	gsc_label("Time Stamp");
	printf("%s", psz);
}



/******************************************************************************
*
*	Function:	main
*
*	Purpose:
*
*		Control the overall flow of the application.
*
*	Arguments:
*
*		argc			The number of command line arguments.
*
*		argv			The list of command line arguments.
*
*	Returned:
*
*		EXIT_SUCCESS	No errors were encounterred.
*		EXIT_FAILURE	One or more problems were encounterred.
*
******************************************************************************/

int main(int argc, char** argv)
{
	args_t	args;
	int		errs;
	time_t	exec		= time(NULL);
	long	failures	= 0;
	long	hours;
	long	mins;
	time_t	now;
	long	passes		= 0;
	int		ret;
	long	secs;
	time_t	test;
	long	tests		= 0;

	for (;;)	// A convenience loop.
	{
		gsc_label_init(26);
		test	= time(NULL);
		errs	= _parse_args(argc, argv, &args);

		// Introduction

		os_id_host();
		_show_time_stamp();
		_show_access_index(&args);
		errs	+= sio4_init_util(1);
		errs	+= os_id_driver(sio4_open, sio4_read, sio4_close);
		errs	+= sio4_count_boards(1, &args.qty);

		if (args.qty <= 0)
			errs++;

		// Open access to device

		if (errs == 0)
			errs	= sio4_open_util(args.index, 0, -1, 1, &args.fd);

		if (errs == 0)
		{
			// Device identification

			errs	= sio4_id_device(args.fd, -1, 1);

			// Perform testing

			errs	+= perform_tests(&args);

			// Close device access

			errs	+= sio4_close_util(args.fd, -1, 1);
		}

		// End of processing.

		now	= time(NULL);
		tests++;

		if (errs)
		{
			failures++;
			printf(	"\nRESULTS: FAIL <---  (%d error%s)",
					errs,
					(errs == 1) ? "" : "s");
		}
		else
		{
			passes++;
			printf("\nRESULTS: PASS");
		}

		secs	= (long) (now - test);
		hours	= secs / 3600;
		secs	= secs % 3600;
		mins	= secs / 60;
		secs	= secs % 60;
		printf(" (duration %ld:%ld:%02ld)\n", hours, mins, secs);

		secs	= (long) (now - exec);
		hours	= secs / 3600;
		secs	= secs % 3600;
		mins	= secs / 60;
		secs	= secs % 60;
		printf(	"SUMMARY: tests %ld, pass %ld, fail %ld"
				" (duration %ld:%ld:%02ld)\n\n",
				tests,
				passes,
				failures,
				hours,
				mins,
				secs);
		break;
	}

	ret	= ((args.qty <= 0) || (errs) || (failures)) ? EXIT_FAILURE : EXIT_SUCCESS;
	return(ret);
}


