// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_2.x.x_GSC_DN/samples/sio4flash/file.c $
// $Rev: 54418 $
// $Date: 2024-05-15 14:14:30 -0500 (Wed, 15 May 2024) $

// SIO4: Sample Application: source file

#include "main.h"



//*****************************************************************************
FILE* file_close(FILE* file)
{
	int	i;

	if (file)
	{
		i	= fclose(file);

		if (i == 0)
			file	= NULL;
	}

	return(file);
}



/******************************************************************************
*
*	Function:	file_create
*
*	Purpose:
*
*		Create a file with the given name. The file is opened in binary write
*		mode.
*
*	Arguments:
*
*		name	The name of the file to create.
*
*	Returned:
*
*		NULL	The operation failed.
*		else	A valid file handle.
*
******************************************************************************/

FILE* file_create(const char* name)
{
	FILE*	file;

	for (;;)
	{
		if ((name == NULL) || (name[0] == 0))
		{
			file	= NULL;
			break;
		}

		file	= fopen(name, "w+b");

		if (file)
			break;

		break;
	}

	return(file);
}



/******************************************************************************
*
*	Function:	file_eof
*
*	Purpose:
*
*		Check to see if we're at the end of the file.
*
*	Arguments:
*
*		file	An open file handle.
*
*		eof		Set to 1 if we're at the end and 0 if not.
*
*	Returned:
*
*		0		All went well.
*		-1		There was a problem.
*
******************************************************************************/

int file_eof(FILE* file, int* eof)
{
	int	ret;

	if ((file == NULL) || (eof == NULL))
	{
		ret	= -EINVAL;
	}
	else
	{
		eof[0]	= feof(file);
		eof[0]	= eof[0] ? 1 : 0;
		ret		= 0;
	}

	return(ret);
}



//*****************************************************************************
FILE* file_open(const char* name)
{
	FILE*	file;

	if ((name == NULL) || (name[0] == 0))
	{
		file	= NULL;
	}
	else
	{
		file	= fopen(name, "r+b");
	}

	return(file);
}



/******************************************************************************
*
*	Function:	file_read
*
*	Purpose:
*
*		Read data from an open file.
*
*	Arguments:
*
*		file	An open file handle.
*
*		buf		Put the data here.
*
*		size	The number of bytes to read.
*
*	Returned:
*
*		>= 0	The number of bytes read.
*		-1		There was a problem.
*
******************************************************************************/

long file_read(FILE* file, char* buf, size_t size)
{
	int		i;
	long	l;

	for (;;)
	{
		if ((file == NULL) || (buf == NULL) || (size == 0))
		{
			l	= 0;
			break;
		}

		memset(buf, 0, size);
		l	= (long) fread(buf, 1, size, file);

		if (l == (long) size)
			break;

		i	= feof(file);

		if (i)
			break;

		l	= -1;
		break;
	}

	return(l);
}



/******************************************************************************
*
*	Function:	file_read_bin
*
*	Purpose:
*
*		Read a fixed number of bytes from the file.
*
*	Arguments:
*
*		file	An open file handle.
*
*		dst		Put the line of text here. The line may be terminated in DOS
*				or UNIX format. The returned line is always null terminated.
*				The EOL characters are removed.
*
*		bytes	The number of bytes to read.
*
*	Returned:
*
*		>= 0	The number of bytes read.
*		-1		There was a problem.
*
******************************************************************************/

long file_read_bin(FILE* file, char* dst, size_t bytes)
{
	int		eof;
	int		i;
	long	qty;

	memset(dst, 0, bytes);
	qty	= file_read(file, dst, bytes);

	if (qty == 0)
	{
		i	= file_eof(file, &eof);

		if (i == -1)
			qty	= -1;
		else if (eof)
			qty	= 0;
		else
			qty	= -1;

		dst[0]	= 0;
	}

	return(qty);
}



/******************************************************************************
*
*	Function:	file_read_line
*
*	Purpose:
*
*		Read a line of text from an open file.
*
*	Arguments:
*
*		file	An open file handle.
*
*		dst		Put the line of text here. The line may be terminated in DOS
*				or UNIX format. The returned line is always null terminated.
*				The EOL characters are removed.
*
*		size	The size of the buffer.
*
*	Returned:
*
*		>= 0	The number of bytes read.
*		-1		There was a problem.
*
******************************************************************************/

long file_read_line(FILE* file, char* dst, size_t size)
{
	char	buf[4096];
	int		eof;
	int		i;
	int		len;
	long	pos;
	char*	psz;
	long	qty;
	int		ret;

	for (;;)	// A convenience loop.
	{
		pos	= file_tell(file);

		if (pos < 0)
		{
			ret	= -1;
			break;
		}

		memset(dst, 0, size);
		qty	= file_read(file, buf, sizeof(buf));

		if (qty == 0)
		{
			i	= file_eof(file, &eof);

			if (i == -1)
				ret	= -1;
			else if (eof)
				ret	= 0;
			else
				ret	= -1;

			dst[0]	= 0;
			break;
		}

		buf[sizeof(buf) - 1]	= 0;
		psz	= strchr(buf, '\n');

		if (psz)
			psz[1]	= 0;

		psz	= strchr(buf, '\r');

		if (psz == NULL)
			;
		else if (psz[1] != '\n')
			psz[1]	= 0;

		len	= (int) strlen(buf);
		i	= file_seek(file, pos + len, SEEK_SET);

		if (i)
		{
			ret	= -1;
		}
		else
		{
			ret	= len;

			if (len >= (int) size)
			{
				strncpy(dst, buf, size);
				dst[size - 1]	= 0;
			}
			else
			{
				strncpy(dst, buf, len);
				dst[len]	= 0;
			}

			psz	= strchr(dst, '\n');

			if (psz)
				psz[0]	= 0;

			psz	= strchr(dst, '\r');

			if (psz)
				psz[0]	= 0;
		}

		break;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	file_seek
*
*	Purpose:
*
*		Report the current position in a file.
*
*	Arguments:
*
*		file	An open file handle.
*
*		pos		The absolute position to seek to.
*
*		wence	Seek to POS relative to this position.
*
*	Returned:
*
*		>= 0	The number of errors seen.
*
******************************************************************************/

int file_seek(FILE* file, long pos, int wence)
{
	int	errs	= 0;
	int	i;

	i	= fseek(file, pos, wence);

	if (i)
	{
		errs	= 1;
	}
	else
	{
		errs	= 0;
	}

	return(errs);
}



/******************************************************************************
*
*	Function:	file_size
*
*	Purpose:
*
*		Report the size of the file.
*
*	Arguments:
*
*		file	An open file handle.
*
*	Returned:
*
*		>= 0	The file size.
*		-1		There was a problem.
*
******************************************************************************/

long file_size(FILE* file)
{
	int		i;
	long	pos;
	long	size;

	for (;;)	// A convenience loop.
	{
		// Obtain the starting position so we can restore it later.
		pos	= ftell(file);

		if (pos == -1)
		{
			size	= -1;
			break;
		}

		// Advance to the end of the file.
		i	= fseek(file, 0, SEEK_END);

		if (i)
		{
			size	= -1;
			fseek(file, pos, SEEK_SET);
			break;
		}

		// Retrieve the current position, which is the file size.
		size	= ftell(file);

		if (size == -1)
		{
			break;
		}

		// Restore the starting position.
		i	= fseek(file, pos, SEEK_SET);

		if (i)
		{
			size	= -1;
			break;
		}

		break;
	}

	return(size);
}



/******************************************************************************
*
*	Function:	file_tell
*
*	Purpose:
*
*		Report the current position in a file.
*
*	Arguments:
*
*		file	An open file handle.
*
*	Returned:
*
*		>= 0	The current location.
*		-1		There was a problem.
*
******************************************************************************/

long file_tell(FILE* file)
{
	long	pos;

	pos	= ftell(file);
	return(pos);
}



/******************************************************************************
*
*	Function:	file_write_bin
*
*	Purpose:
*
*		Write a binary data image to the file.
*
*	Arguments:
*
*		file	An open file handle.
*
*		src		The line of data to write.
*
*		bytes	The number of bytes to write.
*
*	Returned:
*
*		>= 0	The number of bytes read.
*		-1		There was a problem.
*
******************************************************************************/

long file_write_bin(FILE* file, const char* src, size_t bytes)
{
	long	l;

	for (;;)
	{
		if ((file == NULL) || (src == NULL))
		{
			l	= 0;
			break;
		}

		if (bytes == 0)
		{
			l	= 0;
			break;
		}

		l	= fwrite(src, 1, bytes, file);

		if (l == bytes)
			break;

		l	= -1;
		break;
	}

	return(l);
}



/******************************************************************************
*
*	Function:	file_write_str
*
*	Purpose:
*
*		Write a NULL terminated line of data to an open file.
*
*	Arguments:
*
*		file	An open file handle.
*
*		src		The line of data to write.
*
*	Returned:
*
*		>= 0	The number of bytes read.
*		-1		There was a problem.
*
******************************************************************************/

long file_write_str(FILE* file, const char* src)
{
	long	l;
	int		len;

	for (;;)
	{
		if ((file == NULL) || (src == NULL))
		{
			l	= 0;
			break;
		}

		len	= strlen(src);

		if (len == 0)
		{
			l	= 0;
			break;
		}

		l	= fwrite(src, 1, len, file);

		if (l == len)
			break;

		l	= -1;
		break;
	}

	return(l);
}


