// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_2.x.x_GSC_DN/txrate/osc.c $
// $Rev: 32669 $
// $Date: 2015-07-08 17:16:12 -0500 (Wed, 08 Jul 2015) $

#include "main.h"



// variables ******************************************************************

static	u32			_poscr	= 0;

		dev_data_t	osc_dev;



//*****************************************************************************
int osc_board_startup(dev_data_t* dev, int board)
{
	gsc_label("OSC Startup");

	memset(dev, 0, sizeof(dev[0]));
	dev->cache.osc_chip		= SIO4_OSC_CHIP_CY22393;
	dev->cache.osc_pd_max	= 15;
	dev->vaddr.gsc_pocsr_32	= SIO4_GSC_POCSR;
	dev->channel[0].dev		= dev;
	dev->channel[1].dev		= dev;
	dev->channel[2].dev		= dev;
	dev->channel[3].dev		= dev;
	dev->channel[0].index	= 0;
	dev->channel[1].index	= 1;
	dev->channel[2].index	= 2;
	dev->channel[3].index	= 3;
	osc_cy22393_sio4_create(&osc_dev);

	printf("Done\n");
	return(0);
}



//*****************************************************************************
//	Compute the configured frequency for the specified channel.
static float _calc_measure(dev_data_t* dev, int chan, int verbose, long rate)
{
	float				err;
	float				got;
	int					p;
	int					p0;
	osc_cy22393_pll_t*	pll	= NULL;
	int					q;

	if (dev->osc.cy22393.osc.clock[chan].in_use == 0)
	{
		got	= 0;

		if (verbose)
			printf("DISABLED");
	}
	else if (dev->osc.cy22393.osc.clock[chan].freq_src == CY22393_FREQ_SRC_REF)
	{
		got	= (float) dev->osc.cy22393.osc.freq_ref
			/ (float) dev->osc.cy22393.osc.clock[chan].post_div
			/ (float) (0x1L << dev->osc.cy22393.chan[chan].pd);

		if (verbose == 0)
		{
		}
		else if (rate)
		{
			err	= (got - rate) / rate * 100;

			printf("ref ");
			gsc_label_long_comma(dev->osc.cy22393.osc.freq_ref);
			printf(", PD %d", (int) dev->osc.cy22393.osc.clock[chan].post_div);
			printf(", FWPD %d", (int) (0x1L << dev->osc.cy22393.chan[chan].pd));
			printf(" -> %.3f Hz", got);
			printf(", err %+.3f%%", err);
		}
		else
		{
			printf("%.3f Hz", got);
			printf(", PD %d", (int) dev->osc.cy22393.osc.clock[chan].post_div);
			printf(", FWPD %d", (int) (0x1L << dev->osc.cy22393.chan[chan].pd));
		}
	}
	else
	{
		switch (dev->osc.cy22393.osc.clock[chan].freq_src)
		{
			case CY22393_FREQ_SRC_REF:

				printf("INTERNAL ERROR: %s, %d", __FILE__, __LINE__);
				exit(EXIT_FAILURE);
				break;

			case CY22393_FREQ_SRC_PLL1:

				pll	= &dev->osc.cy22393.osc.pll[0];
				break;

			case CY22393_FREQ_SRC_PLL2:

				pll	= &dev->osc.cy22393.osc.pll[1];
				break;

			case CY22393_FREQ_SRC_PLL3:

				pll	= &dev->osc.cy22393.osc.pll[2];
				break;
		}

		p	= pll->p;
		p0	= pll->p0;
		q	= pll->q;

		got	= (float) dev->osc.cy22393.osc.freq_ref
			* (float) ((2 * (p + 3)) + p0)
			/ (float) (q + 2)
			/ (float) dev->osc.cy22393.osc.clock[chan].post_div
			/ (float) (0x1L << dev->osc.cy22393.chan[chan].pd);

		if (verbose)
		{
			err	= (got - rate) / rate * 100;

			printf("ref ");
			gsc_label_long_comma(dev->osc.cy22393.osc.freq_ref);
			printf(", P %d", p);
			printf(", P0 %d", p0);
			printf(", Q %d", q);
			printf(", PD %d", (int) dev->osc.cy22393.osc.clock[chan].post_div);
			printf(", FWPD %d", (int) (0x1L << dev->osc.cy22393.chan[chan].pd));
			printf(" -> %.3f Hz", got);
			printf(", err %+.3f%%", err);
		}
	}

	return(got);
}



//*****************************************************************************
void osc_dev_dump(void)
{
	printf("\n\n\n");
	printf("osc_dev.cache.osc_pd_max                   %ld\n", (long) osc_dev.cache.osc_pd_max);
	printf("osc_dev.cache.osc_chip                     %ld\n", (long) osc_dev.cache.osc_chip);

	printf("osc_dev.osc.cy22393.osc.freq_ref           %ld\n", (long) osc_dev.osc.cy22393.osc.freq_ref);

	printf("osc_dev.osc.cy22393.osc.clock[0].id        %ld\n", (long) osc_dev.osc.cy22393.osc.clock[0].id);
	printf("osc_dev.osc.cy22393.osc.clock[0].in_use    %ld\n", (long) osc_dev.osc.cy22393.osc.clock[0].in_use);
	printf("osc_dev.osc.cy22393.osc.clock[0].freq_want %ld\n", (long) osc_dev.osc.cy22393.osc.clock[0].freq_want);
	printf("osc_dev.osc.cy22393.osc.clock[0].freq_got  %ld\n", (long) osc_dev.osc.cy22393.osc.clock[0].freq_got);
	printf("osc_dev.osc.cy22393.osc.clock[0].freq_src  %ld\n", (long) osc_dev.osc.cy22393.osc.clock[0].freq_src);
	printf("osc_dev.osc.cy22393.osc.clock[0].post_div  %ld\n", (long) osc_dev.osc.cy22393.osc.clock[0].post_div);

	printf("osc_dev.osc.cy22393.osc.clock[1].id        %ld\n", (long) osc_dev.osc.cy22393.osc.clock[1].id);
	printf("osc_dev.osc.cy22393.osc.clock[1].in_use    %ld\n", (long) osc_dev.osc.cy22393.osc.clock[1].in_use);
	printf("osc_dev.osc.cy22393.osc.clock[1].freq_want %ld\n", (long) osc_dev.osc.cy22393.osc.clock[1].freq_want);
	printf("osc_dev.osc.cy22393.osc.clock[1].freq_got  %ld\n", (long) osc_dev.osc.cy22393.osc.clock[1].freq_got);
	printf("osc_dev.osc.cy22393.osc.clock[1].freq_src  %ld\n", (long) osc_dev.osc.cy22393.osc.clock[1].freq_src);
	printf("osc_dev.osc.cy22393.osc.clock[1].post_div  %ld\n", (long) osc_dev.osc.cy22393.osc.clock[1].post_div);

	printf("osc_dev.osc.cy22393.osc.clock[2].id        %ld\n", (long) osc_dev.osc.cy22393.osc.clock[2].id);
	printf("osc_dev.osc.cy22393.osc.clock[2].in_use    %ld\n", (long) osc_dev.osc.cy22393.osc.clock[2].in_use);
	printf("osc_dev.osc.cy22393.osc.clock[2].freq_want %ld\n", (long) osc_dev.osc.cy22393.osc.clock[2].freq_want);
	printf("osc_dev.osc.cy22393.osc.clock[2].freq_got  %ld\n", (long) osc_dev.osc.cy22393.osc.clock[2].freq_got);
	printf("osc_dev.osc.cy22393.osc.clock[2].freq_src  %ld\n", (long) osc_dev.osc.cy22393.osc.clock[2].freq_src);
	printf("osc_dev.osc.cy22393.osc.clock[2].post_div  %ld\n", (long) osc_dev.osc.cy22393.osc.clock[2].post_div);

	printf("osc_dev.osc.cy22393.osc.clock[3].id        %ld\n", (long) osc_dev.osc.cy22393.osc.clock[3].id);
	printf("osc_dev.osc.cy22393.osc.clock[3].in_use    %ld\n", (long) osc_dev.osc.cy22393.osc.clock[3].in_use);
	printf("osc_dev.osc.cy22393.osc.clock[3].freq_want %ld\n", (long) osc_dev.osc.cy22393.osc.clock[3].freq_want);
	printf("osc_dev.osc.cy22393.osc.clock[3].freq_got  %ld\n", (long) osc_dev.osc.cy22393.osc.clock[3].freq_got);
	printf("osc_dev.osc.cy22393.osc.clock[3].freq_src  %ld\n", (long) osc_dev.osc.cy22393.osc.clock[3].freq_src);
	printf("osc_dev.osc.cy22393.osc.clock[3].post_div  %ld\n", (long) osc_dev.osc.cy22393.osc.clock[3].post_div);

	printf("osc_dev.osc.cy22393.osc.pll[0].enable      %ld\n", (long) osc_dev.osc.cy22393.osc.pll[0].enable);
	printf("osc_dev.osc.cy22393.osc.pll[0].freq_got    %ld\n", (long) osc_dev.osc.cy22393.osc.pll[0].freq_got);
	printf("osc_dev.osc.cy22393.osc.pll[0].freq_src    %ld\n", (long) osc_dev.osc.cy22393.osc.pll[0].freq_src);
	printf("osc_dev.osc.cy22393.osc.pll[0].p           %ld\n", (long) osc_dev.osc.cy22393.osc.pll[0].p);
	printf("osc_dev.osc.cy22393.osc.pll[0].p0          %ld\n", (long) osc_dev.osc.cy22393.osc.pll[0].p0);
	printf("osc_dev.osc.cy22393.osc.pll[0].lf          %ld\n", (long) osc_dev.osc.cy22393.osc.pll[0].lf);
	printf("osc_dev.osc.cy22393.osc.pll[0].q           %ld\n", (long) osc_dev.osc.cy22393.osc.pll[0].q);

	printf("osc_dev.osc.cy22393.osc.pll[1].enable      %ld\n", (long) osc_dev.osc.cy22393.osc.pll[1].enable);
	printf("osc_dev.osc.cy22393.osc.pll[1].freq_got    %ld\n", (long) osc_dev.osc.cy22393.osc.pll[1].freq_got);
	printf("osc_dev.osc.cy22393.osc.pll[1].freq_src    %ld\n", (long) osc_dev.osc.cy22393.osc.pll[1].freq_src);
	printf("osc_dev.osc.cy22393.osc.pll[1].p           %ld\n", (long) osc_dev.osc.cy22393.osc.pll[1].p);
	printf("osc_dev.osc.cy22393.osc.pll[1].p0          %ld\n", (long) osc_dev.osc.cy22393.osc.pll[1].p0);
	printf("osc_dev.osc.cy22393.osc.pll[1].lf          %ld\n", (long) osc_dev.osc.cy22393.osc.pll[1].lf);
	printf("osc_dev.osc.cy22393.osc.pll[1].q           %ld\n", (long) osc_dev.osc.cy22393.osc.pll[1].q);

	printf("osc_dev.osc.cy22393.osc.pll[2].enable      %ld\n", (long) osc_dev.osc.cy22393.osc.pll[2].enable);
	printf("osc_dev.osc.cy22393.osc.pll[2].freq_got    %ld\n", (long) osc_dev.osc.cy22393.osc.pll[2].freq_got);
	printf("osc_dev.osc.cy22393.osc.pll[2].freq_src    %ld\n", (long) osc_dev.osc.cy22393.osc.pll[2].freq_src);
	printf("osc_dev.osc.cy22393.osc.pll[2].p           %ld\n", (long) osc_dev.osc.cy22393.osc.pll[2].p);
	printf("osc_dev.osc.cy22393.osc.pll[2].p0          %ld\n", (long) osc_dev.osc.cy22393.osc.pll[2].p0);
	printf("osc_dev.osc.cy22393.osc.pll[2].lf          %ld\n", (long) osc_dev.osc.cy22393.osc.pll[2].lf);
	printf("osc_dev.osc.cy22393.osc.pll[2].q           %ld\n", (long) osc_dev.osc.cy22393.osc.pll[2].q);

	printf("osc_dev.osc.cy22393.pd_max                 %ld\n", (long) osc_dev.osc.cy22393.pd_max);

	printf("osc_dev.osc.cy22393.chan[0].freq_want      %ld\n", (long) osc_dev.osc.cy22393.chan[0].freq_want);
	printf("osc_dev.osc.cy22393.chan[0].freq_got       %ld\n", (long) osc_dev.osc.cy22393.chan[0].freq_got);
	printf("osc_dev.osc.cy22393.chan[0].pd             %ld\n", (long) osc_dev.osc.cy22393.chan[0].pd);

	printf("osc_dev.osc.cy22393.chan[1].freq_want      %ld\n", (long) osc_dev.osc.cy22393.chan[1].freq_want);
	printf("osc_dev.osc.cy22393.chan[1].freq_got       %ld\n", (long) osc_dev.osc.cy22393.chan[1].freq_got);
	printf("osc_dev.osc.cy22393.chan[1].pd             %ld\n", (long) osc_dev.osc.cy22393.chan[1].pd);

	printf("osc_dev.osc.cy22393.chan[2].freq_want      %ld\n", (long) osc_dev.osc.cy22393.chan[2].freq_want);
	printf("osc_dev.osc.cy22393.chan[2].freq_got       %ld\n", (long) osc_dev.osc.cy22393.chan[2].freq_got);
	printf("osc_dev.osc.cy22393.chan[2].pd             %ld\n", (long) osc_dev.osc.cy22393.chan[2].pd);

	printf("osc_dev.osc.cy22393.chan[3].freq_want      %ld\n", (long) osc_dev.osc.cy22393.chan[3].freq_want);
	printf("osc_dev.osc.cy22393.chan[3].freq_got       %ld\n", (long) osc_dev.osc.cy22393.chan[3].freq_got);
	printf("osc_dev.osc.cy22393.chan[3].pd             %ld\n", (long) osc_dev.osc.cy22393.chan[3].pd);

	printf("\n");
}



//*****************************************************************************
// Program the specified channel to produce the specified rate.
int osc_calc_program(dev_data_t* dev, int chan, int verbose, long rate, FILE* file, float* got)
{
	int	errs	= 0;
	int	ret;
	s32	set;

	if (file)
		fprintf(file, "%10ld", (long) rate);

	if (rate)
	{
		set		= rate;
		ret		= osc_cy22393_sio4_program_ioctl(&dev->channel[chan], &set);
		errs	+= ret ? 1 : 0;
		printf("%s", ret ? "FAIL <---  (" : "");

		if (ret == 0)
			got[0]	= _calc_measure(dev, chan, verbose, rate);

		if (ret)
			printf(")");
	}
	else
	{
		printf("ERROR");
		errs++;
	}

	if (file)
		fprintf(file, "%14.3f", (float) got[0]);

	return(errs);
}



//*****************************************************************************
static int _wait(int fd, int errs, int hold)
{

	gsc_label("Waiting");

	if (errs)
	{
		printf("Aborting due to errors.\n");
	}
	else
	{
		printf("%d second%s ... ", hold, (hold == 1) ? "" : "s");
		fflush(stdout);
		os_sleep_ms(1000L * hold);
		printf("Done\n");
	}

	return(0);
}



//*****************************************************************************
static int _mp_setup(int fd)
{
	static const s32	list[]	=
	{
		SIO4_XCVR_PROTOCOL_RS422_RS485,
		SIO4_XCVR_PROTOCOL_RS423,
		SIO4_XCVR_PROTOCOL_RS232,
		SIO4_XCVR_PROTOCOL_RS530,
		SIO4_XCVR_PROTOCOL_RS530A,
		SIO4_XCVR_PROTOCOL_V35,
		SIO4_XCVR_PROTOCOL_V35A,
		SIO4_XCVR_PROTOCOL_RS422_RS423_1,
		SIO4_XCVR_PROTOCOL_RS422_RS423_2,
		-1
	};

	int	errs	= 0;
	int	i;
	s32	got;

	for (i = 0;; i++)
	{
		if (list[i] == -1)
		{
			errs	= 1;
			printf(	"FAIL <---  (MP: unable to make a selection.)\n");
			break;
		}

		errs	= sio4_xcvr_protocol(fd, -1, 0, list[i], &got);

		if (errs)
			break;

		if (list[i] == got)
			break;
	}

	return(errs);
}



//*****************************************************************************
static int _drive_output(int fd, int hold, int z16c30, int mp_prog, long rate)
{
	char	buf[64];
	int		errs	= 0;
	float	got		= 0;

	if (rate > 0)
	{
		errs	+= osc_board_startup(&osc_dev, 0);

		strcpy(buf, "Calculate ");
		gsc_label_long_comma_buf(rate, buf + strlen(buf));
		strcat(buf, " Hz");
		gsc_label(buf);

		errs	+= osc_calc_program(&osc_dev, 0, 1, rate, NULL, &got);
		printf("\n");

		errs	+= sio4_osc_program(fd, -1, 1, rate, NULL);
	}

	if (z16c30)
	{
		errs	+= sio4_z16_cbl_txc_cfg		(fd, -1, 1, SIO4_Z16_CBL_TXC_CFG_OUT_OSC, NULL);
		errs	+= sio4_z16_cbl_txauxc_cfg	(fd, -1, 1, SIO4_Z16_CBL_TXAUXC_CFG_OUT_OSC, NULL);
	}
	else
	{
		errs	+= sio4_sync_txc_idle_cfg	(fd, -1, 1, SIO4_SYNC_TXC_IDLE_CFG_ACTIVE, NULL);
		errs	+= sio4_sync_txc_src		(fd, -1, 1, SIO4_SYNC_TXC_SRC_OSC_HALF_RISE, NULL);
		errs	+= sio4_sync_txauxc_cfg		(fd, -1, 1, SIO4_SYNC_TXAUXC_CFG_OSC_HALF, NULL);
	}

	errs	+= _mp_setup		(fd);
	errs	+= sio4_cbl_mode	(fd, -1, 1, SIO4_CBL_MODE_DTE, NULL);
	errs	+= sio4_xcvr_enable	(fd, -1, 1, SIO4_XCVR_ENABLE_YES, NULL);
	errs	+= sio4_osc_measure	(fd, -1, 1, 1, NULL);
	errs	+= _wait			(fd, errs, hold);

	return(errs);
}



//*****************************************************************************
static int _rate_default(int fd, int hold, int z16c30, int mp_prog)
{
	int	errs	= 0;

	gsc_label("Current Osc Rate");
	printf("\n");
	gsc_label_level_inc();

	errs	+= _drive_output(fd, hold, z16c30, mp_prog, -1);

	gsc_label_level_dec();
	return(errs);
}



//*****************************************************************************
static int _rate_single(int fd, long rate, int hold, int z16c30, int mp_prog)
{
	int	errs	= 0;

	gsc_label("Osc Programming");
	gsc_label_long_comma(rate);
	printf(" Hz");

	if (z16c30 == 0)
		printf("  (Cable rate is half this.)");

	printf("\n");
	gsc_label_level_inc();

	errs	+= _drive_output(fd, hold, z16c30, mp_prog, rate);

	gsc_label_level_dec();
	return(errs);
}



//*****************************************************************************
static int _rate_scan(int fd, long begin, long end, int save, int program, int measure)
{
	char	buf[64];
	int		errs	= 0;
	FILE*	file	= NULL;
	s32		got;
	long	l;
	float	v		= 0;

	gsc_label("Rate Scan");
	gsc_label_long_comma(begin);
	printf(" to ");
	gsc_label_long_comma(end);
	printf("\n");
	gsc_label_level_inc();
	errs	+= osc_board_startup(&osc_dev, 0);

	if (save)
	{
		file	= fopen(FILE_NAME, "w+b");

		if (file == NULL)
		{
			printf("FAIL <---  (unable to create %s)\n", FILE_NAME);
			errs	= 1;
		}
	}

	if (file)
	{
		fprintf(file, "# OSC Scan Rate\n");
		fprintf(file, "# Request   Computed      Program   Measure\n");
		fprintf(file, "# ========  ============  ========  ========\n");
	}

	for (l = begin; l <= end; l++)
	{
		strcpy(buf, "Calculate ");
		gsc_label_long_comma_buf(l, buf + strlen(buf));
		strcat(buf, " Hz");
		gsc_label(buf);

		errs	+= osc_calc_program(&osc_dev, 0, 1, l, file, &v);

		if (file && program)
		{
			errs	+= sio4_osc_program(fd, -1, 0, l, &got);
			fprintf(file, "%10ld", (long) got);
			printf("(Prog %ld)", (long) got);

			if (measure)
			{
				errs	+= sio4_osc_measure(fd, -1, 0, 1, &got);
				fprintf(file, "%10ld", (long) got);
				printf("(Meas %ld)", (long) got);
			}
		}

		if (file)
			fprintf(file, "\r\n");

		printf("\n");
	}

	if (file)
		fclose(file);

	gsc_label_level_dec();
	return(errs);
}



//*****************************************************************************
int osc_test(
	int		fd,
	long	begin,
	long	end,
	int		hold,
	int		z16c30,
	int		mp_prog,
	int		osc_prog,
	int		save,
	int		program,	// only if (save != 0)
	int		measure)	// only if (program != 0)
{
	int	errs	= 0;

	if (begin > 20000000L)
	{
		errs++;
		printf("ERROR: Oscillator: The begin value is too large. The limit is 20M.\n");
	}
	else if (end > 20000000L)
	{
		errs++;
		printf("ERROR: Oscillator: The end value is too large. The limit is 20M.\n");
	}
	else if ((begin > end) && (end >= 0))
	{
		errs++;
		printf("ERROR: Oscillator: The begin value is larger than the end value.\n");
	}
	else if (begin == 0)
	{
		errs++;
		printf("ERROR: Oscillator: The begin value is too small. The limit is 1.\n");
	}
	else if (end == 0)
	{
		errs++;
		printf("ERROR: Oscillator: The end value is too small. The limit is 1.\n");
	}
	else if ((begin < 0) && (end < 0))
	{
		errs	+= _rate_default(fd, hold, z16c30, mp_prog);
	}
	else if (begin < 0)
	{
		errs++;
		printf("ERROR: Oscillator: The begin value is not given.\n");
	}
	else if (end < 0)
	{
		if (osc_prog)
		{
			errs	+= _rate_single(fd, begin, hold, z16c30, mp_prog);
		}
		else
		{
			errs++;
			printf("ERROR: Oscillator: The oscillator isn't programmable.\n");
		}
	}
	else if (begin > end)
	{
		errs++;
		printf("ERROR: Oscillator: The begin value is greater than the end value.\n");
	}
	else
	{
		errs	+= _rate_scan(fd, begin, end, save, program, measure);
	}

	return(errs);
}



//*****************************************************************************
static u32 _reg_read_mem_u32(VADDR_T va)
{
	u32	value;

	if (va == SIO4_GSC_POCSR)
	{
		if		(_poscr & 0x02)	value	= (u32) _calc_measure(&osc_dev, 0, 0, 0);
		else if (_poscr & 0x04)	value	= (u32) _calc_measure(&osc_dev, 1, 0, 0);
		else if (_poscr & 0x08)	value	= (u32) _calc_measure(&osc_dev, 2, 0, 0);
		else if (_poscr & 0x10)	value	= (u32) _calc_measure(&osc_dev, 3, 0, 0);
		else					value	= 0;

		value	= ((value / 10) << 8)
				| SIO4_GSC_POCSR_PRG_DONE
				| SIO4_GSC_POCSR_MSR_DONE;
	}
	else
	{
		value	= 0;
	}

	return(value);
}



//*****************************************************************************
u32 os_reg_mem_rx_u32(dev_data_t* dev, VADDR_T va)
{
	u32	ret;

	ret	= _reg_read_mem_u32(va);
	return(ret);
}



//*****************************************************************************
void os_reg_mem_tx_u32(dev_data_t* dev, VADDR_T va, u32 val)
{
	if (va == SIO4_GSC_POCSR)
		_poscr	= val;
}



//*****************************************************************************
void gsc_irq_access_lock(dev_data_t* dev)
{
}



//*****************************************************************************
void gsc_irq_access_unlock(dev_data_t* dev)
{
}



//*****************************************************************************
int os_time_tick_rate(void)
{
	return(1000);
}



//*****************************************************************************
void os_time_tick_sleep(int ticks)
{
}



//*****************************************************************************
void os_time_us_delay(long us)
{
}


