// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_2.x.x_GSC_DN/txrate/main.c $
// $Rev: 33952 $
// $Date: 2015-11-01 15:19:44 -0600 (Sun, 01 Nov 2015) $

#include "main.h"



// variables ******************************************************************

static	int			_async			= 0;
static	long		_begin			= 1024;
static	int			_best			= 0;
static	int			_def_index		= 0;
static	long		_end			= 1034;
static	int			_hold			= 30;
static	int			_measure		= 0;
static	int			_program		= 0;
static	long		_osc			= 0;
static	int			_save			= 0;
static	int			_sync			= 0;



//*****************************************************************************
static int _parse_args(int argc, char** argv)
{
	char	buf[32];
	char	c;
	int		errs	= 0;
	int		i;
	int		j;
	int		k;

	printf("txrate: Tx Bit Rate Test Application (Version %s)\n", VERSION);
	printf("USAGE: txrate <-async> <-b#> <-best> <-e#> <-h#> <-m> <-osc> <-p> <-s> <-sync> index\n");
	printf("  -async  Configure the board for asynchronous operation (Async).\n");
	printf("  -b#     Begin a sequential rate scan here. (Default is %ld.)\n", (long) _begin);
	printf("  -best   Find the best oscillator frequency (Zilog -sync specific, slower).\n");
	printf("  -e#     End a sequential rate scan here. (Default is %ld.)\n", (long) _end);
	printf("  -h#     Hold the -b# output active for this long (seconds).\n");
	printf("  -m      Measure the oscillator for each scan value. (-osc & -p only)\n");
	printf("  -osc    Report oscillator configuration information.\n");
	printf("  -p      Program the oscillator for each scan value. (-osc only)\n");
	printf("  -s      Save scan data to file (txrate.txt).\n");
	printf("  -sync   Configure the board for synchronous operation (HDLC/-SYNC).\n");
	printf("  index   The zero based index of the channel to access (default is 0).\n");
	printf("  Note: Use -b and -e to compute the operation for the given range.\n");
	printf("  Note: Omit -e to make the -b value appear at the cable interface.\n");
	printf("  Note: Use -osc without -b and -e to output the oscillator at the cable interface.\n");

	gsc_label_level_inc();

	for (i = 0; i < argc; i++)
	{
		sprintf(buf, "Argument %d", i);
		gsc_label(buf);
		printf("%s\n", argv[i]);
	}

	for (i = 1; i < argc; i++)
	{
		if (strcmp(argv[i], "-best") == 0)
		{
			_best	= 1;
			continue;
		}

		if (strcmp(argv[i], "-m") == 0)
		{
			_measure	= 1;
			continue;
		}

		if (strcmp(argv[i], "-p") == 0)
		{
			_program	= 1;
			continue;
		}

		if (strcmp(argv[i], "-s") == 0)
		{
			_save	= 1;
			continue;
		}

		if (strcmp(argv[i], "-async") == 0)
		{
			_async	= 1;
			_sync	= 0;
			_osc	= 0;
			continue;
		}

		if (strcmp(argv[i], "-sync") == 0)
		{
			_async	= 0;
			_sync	= 1;
			_osc	= 0;
			continue;
		}

		if (strcmp(argv[i], "-osc") == 0)
		{
			_async	= 0;
			_sync	= 0;
			_osc	= 1;
			continue;
		}

		if ((argv[i][0] == '-') &&
			(argv[i][1] == 'h') &&
			(argv[i][2]))
		{
			j	= sscanf(&argv[i][2], "%d%c", &k, &c);

			if ((j == 1) && (k >=  1) && (k <= 3600))
			{
				_hold	= k;
				continue;
			}

			errs++;
			printf("ERROR: invalid argument: %s\n", argv[i]);
			break;
		}

		if ((argv[i][0] == '-') &&
			(argv[i][1] == 'b') &&
			(argv[i][2]))
		{
			j	= sscanf(&argv[i][2], "%d%c", &k, &c);

			if ((j == 1) && (k >=  1) && (k <= 20000000L))
			{
				_begin	= k;
				continue;
			}

			errs++;
			printf("ERROR: invalid argument: %s\n", argv[i]);
			break;
		}

		if ((argv[i][0] == '-') &&
			(argv[i][1] == 'e') &&
			(argv[i][2]))
		{
			j	= sscanf(&argv[i][2], "%d%c", &k, &c);

			if ((j == 1) && (k >=  1) && (k <= 20000000L))
			{
				_end	= k;
				continue;
			}

			errs++;
			printf("ERROR: invalid argument: %s\n", argv[i]);
			break;
		}

		j	= sscanf(argv[i], "%d%c", &k, &c);

		if ((j == 1) && (k >= 0))
		{
			_def_index	= k;
			continue;
		}

		errs	= 1;
		printf("ERROR: invalid channel index: %s\n", argv[i]);
		break;
	}

	if (errs)
	{
		// We've already reported a problem.
	}
	else if ((_begin >= 0) && (_end < 0))
	{
		// Output the beginning value to the cable interface
	}
	else if ((_begin >= 0) && (_end >= _begin))
	{
		// Report the configuration from beginning to end.
	}
	else if ((_begin < 0) && (_end >= 0))
	{
		errs++;
		printf("ERROR: argument '-b#' not given.\n");
	}
	else if ((_begin > _end) && (_end >= 0))
	{
		errs++;
		printf("ERROR: argument '-b#' greater than argument '-e#'.\n");
	}
	else if (_osc)
	{
	}
	else if (_begin < 0)
	{
		errs++;
		printf("ERROR: argument '-b#' not given.\n");
	}

	return(errs);
}



//*****************************************************************************
static int _perform_tests(int fd, s32 z16c30, s32 mp_prog, s32 osc_prog)
{
	int			errs	= 0;
	const char*	psz;
	struct tm*	stm;
	time_t		tt;

	time(&tt);
	stm	= localtime(&tt);
	psz	= asctime(stm);
	gsc_label("Performing Test");
	printf("%s", psz);

	errs	+= sio4_id_board(fd, -1);

	if (_osc)
	{
		errs	+= osc_test(fd, _begin, _end, _hold, z16c30, mp_prog, osc_prog, _save, _program, _measure);
	}
	else if (z16c30)
	{
		if (_async)
			errs	+= async_test(fd, _begin, _end, _hold);
		else if (_sync)
			errs	+= hdlc_test(fd, _begin, _end, _hold, _save, _best);
		else
			printf("Must specify -async or -sync on the command line.\n");
	}
	else
	{
		if (_async)
			printf("-async command line option not available with -SYNC boards.\n");
		else
			errs	+= sync_test(fd, _begin, _end, _hold, _save);
	}

	return(errs);
}



/******************************************************************************
*
*	Function:	main
*
*	Purpose:
*
*		Control the overall operation of the application.
*
*	Arguments:
*
*		argc	The number of command line arguments.
*
*		argv	The list of command line arguments.
*
*	Returned:
*
*		EXIT_SUCCESS	No errors were encounterred.
*		EXIT_FAILURE	One or more errors were encounterred.
*
******************************************************************************/

int main(int argc, char** argv)
{
	int		errs		= 0;
	time_t	exec		= time(NULL);
	long	failures	= 0;
	int		fd			= 0;
	long	hours;
	long	mins;
	s32		mp_prog		= 0;
	time_t	now;
	long	passes		= 0;
	s32		osc_prog	= 0;
	int		qty			= 0;
	int		ret			= EXIT_SUCCESS;
	long	secs;
	time_t	test;
	long	tests		= 0;
	s32		z16c30		= 0;

	for (;;)
	{
		gsc_label_init(29);
		test	= time(NULL);
		errs	= _parse_args(argc, argv);

		if (errs)
			break;

		os_id_host();
		errs	+= os_id_driver(SIO4_BASE_NAME);

		if (errs)
			break;

		qty		= os_count_boards(SIO4_BASE_NAME);

		if (qty <= 0)
			break;

		gsc_label("Testing Device Index");
		printf("%d\n", _def_index);

		// Obtain some initial information about the channel.
		fd		= sio4_dev_open(_def_index, -1, 0);

		if (fd < 0)
		{
			errs++;
		}
		else
		{
			errs	+= sio4_query(fd, SIO4_QUERY_MODEL_Z16C30, &z16c30);
			errs	+= sio4_query(fd, SIO4_QUERY_MP_PROGRAM, &mp_prog);
			errs	+= sio4_query(fd, SIO4_QUERY_OSC_PROGRAM, &osc_prog);
			errs	+= sio4_dev_close(fd, -1, 0);
		}

		if (errs)
		{
			printf(	"ERROR: channel %d access error.\n", _def_index);
			break;
		}

		// Open access to the channel.

		if ((z16c30) && (_sync))
		{
			errs	+= sio4_hdlc_lib_init();

			if (errs)
				break;

			fd	= hdlc_util_open(_def_index, -1, 1);
		}
		else
		{
			fd	= sio4_dev_open(_def_index, -1, 1);
		}

		if (fd < 0)
		{
			errs	= 1;
			printf("ERROR: channel %d access error\n", _def_index);
		}

		if (errs == 0)
		{
			errs	= _perform_tests(fd, z16c30, mp_prog, osc_prog);
		}

		if ((z16c30) && (_sync))
		{
			errs	+= hdlc_util_close(fd, -1, 1);
		}
		else
		{
			errs	+= sio4_dev_close(fd, -1, 0);
		}

		now	= time(NULL);
		tests++;

		if (errs)
		{
			failures++;
			printf(	"\nRESULTS: FAIL <---  (%d error%s)",
					errs,
					(errs == 1) ? "" : "s");
		}
		else
		{
			passes++;
			printf("\nRESULTS: PASS");
		}

		secs	= (long) (now - test);
		hours	= secs / 3600;
		secs	= secs % 3600;
		mins	= secs / 60;
		secs	= secs % 60;
		printf(" (duration %ld:%ld:%02ld)\n", hours, mins, secs);

		secs	= (long) (now - exec);
		hours	= secs / 3600;
		secs	= secs % 3600;
		mins	= secs / 60;
		secs	= secs % 60;
		printf(	"SUMMARY: tests %ld, pass %ld, fail %ld"
				" (duration %ld:%ld:%02ld)\n\n",
				tests,
				passes,
				failures,
				hours,
				mins,
				secs);
		break;
	}

	if ((qty <= 0) || (errs) || (failures))
		ret	= EXIT_FAILURE;

	return(ret);
}


