// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_2.x.x_GSC_DN/txrate/hdlc.c $
// $Rev: 25443 $
// $Date: 2014-02-26 14:25:34 -0600 (Wed, 26 Feb 2014) $

#include "main.h"



// #defines *******************************************************************

#define ABS(v)		(((v) < 0) ? -(v) : (v))



//*****************************************************************************
static int _show_disabled(float* rate)
{
	printf("Disabled");
	rate[0]	= 0;
	return(0);
}



//*****************************************************************************
static int _show_drive_high(float* rate)
{
	printf("Drive High (0)");
	rate[0]	= 0;
	return(0);
}



//*****************************************************************************
static int _show_drive_low(float* rate)
{
	printf("Drive Low (0)");
	rate[0]	= 0;
	return(0);
}



//*****************************************************************************
static int _show_osc(const sio4_hdlc_t* hdlc, float* rate)
{
	printf("Osc ");
	gsc_label_long_comma(hdlc->osc.prog);
	rate[0]	= (float) hdlc->osc.prog;
	return(0);
}



//*****************************************************************************
static int _show_rxc_pin(const sio4_hdlc_t* hdlc, float* rate)
{
	int	errs	= 0;

	switch (hdlc->usc.rx.rxc)
	{
		default:

			errs++;
			printf("FAIL <--- UNKNOWN");
			rate[0]	= 0;
			break;

		case SIO4_HDLC_USC_RX_RXC_IN_OSC:
		case SIO4_HDLC_USC_RX_RXC_IN_OSC_INV:

			errs	+= _show_osc(hdlc, rate);
			break;

		case SIO4_HDLC_USC_RX_RXC_IN_0:

			errs	+= _show_drive_low(rate);
			break;

		case SIO4_HDLC_USC_RX_RXC_IN_1:

			errs	+= _show_drive_high(rate);
			break;

		case SIO4_HDLC_USC_RX_RXC_IN_CBL_RXC:

			printf("Cable RxClk (0)");
			rate[0]	= 0;
			break;

		case SIO4_HDLC_USC_RX_RXC_IN_CBL_RXAUX:

			printf("Cable RxAuxC (0)");
			rate[0]	= 0;
			break;

		case SIO4_HDLC_USC_RX_RXC_OUT_CLK:

			printf("USC Rx Out Clock (0)");
			rate[0]	= 0;
			break;

		case SIO4_HDLC_USC_RX_RXC_OUT_BYTE_CLK:

			printf("USC Rx Byte Clock (0)");
			rate[0]	= 0;
			break;

		case SIO4_HDLC_USC_RX_RXC_OUT_SYNC:

			printf("USC Rx Out SYNC (0)");
			rate[0]	= 0;
			break;

		case SIO4_HDLC_USC_RX_RXC_OUT_BRG0:

			errs++;
			printf("ERROR <--- BRG0");
			rate[0]	= 0;
			break;

		case SIO4_HDLC_USC_RX_RXC_OUT_BRG1:

			errs++;
			printf("ERROR <--- BRG1");
			rate[0]	= 0;
			break;

		case SIO4_HDLC_USC_RX_RXC_OUT_CTR0:

			errs++;
			printf("ERROR <--- CTR0");
			rate[0]	= 0;
			break;

		case SIO4_HDLC_USC_RX_RXC_OUT_DPLL_RX:

			errs++;
			printf("ERROR <--- DPLL");
			rate[0]	= 0;
			break;
	}

	printf(", USC RxC");
	return(errs);
}



//*****************************************************************************
static int _show_txc_pin(const sio4_hdlc_t* hdlc, float* rate)
{
	int	errs	= 0;

	switch (hdlc->usc.tx.txc)
	{
		default:

			errs++;
			printf("FAIL <--- UNKNOWN");
			rate[0]	= 0;
			break;

		case SIO4_HDLC_USC_TX_TXC_IN_OSC:
		case SIO4_HDLC_USC_TX_TXC_IN_OSC_INV:

			errs	+= _show_osc(hdlc, rate);
			break;

		case SIO4_HDLC_USC_TX_TXC_IN_0:

			errs	+= _show_drive_low(rate);
			break;

		case SIO4_HDLC_USC_TX_TXC_IN_1:

			errs	+= _show_drive_high(rate);
			break;

		case SIO4_HDLC_USC_TX_TXC_IN_CBL_RXC:

			printf("Cable RxClk (0)");
			rate[0]	= 0;
			break;

		case SIO4_HDLC_USC_TX_TXC_IN_CBL_RXAUX:

			printf("Cable RxAuxC (0)");
			rate[0]	= 0;
			break;

		case SIO4_HDLC_USC_TX_TXC_OUT_CLK:

			printf("USC Tx Out Clock (0)");
			rate[0]	= 0;
			break;

		case SIO4_HDLC_USC_TX_TXC_OUT_BYTE_CLK:

			printf("USC Tx Byte Clock (0)");
			rate[0]	= 0;
			break;

		case SIO4_HDLC_USC_TX_TXC_OUT_COMP:

			printf("USC Tx Complete (0)");
			rate[0]	= 0;
			break;

		case SIO4_HDLC_USC_TX_TXC_OUT_BRG0:

			errs++;
			printf("ERROR <--- BRG0");
			rate[0]	= 0;
			break;

		case SIO4_HDLC_USC_TX_TXC_OUT_BRG1:

			errs++;
			printf("ERROR <--- BRG1");
			rate[0]	= 0;
			break;

		case SIO4_HDLC_USC_TX_TXC_OUT_CTR1:

			errs++;
			printf("ERROR <--- CTR1");
			rate[0]	= 0;
			break;

		case SIO4_HDLC_USC_TX_TXC_OUT_DPLL_TX:

			errs++;
			printf("ERROR <--- DPLL");
			rate[0]	= 0;
			break;
	}

	printf(", USC TxC");
	return(errs);
}



//*****************************************************************************
static int _show_ctr0_rate(const sio4_hdlc_t* hdlc, float* rate)
{
	int	errs	= 0;

	switch (hdlc->usc.ctr0.rate)
	{
		default:

			errs++;
			printf("FAIL <--- UNKNOWN");
			rate[0]	= 0;
			break;

		case SIO4_HDLC_USC_CTR0_RATE_32X:

			printf("(/32)");
			rate[0]	/= 32;
			break;

		case SIO4_HDLC_USC_CTR0_RATE_16X:

			printf("(/16)");
			rate[0]	/= 16;
			break;

		case SIO4_HDLC_USC_CTR0_RATE_8X:

			printf("(/8)");
			rate[0]	/= 8;
			break;

		case SIO4_HDLC_USC_CTR0_RATE_4X:

			printf("(/4)");
			rate[0]	/= 4;
			break;
	}

	return(errs);
}



//*****************************************************************************
static int _show_dpll_rate(const sio4_hdlc_t* hdlc, float* rate, int four_is_ok)
{
	int	errs	= 0;

	switch (hdlc->usc.dpll.rate)
	{
		default:

			errs++;
			printf("FAIL <--- UNKNOWN");
			rate[0]	= 0;
			break;


		case SIO4_HDLC_USC_DPLL_RATE_32X:

			printf("(/32)");
			rate[0]	/= 32;
			break;

		case SIO4_HDLC_USC_DPLL_RATE_16X:

			printf("(/16)");
			rate[0]	/= 16;
			break;

		case SIO4_HDLC_USC_DPLL_RATE_8X:

			printf("(/8)");
			rate[0]	/= 8;
			break;

		case SIO4_HDLC_USC_DPLL_RATE_CTR1_4X:

			printf("(/4)");
			rate[0]	/= 4;

			if (four_is_ok == 0)
			{
				errs++;
				printf(" FAIL <--- INVALID");
			}

			break;
	}

	return(errs);
}



//*****************************************************************************
static int _show_ctr0(const sio4_hdlc_t* hdlc, float* rate)
{
	int	errs	= 0;

	switch (hdlc->usc.ctr0.clk_src)
	{
		default:

			errs++;
			printf("FAIL <--- UNKNOWN");
			rate[0]	= 0;
			break;

		case SIO4_HDLC_USC_CTR0_CLK_SRC_DISABLE:

			errs	+= _show_disabled(rate);
			break;

		case SIO4_HDLC_USC_CTR0_CLK_SRC_RXC_PIN:

			errs	+= _show_rxc_pin(hdlc, rate);
			break;

		case SIO4_HDLC_USC_CTR0_CLK_SRC_TXC_PIN:

			errs	+= _show_txc_pin(hdlc, rate);
			break;
	}

	printf(", USC CTR0 ");
	errs	+= _show_ctr0_rate(hdlc, rate);
	return(errs);
}



//*****************************************************************************
static int _show_ctr1(const sio4_hdlc_t* hdlc, float* rate)
{
	int	errs	= 0;

	switch (hdlc->usc.ctr1.clk_src)
	{
		default:

			errs++;
			printf("FAIL <--- UNKNOWN");
			rate[0]	= 0;
			break;

		case SIO4_HDLC_USC_CTR1_CLK_SRC_DISABLE:

			errs	+= _show_disabled(rate);
			break;

		case SIO4_HDLC_USC_CTR1_CLK_SRC_RXC_PIN:

			errs	+= _show_rxc_pin(hdlc, rate);
			break;

		case SIO4_HDLC_USC_CTR1_CLK_SRC_TXC_PIN:

			errs	+= _show_txc_pin(hdlc, rate);
			break;
	}

	printf(", USC CTR1 ");

	if (hdlc->usc.ctr1.rate_src == SIO4_HDLC_USC_CTR1_RATE_SRC_CTR0)
		errs	+= _show_ctr0_rate(hdlc, rate);
	else
		errs	+= _show_dpll_rate(hdlc, rate, 1);

	return(errs);
}



//*****************************************************************************
static int _show_brg0(const sio4_hdlc_t* hdlc, float* rate)
{
	int	errs	= 0;

	switch (hdlc->usc.brg0.clk_src)
	{
		default:

			errs++;
			printf("FAIL <--- UNKNOWN");
			rate[0]	= 0;
			break;

		case SIO4_HDLC_USC_BRG0_CLK_SRC_CTR0:

			errs	+= _show_ctr0(hdlc, rate);
			break;

		case SIO4_HDLC_USC_BRG0_CLK_SRC_CTR1:

			errs	+= _show_ctr1(hdlc, rate);
			break;

		case SIO4_HDLC_USC_BRG0_CLK_SRC_RXC_PIN:

			errs	+= _show_rxc_pin(hdlc, rate);
			break;

		case SIO4_HDLC_USC_BRG0_CLK_SRC_TXC_PIN:

			errs	+= _show_txc_pin(hdlc, rate);
			break;
	}

	printf(", BRG0 (%ld)", (long) hdlc->usc.brg0.divider);
	rate[0]	/= hdlc->usc.brg0.divider + 1;
	return(errs);
}



//*****************************************************************************
static int _show_brg1(const sio4_hdlc_t* hdlc, float* rate)
{
	int	errs	= 0;

	switch (hdlc->usc.brg1.clk_src)
	{
		default:

			errs++;
			printf("FAIL <--- UNKNOWN");
			rate[0]	= 0;
			break;

		case SIO4_HDLC_USC_BRG1_CLK_SRC_CTR0:

			errs	+= _show_ctr0(hdlc, rate);
			break;

		case SIO4_HDLC_USC_BRG1_CLK_SRC_CTR1:

			errs	+= _show_ctr1(hdlc, rate);
			break;

		case SIO4_HDLC_USC_BRG1_CLK_SRC_RXC_PIN:

			errs	+= _show_rxc_pin(hdlc, rate);
			break;

		case SIO4_HDLC_USC_BRG1_CLK_SRC_TXC_PIN:

			errs	+= _show_txc_pin(hdlc, rate);
			break;
	}

	printf(", BRG1 (%ld)", (long) hdlc->usc.brg1.divider);
	rate[0]	/= hdlc->usc.brg1.divider + 1;
	return(errs);
}



//*****************************************************************************
static int _show_dpll(const sio4_hdlc_t* hdlc, float* rate)
{
	int	errs	= 0;

	switch (hdlc->usc.dpll.clk_src)
	{
		default:

			errs++;
			printf("FAIL <--- UNKNOWN");
			rate[0]	= 0;
			break;

		case SIO4_HDLC_USC_DPLL_CLK_SRC_BRG0:

			errs	+= _show_brg0(hdlc, rate);
			break;

		case SIO4_HDLC_USC_DPLL_CLK_SRC_BRG1:

			errs	+= _show_brg1(hdlc, rate);
			break;

		case SIO4_HDLC_USC_DPLL_CLK_SRC_RXC_PIN:

			errs	+= _show_rxc_pin(hdlc, rate);
			break;

		case SIO4_HDLC_USC_DPLL_CLK_SRC_TXC_PIN:

			errs	+= _show_txc_pin(hdlc, rate);
			break;
	}

	printf(", USC DPLL ");
	errs	+= _show_dpll_rate(hdlc, rate, 0);
	return(errs);
}



//*****************************************************************************
static int _show_usc_txclk_src(const sio4_hdlc_t* hdlc, float* rate)
{
	int	errs	= 0;

	switch (hdlc->usc.tx.clk_src)
	{
		default:

			errs++;
			printf("FAIL <--- UNKNOWN");
			rate[0]	= 0;
			break;

		case SIO4_HDLC_USC_TX_CLK_SRC_DISABLE:

			errs	+= _show_disabled(rate);
			break;

		case SIO4_HDLC_USC_TX_CLK_SRC_RXC_PIN:

			errs	+= _show_rxc_pin(hdlc, rate);
			break;

		case SIO4_HDLC_USC_TX_CLK_SRC_TXC_PIN:

			errs	+= _show_txc_pin(hdlc, rate);
			break;

		case SIO4_HDLC_USC_TX_CLK_SRC_DPLL:

			errs	+= _show_dpll(hdlc, rate);
			break;

		case SIO4_HDLC_USC_TX_CLK_SRC_BRG0:

			errs	+= _show_brg0(hdlc, rate);
			break;

		case SIO4_HDLC_USC_TX_CLK_SRC_BRG1:

			errs	+= _show_brg1(hdlc, rate);
			break;

		case SIO4_HDLC_USC_TX_CLK_SRC_CTR0:

			errs	+= _show_ctr0(hdlc, rate);
			break;

		case SIO4_HDLC_USC_TX_CLK_SRC_CTR1:

			errs	+= _show_ctr1(hdlc, rate);
			break;
	}

	printf(", USC TxClk");
	return(errs);
}



//*****************************************************************************
static int _show_tx_rate(long rate, const sio4_hdlc_t* hdlc, FILE* file)
{
	float	err;
	int		errs	= 0;
	float	results	= 0;

	errs	+= _show_usc_txclk_src(hdlc, &results);
	err		= (results - rate) / rate * 100;
	printf(" -> %.3f b/s, err %.3f%%\n", results, err);

	if (file)
		fprintf(file, "%10ld%14.3f\n", (long) rate, (float) results);

	return(errs);
}



//*****************************************************************************
static int _rate_compute(
	int					fd,
	long				rate,
	sio4_hdlc_init_t*	init,
	sio4_hdlc_t*		hdlc,
	int					best)
{
	float		delta;
	float		delta_best	= 0;
	const char*	err			= NULL;
	int			errs		= 0;
	s32			factor;
	s32			factor_best	= 1;
	int			first		= 0;
	float		got;
	s32			prog;

	errs	+= sio4_query(fd, SIO4_QUERY_OSC_PROGRAM, &prog);

	memset(init, 0, sizeof(init[0]));
	init->tx_bit_rate		= rate;
	init->rx_bit_rate		= (rate > 1000) ? 1000 : 1200;
	init->rx_uses_cbl_rxc	= SIO4_HDLC_RX_USES_CBL_RXC_NO;

	if (prog == 0)
	{
		init->osc_prog	= 20000000L;
	}
	else if (best)
	{
		factor	= 20000000L / rate;

		for (; factor >= 1; factor--)
		{
			init->osc_prog	= factor * rate;

			errs	+= osc_calc_program(&osc_dev, 0, 0, init->osc_prog, NULL, &got);

			if (errs)
				break;

			delta	= (got / factor) - rate;
			delta	= ABS(delta);

			if ((first == 0) || (delta <= delta_best))
			{
				first		= 1;
				delta_best	= delta;
				factor_best	= factor;

				if (delta < 0.000001)
					break;
			}
		}

		init->osc_prog	= factor_best * rate;
	}
	else
	{
		factor	= (s32) (0.5 + ((float) 20000000L / rate));

		if (factor > (0xFFFF * 32))
			factor	= 0xFFFF * 32;

		init->osc_prog	= factor * rate;

		if (init->osc_prog > 20000000L)
			init->osc_prog	-= rate;

		errs	+= osc_calc_program(&osc_dev, 0, 0, init->osc_prog, NULL, &got);

		if (errs == 0)
			init->osc_prog	= (s32) (got + 0.5);
	}

	errs	+= sio4_hdlc_init(fd, init, hdlc, &err);

	if ((errs) && (err))
		printf("FAIL <---  (%d error%s, %s)\n", errs, (errs == 1) ? "" : "s", err);
	else if (errs)
		printf("FAIL <---  (%d error%s)\n", errs, (errs == 1) ? "" : "s");
	else if (err)
		printf("FAIL <---  (%s)\n", err);

	return(errs);
}



//*****************************************************************************
static int _apply(int fd, const sio4_hdlc_t* hdlc)
{
	const char*	err		= NULL;
	int			errs	= 0;

	gsc_label("Applying");
	errs	+= sio4_hdlc_set(fd, hdlc, &err);

	if ((errs) && (err))
		printf("FAIL <---  (%d error%s, %s)\n", errs, (errs == 1) ? "" : "s", err);
	else if (errs)
		printf("FAIL <---  (%d error%s)\n", errs, (errs == 1) ? "" : "s");
	else if (err)
		printf("FAIL <---  (%s)\n", err);
	else
		printf("PASS\n");

	return(errs);
}



//*****************************************************************************
static int _wait(int errs, int hold)
{
	gsc_label("Waiting");

	if (errs)
	{
		printf("Aborting due to errors.\n");
	}
	else
	{
		printf("%d second%s ... ", hold, (hold == 1) ? "" : "s");
		fflush(stdout);
		os_sleep_ms(100L * hold);
		printf("Done\n");
	}

	return(0);
}



//*****************************************************************************
static int _mp_setup(int fd)
{
	static const s32	list[]	=
	{
		SIO4_XCVR_PROTOCOL_RS422_RS485,
		SIO4_XCVR_PROTOCOL_RS423,
		SIO4_XCVR_PROTOCOL_RS232,
		SIO4_XCVR_PROTOCOL_RS530,
		SIO4_XCVR_PROTOCOL_RS530A,
		SIO4_XCVR_PROTOCOL_V35,
		SIO4_XCVR_PROTOCOL_V35A,
		SIO4_XCVR_PROTOCOL_RS422_RS423_1,
		SIO4_XCVR_PROTOCOL_RS422_RS423_2,
		-1
	};

	int	errs	= 0;
	int	i;
	s32	got;

	for (i = 0;; i++)
	{
		if (list[i] == -1)
		{
			errs	= 1;
			printf(	"FAIL <---  (MP: unable to make a selection.)\n");
			break;
		}

		errs	= sio4_xcvr_protocol(fd, -1, 0, list[i], &got);

		if (errs)
			break;

		if (list[i] == got)
			break;
	}

	return(errs);
}



//*****************************************************************************
static int _rate_single(int fd, long rate, int hold, int best)
{
	char				buf[64];
	int					errs	= 0;
	sio4_hdlc_t			hdlc;
	sio4_hdlc_init_t	init;

	gsc_label("Rate Calculation");
	printf("\n");
	gsc_label_level_inc();
	errs	+= osc_board_startup(&osc_dev, 0);

	strcpy(buf, "Calculate ");
	gsc_label_long_comma_buf(rate, buf + strlen(buf));
	strcat(buf, " b/s");
	gsc_label(buf);

	errs				+= _rate_compute(fd, rate, &init, &hdlc, best);
	hdlc.cable.txaux	= SIO4_HDLC_CABLE_TXAUX_OUT_OSC;
	errs				+= _show_tx_rate(rate, &hdlc, NULL);

	errs	+= _apply(fd, &hdlc);
	errs	+= _mp_setup(fd);
	errs	+= _wait(errs, hold);
	gsc_label_level_dec();
	return(errs);
}



//*****************************************************************************
static int _rate_scan(int fd, long begin, long end, int save, int best)
{
	char				buf[64];
	int					errs	= 0;
	FILE*				file	= NULL;
	sio4_hdlc_t			hdlc;
	sio4_hdlc_init_t	init;
	long				l;

	gsc_label("Rate Scan");
	gsc_label_long_comma(begin);
	printf(" to ");
	gsc_label_long_comma(end);
	printf("\n");
	gsc_label_level_inc();
	errs	+= osc_board_startup(&osc_dev, 0);

	if (save)
	{
		file	= fopen(FILE_NAME, "w+b");

		if (file == NULL)
		{
			printf("FAIL <---  (unable to create %s)\n", FILE_NAME);
			errs	= 1;
		}
	}

	if (file)
	{
		fprintf(file, "# Zilog Sync Scan Rate\n");
		fprintf(file, "# Request   Computed\n");
		fprintf(file, "# ========  ============\n");
	}

	for (l = begin; l <= end; l++)
	{
		strcpy(buf, "Calculate ");
		gsc_label_long_comma_buf(l, buf + strlen(buf));
		strcat(buf, " b/s");
		gsc_label(buf);

		errs	+= _rate_compute(fd, l, &init, &hdlc, best);
		errs	+= _show_tx_rate(l, &hdlc, file);
	}

	if (file)
		fclose(file);

	gsc_label_level_dec();
	return(errs);
}



//*****************************************************************************
int hdlc_test(int fd, long begin, long end, int hold, int save, int best)
{
	int	errs	= 0;

	hdlc_util_version(-1, 1, NULL, NULL);

	if (begin < 0)
	{
		errs++;
		printf("ERROR: HDLC: The begin value isn't specified\n");
	}
	else if (begin > 10000000L)
	{
		errs++;
		printf("ERROR: HDLC: The begin value is too large. The limit is 10M.\n");
	}
	else if (end > 10000000L)
	{
		errs++;
		printf("ERROR: HDLC: The end value is too large. The limit is 10M.\n");
	}
	else if (end < 0)
	{
		errs	+= _rate_single(fd, begin, hold, best);
	}
	else
	{
		errs	+= _rate_scan(fd, begin, end, save, best);
	}

	return(errs);
}


