// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_2.x.x_GSC_DN/sync/syncc2c/io.c $
// $Rev: 33933 $
// $Date: 2015-10-07 15:15:32 -0500 (Wed, 07 Oct 2015) $

#include "main.h"



// data types *****************************************************************

typedef struct
{
	int	index;
	s32	mp;
	s32	program;
	s32	current;
	s32	_232;
	s32	_422;
	s32	_423;
} mp_t;



// variables ******************************************************************

static	int	_pipeline_limit	= 512;		// modified during init.
static	int	_pipeline_qty	= 0;
static	u8	_rx_buf[256];				// Must be 256!!!
static	u8	_tx_buf[sizeof(_rx_buf)];



//*****************************************************************************
static void _connection_type(int tx_index, int rx_index)
{
	int	rx_b	= rx_index / 4;
	int	rx_c	= rx_index % 4;
	int	tx_b	= tx_index / 4;
	int	tx_c	= tx_index % 4;

	gsc_label("Connection Type");

	if (tx_index == rx_index)
	{
		printf("External Loopback (no cable)\n");
	}
	else if (tx_b == rx_b)
	{
		printf(	"Channel-to-Channel Loopback Cable"
				" (B# %d, C#s %d+%d)\n",
				tx_b,
				tx_c,
				rx_c);
	}
	else
	{
		printf(	"Channel-to-Channel Passthrough Cable"
				" (B%d C%d, B%d C%d)\n",
				tx_b,
				tx_c,
				rx_b,
				rx_c);
	}
}



//*****************************************************************************
static int _verify_model(int fd, const char* desc)
{
	int	errs;
	s32	sync;

	gsc_label(desc);

	errs	= sio4_query(fd, SIO4_QUERY_MODEL_SYNC, &sync);

	if (errs)
	{
	}
	else if (sync)
	{
		printf("PASS  (SYNC)\n");
	}
	else
	{
		errs++;
		printf("FAIL <---  (Not a SYNC model SIO4)\n");
	}

	return(errs);
}



//****************************************************************************
static int _mp_test(int fd, s32 set, s32* supported)
{
	int	errs	= 0;
	s32	got;
	s32	init;
	int	ret;

	// Get the starting protocol.
	init	= -1;
	ret		= sio4_ioctl(fd, SIO4_IOCTL_XCVR_PROTOCOL, &init);
	errs	+= (ret < 0) ? 1 : 0;

	// Select the protocol in question.
	ret		= sio4_ioctl(fd, SIO4_IOCTL_XCVR_PROTOCOL, &set);
	errs	+= (ret < 0) ? 1 : 0;

	// Now see what we got.
	got		= -1;
	ret		= sio4_ioctl(fd, SIO4_IOCTL_XCVR_PROTOCOL, &got);
	errs	+= (ret < 0) ? 1 : 0;

	// Restore the initial setting.
	ret		= sio4_ioctl(fd, SIO4_IOCTL_XCVR_PROTOCOL, &init);
	errs	+= (ret < 0) ? 1 : 0;

	// Now record the results.
	supported[0]	= (got == set) ? 1 : 0;
	return(errs);
}



//*****************************************************************************
static int _xcvr_support(int fd, int index, const char* desc, mp_t* mp)
{
	int	errs	= 0;
	int	ret;

	mp->index	= index;
	gsc_label(desc);
	errs	+= sio4_query(fd, SIO4_QUERY_MP, &mp->mp);
	errs	+= sio4_query(fd, SIO4_QUERY_MP_PROGRAM, &mp->program);
	errs	+= _mp_test(fd, SIO4_XCVR_PROTOCOL_RS232, &mp->_232);
	errs	+= _mp_test(fd, SIO4_XCVR_PROTOCOL_RS422_RS485, &mp->_422);
	errs	+= _mp_test(fd, SIO4_XCVR_PROTOCOL_RS423, &mp->_423);

	mp->current	= -1;
	ret			= sio4_ioctl(fd, SIO4_IOCTL_XCVR_PROTOCOL, &mp->current);
	errs		+= (ret < 0) ? 1 : 0;

	if (errs == 0)
		printf("Done\n");

	return(errs);
}



//*****************************************************************************
static int _verify_xcvr(const mp_t* tx_xcvr, const mp_t* rx_xcvr, s32* xcvr)
{
	int	errs	= 0;

	for (;;)	// A convenience loop.
	{
		gsc_label("Transceiver Selection");

		if ((tx_xcvr->mp == 0) || (rx_xcvr->mp == 0))
		{
			if (tx_xcvr->index == rx_xcvr->index)
			{
				// The channel has fixed transceivers.
				xcvr[0]	= -1;
				printf("PASS  (using installed transceivers)\n");
				break;
			}

			if ((tx_xcvr->index / 4) == (rx_xcvr->index / 4))
			{
				// The board has fixed transceivers.
				// All channels should be the same.
				xcvr[0]	= -1;
				printf("PASS  (using installed transceivers)\n");
				break;
			}

			// There is no way of knowing if they are both the same.
			errs++;
			xcvr[0]	= -1;
			printf("FAIL <---  (insufficient MP support)\n");
			break;
		}

		if (tx_xcvr->current != rx_xcvr->current)
		{
		}
		else if (tx_xcvr->current == SIO4_XCVR_PROTOCOL_RS422_RS485)
		{
			xcvr[0]	= SIO4_XCVR_PROTOCOL_RS422_RS485;
			printf("PASS  (RS-423)\n");
			break;
		}
		else if (tx_xcvr->current == SIO4_XCVR_PROTOCOL_RS423)
		{
			xcvr[0]	= SIO4_XCVR_PROTOCOL_RS423;
			printf("PASS  (RS-423)\n");
			break;
		}
		else if (tx_xcvr->current == SIO4_XCVR_PROTOCOL_RS232)
		{
			xcvr[0]	= SIO4_XCVR_PROTOCOL_RS232;
			printf("PASS  (RS-232)\n");
			break;
		}

		if ((tx_xcvr->_422) && (rx_xcvr->_422))
		{
			// Fastest of the supported protocols.
			xcvr[0]	= SIO4_XCVR_PROTOCOL_RS422_RS485;
			printf("PASS  (RS-422/485)\n");
			break;
		}

		if ((tx_xcvr->_423) && (rx_xcvr->_423))
		{
			xcvr[0]	= SIO4_XCVR_PROTOCOL_RS423;
			printf("PASS  (RS-423)\n");
			break;
		}

		if ((tx_xcvr->_232) && (rx_xcvr->_232))
		{
			// Slowest of the supported protocols.
			xcvr[0]	= SIO4_XCVR_PROTOCOL_RS232;
			printf("PASS  (RS-232)\n");
			break;
		}

		errs++;
		xcvr[0]	= SIO4_XCVR_PROTOCOL_UNKNOWN;
		printf("FAIL <---  (common protocol not supported)\n");
		break;
	}

	return(errs);
}



//*****************************************************************************
static int _setup_common(
	int			fd,
	const char*	title,
	int			verbose,
	int			dce,
	int			lb,
	s32			mp,
	s32			clk)
{
	char		buf[64];
	int			errs	= 0;
	s32			got;
	int			status;
	sio4_sync_t	sync;

	for (;;)	// A convenience loop.
	{
		sprintf(buf, "Common Setup (%s)", title);
		gsc_label(buf);

		if (mp == SIO4_XCVR_PROTOCOL_UNKNOWN)
		{
			errs	= 1;
			printf("FAIL <---  (MP support)\n");
			break;
		}

		errs	+= sio4_osc_program(fd, -1, 0, clk, NULL);

		if (errs)
			break;

		memset(&sync, 0, sizeof(sync));
		sync.dce_enable		= dce ? 1 : 0;
		sync.dte_enable		= dce ? 0 : 1;
		sync.lb.enable		= lb;
		sync.lb.internal	= 0;
		status				= sio4_sync_set(fd, &sync);

		if (status)
		{
			errs	= 1;
			printf("FAIL <--- (sio4_sync_set())\n");
			break;
		}

		if (mp != -1)
		{
			status	= sio4_xcvr_protocol(fd, -1, 0, mp, &got);

			if (status)
			{
				errs	= 1;
				printf("FAIL <--- (sio4_mp_config())\n");
				break;
			}

			if (mp != got)
			{
				errs	= 1;
				printf(	"FAIL <---  (MP: wanted %d, got %d)\n",
						(int) mp,
						(int) got);
				break;
			}
		}

		printf("PASS");

		if (verbose)
		{
			printf("  (");
			gsc_label_long_comma(clk);
			printf(" Hz clock -> ");
			gsc_label_long_comma(clk / 2);
			printf(" bits/sec)");
		}

		printf("\n");
		break;
	}

	return(errs);
}



//*****************************************************************************
static int _setup_tx(int fd, int verbose, long clk)
{
	int				errs	= 0;
	int				status;
	sio4_sync_tx_t	tx;

	for (;;)	// A convenience loop.
	{
		gsc_label("Tx Setup");
		errs	+= sio4_tx_fifo_reset(fd, -1, 0, 1, NULL);

		memset(&tx, 0, sizeof(tx));
		tx.enable			= SIO4_SYNC_ENABLE_YES;
		tx.empty_cfg		= SIO4_TX_FIFO_EMPTY_CFG_IGNORE;
		tx.bit_order		= SIO4_SYNC_BIT_ORDER_MSB;
		tx.word_size		= 8;
		tx.gap_size			= 0;
		tx.clock.cfg		= SIO4_SYNC_TXC_CFG_INT;
		tx.clock.pol		= SIO4_SYNC_CLOCK_POL_RISE;
		tx.clock.src		= SIO4_SYNC_TXC_SRC_OSC_HALF_RISE;
		tx.clock.idle		= SIO4_SYNC_TXC_IDLE_NO;
		tx.clock.idle_cfg	= SIO4_SYNC_TXC_IDLE_CFG_IDLE_0;
		tx.env.cfg			= SIO4_SYNC_TXE_CFG_ACTIVE_HI;
		tx.env.pol			= SIO4_SYNC_ENV_POL_ACTIVE_HI;
		tx.data.cfg			= SIO4_SYNC_TXD_CFG_ACTIVE_HI;
		tx.data.idle		= SIO4_SYNC_TXD_IDLE_CFG_OUT_0;
		tx.data.legacy		= SIO4_SYNC_LEG_TXD_CFG_LOW;
		tx.aux_clock		= SIO4_SYNC_TXAUXC_CFG_TRI;
		tx.spare			= SIO4_SYNC_TXSP_CFG_DISABLE;
		status				= sio4_sync_tx_set(fd, &tx);

		if (status)
		{
			errs	= 1;
			printf("FAIL <--- (sio4_sync_tx_set())\n");
			break;
		}

		errs	+= sio4_tx_io_mode(fd, -1, 0, GSC_IO_MODE_PIO, NULL);
		errs	+= sio4_tx_io_timeout(fd, -1, 0, 0, NULL);	// zero timeout with PIO

		// Make sure everything is flushed before we procede.
		os_sleep_ms(10);
		errs	+= sio4_tx_fifo_reset(fd, -1, 0, 1, NULL);

		if (errs == 0)
			printf(	"PASS\n");

		break;
	}

	return(errs);
}



//*****************************************************************************
static int _setup_rx(int fd)
{
	int				errs	= 0;
	sio4_sync_rx_t	rx;
	int				status;

	for (;;)	// A convenience loop.
	{
		gsc_label("Rx Setup");
		errs	+= sio4_rx_fifo_reset(fd, -1, 0, 1, NULL);

		memset(&rx, 0, sizeof(rx));
		rx.enable		= 1;
		rx.bit_order	= SIO4_SYNC_BIT_ORDER_MSB;
		rx.clock.cfg	= SIO4_SYNC_RXC_CFG_FALL_EDGE;
		rx.clock.pol	= SIO4_SYNC_RXD_CFG_ACTIVE_HI;
		rx.env.cfg		= SIO4_SYNC_RXE_CFG_ACTIVE_HI;
		rx.env.pol		= SIO4_SYNC_ENV_POL_ACTIVE_HI;
		rx.data.cfg		= SIO4_SYNC_RXD_CFG_ACTIVE_HI;
		rx.data.legacy	= SIO4_SYNC_LEG_RXD_CFG_LOW;
		status			= sio4_sync_rx_set(fd, &rx);

		if (status)
		{
			errs	= 1;
			printf("FAIL <--- (sio4_sync_rx_set())\n");
			break;
		}

		errs	+= sio4_rx_io_mode(fd, -1, 0, GSC_IO_MODE_PIO, NULL);
		errs	+= sio4_rx_io_timeout(fd, -1, 0, 0, NULL);	// zero timeout with PIO

		// Make sure everything is flushed before we procede.
		os_sleep_ms(10);
		errs	+= sio4_rx_fifo_reset(fd, -1, 0, 1, NULL);

		if (errs == 0)
			printf("PASS\n");

		break;
	}

	return(errs);
}



//*****************************************************************************
static int _signal_test(int tx, int rx)
{
	static const struct
	{
		const char*	name;	// NULL terminates list.
		u32			src_msk;
		u32			src_val;
		u32			sts_msk;
		u32			sts_val;
	} list[]	=
	{
		// name				src_msk	src_val	sts_msk	sts_val
		{ "TxC 0 -> RxC",	0x007,	0x006,	0x01,	0x00	},
		{ "TxC 1 -> RxC",	0x007,	0x007,	0x01,	0x01	},
		{ "TxD 0 -> RxD",	0x1C0,	0x180,	0x02,	0x00	},
		{ "TxD 1 -> RxD",	0x1C0,	0x1C0,	0x02,	0x02	},
		{ "TxE 0 -> RxE",	0x030,	0x020,	0x04,	0x00	},
		{ "TxE 1 -> RxE",	0x030,	0x030,	0x04,	0x04	},
		{ NULL,				0x000,	0x000,	0x00,	0x00	}
	};

	int	err;
	int	errs	= 0;
	int	i;
	u32	psrcr;
	u32	pstsr;

	gsc_label("Signal Test");
	printf("\n");
	gsc_label_level_inc();

	for (i = 0; list[i].name; i++)
	{
		err	= 0;
		gsc_label(list[i].name);
		err	+= sio4_reg_read(tx, SIO4_GSC_PSRCR, &psrcr);
		err	+= sio4_reg_mod(tx,
							SIO4_GSC_PSRCR,
							list[i].src_val,
							list[i].src_msk);

		os_sleep_ms(10);

		err	+= sio4_reg_read(rx, SIO4_GSC_PSTSR, &pstsr);
		err	+= sio4_reg_write(tx,SIO4_GSC_PSRCR, psrcr);

		if (err)
			break;

		if ((pstsr & list[i].sts_msk) != list[i].sts_val)
			err++;

		printf("%s\n", err ? "FAIL <---" : "PASS");
		errs	+= err;
	}

	gsc_label_level_dec();
	return(errs);
}



//*****************************************************************************
static long _tx_data(int fd)
{
	int		idle	= 0;
	int		left;
	int		sent;
	u8*		src;
	long	tx_qty	= 0;

	for (;;)
	{
		if (_pipeline_qty > (_pipeline_limit - (int) sizeof(_tx_buf)))
			break;

		// Write out a single block of data.
		src		= _tx_buf;
		left	= sizeof(_tx_buf);

		for (;;)
		{
			if (left == 0)
				break;

			sent	= sio4_write(fd, src, left);

			if (sent < 0)
			{
				printf(	"FAIL <---  (sio4_write() return %ld)\n",
						(long) sent);
				tx_qty	= -1;
				break;
			}

			if (sent == 0)
			{
				idle++;

				if (idle >= 3)
				{
					// This is 3x the Tx I/O Timeout,
					// which should be 30 seconds.
					printf(	"FAIL <---  (failed to write any data)\n");
					tx_qty	= -1;
					break;
				}

				continue;
			}

			tx_qty	+= sent;
			src		+= sent;
			left	-= sent;
			idle	= 0;

			if (left <= 0)
			{
				_pipeline_qty	+= sizeof(_tx_buf);
				break;
			}
		}

		if (tx_qty < 0)
			break;
	}

	return(tx_qty);
}



//*****************************************************************************
static long _rx_data(int fd)
{
	u8*		dst;
	int		got;
	int		i;
	int		idle	= 0;
	int		left;
	long	rx_qty	= 0;

	for (;;)
	{
		if (_pipeline_qty < sizeof(_rx_buf))
			break;

		// Read in a single block of data.
		dst		= _rx_buf;
		left	= sizeof(_rx_buf);

		for (;;)
		{
			if (left == 0)
				break;

			got	= sio4_read(fd, dst, left);

			if (got < 0)
			{
				printf(	"FAIL <---  (sio4_read() return %ld)\n",
						(long) got);
				rx_qty	= -1;
				break;
			}

			if (got == 0)
			{
				idle++;

				if (idle >= 3)
				{
					// This is 3x the Rx I/O Timeout,
					// which should be 30 seconds.
					printf(	"FAIL <---  (failed to read any data)\n");
					rx_qty	= -1;
					break;
				}

				continue;
			}

			rx_qty	+= got;
			dst		+= got;
			left	-= got;
			idle	= 0;

			if (left <= 0)
			{
				_pipeline_qty	-= sizeof(_rx_buf);

				if (memcmp(_rx_buf, _tx_buf, sizeof(_rx_buf)))
				{
					printf("FAIL <---  (data mismatch)\n");
					rx_qty	= -1;

					for (i = 0; i < sizeof(_rx_buf); i++)
					{
						printf(	"%5d. Tx 0x%02lX, Rx 0x%02lX",
								i,
								(long) _tx_buf[i],
								(long) _rx_buf[i]);

						if (_rx_buf[i] != _tx_buf[i])
							printf(" <--- mismatch");

						printf("\n");
					}
				}

				break;
			}
		}

		break;
	}

	return(rx_qty);
}



//*****************************************************************************
static int _data_transfer_test(int tx, int rx, int errs, int secs)
{
	time_t		end;
	int			i;
	time_t		now;
	long		rx_qty;
	long long	rx_total	= 0;
	s32			size		= 0;
	int			tx_done		= 0;
	long		tx_qty;
	long long	tx_total	= 0;

	for (;;)	// A convenience loop.
	{
		gsc_label("Data Transfer");

		if (errs)
		{
			errs	= 0;
			printf("SKIPPED  (setup errors)\n");
			break;
		}

		// Initialization.

		for (i = 0; i < sizeof(_tx_buf); i++)
			_tx_buf[i]	= (u8) i;

		i		= sio4_query(rx, SIO4_QUERY_FIFO_SIZE_RX, &size);
		size	= (size <= 256) ? 256 : size;

		if (i < 0)
		{
			errs++;
			printf("FAIL <---  (sio4_query())\n");
			break;
		}

		_pipeline_qty	= 0;
		_pipeline_limit	= size;
		end				= time(NULL) + secs;

		for (;;)
		{
			if (tx_done == 0)
			{
				tx_qty	= _tx_data(tx);

				if (tx_qty < 0)
				{
					errs++;
					break;
				}

				tx_total	+= tx_qty;
			}

			rx_qty		= _rx_data(rx);

			if (rx_qty < 0)
			{
				errs++;
				break;
			}

			rx_total	+= rx_qty;
			now			= time(NULL);

			if (now > end)
				tx_done	= 1;

			if ((tx_done) && (_pipeline_qty < sizeof(_rx_buf)))
				break;
		}

		if (errs == 0)
			printf("PASS");

		// Report the accumulated transfer quantities.
		printf(	"  (Tx Total ");
		gsc_label_long_comma(tx_total);
		printf(" bytes, Rx Total ");
		gsc_label_long_comma(rx_total);
		printf(" bytes)\n");
		break;
	}

	return(errs);
}



/*****************************************************************************
*
*	Function:	io_test
*
*	Purpose:
*
*		Perform a data I/O write/read test.
*
*	Arguments:
*
*		tx			The Tx device handle.
*
*		tx_index	The device index of the Tx channel.
*
*		rx			The Rx device handle.
*
*		rx_index	The device index of the Rx channel.
*
*	Returned:
*
*		>= 0		The number of errors encounterred.
*
*****************************************************************************/

int io_test(int tx, int tx_index, int rx, int rx_index)
{
	s32		clk		= 100 * 1000;
	int		errs	= 0;
	int		lb		= tx == rx;
	s32		mp		= SIO4_XCVR_PROTOCOL_UNKNOWN;
	mp_t	rx_xcvr;
	int		secs	= 30;
	mp_t	tx_xcvr;
	s32		xcvr;

	for (;;)
	{
		gsc_label("Data I/O Test");
		printf("\n");
		gsc_label_level_inc();

		_connection_type(tx_index, rx_index);
		errs	+= _verify_model(tx, "Tx Model Type");
		errs	+= _verify_model(rx, "Rx Model Type");

		if (errs == 0)
		{
			errs	+= _xcvr_support(tx, tx_index, "Tx Transceiver Support", &tx_xcvr);
			errs	+= _xcvr_support(rx, rx_index, "Rx Transceiver Support", &rx_xcvr);
			errs	+= _verify_xcvr(&tx_xcvr, &rx_xcvr, &xcvr);
		}

		if (errs == 0)
		{
			gsc_label("Signal Test");
			printf("\n");
			gsc_label_level_inc();

			errs	+= sio4_initialize(tx, tx_index, 1, 1, NULL);
			errs	+= sio4_initialize(rx, rx_index, 1, 1, NULL);
			errs	+= _setup_common(tx, "Tx", 0, 1, lb, mp, clk);
			errs	+= _setup_tx(tx, 0, clk);
			errs	+= _setup_common(rx, "Rx", 0, 0, lb, mp, clk);
			errs	+= _setup_rx(rx);
			errs	+= _signal_test(tx, rx);

			gsc_label_level_dec();
		}

		if (errs == 0)
		{
			gsc_label("Data Transfer Test");
			printf("(%ld secs)\n", (long) secs);
			gsc_label_level_inc();

			errs	+= sio4_initialize(tx, tx_index, 1, 1, NULL);
			errs	+= sio4_initialize(rx, rx_index, 1, 1, NULL);
			errs	+= _setup_common(tx, "Tx", 1, 1, lb, mp, clk);
			errs	+= _setup_tx(tx, 1, clk);
			errs	+= _setup_common(rx, "Rx", 1, 0, lb, mp, clk);
			errs	+= _setup_rx(rx);
			errs	+= _data_transfer_test(tx, rx, errs, secs);

			gsc_label_level_dec();
		}

		gsc_label_level_dec();
		break;
	}

	return(errs);
}


