// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_2.x.x_GSC_DN/sync/gpio/main.c $
// $Rev: 33970 $
// $Date: 2015-11-05 18:50:55 -0600 (Thu, 05 Nov 2015) $

#include "main.h"



// variables ******************************************************************

static	int		_continuous		= 0;
static	int		_def_index		= 0;
static	int		_ignore_errors	= 0;
static	long	_minute_limit	= 0;
static	int		_seconds		= 0;
static	int		_test_limit		= -1;
static	int		_xcvr			= SIO4_XCVR_PROTOCOL_UNKNOWN;



//*****************************************************************************
static int _parse_args(int argc, char** argv)
{
	char	buf[32];
	char	c;
	int		errs	= 0;
	int		i;
	int		j;
	int		k;

	printf(	"GPIO Test Utility (Version %s)\n", VERSION);
	printf("USAGE: gpio <-232> <-422> <-423> <-c> <-C> <-m#> <-n#> <-s#> <index>\n");
	printf("  -232   Select RS-232 cable transceivers.\n");
	printf("  -422   Select RS-422/RS-485 cable transceivers.\n");
	printf("  -423   Select RS-423 cable transceivers.\n");
	printf("  -c     Continue testing until an error occurs.\n");
	printf("  -C     Continue testing even if errors occur.\n");
	printf("  -m#    Run for at most # minutes (a decimal number).\n");
	printf("  -n#    Repeat test at most # times (a decimal number).\n");
	printf("  -s#    Hold output and monitor input for this long (seconds).\n");
	printf("  index  The zero based index of the channel to access (default is 0).\n");

	gsc_label_level_inc();

	for (i = 0; i < argc; i++)
	{
		sprintf(buf, "Argument %d", i);
		gsc_label(buf);
		printf("%s\n", argv[i]);
	}

	gsc_label_level_dec();

	for (i = 1; i < argc; i++)
	{
		if (strcmp(argv[i], "-232") == 0)
		{
			_xcvr	= SIO4_XCVR_PROTOCOL_RS232;
			continue;
		}

		if (strcmp(argv[i], "-422") == 0)
		{
			_xcvr	= SIO4_XCVR_PROTOCOL_RS422_RS485;
			continue;
		}

		if (strcmp(argv[i], "-423") == 0)
		{
			_xcvr	= SIO4_XCVR_PROTOCOL_RS423;
			continue;
		}

		if (strcmp(argv[i], "-c") == 0)
		{
			_continuous		= 1;
			_ignore_errors	= 0;
			continue;
		}

		if (strcmp(argv[i], "-C") == 0)
		{
			_continuous		= 1;
			_ignore_errors	= 1;
			continue;
		}

		if ((argv[i][0] == '-') && (argv[i][1] == 'm') && (argv[i][2]))
		{
			j	= sscanf(&argv[i][2], "%d%c", &k, &c);

			if ((j == 1) && (k > 0))
			{
				_minute_limit	= k;
				continue;
			}

			errs	= 1;
			printf("ERROR: invalid argument: %s\n", argv[i]);
			break;
		}

		if ((argv[i][0] == '-') && (argv[i][1] == 'n') && (argv[i][2]))
		{
			j	= sscanf(&argv[i][2], "%d%c", &k, &c);

			if ((j == 1) && (k > 0))
			{
				_test_limit	= k;
				continue;
			}

			errs	= 1;
			printf("ERROR: invalid argument: %s\n", argv[i]);
			break;
		}

		if ((argv[i][0] == '-') && (argv[i][1] == 's') && (argv[i][2]))
		{
			j	= sscanf(&argv[i][2], "%d%c", &k, &c);

			if ((j == 1) && (k >= 0) && (k <= 3600))
			{
				_seconds	= k;
				continue;
			}

			errs	= 1;
			printf("ERROR: invalid argument: %s\n", argv[i]);
			break;
		}

		j	= sscanf(argv[i], "%d%c", &k, &c);

		if ((j == 1) && (k >= 0))
		{
			_def_index	= k;
			continue;
		}

		errs	= 1;
		printf("ERROR: extraneous argument: %s\n", argv[i]);
		break;
	}

	return(errs);
}



//*****************************************************************************
static int _perform_tests(int fd)
{
	int			cable_legacy	= 0;
	int			errs			= 0;
	const char*	psz;
	struct tm*	stm;
	s32			sync;
	time_t		tt;

	time(&tt);
	stm	= localtime(&tt);
	psz	= asctime(stm);
	gsc_label("Performing Test");
	printf("%s", psz);

	gsc_label("Firmware Type");
	errs	+= sio4_query(fd, SIO4_QUERY_MODEL_SYNC, &sync);

	if (sync == 0)
	{
		printf("SKIPPING TESTS  (Not a SYNC model SIO4.)\n");
	}
	else
	{
		printf("PASS  (SYNC)\n");

		errs	+= sio4_initialize(fd, -1, 1, 1, NULL);
		errs	+= xcvr_test(fd, _xcvr);
		errs	+= cable_test(fd, &cable_legacy);

		if (errs)
		{
			printf("ABORTING TEST DUE TO ERRORS\n");
		}
		else
		{
			errs	+= txc_test		(fd, cable_legacy, _seconds);
			errs	+= txd_test		(fd, cable_legacy, _seconds);
			errs	+= txe_test		(fd, cable_legacy, _seconds);
			errs	+= txauxc_test	(fd, cable_legacy, _seconds);
			errs	+= txspare_test	(fd, cable_legacy, _seconds);
			errs	+= rxc_test		(fd, cable_legacy, _seconds);
			errs	+= rxd_test		(fd, cable_legacy, _seconds);
			errs	+= rxe_test		(fd, cable_legacy, _seconds);
			errs	+= rxauxc_test	(fd, cable_legacy, _seconds);
			errs	+= rxspare_test	(fd, cable_legacy, _seconds);
		}
	}

	return(errs);
}



/******************************************************************************
*
*	Function:	main
*
*	Purpose:
*
*		Control the overall operation of the application.
*
*	Arguments:
*
*		argc	The number of command line arguments.
*
*		argv	The list of command line arguments.
*
*	Returned:
*
*		EXIT_SUCCESS	No errors were encounterred.
*		EXIT_FAILURE	One or more errors were encounterred.
*
******************************************************************************/

int main(int argc, char **argv)
{
	int		errs		= 0;
	time_t	exec		= time(NULL);
	long	failures	= 0;
	int		fd;
	long	hours;
	long	mins;
	time_t	now;
	long	passes		= 0;
	int		qty;
	int		ret			= EXIT_SUCCESS;
	long	secs;
	time_t	t_limit;
	time_t	test;
	long	tests		= 0;

	for (;;)
	{
		gsc_label_init(26);
		test		= time(NULL);
		errs		= _parse_args(argc, argv);

		if (errs)
			break;

		os_id_host();
		errs	+= os_id_driver(SIO4_BASE_NAME);

		if (errs)
			break;

		sio4_sync_id_library();
		t_limit	= exec + (_minute_limit * 60);
		qty		= os_count_boards(SIO4_BASE_NAME);

		if (qty <= 0)
			break;

		fd		= sio4_dev_open(_def_index, _def_index, 1);
		errs	+= (fd == -1) ? 1 : 0;

		if (errs == 0)
		{
			gsc_label_level_inc();
			errs	+= sio4_id_board(fd, -1);
			gsc_label_level_dec();
		}

		if (errs)
		{
			printf(	"Skipping tests (device access error%s).\n",
					(errs == 1) ? "" : "s");
		}
		else
		{
			errs	+= _perform_tests(fd);
		}

		errs	+= sio4_dev_close(fd, _def_index, 1);
		now		= time(NULL);
		tests++;

		if (errs)
		{
			failures++;
			printf(	"\nRESULTS: FAIL <---  (%d error%s)",
					errs,
					(errs == 1) ? "" : "s");
		}
		else
		{
			passes++;
			printf("\nRESULTS: PASS");
		}

		secs	= (long) (now - test);
		hours	= secs / 3600;
		secs	= secs % 3600;
		mins	= secs / 60;
		secs	= secs % 60;
		printf(" (duration %ld:%ld:%02ld)\n", hours, mins, secs);

		secs	= (long) (now - exec);
		hours	= secs / 3600;
		secs	= secs % 3600;
		mins	= secs / 60;
		secs	= secs % 60;
		printf(	"SUMMARY: tests %ld, pass %ld, fail %ld"
			" (duration %ld:%ld:%02ld)\n\n",
			tests,
			passes,
			failures,
			hours,
			mins,
			secs);

		if ((_test_limit > 0) && (_test_limit <= tests))
			break;

		if (_continuous == 0)
			break;

		if ((_ignore_errors == 0) && (errs))
			break;

		if ((_minute_limit) && (now >= t_limit))
			break;
	}

	if ((errs) || (failures))
		ret	= EXIT_FAILURE;

	return(ret);
}


