// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_2.x.x_GSC_DN/services/file.c $
// $Rev: 23193 $
// $Date: 2013-09-30 17:59:10 -0500 (Mon, 30 Sep 2013) $

#include "file.h"



/*******************************************************************************
*
*	Function: file_close
*
*	Purpose:
*
*		Perform a close on an open file.
*
*	Arguments:
*
*		name	The name of the open file.
*
*		file	An open file handle.
*
*	Returned:
*
*		else	The operation failed.
*		NULL	The operation succeeded.
*
*******************************************************************************/

FILE* file_close(const char* name, FILE* file)
{
	int	i;

	if (file)
	{
		i	= fclose(file);

		if (i)
		{
			printf("ERROR: unable to close '%s'\n", name);
		}
		else
		{
			file	= NULL;
		}
	}

	return(file);
}



/*******************************************************************************
*
*	Function: file_eof
*
*	Purpose:
*
*		Check to see if we're at the end of the file.
*
*	Arguments:
*
*		name	The name of the open file.
*
*		file	An open file handle.
*
*		eof		Set to 1 if we're at the end and 0 if not.
*
*	Returned:
*
*		0		All went well.
*		-1		There was a problem.
*
*******************************************************************************/

int file_eof(const char* name, FILE* file, int* eof)
{
	if (name)
	{
	}

	if ((file) && (eof))
	{
		eof[0]	= feof(file);
		eof[0]	= eof[0] ? 1 : 0;
	}
	else if (eof)
	{
		eof[0]	= 0;
	}

	return(0);
}



/*******************************************************************************
*
*	Function: file_open
*
*	Purpose:
*
*		Perform an open given a file name. The file is opened in binary read
*		mode.
*
*	Arguments:
*
*		name	The name of the file to open.
*
*	Returned:
*
*		NULL	The operation failed.
*		else	A valid file handle.
*
*******************************************************************************/

FILE* file_open(const char* name)
{
	FILE*	file;

	for (;;)
	{
		if ((name == NULL) || (name[0] == 0))
		{
			file	= NULL;
			break;
		}

		file	= fopen(name, "rb");

		if (file)
			break;

		printf("ERROR: unable to open '%s'\n", name);
		break;
	}

	return(file);
}



/*******************************************************************************
*
*	Function: file_read
*
*	Purpose:
*
*		Read data from an open file.
*
*	Arguments:
*
*		name	The name of the open file.
*
*		file	An open file handle.
*
*		buf		Put the data here.
*
*		size	The number of bytes to read.
*
*	Returned:
*
*		>- 0	The number of bytes read.
*		-1		There was a problem.
*
*******************************************************************************/

long file_read(const char* name, FILE* file, char* buf, size_t size)
{
	int		i;
	long	l;

	for (;;)
	{
		if ((name == NULL) || (file == NULL) || (buf == NULL) || (size == 0))
		{
			l	= 0;
			break;
		}

		l	= fread(buf, 1, size, file);

		if (l)
			break;

		i	= feof(file);

		if (i)
			break;

		printf("ERROR: read failure of %s bytes on '%ld'\n", name, (long) size);
		l	= -1;
		break;
	}

	return(l);
}



/*******************************************************************************
*
*	Function: file_read_line
*
*	Purpose:
*
*		Read a line of text from an open file.
*
*	Arguments:
*
*		name	The name of the open file.
*
*		file	An open file handle.
*
*		dst		Put the line of text here. The line may be terminated in DOS
*				or UNIX format.
*
*		size	The size of the buffer.
*
*	Returned:
*
*		>= 0	The number of bytes read.
*		-1		There was a problem.
*
*******************************************************************************/

long file_read_line(const char* name, FILE* file, char* dst, size_t size)
{
	char	buf[4096];
	int		eof;
	int		i;
	int		len;
	long	pos;
	char*	psz;
	long	qty;
	int		ret;

	for (;;)	// A convenience loop.
	{
		pos	= file_tell(name, file);

		if (pos < 0)
		{
			ret	= -1;
			break;
		}

		qty	= file_read(name, file, buf, sizeof(buf));

		if (qty == 0)
		{
			i	= file_eof(name, file, &eof);

			if (i == -1)
				ret	= -1;
			else if (eof)
				ret	= 0;
			else
				ret	= -1;

			dst[0]	= 0;
			break;
		}

		buf[sizeof(buf) - 1]	= 0;
		psz	= strchr(buf, '\n');

		if (psz)
			psz[1]	= 0;

		psz	= strchr(buf, '\r');

		if (psz == NULL)
			;
		else if (psz[1] != '\n')
			psz[1]	= 0;

		len	= strlen(buf);
		i	= file_seek(name, file, pos + len);

		if (i)
		{
			ret	= -1;
		}
		else
		{
			ret	= len;

			if (len > (int) size)
				len	= size;

			strncpy(dst, buf, len);
			dst[len - 1]	= 0;
		}

		break;
	}

	return(ret);
}



/*******************************************************************************
*
*	Function: file_seek
*
*	Purpose:
*
*		Report the current position in a file.
*
*	Arguments:
*
*		name	The name of the open file.
*
*		file	An open file handle.
*
*		pos		The absolute position to seek to.
*
*	Returned:
*
*		>= 0	The number of errors seen.
*
*******************************************************************************/

int file_seek(const char* name, FILE* file, long pos)
{
	int	errs;
	int	i;

	i	= fseek(file, pos, SEEK_SET);

	if (i)
	{
		errs	= 1;
		printf("ERROR: fseek(): file access error: %s\n", name);
	}
	else
	{
		errs	= 0;
	}

	return(errs);
}



/*******************************************************************************
*
*	Function: file_tell
*
*	Purpose:
*
*		Report the current position in a file.
*
*	Arguments:
*
*		name	The name of the open file.
*
*		file	An open file handle.
*
*	Returned:
*
*		>= 0	The current location.
*		-1		There was a problem.
*
*******************************************************************************/

long file_tell(const char* name, FILE* file)
{
	long	pos;

	pos	= ftell(file);

	if (pos == -1)
		printf("ERROR: ftell(): file access error: %s\n", name);

	return(pos);
}



/*******************************************************************************
*
*	Function: file_test
*
*	Purpose:
*
*		Test for the existence of the named file.
*
*	Arguments:
*
*		name	The name of the file to test.
*
*	Returned:
*
*		1		The file could be opened.
*		0		The file could not be opened.
*
*******************************************************************************/

int file_test(const char* name)
{
	FILE*	file;
	int		ret;

	for (;;)
	{
		if ((name == NULL) || (name[0] == 0))
		{
			ret	= 0;
			break;
		}

		file	= fopen(name, "rb");

		if (file)
		{
			fclose(file);
			ret	= 1;
			break;
		}

		ret	= 0;
		break;
	}

	return(ret);
}


