// $URL: http://subversion:8080/svn/gsc/trunk/drivers/gsc_common/lib/rtx/os_res.c $
// $Rev: 33966 $
// $Date: 2015-11-05 18:39:22 -0600 (Thu, 05 Nov 2015) $

// RTX 2012 library

#include "main.h"




//*****************************************************************************
// returns 0 = ok, <0 = -errno, >0 = error code for Jungo or Windows
static int _res_mem_acquire(const char* name, os_mem_t* mem, int read_write)
{
	int	access;
	int	ret;

	if (read_write)
		access	= SHM_MAP_READ | SHM_MAP_WRITE;
	else
		access	= SHM_MAP_READ;

	// Access the named resource.
	mem->handle	= RtOpenSharedMemory(access, 0, name, &mem->ptr);

	if (mem->handle == NULL)
	{
		ret	= (int) RtGetLastError();
	}
	else if (mem->ptr == NULL)
	{
		RtCloseHandle(mem->handle);
		mem->handle	= NULL;
		ret			= -ENOMEM;
	}
	else
	{
		ret	= 0;
	}

	return(ret);
}



//*****************************************************************************
// returns 0 = ok, <0 = -errno, >0 = error code for Jungo or Windows
static int _res_mem_acquire_by_id(const char* base, u32 suffix, os_mem_t* mem)
{
	int		len;
	char	name[OS_RESOURCE_NAME_MAX_LEN];
	int		ret;

	len	= strlen(base);

	if (len > (OS_RESOURCE_NAME_MAX_LEN - 15))
	{
		ret	= -ENOSPC;
	}
	else
	{
		sprintf(name, "%s.mem.%08lX", base, (long) suffix);
		ret	= _res_mem_acquire(name, mem, 1);
	}

	return(ret);
}



//*****************************************************************************
static void _res_mem_release(os_mem_t* mem)
{
	if (mem->handle)
		RtCloseHandle(mem->handle);

	memset(mem, 0, sizeof(os_mem_t));
}



//*****************************************************************************
// returns 0 = ok, <0 = -errno, >0 = error code for Jungo or Windows
static int _res_sem_acquire_by_suffix(const char* base, u32 suffix, os_sem_t* sem)
{
	int		len;
	char	name[OS_RESOURCE_NAME_MAX_LEN];
	int		ret;

	len	= strlen(base);

	if (len > (OS_RESOURCE_NAME_MAX_LEN - 15))
	{
		ret	= -ENOSPC;
	}
	else
	{
		sprintf(name, "%s.sem.%08lX", base, (long) suffix);
		sem->handle	= RtOpenSemaphore(0, FALSE, name);

		if (sem->handle == NULL)
			ret	= RtGetLastError();
		else
			ret	= 0;
	}

	return(ret);
}



//*****************************************************************************
static void _res_sem_release(os_sem_t* sem)
{
	if (sem->handle)
		RtCloseHandle(sem->handle);

	memset(sem, 0, sizeof(os_sem_t));
}



//*****************************************************************************
// returns 0 = ok, <0 = -errno, >0 = error code for Jungo or Windows
static int _res_main_mem_acquire(os_file_t* fp, const char* name)
{
	int	ret;

	ret	= _res_mem_acquire(name, &fp->main_mem.mem, 0);

	if (ret)
		;
	else if (fp->main_mem.mem.ptr)
		fp->main_mem.ptr	= fp->main_mem.mem.ptr;
	else
		ret	= -ENOMEM;

	return(ret);
}



//*****************************************************************************
static void _res_main_mem_release(os_file_t* fp)
{
	_res_mem_release(&fp->main_mem.mem);
	fp->main_mem.ptr	= NULL;
}



//*****************************************************************************
// returns 0 = ok, <0 = -errno, >0 = error code for Jungo or Windows
static int _res_primary_acquire(os_file_t* fp, const char* base)
{
	os_main_mem_t*	mm;
	int				ret		= 0;
	int				tmp;

	mm	= fp->main_mem.ptr;

	tmp	= _res_sem_acquire_by_suffix(
			base,
			fp->main_mem.ptr->primary.xfer,
			&fp->primary.msg_set.xfer);
	ret	= ret ? ret : tmp;

	tmp	= _res_sem_acquire_by_suffix(
			base,
			fp->main_mem.ptr->primary.to_dev,
			&fp->primary.msg_set.to_dev);
	ret	= ret ? ret : tmp;

	tmp	= _res_sem_acquire_by_suffix(
			base,
			fp->main_mem.ptr->primary.to_app,
			&fp->primary.msg_set.to_app);
	ret	= ret ? ret : tmp;

	tmp	= _res_mem_acquire_by_id(
			base,
			fp->main_mem.ptr->primary.mem,
			&fp->primary.mem);
	ret	= ret ? ret : tmp;

	if (ret == 0)
		fp->primary.msg_set.msg	= (void*) fp->primary.mem.ptr;

	return(ret);
}



//*****************************************************************************
static void _res_primary_release(os_file_t* fp)
{
	_res_sem_release(&fp->primary.msg_set.xfer);
	_res_sem_release(&fp->primary.msg_set.to_dev);
	_res_sem_release(&fp->primary.msg_set.to_app);
	_res_mem_release(&fp->primary.mem);
	fp->primary.msg_set.msg	= NULL;
}



//*****************************************************************************
// returns 0 = ok, <0 = -errno, >0 = error code for Jungo or Windows
static int _res_msg_mem_acquire(os_file_t* fp, const char* base)
{
	int	ret;

	ret	= _res_mem_acquire_by_id(
			base,
			fp->main_mem.ptr->msg_mem.mem,
			&fp->msg_mem.mem);
	fp->msg_mem.msgs	= (void*) fp->msg_mem.mem.ptr;
	return(ret);
}



//*****************************************************************************
static void _res_msg_mem_release(os_file_t* fp)
{
	_res_mem_release(&fp->msg_mem.mem);
	fp->msg_mem.msgs	= NULL;
}



//*****************************************************************************
// returns 0 = ok, <0 = -errno, >0 = error code for Jungo or Windows
static int _res_msg_list_acquire(os_file_t* fp, const char* base)
{
	int	i;
	int	ret	= 0;
	int	tmp;

	for (i = 0; i < OS_MSG_LIST_SIZE; i++)
	{
		tmp	= _res_sem_acquire_by_suffix(
				base,
				fp->main_mem.ptr->msg_list[i].xfer,
				&fp->msg_list[i].xfer);
		ret	= ret ? ret : tmp;

		tmp	= _res_sem_acquire_by_suffix(
				base,
				fp->main_mem.ptr->msg_list[i].to_dev,
				&fp->msg_list[i].to_dev);
		ret	= ret ? ret : tmp;

		tmp	= _res_sem_acquire_by_suffix(
				base,
				fp->main_mem.ptr->msg_list[i].to_app,
				&fp->msg_list[i].to_app);
		ret	= ret ? ret : tmp;

		fp->msg_list[i].msg	= &fp->msg_mem.msgs[i];
	}

	return(ret);
}



//*****************************************************************************
static void _res_msg_list_release(os_file_t* fp)
{
	int	i;

	for (i = 0; i < OS_MSG_LIST_SIZE; i++)
	{
		_res_sem_release(&fp->msg_list[i].xfer);
		_res_sem_release(&fp->msg_list[i].to_dev);
		_res_sem_release(&fp->msg_list[i].to_app);
		fp->msg_list[i].msg	= NULL;
	}
}



//*****************************************************************************
// returns 0 = ok, <0 = -errno, >0 = error code for Jungo or Windows
static int _res_rx_buf_acquire(os_file_t* fp, const char* base)
{
	u32				adrs	= fp->main_mem.ptr->rx.adrs;
	LARGE_INTEGER	li;
	os_mem_t*		mem		= &fp->rx;
	int				ret		= 0;
	u32				size	= fp->main_mem.ptr->rx.size;

	// It is acceptible if there isn't an Rx buffer.
	// This just means read() is unsupported.

	if ((adrs == 0) && (size == 0))
	{
		ret	= 0;
	}
	else if ((adrs) && (size))
	{
		li.QuadPart	= adrs;
		mem->ptr	= RtMapMemory(li, size, MmNonCached);

		if (mem->ptr)
			mem->size	= size;
		else
			ret	= -ENOMEM;
	}
	else
	{
		ret	= -EINVAL;
	}

	return(ret);
}



//*****************************************************************************
static void _res_rx_buf_release(os_file_t* fp)
{
	os_mem_t*	mem	= &fp->rx;

	if (mem->ptr)
		RtUnmapMemory(mem->ptr);

	memset(mem, 0, sizeof(os_mem_t));
}



//*****************************************************************************
// returns 0 = ok, <0 = -errno, >0 = error code for Jungo or Windows
static int _res_tx_buf_acquire(os_file_t* fp, const char* base)
{
	u32				adrs	= fp->main_mem.ptr->tx.adrs;
	LARGE_INTEGER	li;
	os_mem_t*		mem		= &fp->tx;
	int				ret		= 0;
	u32				size	= fp->main_mem.ptr->tx.size;

	// It is acceptible if there isn't an Tx buffer.
	// This just means read() is unsupported.

	if ((adrs == 0) && (size == 0))
	{
		ret	= 0;
	}
	else if ((adrs) && (size))
	{
		li.QuadPart	= adrs;
		mem->ptr	= RtMapMemory(li, size, MmNonCached);

		if (mem->ptr)
			mem->size	= size;
		else
			ret	= -ENOMEM;
	}
	else
	{
		ret	= -EINVAL;
	}

	return(ret);
}



//*****************************************************************************
static void _res_tx_buf_release(os_file_t* fp)
{
	os_mem_t*	mem	= &fp->tx;

	if (mem->ptr)
		RtUnmapMemory(mem->ptr);

	memset(mem, 0, sizeof(os_mem_t));
}



//*****************************************************************************
// returns 0 = ok, <0 = -errno, >0 = error code for Jungo or Windows
int os_res_acquire(os_file_t* fp, const char* base, const char* name)
{
	int	ret;

	// Gain access to the driver resources required for passing mssages
	// between the driver and the library.

	for (;;)	// A convenience loop.
	{
		// Validate the arguments.
		if ((fp == NULL) || (base == NULL) || (name == NULL))
		{
			ret	= -EINVAL;
			break;
		}

		ret	= _res_main_mem_acquire(fp, name);

		if (ret)
			break;

		ret	= _res_primary_acquire(fp, base);

		if (ret)
			break;

		ret	= _res_msg_mem_acquire(fp, base);

		if (ret)
			break;

		ret	= _res_msg_list_acquire(fp, base);

		if (ret)
			break;

		ret	= _res_rx_buf_acquire(fp, base);

		if (ret)
			break;

		ret	= _res_tx_buf_acquire(fp, base);
		break;
	}

	return(ret);
}



//*****************************************************************************
// returns 0 = ok, <0 = -errno, >0 = error code for Jungo or Windows
void os_res_release(os_file_t* fp)
{
	// Release access to the driver resources to which we previously acquired
	// access.

	if (fp)
	{
		_res_tx_buf_release(fp);
		_res_rx_buf_release(fp);
		_res_msg_list_release(fp);
		_res_msg_mem_release(fp);
		_res_primary_release(fp);
		_res_main_mem_release(fp);
	}
}


