// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_2.x.x_GSC_DN/isoc/isocc2c/io.c $
// $Rev: 33932 $
// $Date: 2015-10-07 15:14:07 -0500 (Wed, 07 Oct 2015) $

#include "main.h"



// #defines *******************************************************************

#define	RXCLK		0x1
#define	RXD			0x2



// data types *****************************************************************

typedef struct
{
	int	index;
	s32	mp;
	s32	program;
	s32	current;
	s32	_232;
	s32	_422;
	s32	_423;
} mp_t;



// variables ******************************************************************

static	int	_pipeline_limit	= 512;		// modified during init.
static	int	_pipeline_qty	= 0;
static	u8	_rx_buf[256];				// Must be 256!!!
static	u8	_tx_buf[sizeof(_rx_buf)];



//*****************************************************************************
static void _connection_type(int tx_index, int rx_index)
{
	int	rx_b	= rx_index / 4;
	int	rx_c	= rx_index % 4;
	int	tx_b	= tx_index / 4;
	int	tx_c	= tx_index % 4;

	gsc_label("Connection Type");

	if (tx_index == rx_index)
	{
		printf("External Loopback (no cable)\n");
	}
	else if (tx_b == rx_b)
	{
		printf(	"Channel-to-Channel Loopback Cable"
				" (B# %d, C#s %d+%d)\n",
				tx_b,
				tx_c,
				rx_c);
	}
	else
	{
		printf(	"Channel-to-Channel Passthrough Cable"
				" (B%d C%d, B%d C%d)\n",
				tx_b,
				tx_c,
				rx_b,
				rx_c);
	}
}



//*****************************************************************************
static int _verify_model(int fd, const char* desc)
{
	int	errs;
	s32	z16c30;

	gsc_label(desc);
	errs	= sio4_query(fd, SIO4_QUERY_MODEL_Z16C30, &z16c30);

	if (errs)
	{
	}
	else if (z16c30)
	{
		printf("PASS  (Z16C30)\n");
	}
	else
	{
		errs++;
		printf("FAIL <---  (Not a Z16C30 model SIO4)\n");
	}

	return(errs);
}



//****************************************************************************
static int _mp_test(int fd, s32 set, s32* supported)
{
	int	errs	= 0;
	s32	got;
	s32	init;
	int	ret;

	// Get the starting protocol.
	init	= -1;
	ret		= sio4_ioctl(fd, SIO4_IOCTL_XCVR_PROTOCOL, &init);
	errs	+= (ret < 0) ? 1 : 0;

	// Select the protocol in question.
	ret		= sio4_ioctl(fd, SIO4_IOCTL_XCVR_PROTOCOL, &set);
	errs	+= (ret < 0) ? 1 : 0;

	// Now see what we got.
	got		= -1;
	ret		= sio4_ioctl(fd, SIO4_IOCTL_XCVR_PROTOCOL, &got);
	errs	+= (ret < 0) ? 1 : 0;

	// Restore the initial setting.
	ret		= sio4_ioctl(fd, SIO4_IOCTL_XCVR_PROTOCOL, &init);
	errs	+= (ret < 0) ? 1 : 0;

	// Now record the results.
	supported[0]	= (got == set) ? 1 : 0;
	return(errs);
}



//*****************************************************************************
static int _xcvr_support(int fd, int index, const char* desc, mp_t* mp)
{
	int	errs	= 0;
	int	ret;

	mp->index	= index;
	gsc_label(desc);
	errs	+= sio4_query(fd, SIO4_QUERY_MP, &mp->mp);
	errs	+= sio4_query(fd, SIO4_QUERY_MP_PROGRAM, &mp->program);
	errs	+= _mp_test(fd, SIO4_XCVR_PROTOCOL_RS232, &mp->_232);
	errs	+= _mp_test(fd, SIO4_XCVR_PROTOCOL_RS422_RS485, &mp->_422);
	errs	+= _mp_test(fd, SIO4_XCVR_PROTOCOL_RS423, &mp->_423);

	mp->current	= -1;
	ret			= sio4_ioctl(fd, SIO4_IOCTL_XCVR_PROTOCOL, &mp->current);
	errs		+= (ret < 0) ? 1 : 0;

	if (errs == 0)
		printf("Done\n");

	return(errs);
}



//*****************************************************************************
static int _verify_xcvr(const mp_t* tx_xcvr, const mp_t* rx_xcvr, s32* xcvr)
{
	int	errs	= 0;

	for (;;)	// A convenience loop.
	{
		gsc_label("Transceiver Selection");

		if ((tx_xcvr->mp == 0) || (rx_xcvr->mp == 0))
		{
			if (tx_xcvr->index == rx_xcvr->index)
			{
				// The channel has fixed transceivers.
				xcvr[0]	= -1;
				printf("PASS  (using installed transceivers)\n");
				break;
			}

			if ((tx_xcvr->index / 4) == (rx_xcvr->index / 4))
			{
				// The board has fixed transceivers.
				// All channels should be the same.
				xcvr[0]	= -1;
				printf("PASS  (using installed transceivers)\n");
				break;
			}

			// There is no way of knowing if they are both the same.
			errs++;
			xcvr[0]	= -1;
			printf("FAIL <---  (insufficient MP support)\n");
			break;
		}

		if (tx_xcvr->current != rx_xcvr->current)
		{
		}
		else if (tx_xcvr->current == SIO4_XCVR_PROTOCOL_RS422_RS485)
		{
			xcvr[0]	= SIO4_XCVR_PROTOCOL_RS422_RS485;
			printf("PASS  (RS-423)\n");
			break;
		}
		else if (tx_xcvr->current == SIO4_XCVR_PROTOCOL_RS423)
		{
			xcvr[0]	= SIO4_XCVR_PROTOCOL_RS423;
			printf("PASS  (RS-423)\n");
			break;
		}
		else if (tx_xcvr->current == SIO4_XCVR_PROTOCOL_RS232)
		{
			xcvr[0]	= SIO4_XCVR_PROTOCOL_RS232;
			printf("PASS  (RS-232)\n");
			break;
		}

		if ((tx_xcvr->_422) && (rx_xcvr->_422))
		{
			// Fastest of the supported protocols.
			xcvr[0]	= SIO4_XCVR_PROTOCOL_RS422_RS485;
			printf("PASS  (RS-422/485)\n");
			break;
		}

		if ((tx_xcvr->_423) && (rx_xcvr->_423))
		{
			xcvr[0]	= SIO4_XCVR_PROTOCOL_RS423;
			printf("PASS  (RS-423)\n");
			break;
		}

		if ((tx_xcvr->_232) && (rx_xcvr->_232))
		{
			// Slowest of the supported protocols.
			xcvr[0]	= SIO4_XCVR_PROTOCOL_RS232;
			printf("PASS  (RS-232)\n");
			break;
		}

		errs++;
		xcvr[0]	= SIO4_XCVR_PROTOCOL_UNKNOWN;
		printf("FAIL <---  (common protocol not supported)\n");
		break;
	}

	return(errs);
}



//*****************************************************************************
static int _setup_apply(
	int						fd,
	int						index,
	const char*				desc,
	const sio4_isoc_init_t*	init,
	s32						xcvr,
	int						dce,	// DCE or DTE => show Tx or Rx bitrate
	int						lb)
{
	const char*			err		= "INTERNAL ERROR";
	int					errs	= 0;
	sio4_isoc_t			prot;
	int					sts;
	int					xmit	= dce ? 1 : 0;

	gsc_label_index(desc, index);
	sts		= sio4_isoc_init(fd, init, &prot, &err);

	if (sts)
	{
		errs++;
		printf("FAIL <---  (sio4_isoc_init(): %s\n", err);
	}
	else
	{
		prot.cable.mode				= dce
									? SIO4_ISOC_CABLE_MODE_DCE
									: SIO4_ISOC_CABLE_MODE_DTE;
		prot.cable.loopback.mode	= lb;
		prot.cable.protocol			= xcvr;

		if (lb != SIO4_ISOC_LOOPBACK_MODE_DISABLE)
		{
			prot.cable.legacy.txc		= SIO4_ISOC_CABLE_LEGACY_TXC_UP;
			prot.cable.legacy.txd_cts	= SIO4_ISOC_CABLE_LEGACY_TXD_CTS_UP;
			prot.cable.legacy.rxc		= SIO4_ISOC_CABLE_LEGACY_RXC_UP;
			prot.cable.legacy.rxd_dcd	= SIO4_ISOC_CABLE_LEGACY_RXD_DCD_UP;
		}
		else if (xmit)
		{
			prot.cable.legacy.txc		= SIO4_ISOC_CABLE_LEGACY_TXC_UP;
			prot.cable.legacy.txd_cts	= SIO4_ISOC_CABLE_LEGACY_TXD_CTS_UP;
			prot.cable.legacy.rxc		= SIO4_ISOC_CABLE_LEGACY_RXC_LOW;
			prot.cable.legacy.rxd_dcd	= SIO4_ISOC_CABLE_LEGACY_RXD_DCD_LOW;
		}
		else
		{
			prot.cable.legacy.txc		= SIO4_ISOC_CABLE_LEGACY_TXC_LOW;
			prot.cable.legacy.txd_cts	= SIO4_ISOC_CABLE_LEGACY_TXD_CTS_LOW;
			prot.cable.legacy.rxc		= SIO4_ISOC_CABLE_LEGACY_RXC_UP;
			prot.cable.legacy.rxd_dcd	= SIO4_ISOC_CABLE_LEGACY_RXD_DCD_UP;
		}

		sts	= sio4_isoc_set(fd, &prot, &err);

		if (sts)
		{
			errs++;
			printf("FAIL <---  (sio4_isoc_set(): %s\n", err);
		}
	}

	if (errs == 0)
	{
		printf("PASS");

		if (xmit)
		{
			printf("  (");
			gsc_label_long_comma(init->tx_bit_rate);
			printf(" bits/second)");
		}

		printf("\n");
	}

	return(errs);
}



//*****************************************************************************
static int _setup(int tx_index, int tx, int rx_index, int rx, s32 xcvr)
{
	int					errs	= 0;
	sio4_isoc_init_t	init	= { 50000, 50000 };

	if (tx == rx)
	{
		errs	+= _setup_apply(tx, tx_index, "Tx/Rx Setup", &init, xcvr, 1, SIO4_ISOC_LOOPBACK_MODE_EXTERNAL);
	}
	else
	{
		errs	+= _setup_apply(tx, tx_index, "Tx Setup", &init, xcvr, 1, SIO4_ISOC_LOOPBACK_MODE_DISABLE);
		errs	+= _setup_apply(rx, rx_index, "Rx Setup", &init, xcvr, 0, SIO4_ISOC_LOOPBACK_MODE_DISABLE);
	}

	return(errs);
}



//*****************************************************************************
static int _txc_to_rxc_test(int tx, int rx)
{
	s32			arg;
	s32			bitmap;
	const char*	err		= NULL;
	int			errs	= 0;
	int			i;
	u32			pstsr	= 0;	// Pin Status Register bitmap
	int			sts;

	for (;;)	// A convenience loop.
	{
		gsc_label("Tx Clock -> Rx Clock");

		// Find out if we can configure TxClk to go high and low.
		sio4_isoc_t_cable_txc(tx, &arg, SIO4_ISOC_ACTION_GET, &err);

		if (err)
		{
			errs++;
			printf("FAIL <---  (sio4_isoc_t_cable_txc(): %s)\n", err);
			break;
		}

		if (arg == -1)
		{
			printf("SKIPPED  (Cable TxClk not configurable)\n");
			break;
		}

		// Find out if we can monitor the RxClk signal.
		errs	= sio4_query(rx, SIO4_QUERY_REG_PSTSR_BITS, &bitmap);

		if (errs)
		{
			printf("FAIL <---  (sio4_query())\n");
			break;
		}

		if ((bitmap & RXCLK) == 0)
		{
			printf("SKIPPED  (Can't read RxClk status)\n");
			break;
		}

		for (i = 0; i <= 5; i++)
		{
			// Drive TxClk low.
			arg	= SIO4_ISOC_CABLE_TXC_OUT_0;
			sio4_isoc_t_cable_txc(tx, &arg, SIO4_ISOC_ACTION_SET, &err);

			if (err)
			{
				errs++;
				printf("FAIL <---  (sio4_isoc_t_cable_txc(): %s)\n", err);
			}

			// Check that RxClk is driven low.
			os_sleep_ms(10);
			sts	= sio4_reg_read(rx, SIO4_GSC_PSTSR, &pstsr);

			if (sts)
			{
				errs++;
				printf("FAIL <---  (sio4_reg_read(): PSTSR)\n");
			}
			else if ((pstsr & RXCLK) != 0)
			{
				errs++;
				printf("FAIL <---  (RxClk did not go low)\n");
			}

			// Drive TxClk high.
			arg	= SIO4_ISOC_CABLE_TXC_OUT_1;
			sio4_isoc_t_cable_txc(tx, &arg, SIO4_ISOC_ACTION_SET, &err);

			if (err)
			{
				errs++;
				printf("FAIL <---  (sio4_isoc_t_cable_txc(): %s)\n", err);
			}

			// Check that RxClk is driven hi.
			os_sleep_ms(10);
			sts	= sio4_reg_read(rx, SIO4_GSC_PSTSR, &pstsr);

			if (sts)
			{
				errs++;
				printf("FAIL <---  (sio4_reg_read(): PSTSR)\n");
			}
			else if ((pstsr & RXCLK) == 0)
			{
				errs++;
				printf("FAIL <---  (RxClk did not go high)\n");
			}
		}

		if (errs == 0)
			printf("PASS\n");

		break;
	}

	return(errs);
}



//*****************************************************************************
static int _txd_to_rxd_test(int tx, int rx)
{
	s32			arg;
	s32			bitmap;
	const char*	err		= NULL;
	int			errs	= 0;
	int			i;
	u32			pstsr	= 0;	// Pin Status Register bitmap
	int			sts;

	for (;;)	// A convenience loop.
	{
		gsc_label("Tx Data -> Rx Data");

		// Find out if we can configure TxD to go high and low.
		sio4_isoc_t_cable_txd(tx, &arg, SIO4_ISOC_ACTION_GET, &err);

		if (err)
		{
			errs++;
			printf("FAIL <---  (sio4_isoc_t_cable_txd(): %s)\n", err);
			break;
		}

		if (arg == -1)
		{
			printf("SKIPPED  (Cable TxD not configurable)\n");
			break;
		}

		// Find out if we can monitor the RxD signal.
		errs	= sio4_query(rx, SIO4_QUERY_REG_PSTSR_BITS, &bitmap);

		if (errs)
		{
			printf("FAIL <---  (sio4_query())\n");
			break;
		}

		if ((bitmap & RXD) == 0)
		{
			printf("SKIPPED  (Can't read RxD status)\n");
			break;
		}

		for (i = 0; i <= 5; i++)
		{
			// Drive TxD low.
			arg	= SIO4_ISOC_CABLE_TXD_OUT_0;
			sio4_isoc_t_cable_txd(tx, &arg, SIO4_ISOC_ACTION_SET, &err);

			if (err)
			{
				errs++;
				printf("FAIL <---  (sio4_isoc_t_cable_txd(): %s)\n", err);
			}

			// Check that RxD is driven low.
			os_sleep_ms(10);
			sts	= sio4_reg_read(rx, SIO4_GSC_PSTSR, &pstsr);

			if (sts)
			{
				errs++;
				printf("FAIL <---  (sio4_reg_read(): PSTSR)\n");
			}
			else if ((pstsr & RXD) != 0)
			{
				errs++;
				printf("FAIL <---  (RxD did not go low)\n");
			}

			// Drive TxD high.
			arg	= SIO4_ISOC_CABLE_TXD_OUT_1;
			sio4_isoc_t_cable_txd(tx, &arg, SIO4_ISOC_ACTION_SET, &err);

			if (err)
			{
				errs++;
				printf("FAIL <---  (sio4_isoc_t_cable_txd(): %s)\n", err);
			}

			// Check that RxD is driven hi.
			os_sleep_ms(10);
			sts	= sio4_reg_read(rx, SIO4_GSC_PSTSR, &pstsr);

			if (sts)
			{
				errs++;
				printf("FAIL <---  (sio4_reg_read(): PSTSR)\n");
			}
			else if ((pstsr & RXD) == 0)
			{
				errs++;
				printf("FAIL <---  (RxD did not go high)\n");
			}
		}

		if (errs == 0)
			printf("PASS\n");

		break;
	}

	return(errs);
}



//*****************************************************************************
static long _tx_data(int fd)
{
	int		idle	= 0;
	int		left;
	int		sent;
	u8*		src;
	long	tx_qty	= 0;

	for (;;)
	{
		if (_pipeline_qty > (_pipeline_limit - (int) sizeof(_tx_buf)))
			break;

		// Write out a single block of data.
		src		= _tx_buf;
		left	= sizeof(_tx_buf);

		for (;;)
		{
			if (left == 0)
				break;

			sent	= sio4_isoc_write(fd, src, left);	// returns >= 0 or -errno

			if (sent < 0)
			{
				printf("FAIL <---  (sio4_isoc_write())\n");
				tx_qty	= -1;
				break;
			}

			if (sent == 0)
			{
				idle++;

				if (idle >= 3)
				{
					// This is 3x the Tx I/O Timeout,
					// which should be 30 seconds.
					printf(	"FAIL <---  (failed to write any data)\n");
					tx_qty	= -1;
					break;
				}

				continue;
			}

			tx_qty	+= sent;
			src		+= sent;
			left	-= sent;
			idle	= 0;

			if (left <= 0)
			{
				_pipeline_qty	+= sizeof(_tx_buf);
				break;
			}
		}

		if (tx_qty < 0)
			break;
	}

	return(tx_qty);
}



//*****************************************************************************
static long _rx_data(int fd)
{
	u8*		dst;
	int		got;
	int		i;
	int		idle	= 0;
	int		left;
	long	rx_qty	= 0;

	for (;;)
	{
		if (_pipeline_qty < sizeof(_rx_buf))
			break;

		// Read in a single block of data.
		dst		= _rx_buf;
		left	= sizeof(_rx_buf);

		for (;;)
		{
			if (left == 0)
				break;

			got	= sio4_isoc_read(fd, dst, left);	// returns >= 0 or -errno

			if (got < 0)
			{
				printf(	"FAIL <---  (sio4_isoc_read())\n");
				rx_qty	= -1;
				break;
			}

			if (got == 0)
			{
				idle++;

				if (idle >= 3)
				{
					// This is 3x the Rx I/O Timeout,
					// which should be 30 seconds.
					printf(	"FAIL <---  (failed to read any data)\n");
					rx_qty	= -1;
					break;
				}

				continue;
			}

			rx_qty	+= got;
			dst		+= got;
			left	-= got;
			idle	= 0;

			if (left <= 0)
			{
				_pipeline_qty	-= sizeof(_rx_buf);

				if (memcmp(_rx_buf, _tx_buf, sizeof(_rx_buf)))
				{
					printf("FAIL <---  (data mismatch)\n");
					rx_qty	= -1;

					for (i = 0; i < sizeof(_rx_buf); i++)
					{
						printf(	"%5d. Tx 0x%02lX, Rx 0x%02lX",
								i,
								(long) _tx_buf[i],
								(long) _rx_buf[i]);

						if (_rx_buf[i] != _tx_buf[i])
							printf(" <--- mismatch");

						printf("\n");
					}
				}

				break;
			}
		}

		break;
	}

	return(rx_qty);
}



//*****************************************************************************
static int _data_transfer_test(int tx, int rx, int errs, int secs)
{
	time_t		end;
	int			i;
	time_t		now;
	long		rx_qty;
	long long	rx_total	= 0;
	s32			size		= 0;
	int			tx_done		= 0;
	long		tx_qty;
	long long	tx_total	= 0;

	for (;;)	// A convenience loop.
	{
		gsc_label("Data Transfer");

		if (errs)
		{
			errs	= 0;
			printf("SKIPPED  (setup errors)\n");
			break;
		}

		// Initialization.

		for (i = 0; i < sizeof(_tx_buf); i++)
			_tx_buf[i]	= (u8) i;

		i		= sio4_query(rx, SIO4_QUERY_FIFO_SIZE_RX, &size);
		size	= (size <= 256) ? 256 : size;

		if (i < 0)
		{
			errs++;
			printf("FAIL <---  (sio4_query())\n");
			break;
		}

		_pipeline_qty	= 0;
		_pipeline_limit	= size;
		end				= time(NULL) + secs;

		for (;;)
		{
			if (tx_done == 0)
			{
				tx_qty	= _tx_data(tx);

				if (tx_qty < 0)
				{
					errs++;
					break;
				}

				tx_total	+= tx_qty;
			}

			rx_qty		= _rx_data(rx);

			if (rx_qty < 0)
			{
				errs++;
				break;
			}

			rx_total	+= rx_qty;
			now			= time(NULL);

			if (now > end)
				tx_done	= 1;

			if ((tx_done) && (_pipeline_qty < sizeof(_rx_buf)))
				break;
		}

		if (errs == 0)
			printf("PASS");

		// Report the accumulated transfer quantities.
		printf(	"  (Tx Total ");
		gsc_label_long_comma(tx_total);
		printf(" bytes, Rx Total ");
		gsc_label_long_comma(rx_total);
		printf(" bytes)\n");
		break;
	}

	return(errs);
}



/*****************************************************************************
*
*	Function:	io_test
*
*	Purpose:
*
*		Perform a data I/O write/read test.
*
*	Arguments:
*
*		tx_index	The device index of the Tx channel.
*
*		tx			The Tx device handle.
*
*		rx_index	The device index of the Rx channel.
*
*		rx			The Rx device handle.
*
*	Returned:
*
*		>= 0		The number of errors encounterred.
*
*****************************************************************************/

int io_test(int tx_index, int tx, int rx_index, int rx)
{
	int		errs	= 0;
	mp_t	rx_xcvr;
	int		secs	= 30;	// Perform the data transfer for this long.
	mp_t	tx_xcvr;
	s32		xcvr	= 0;

	for (;;)
	{
		gsc_label("Data I/O Test");
		printf("\n");
		gsc_label_level_inc();

		_connection_type(tx_index, rx_index);
		errs	+= _verify_model(tx, "Tx Model Type");
		errs	+= _verify_model(rx, "Rx Model Type");

		if (errs == 0)
		{
			errs	+= _xcvr_support(tx, tx_index, "Tx Transceiver Support", &tx_xcvr);
			errs	+= _xcvr_support(rx, rx_index, "Rx Transceiver Support", &rx_xcvr);
			errs	+= _verify_xcvr(&tx_xcvr, &rx_xcvr, &xcvr);
		}

		if (errs == 0)
		{
			gsc_label("Signal Test");
			printf("\n");
			gsc_label_level_inc();

			errs	+= _setup(tx_index, tx, rx_index, rx, xcvr);
			errs	+= _txc_to_rxc_test(tx, rx);
			errs	+= _txd_to_rxd_test(tx, rx);

			gsc_label_level_dec();
		}

		if (errs == 0)
		{
			gsc_label("Data Transfer Test");
			printf("(%ld secs)\n", (long) secs);
			gsc_label_level_inc();

			errs	+= _setup(tx_index, tx, rx_index, rx, xcvr);
			errs	+= _data_transfer_test(tx, rx, errs, secs);

			gsc_label_level_dec();
		}

		gsc_label_level_dec();
		break;
	}

	return(errs);
}


