// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_2.x.x_GSC_DN/irq/irq.c $
// $Rev: 33936 $
// $Date: 2015-10-07 15:24:29 -0500 (Wed, 07 Oct 2015) $

#include "main.h"



// data types *****************************************************************

typedef struct
{
	const char*	name;
	s32			main;
	s32			gsc;
	s32			alt;
	s32			io;
	int			(*func)(int fd, int supported);
} irq_test_t;



// variables ******************************************************************

static	char	_buf_err[256];
static	int		_fd;
static	int		_thread_done;

static const irq_test_t	_tests[]	=
{
	// Main interrupts
	// name						main									gsc	alt	io	func
	{ "PCI",					GSC_WAIT_MAIN_PCI,						0,	0,	0,	main_pci_irq_test			},
	{ "DMA0/DMA1",				GSC_WAIT_MAIN_DMA0 | GSC_WAIT_MAIN_DMA1,0,	0,	0,	main_dma_irq_test			},
	{ "GSC",					GSC_WAIT_MAIN_GSC,						0,	0,	0,	main_gsc_irq_test			},

	// GSC Interrupts
	// name						main	gsc									alt	io	func
	{ "Rx FIFO Empty",			0,		SIO4_WAIT_GSC_RX_FIFO_E,			0,	0,	rx_fifo_empty_test			},
	{ "Rx FIFO Almost Full",	0,		SIO4_WAIT_GSC_RX_FIFO_AF,			0,	0,	rx_fifo_almost_full_test	},
	{ "Rx FIFO Full",			0,		SIO4_WAIT_GSC_RX_FIFO_F,			0,	0,	rx_fifo_full_test			},
	{ "Rx Envelope",			0,		SIO4_WAIT_GSC_RX_ENV,				0,	0,	rx_envelope_test			},
	{ "Rx Spare",				0,		SIO4_WAIT_GSC_RX_SPARE,				0,	0,	rx_spare_test				},
	{ "SYNC Detected",			0,		SIO4_WAIT_GSC_SYNC_BYTE,			0,	0,	sync_detected_test			},
	{ "Tx FIFO Empty",			0,		SIO4_WAIT_GSC_TX_FIFO_E,			0,	0,	tx_fifo_empty_test			},
	{ "Tx FIFO Almost Empty",	0,		SIO4_WAIT_GSC_TX_FIFO_AE,			0,	0,	tx_fifo_almost_empty_test	},
	{ "Tx FIFO Full",			0,		SIO4_WAIT_GSC_TX_FIFO_F,			0,	0,	tx_fifo_full_test			},
	{ "USC",					0,		SIO4_WAIT_GSC_USC,					0,	0,	usc_test					},

	{ NULL }
};

static const irq_test_t	_usc_tests[]	=
{
	// USC Interrupts
	// name						main	gsc	alt									io	func
	{ "Rx Exited Hunt",			0,		0,	SIO4_WAIT_USC_RX_EXITED_HUNT,		0,	usc_rx_exited_hunt_test		},
	{ "Rx Idle Received",		0,		0,	SIO4_WAIT_USC_RX_IDLE_RECEIVED,		0,	usc_rx_idle_received_test	},
	{ "Rx Break/Abort",			0,		0,	SIO4_WAIT_USC_RX_BREAK_ABORT,		0,	usc_rx_break_abort_test		},
	{ "Rx Bound",				0,		0,	SIO4_WAIT_USC_RX_BOUND,				0,	usc_rx_bound_test			},
	{ "Rx Abort/Par Error",		0,		0,	SIO4_WAIT_USC_RX_ABORT_PAR_ERROR,	0,	usc_rx_abort_pe_test		},
	{ "Rx Overrun",				0,		0,	SIO4_WAIT_USC_RX_OVERRUN,			0,	usc_rx_overrun_test			},
	{ "Rx Data",				0,		0,	SIO4_WAIT_USC_RX_DATA,				0,	usc_rx_data_test			},
	{ "Tx Preamble Sent",		0,		0,	SIO4_WAIT_USC_TX_PREAMBLE_SENT,		0,	usc_tx_preamble_sent_test	},
	{ "Tx Idle Sent",			0,		0,	SIO4_WAIT_USC_TX_IDLE_SENT,			0,	usc_tx_idle_sent_test		},
	{ "Tx Abort Sent",			0,		0,	SIO4_WAIT_USC_TX_ABORT_SENT,		0,	usc_tx_abort_sent_test		},
	{ "Tx EOF/EOM Sent",		0,		0,	SIO4_WAIT_USC_TX_END_SENT,			0,	usc_tx_eof_eom_sent_test	},
	{ "Tx CRC Sent",			0,		0,	SIO4_WAIT_USC_TX_CRC_SENT,			0,	usc_tx_crc_sent_test		},
	{ "Tx Underrun",			0,		0,	SIO4_WAIT_USC_TX_UNDERRUN,			0,	usc_tx_underrun_test		},
	{ "Tx Data",				0,		0,	SIO4_WAIT_USC_TX_DATA,				0,	usc_tx_data_test			},
	{ "I/O Pin RxC Fall",		0,		0,	SIO4_WAIT_USC_IOP_RXC_FALL,			0,	usc_rxc_down_test			},
	{ "I/O Pin RxC Rise",		0,		0,	SIO4_WAIT_USC_IOP_RXC_RISE,			0,	usc_rxc_up_test				},
	{ "I/O Pin TxC Fall",		0,		0,	SIO4_WAIT_USC_IOP_TXC_FALL,			0,	usc_txc_down_test			},
	{ "I/O Pin TxC Rise",		0,		0,	SIO4_WAIT_USC_IOP_TXC_RISE,			0,	usc_txc_up_test				},
	{ "I/O Pin RxReq Fall",		0,		0,	SIO4_WAIT_USC_IOP_RXREQ_FALL,		0,	usc_rxreq_down_test			},
	{ "I/O Pin RxReq Rise",		0,		0,	SIO4_WAIT_USC_IOP_RXREQ_RISE,		0,	usc_rxreq_up_test			},
	{ "I/O Pin TxReq Fall",		0,		0,	SIO4_WAIT_USC_IOP_TXREQ_FALL,		0,	usc_txreq_down_test			},
	{ "I/O Pin TxReq Rise",		0,		0,	SIO4_WAIT_USC_IOP_TXREQ_RISE,		0,	usc_txreq_up_test			},
	{ "I/O Pin DCD Rise",		0,		0,	SIO4_WAIT_USC_IOP_DCD_FALL,			0,	usc_dcd_down_test			},
	{ "I/O Pin DCD Fall",		0,		0,	SIO4_WAIT_USC_IOP_DCD_RISE,			0,	usc_dcd_up_test				},
	{ "I/O Pin CTS Fall",		0,		0,	SIO4_WAIT_USC_IOP_CTS_FALL,			0,	usc_cts_down_test			},
	{ "I/O Pin CTS Rise",		0,		0,	SIO4_WAIT_USC_IOP_CTS_RISE,			0,	usc_cts_up_test				},
	{ "Misc RCC Underrun",		0,		0,	SIO4_WAIT_USC_MISC_RCC_UNDERRUN,	0,	usc_rcc_underrun_test		},
	{ "Misc DPLL Desync",		0,		0,	SIO4_WAIT_USC_MISC_DPLL_DESYNC,		0,	usc_dpll_desync_test		},
	{ "Misc BRG1 Zero",			0,		0,	SIO4_WAIT_USC_MISC_BRG1_ZERO,		0,	usc_brg1_zero_test			},
	{ "Misc BRG0 Zero",			0,		0,	SIO4_WAIT_USC_MISC_BRG0_ZERO,		0,	usc_brg0_zero_test			},

	{ NULL }
};



//*****************************************************************************
static int _wait_thread(void* arg)
{
	int			errs	= 0;
	irq_test_t*	ptr	= (void*) arg;
	gsc_wait_t	wait;

	memset(&wait, 0, sizeof(wait));
	wait.main		= ptr->main;
	wait.gsc		= ptr->gsc;
	wait.alt		= ptr->alt;
	wait.io			= ptr->io;
	wait.timeout_ms	= 15000;
	errs			= sio4_wait_event(_fd, &wait);

	if (errs)
	{
		strcpy(_buf_err, "FAIL <---  (Wait request failed.)");
	}
	else if ((wait.flags & GSC_WAIT_FLAG_DONE) == 0)
	{
		sprintf(_buf_err,
				"FAIL <---  (Wait request did not complete. Got 0x%lX.)",
				(long) wait.flags);
	}

	_thread_done	= 1;
	return(errs);
}



//*****************************************************************************
static int _check_for_support(int fd, const irq_test_t* ptr, int* supported)
{
	int	errs	= 0;
	int	sts;

	sts	= ptr->func(fd, 1);

	switch (sts)
	{
		default:

				// errors
				errs++;
				supported[0]	= 0;
				break;

		case STATUS_NOT_IMPLEMENTED:

				printf("Skipped  (Not implemented.)\n");
				supported[0]	= 0;
				break;

		case STATUS_NOT_SUPPORTED:

				printf("Skipped  (Not supported on this board.)\n");
				supported[0]	= 0;
				break;

		case STATUS_SUPPORTED:

				supported[0]	= 1;
				break;
	}

	return(errs);
}



//*****************************************************************************
static int _wait_for_waiting(int fd, const irq_test_t* ptr)
{
	int			errs	= 0;
	int			i;
	int			limit	= 1000;
	gsc_wait_t	wait;

	memset(&wait, 0, sizeof(wait));
	wait.main	= ptr->main;
	wait.gsc	= ptr->gsc;
	wait.alt	= ptr->alt;
	wait.io		= ptr->io;

	for (i = 0;; i++)
	{
		errs	= sio4_wait_status(fd, &wait);

		if (errs)
		{
			printf("FAIL <---  (Wait status request failed.)\n");
			break;
		}
		else if (_thread_done)
		{
			printf("FAIL <---  (Thread completed prematurely.)\n");
			errs++;
			break;
		}
		else if (i > limit)
		{
			printf("FAIL <---  (Excessive delay for thread to wait.)\n");
			errs++;
			break;
		}
		else if (wait.count)
		{
			break;
		}
		else
		{
			os_sleep_ms(10);
		}
	}

	return(errs);
}



//*****************************************************************************
static int _wait_for_running(int fd, const irq_test_t* ptr)
{
	int			errs	= 0;
	int			i;
	int			limit	= 1000;
	gsc_wait_t	wait;

	memset(&wait, 0, sizeof(wait));
	wait.main	= ptr->main;
	wait.gsc	= ptr->gsc;
	wait.alt	= ptr->alt;
	wait.io		= ptr->io;

	for (i = 0;; i++)
	{
		errs	= sio4_wait_status(fd, &wait);

		if (errs)
		{
			printf("FAIL <---  (Wait status request failed.)\n");
			break;
		}
		else if (i > limit)
		{
			printf("FAIL <---  (Interrupt not received. Excessive delay for thread to run.)\n");
			errs++;
			break;
		}
		else if (wait.count == 0)
		{
			break;
		}
		else
		{
			os_sleep_ms(10);
		}
	}

	return(errs);
}



//*****************************************************************************
static int _wait_for_done(void)
{
	int	errs	= 0;
	int	i;

	for (i = 0;; i++)
	{
		if (_thread_done)
			break;

		if (i > 1000)
		{
			printf("FAIL <---  (Excessive delay for thread to complete.)\n");
			errs++;
			break;
		}
		else
		{
			os_sleep_ms(10);
		}
	}

	return(errs);
}



//*****************************************************************************
static int _test_irq(int fd, const irq_test_t* ptr)
{
	s32			arg;
	int			err;
	int			errs		= 0;
	int			status;
	int			supported;
	os_thread_t	thread;
	int			thread_ok	= 0;
	gsc_wait_t	wait;

	gsc_label(ptr->name);
	_thread_done	= 0;
	_buf_err[0]		= 0;

	for (;;)	// A cenvenience loop.
	{
		// Check to see if the interrupt is supported.
		errs	+= _check_for_support(fd, ptr, &supported);

		if (errs)
			break;

		if (supported == 0)
			break;

		// Initialize the channel.
		arg		= 1;
		status	= sio4_ioctl(fd, SIO4_IOCTL_INITIALIZE, &arg);

		if (status < 0)
		{
			printf("FAIL <---  (Initialization failed)\n");
			errs++;
			break;
		}

		// Create a thread to wait on the interrupt.
		_fd	= fd;
		err	= os_thread_create(&thread, "wait thtread", _wait_thread, (void*) ptr);

		if (err)
		{
			errs++;
			break;
		}

		thread_ok	= 1;

		// Wait for the thread to become "waiting".
		errs	= _wait_for_waiting(fd, ptr);

		if (errs)
			break;

		// Initiate the interrupt.
		err	= ptr->func(fd, 0);

		if (err)
		{
			// There were errors.
			errs++;
			break;
		}

		// Wait for the thread to become "running".
		errs	= _wait_for_running(fd, ptr);

		if (errs)
			break;

		// Resume the waiting thread, just in case.
		memset(&wait, 0, sizeof(wait));
		wait.main	= 0xFFFFFFFF;
		wait.gsc	= 0xFFFFFFFF;
		wait.alt	= 0xFFFFFFFF;
		wait.io		= 0xFFFFFFFF;
		err			= sio4_wait_cancel(fd, &wait);

		// Wait for the thread to complete.
		errs	= _wait_for_done();

		if (errs)
			break;

		if (_buf_err[0])
		{
			printf("%s\n", _buf_err);
			errs++;
			break;
		}

		break;
	}

	if (thread_ok)
	{
		err	= os_thread_destroy(&thread);

		if (err)
		{
			printf("FAIL <---  (Thread shutdown failed)\n");
			errs++;
		}
	}

	if ((supported) && (errs == 0))
		printf("PASS\n");

	return(errs);
}



//*****************************************************************************
int irq_tests(int fd)
{
	int	err;
	int	errs	= 0;
	int	i;
	s32	z16c30	= 0;

	for (i = 0; _tests[i].name; i++)
		errs	+= _test_irq(fd, &_tests[i]);

	err		= sio4_query(fd, SIO4_QUERY_MODEL_Z16C30, &z16c30);
	errs	+= err;

	if ((err == 0) && (z16c30))
	{
		gsc_label_level_inc();

		for (i = 0; _usc_tests[i].name; i++)
			errs	+= _test_irq(fd, &_usc_tests[i]);

		gsc_label_level_dec();
	}

	return(errs);
}


