// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_2.x.x_GSC_DN/hdlc/lib/tx_wait.c $
// $Rev: 33920 $
// $Date: 2015-10-07 14:47:33 -0500 (Wed, 07 Oct 2015) $

#include "main.h"



//*****************************************************************************
static int _args_validate(int fd, int timeout, int* done, device_t** dev)
{
	int	ret	= -EINVAL;

	for (;;)	// A convenience loop.
	{
		// Validate the "dev" argument.

		if (dev == NULL)
			break;

		dev[0]	= lib_fd_find_inc(fd);

		if (dev[0] == NULL)
		{
			ret	= -ENODEV;
			break;
		}

		// Validate the "timeout" argument.

		if ((timeout < 1) || (timeout > 3600))
			break;

		// Validate the "done" argument.

		if ((done == NULL) || (done[0]))
			break;

		ret	= 0;
		break;
	}

	return(ret);
}



//*****************************************************************************
static int _env_validate(device_t* dev)
{
	int	ret	= 0;

	ret	|= tx_abort_active(dev);
	ret	|= tx_flush_active(dev);
	ret	|= tx_frame_active(dev);
	ret	|= tx_wait_active(dev);
	ret	= ret ? -EBUSY : 0;
	return(ret);
}



//*****************************************************************************
static int _tx_wait(device_t* dev, int timeout, int* done)
{
	s32			arg;
	long		end_ms;
	long		now_ms;
	gsc_reg_t	reg;
	int			ret;

	end_ms	= os_time_delta_ms();
	end_ms	+= 1000L * timeout;

	for (;;)
	{
		// Check for a timeout condition.
		now_ms	= os_time_delta_ms();

		if (now_ms >= end_ms)
		{
			ret	= -ETIMEDOUT;
			break;
		}

		// Make sure the main Tx FIFO is empty.
		arg	= -1;
		ret	= sio4_hdlc_ioctl(dev->fd, SIO4_IOCTL_TX_FIFO_STATUS, &arg);

		if (ret)
			break;

		if (arg != SIO4_FIFO_STATUS_EMPTY)
		{
			// Wait before checking again.
			os_sleep_ms(10);
			continue;
		}

		// Make sure the USC transmitter is empty and done.
		reg.reg		= SIO4_USC_TCSR;
		reg.value	= 0xDEADBEEF;
		reg.mask	= 0;	// unused;
		ret			= sio4_hdlc_ioctl(dev->fd, SIO4_IOCTL_REG_READ, &reg);

		if (ret)
			break;

		if (reg.value & D1)		// Tx Under: 1 = yes, 0 = no
		{
			if (done)
				done[0]	= 1;

			break;				// The Tx FIFO is empty.
		}

		// Wait before checking again.
		os_sleep_ms(10);
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	sio4_hdlc_tx_wait
*
*	Purpose:
*
*		Wait til the Tx side has finished sending out all data.
*
*	Arguments:
*
*		fd		The file descriptor for the SIO4 channel to access.
*
*		timeout	Amount of time to wait for completion?
*
*		done	The results are reported here. It is zero if the transmitter
*				was not done within the time limit and non-zero if it was.
*
*	Returned:
*
*		0		All went well.
*		-errno	There was a problem.
*
******************************************************************************/

int sio4_hdlc_tx_wait(int fd, int timeout, int* done)
{
	device_t*	dev		= NULL;
	int			ret;

	for (;;)	// A convenience loop.
	{
		// Make sure the library is initialized.
		ret	= sio4_hdlc_lib_init();

		if (ret)
			break;

		// Validate the arguments.
		ret	= _args_validate(fd, timeout, done, &dev);

		if (ret)
			break;

		// Gain exclusive access to the device.
		ret	= os_sem_lock(&dev->sem);

		if (ret)
			break;

		// Validate the state.
		ret	= _env_validate(dev);

		if (ret)
		{
			os_sem_unlock(&dev->sem);
			break;
		}

		// Gain exclusive access to the Tx stream.
		ret	= os_sem_lock(&dev->tx.sem);
		os_sem_unlock(&dev->sem);

		if (ret)
			break;

		// Record our presence.
		dev->tx.tx_wait	= 1;
		os_sem_unlock(&dev->tx.sem);

		// Perform the operation.
		ret	= _tx_wait(dev, timeout, done);

		// Cleanup.
		dev->tx.tx_frame	= 0;
		break;
	}

	if (dev)
		lib_dev_dec(dev);

	return(ret);
}



//*****************************************************************************
int tx_wait_active(device_t* dev)
{
	int	ret;

	ret	= dev->tx.tx_wait ? 1 : 0;
	return(ret);
}


