// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_2.x.x_GSC_DN/hdlc/lib/tx_frame.c $
// $Rev: 33920 $
// $Date: 2015-10-07 14:47:33 -0500 (Wed, 07 Oct 2015) $

#include "main.h"



//*****************************************************************************
static int _args_validate(int fd, const sio4_hdlc_tx_frame_t* tx, device_t** dev)
{
	int	ret	= -EINVAL;

	for (;;)	// A convenience loop.
	{
		// Validate the "dev" argument.

		if (dev == NULL)
			break;

		dev[0]	= lib_fd_find_inc(fd);

		if (dev[0] == NULL)
		{
			ret	= -ENODEV;
			break;
		}

		// Validate the "tx" argument.

		if (tx == NULL)
			break;

		if (tx->buf == NULL)
			break;

		if ((tx->count < 0) || (tx->count > 0xFFFF))
			break;

		if (tx->sent)
			break;

		if (tx->flags)
			break;

		switch (tx->last)
		{
			default:

				break;

			case SIO4_HDLC_TX_LAST_CHAR_LEN_1:
			case SIO4_HDLC_TX_LAST_CHAR_LEN_2:
			case SIO4_HDLC_TX_LAST_CHAR_LEN_3:
			case SIO4_HDLC_TX_LAST_CHAR_LEN_4:
			case SIO4_HDLC_TX_LAST_CHAR_LEN_5:
			case SIO4_HDLC_TX_LAST_CHAR_LEN_6:
			case SIO4_HDLC_TX_LAST_CHAR_LEN_7:
			case SIO4_HDLC_TX_LAST_CHAR_LEN_8:

				ret	= 0;
				break;
		}

		break;
	}

	return(ret);
}



//*****************************************************************************
static int _env_validate(device_t* dev)
{
	int	ret	= 0;

	ret	|= tx_abort_active(dev);
	ret	|= tx_flush_active(dev);
	ret	|= tx_frame_active(dev);
	ret	|= tx_wait_active(dev);
	ret	= ret ? -EBUSY : 0;
	return(ret);
}



//*****************************************************************************
static int _setup(device_t* dev)
{
	s32	arg;
	int	ret		= 0;
	int	sts;

	if (dev->tx.stream_init == 0)
	{
		arg	= SIO4_USC_CTRL_BLOCK_2_WORD;
		sts	= sio4_hdlc_ioctl(dev->fd, SIO4_IOCTL_USC_TX_CTRL_BLOCK, &arg);
		ret	= ret ? ret : sts;

		arg	= SIO4_USC_TX_WAIT_UNDERRUN_YES;
		sts	= sio4_hdlc_ioctl(dev->fd, SIO4_IOCTL_USC_TX_WAIT_UNDERRUN, &arg);
		ret	= ret ? ret : sts;

		dev->tx.stream_init	= 1;
	}

	if (dev->tx.tx_underrun)
	{
		arg	= SIO4_USC_TX_CMD_SEND_FRM_MSG;
		sts	= sio4_hdlc_ioctl(dev->fd, SIO4_IOCTL_USC_TX_CMD, &arg);
		ret	= ret ? ret : sts;
		dev->tx.tx_underrun	= 0;
	}

	return(ret);
}



//*****************************************************************************
static int _tx_control_block(device_t* dev, const sio4_hdlc_tx_frame_t* tx)
{
	u32	cmr;
	u8	data[4];
	int	ret;
	u32	txcb;

	// Compose the 32-bit Tx Control Block value.
	txcb	= (u32) tx->count << 16;

	ret		= sio4_reg_read(dev->fd, SIO4_USC_CMR, &cmr);
	txcb	|= cmr & 0xF000;

	txcb	|= (u32) tx->last << 2;

	// Comvert the 32-bit value to an array of bytes;
	data[0]	= (u8) 0xFF & (txcb >> (0 * 8));
	data[1]	= (u8) 0xFF & (txcb >> (1 * 8));
	data[2]	= (u8) 0xFF & (txcb >> (2 * 8));
	data[3]	= (u8) 0xFF & (txcb >> (3 * 8));

	// Write the Tx Control Block to the channel.

	if (ret == 0)
	{
		ret	= sio4_write(dev->fd, data, 4);

		if ((ret >= 0) && (ret != 4))
			ret	= -EIO;
	}

	return(ret);
}



//*****************************************************************************
static int _tx_write(device_t* dev, sio4_hdlc_tx_frame_t* tx)
{
	s32	arg;
	int	ret;
	int	sts;
	u32	tcsr;

	// Write the data and check the results.
	sts	= sio4_write(dev->fd, tx->buf, tx->count);
	ret	= sts;

	if (ret > 0)
		tx->sent	= sts;

	if (sts == tx->count)
		tx->flags	= SIO4_HDLC_TX_FLAG_EOF;

	// Check for a Tx Overrun condition.
	arg	= SIO4_FIFO_OVERRUN_TEST;
	sts	= sio4_hdlc_ioctl(dev->fd, SIO4_IOCTL_TX_FIFO_OVERRUN, &arg);
	ret	= ret ? ret : sts;

	if (arg == SIO4_FIFO_OVERRUN_YES)
	{
		// Report the status.
		tx->flags	|= SIO4_HDLC_TX_FLAG_OVERRUN;

		// Clear the status.
		arg	= SIO4_FIFO_OVERRUN_CLEAR;
		sts	= sio4_hdlc_ioctl(dev->fd, SIO4_IOCTL_TX_FIFO_OVERRUN, &arg);
		ret	= ret ? ret : sts;
	}

	// Check for a USC Tx Underrun condition.
	sts	= sio4_reg_read(dev->fd, SIO4_USC_TCSR, &tcsr);
	ret	= ret ? ret : sts;

	if (tcsr & D1)	// TxUnder
	{
		// Record the status for the user.
		tx->flags	|= SIO4_HDLC_TX_FLAG_UNDERRUN;

		// Clear the TxUnder bit.
		tcsr		= (tcsr & 0x0F00) | D1;
		sts			= sio4_reg_write(dev->fd, SIO4_USC_TCSR, tcsr);
		ret			= ret ? ret : sts;

		// Record the condition so we can deal with it.
		dev->tx.tx_underrun	= 1;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	sio4_hdlc_tx_frame
*
*	Purpose:
*
*		Send out a frame of data.
*
*	Arguments:
*
*		fd		The file descriptor for the SIO4 channel to access.
*
*		tx		The structure describing the frame to send.
*
*	Returned:
*
*		0		All went well.
*		-errno	There was a problem.
*
******************************************************************************/

int sio4_hdlc_tx_frame(int fd, sio4_hdlc_tx_frame_t* tx)
{
	device_t*	dev		= NULL;
	int			ret;
	int			sts;

	for (;;)	// A convenience loop.
	{
		// Make sure the library is initialized.
		ret	= sio4_hdlc_lib_init();

		if (ret)
			break;

		// Validate the arguments.
		ret	= _args_validate(fd, tx, &dev);

		if (ret)
			break;

		// Gain exclusive access to the device.
		ret	= os_sem_lock(&dev->sem);

		if (ret)
			break;

		// Validate the state.
		ret	= _env_validate(dev);

		if (ret)
		{
			os_sem_unlock(&dev->sem);
			break;
		}

		// Gain exclusive access to the Tx stream.
		ret	= os_sem_lock(&dev->tx.sem);
		os_sem_unlock(&dev->sem);

		if (ret)
			break;

		// Record our presence.
		dev->tx.tx_frame	= 1;
		os_sem_unlock(&dev->tx.sem);

		// Perform the operation.
		sts	= _setup(dev);
		ret	= ret ? ret : sts;

		sts	= _tx_control_block(dev, tx);
		ret	= ret ? ret : sts;

		sts	= _tx_write(dev, tx);
		ret	= ret ? ret : sts;

		if (ret)
			tx->flags	|= SIO4_HDLC_TX_FLAG_ERROR;

		// Cleanup.
		dev->tx.tx_frame	= 0;
		break;
	}

	if (dev)
		lib_dev_dec(dev);

	return(ret);
}



//*****************************************************************************
int tx_frame_active(device_t* dev)
{
	int	ret;

	ret	= dev->tx.tx_frame ? 1 : 0;
	return(ret);
}


