// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_2.x.x_GSC_DN/hdlc/lib/rx_frame.c $
// $Rev: 33920 $
// $Date: 2015-10-07 14:47:33 -0500 (Wed, 07 Oct 2015) $

#include "main.h"



// data types *****************************************************************

typedef struct
{
	u16	size;
	u32	flags;
} hdlc_rcc_t;

typedef struct
{
	u16	rcsr;
} hdlc_rsb_t;



//*****************************************************************************
static int _args_validate(int fd, const sio4_hdlc_rx_frame_t* rx, device_t** dev)
{
	int	ret	= -EINVAL;

	for (;;)	// A convenience loop.
	{
		// Validate the "dev" argument.

		if (dev == NULL)
			break;

		dev[0]	= lib_fd_find_inc(fd);

		if (dev[0] == NULL)
		{
			ret	= -ENODEV;
			break;
		}

		// Validate the "rx" argument.

		if (rx == NULL)
			break;

		if (rx->buf == NULL)
			break;

		if (rx->size < 1)
			break;

		if (rx->rcvd)
			break;

		if (rx->flags)
			break;

		if (rx->last)
			break;

		ret	= 0;
		break;
	}

	return(ret);
}



//*****************************************************************************
static int _env_validate(device_t* dev)
{
	int	ret	= 0;

	ret	|= rx_flush_active(dev);
	ret	|= rx_frame_active(dev);
	ret	= ret ? -EBUSY : 0;
	return(ret);
}



//*****************************************************************************
static int _rx_read_data_buf(device_t* dev, void* dst, s32 qty)
{
	int	ret;

	// All arguments have already been validated.
	ret	= sio4_read(dev->fd, dst, qty);

	if (ret >= 0)
	{
		dev->rx.put	+= qty;

		if (ret == qty)
			ret	= 0;
		else
			ret	= -EIO;
	}

	return(ret);
}



//*****************************************************************************
static int _rx_read_data(device_t* dev)
{
	s32		bot;	// free buffer space below GET
	s32		data;	// Amount of data available for retrieval.
	void*	dst;
	s32		get;	// relative offset of where to GET data
	s32		put;	// relative offset of where to PUT data
	int		ret;
	s32		space;	// Free space in the receive buffer.
	int		sts;
	s32		top;	// free buffer space above PUT

	for (;;)	// A convenience loop.
	{
		// See how much data is available for retrieval.
		data	= -1;
		ret		= sio4_hdlc_ioctl(dev->fd, SIO4_IOCTL_RX_FIFO_FILL_LEVEL, &data);

		if ((ret) || (data <= 0))
			break;

		// See how much free space we have, and how it is arranged.
		put	= dev->rx.put % sizeof(dev->rx.buf);
		get	= dev->rx.get % sizeof(dev->rx.buf);

		if (put > get)
		{
			// Free space wraps from the top to the buffer of the buffer.
			top	= sizeof(dev->rx.buf) - put;
			bot	= get;
		}
		else
		{
			// Free space is in a single, contiguous block.
			top	= get - put;
			bot	= 0;
		}

		// See how much data we can store.
		space	= dev->rx.put - dev->rx.get;	// buffer data
		space	= sizeof(dev->rx.buf) - space;	// buffer free space

		if (data > space)
		{
			// We'll fill up our data buffer.
		}
		else if (top >= data)
		{
			// We'll put data only in the upper buffer area.
			top	= data;
			bot	= 0;
		}
		else
		{
			// We'll fill up the top buffer area.
			data	-= top;

			if (bot > data)
			{
				// We'll put some data in the bottom buffer area.
				bot	= data;
			}
		}

		// Retrieve data into the buffer.

		if (top)
		{
			dst	= (void*) ((long) dev->rx.buf + put);
			ret	= _rx_read_data_buf(dev, dst, top);
		}

		if (bot)
		{
			dst	= dev->rx.buf;
			sts	= _rx_read_data_buf(dev, dst, bot);
			ret	= ret ? ret : sts;
		}

		break;
	}

	return(ret);
}



//*****************************************************************************
static int _rx_read_rcc(device_t* dev, hdlc_rcc_t* rcc)
{
	s32		arg;
	long	end_ms;
	long	now_ms;
	int		ret		= 0;
	s32		timeout	= -1;

	for (;;)	// A convenience loop.
	{
		ret	= sio4_hdlc_ioctl(dev->fd, SIO4_IOCTL_RX_IO_TIMEOUT, &timeout);

		if (ret)
			break;

		// Loop while trying to read an rcc record.
		end_ms	= os_time_delta_ms();
		end_ms	+= 1000L * timeout;

		for (;;)
		{
			// Try to read an rcc record.
			arg	= 2;	// Wait this number of milleseconds for a record.
			ret	= sio4_hdlc_ioctl(dev->fd, SIO4_IOCTL_RCC_SW_FIFO_READ, &arg);

			if (ret == -ETIMEDOUT)
			{
				arg	= 0;
				ret	= 0;
			}
			else if (ret)
			{
				break;
			}

			if (arg & SIO4_RCC_SW_FIFO_READ_FRAME_END)
			{
				rcc->size	= arg & 0x0000FFFF;
				rcc->flags	= arg & 0xFFFF0000;
				break;
			}

			// Check for a timeout condition.
			now_ms	= os_time_delta_ms();

			if (now_ms >= end_ms)
			{
				ret	= -ETIMEDOUT;
				break;
			}

			// Try to read some data.
			ret	= _rx_read_data(dev);

			if (ret)
				break;
		}

		break;
	}

	return(ret);
}



//*****************************************************************************
static int _rx_read_local_data(device_t* dev, sio4_hdlc_rx_frame_t* rx)
{
	s32		bytes;
	char*	dst;
	s32		get;
	s32		put;
	s32		space;
	void*	src;
	s32		top;

	bytes	= dev->rx.put - dev->rx.get;
	space	= rx->size;

	if (bytes > space)
	{
		dev->rx.remainder	= bytes - space;
		bytes				= space;
	}

	put	= dev->rx.put % sizeof(dev->rx.buf);
	get	= dev->rx.get % sizeof(dev->rx.buf);
	src	= (void*) ((long) dev->rx.buf + get);
	dst	= rx->buf;

	if (get < put)
	{
		memcpy(dst, src, bytes);
		dev->rx.get	+= bytes;
		rx->rcvd	= bytes;
	}
	else
	{
		top	= sizeof(dev->rx.buf) - get;

		if (top)
		{
			memcpy(dst, src, top);
			dev->rx.get	+= top;
			rx->rcvd	= top;
			bytes		-= top;
		}

		if (bytes)
		{
			dst	+= top;
			memcpy(dst, src, bytes);
			dev->rx.get	+= bytes;
			rx->rcvd	+= bytes;
		}
	}

	return(0);
}



//*****************************************************************************
static int _rx_read_rsb(device_t* dev, hdlc_rsb_t* rsb)
{
	u8						data[2]	= { 0, 0 };
	int						ret;
	sio4_hdlc_rx_frame_t	rx;

	ret	= _rx_read_data(dev);

	if (ret == 0)
	{
		memset(&rx, 0, sizeof(rx));
		rx.buf	= data;
		rx.size	= sizeof(data);
		ret		= _rx_read_local_data(dev, &rx);
	}

	if (ret)
		rsb->rcsr	= D6 | D7 | D12 | D13;
	else
		rsb->rcsr	= (((u16) data[0]) << 8) | data[1];

	return(ret);
}



//*****************************************************************************
static int _rx_read_status(
	device_t*				dev,
	sio4_hdlc_rx_frame_t*	rx,
	hdlc_rcc_t*				rcc,
	hdlc_rsb_t*				rsb)
{
	s32			arg;
	s32			qa;			// Queue Abort?
	gsc_reg_t	reg;
	int			ret	= 0;
	int			sts;

	// Are we returning data to the caller?

	if (rx->rcvd)
		rx->flags	|= SIO4_HDLC_RX_FLAG_DATA;

	// Check for an Abort/Parity Error.
	qa	= -1;
	ret	= sio4_hdlc_ioctl(dev->fd, SIO4_IOCTL_USC_RX_QUEUE_ABORT, &qa);

	if ((rsb->rcsr & D2)	||
		(rcc->flags & SIO4_RCC_SW_FIFO_READ_ABORT_PE))
	{
		if (qa == SIO4_USC_RX_QUEUE_ABORT_YES)
			rx->flags	|= SIO4_HDLC_RX_FLAG_ABORT;
		else
			rx->flags	|= SIO4_HDLC_RX_FLAG_PARITY;
	}

	if (rcc->flags & SIO4_RCC_SW_FIFO_READ_BREAK_ABRT)
		rx->flags	|= SIO4_HDLC_RX_FLAG_ABORT;

	// Check for a data overrun.

	if ((rsb->rcsr & D1)	||
		(rcc->flags & SIO4_RCC_SW_FIFO_READ_OVER_DATA))
	{
		rx->flags	|= SIO4_HDLC_RX_FLAG_DATA_OVERRUN;
	}
	else
	{
		arg	= SIO4_FIFO_OVERRUN_TEST;
		sts	= sio4_hdlc_ioctl(dev->fd, SIO4_IOCTL_RX_FIFO_OVERRUN, &arg);
		ret	= ret ? ret : sts;

		if (arg == SIO4_FIFO_OVERRUN_YES)
		{
			rx->flags	|= SIO4_HDLC_RX_FLAG_DATA_OVERRUN;
			arg	= SIO4_FIFO_OVERRUN_CLEAR;
			sts	= sio4_hdlc_ioctl(dev->fd, SIO4_IOCTL_RX_FIFO_OVERRUN, &arg);
			ret	= ret ? ret : sts;
		}
	}

	// Check for a data underrun.
	arg	= SIO4_FIFO_UNDERRUN_TEST;
	sts	= sio4_hdlc_ioctl(dev->fd, SIO4_IOCTL_RX_FIFO_UNDERRUN, &arg);
	ret	= ret ? ret : sts;

	if (arg == SIO4_FIFO_UNDERRUN_YES)
	{
		rx->flags	|= SIO4_HDLC_RX_FLAG_DATA_UNDERRUN;
		arg	= SIO4_FIFO_UNDERRUN_CLEAR;
		sts	= sio4_hdlc_ioctl(dev->fd, SIO4_IOCTL_RX_FIFO_UNDERRUN, &arg);
		ret	= ret ? ret : sts;
	}

	// Check for a frame overrun.

	if ((rsb->rcsr & D5)	||
		(rcc->flags & SIO4_RCC_SW_FIFO_READ_OVER_FRAME))
	{
		rx->flags	|= SIO4_HDLC_RX_FLAG_FRAME_OVERRUN;
	}

	// Check for a DPLL Desync.

	if (rcc->flags & SIO4_RCC_SW_FIFO_READ_DPLL_DESYN)
	{
		reg.reg		= SIO4_USC_CCSR;
		reg.value	= 0;
		reg.mask	= 0;
		sts			= sio4_hdlc_ioctl(dev->fd, SIO4_IOCTL_REG_READ, &reg);
		ret			= ret ? ret : sts;

		if (reg.value & D10)
			reg.mask	|= SIO4_HDLC_RX_FLAG_DPLL_CV1;

		if (reg.value & D11)
			reg.mask	|= SIO4_HDLC_RX_FLAG_DPLL_CV2;

		if (reg.mask)
			rx->flags	|= reg.mask;
		else
			rx->flags	|= SIO4_HDLC_RX_FLAG_DPLL_CV;
	}

	// Check for a CRC Error and a Short Frame error.
	reg.reg		= SIO4_USC_RCSR;
	reg.value	= 0;
	reg.mask	= 0;
	sts			= sio4_hdlc_ioctl(dev->fd, SIO4_IOCTL_REG_READ, &reg);
	ret			= ret ? ret : sts;

	if ((reg.value & D3) || (rsb->rcsr & D3))
		rx->flags	|= SIO4_HDLC_RX_FLAG_CRC_ERR;

	if ((reg.value & D8) || (rsb->rcsr & D8))
		rx->flags	|= SIO4_HDLC_RX_FLAG_FRAME_SHORT;

	// Check for a Frame End indicator.

	if ((dev->rx.remainder == 0)	||
		(rcc->flags & SIO4_RCC_SW_FIFO_READ_FRAME_END))
	{
		rx->flags	|= SIO4_HDLC_RX_FLAG_EOF;
	}

	return(ret);
}



//*****************************************************************************
static int _rx_read(device_t* dev, sio4_hdlc_rx_frame_t* rx)
{
	int			bytes;
	hdlc_rcc_t	rcc		= { 0, 0 };
	int			ret		= 0;
	hdlc_rsb_t	rsb		= { 0 };
	int			sts;

	if (dev->rx.remainder)
	{
		rcc.size			= dev->rx.remainder;
		dev->rx.remainder	= 0;
	}
	else
	{
		ret	= _rx_read_rcc(dev, &rcc);
	}

	sts		= _rx_read_data(dev);
	ret		= ret ? ret : sts;
	bytes	= dev->rx.put - dev->rx.get;

	if (bytes < rcc.size)
		rx->flags	|= SIO4_HDLC_RX_FLAG_DATA_LOSS;

	sts	= _rx_read_local_data(dev, rx);
	ret	= ret ? ret : sts;

	if (dev->rx.remainder == 0)
	{
		sts	= _rx_read_rsb(dev, &rsb);
		ret	= ret ? ret : sts;
	}

	sts	= _rx_read_status(dev, rx, &rcc, &rsb);
	ret	= ret ? ret : sts;

	return(ret);
}



/******************************************************************************
*
*	Function:	sio4_hdlc_rx_frame
*
*	Purpose:
*
*		Read in a frame of data.
*
*	Arguments:
*
*		fd		The file descriptor for the SIO4 channel to access.
*
*		rx		The structure where the received frame is recorded.
*
*	Returned:
*
*		0		All went well.
*		-errno	There was a problem.
*
******************************************************************************/

int sio4_hdlc_rx_frame(int fd, sio4_hdlc_rx_frame_t* rx)
{
	device_t*	dev		= NULL;
	int			ret;

	for (;;)	// A convenience loop.
	{
		// Make sure the library is initialized.
		ret	= sio4_hdlc_lib_init();

		if (ret)
			break;

		// Validate the arguments.
		ret	= _args_validate(fd, rx, &dev);

		if (ret)
			break;

		// Gain exclusive access to the device.
		ret	= os_sem_lock(&dev->sem);

		if (ret)
			break;

		// Validate the state.
		ret	= _env_validate(dev);

		if (ret)
		{
			os_sem_unlock(&dev->sem);
			break;
		}

		// Gain exclusive access to the Rx stream.
		ret	= os_sem_lock(&dev->rx.sem);
		os_sem_unlock(&dev->sem);

		if (ret)
			break;

		// Record our presence.
		dev->rx.rx_frame	= 1;
		os_sem_unlock(&dev->rx.sem);

		// Perform the operation.
		ret	= _rx_read(dev, rx);

		// Cleanup.
		dev->rx.rx_frame	= 0;
		break;
	}

	if (dev)
		lib_dev_dec(dev);

	return(ret);
}



//*****************************************************************************
int rx_frame_active(device_t* dev)
{
	int	ret;

	ret	= dev->rx.rx_frame ? 1 : 0;
	return(ret);
}


