// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_2.x.x_GSC_DN/hdlc/lib/init.c $
// $Rev: 26109 $
// $Date: 2014-05-17 15:56:56 -0500 (Sat, 17 May 2014) $

#include "main.h"



//*****************************************************************************
static const char* _sio4_hdlc_init_t_validate(const sio4_hdlc_init_t* init)
{
	const char*	psz	= NULL;

	for (;;)	// A convenience loop.
	{
		if (init == NULL)
		{
			psz	= "_sio4_hdlc_init_t_validate(init)";
			break;
		}

		if ((init->rx_bit_rate < 1) || (init->rx_bit_rate > 10000000L))
		{
			psz	= "init->rx_bit_rate";
			break;
		}

		if ((init->tx_bit_rate < 1) || (init->tx_bit_rate > 10000000L))
		{
			psz	= "init->tx_bit_rate";
			break;
		}

		if ((init->osc_prog < 0) || (init->osc_prog > 20000000L))
		{
			psz	= "init->osc_prog";
			break;
		}

		switch (init->rx_uses_cbl_rxc)
		{
			default:

				psz	= "init->rx_uses_cbl_rxc";
				break;

			case SIO4_HDLC_RX_USES_CBL_RXC_NO:
			case SIO4_HDLC_RX_USES_CBL_RXC_YES:

				break;
		}

		break;
	}

	return(psz);
}



/******************************************************************************
*
*	Function:	sio4_hdlc_init
*
*	Purpose:
*
*		Initialize the HDLC settings for the specified SIO4.
*
*	Arguments:
*
*		fd		The file descriptor for the SIO4 channel to access.
*
*		init	This provides preliminary settings upon which various device
*				clocking parameters are based.
*
*		hdlc	The initialized settings are recorded here.
*
*		err		If a field value is invalid, this will name the field.
*
*	Returned:
*
*		0		All went well.
*		-errno	There was a problem.
*
******************************************************************************/

int sio4_hdlc_init(
	int						fd,
	const sio4_hdlc_init_t*	init,
	sio4_hdlc_t*			hdlc,
	const char**			err)
{
	device_t*	dev;
	s32			prog;
	const char*	psz		= NULL;
	int			ret;

	for (;;)	// A convenience loop.
	{
		// Make sure the library is initialized.
		ret	= sio4_hdlc_lib_init();

		if (ret)
		{
			psz	= "sio4_hdlc_lib_init";
			break;
		}

		// Validate the received arguments.

		if (init == NULL)
		{
			psz	= "sio4_hdlc_init(,init,,)";
			ret	= -EINVAL;
			break;
		}

		if (hdlc == NULL)
		{
			psz	= "sio4_hdlc_init(,,hdlc,)";
			ret	= -EINVAL;
			break;
		}

		// Gain access to the device.
		dev	= lib_fd_find_inc(fd);

		if (dev == NULL)
		{
			ret	= -ENODEV;
			psz	= "dev";
			break;
		}

		// Begin initialization.
		psz	= _sio4_hdlc_init_t_validate(init);

		if (psz)
		{
			ret	= -EINVAL;
		}
		else
		{
			memset(hdlc, 0, sizeof(sio4_hdlc_t));
			psz	= sio4_hdlc_t_access(dev, hdlc, SIO4_HDLC_ACTION_INIT);
		}

		prog	= SIO4_QUERY_OSC_PROGRAM;
		ret		= sio4_hdlc_ioctl(dev->fd, SIO4_IOCTL_QUERY, &prog);

		if (prog)
			hdlc->osc.prog	= init->osc_prog;

		if (psz)
		{
			ret	= -EINVAL;
		}
		else
		{
			psz	= sio4_hdlc_t_access(dev, hdlc, SIO4_HDLC_ACTION_VERIFY);
		}

		if (psz)
		{
			ret	= -EINVAL;
		}
		else if (init->rx_uses_cbl_rxc == SIO4_HDLC_RX_USES_CBL_RXC_YES)
		{
			if (init->rx_bit_rate == init->tx_bit_rate)
			{
				// test app: case 1
				hdlc->cable.legacy.rxc	= SIO4_HDLC_CABLE_LEGACY_RXC_LOW;
				hdlc->usc.rx.rxc		= SIO4_HDLC_USC_RX_RXC_IN_CBL_RXC;
				hdlc->usc.rx.rxc_legacy	= SIO4_HDLC_USC_RX_RXC_LEG_IN;
				hdlc->usc.rx.clk_src	= SIO4_HDLC_USC_RX_CLK_SRC_RXC_PIN;
				hdlc->usc.tx.clk_src	= SIO4_HDLC_USC_TX_CLK_SRC_RXC_PIN;
				hdlc->usc.tx.txc		= SIO4_HDLC_USC_TX_TXC_OUT_CLK;
				hdlc->usc.tx.txc_legacy	= SIO4_HDLC_USC_TX_TXC_LEG_OUT_CLK;
				hdlc->cable.txc			= SIO4_HDLC_CABLE_TXC_OUT_USC_TXC;
				hdlc->cable.legacy.txc	= SIO4_HDLC_CABLE_LEGACY_TXC_UP;
				hdlc->tx.bit_rate		= init->tx_bit_rate;
				hdlc->rx.bit_rate		= init->rx_bit_rate;
			}
			else
			{
				// test app: case 2
				hdlc->cable.legacy.rxc	= SIO4_HDLC_CABLE_LEGACY_RXC_LOW;
				hdlc->usc.rx.rxc		= SIO4_HDLC_USC_RX_RXC_IN_CBL_RXC;
				hdlc->usc.rx.rxc_legacy	= SIO4_HDLC_USC_RX_RXC_LEG_IN;
				hdlc->usc.rx.clk_src	= SIO4_HDLC_USC_RX_CLK_SRC_RXC_PIN;
				hdlc->cable.txc			= SIO4_HDLC_CABLE_TXC_OUT_OSC;
				hdlc->cable.legacy.txc	= SIO4_HDLC_CABLE_LEGACY_TXC_UP;
				hdlc->usc.tx.txc		= SIO4_HDLC_USC_TX_TXC_IN_OSC;
				hdlc->usc.tx.txc_legacy	= SIO4_HDLC_USC_TX_TXC_LEG_IN;
				hdlc->usc.tx.clk_src	= SIO4_HDLC_USC_TX_CLK_SRC_TXC_PIN;
				hdlc->rx.bit_rate		= init->rx_bit_rate;

				if (hdlc->osc.prog == init->tx_bit_rate)
					hdlc->tx.bit_rate	= init->tx_bit_rate;			// test app: case 2A
				else
					cfg_txc_ctr0_brg0_txclk(hdlc, init->tx_bit_rate);	// test app: case 2B
			}
		}
		else	// init->rx_uses_cbl_rxc == SIO4_HDLC_RX_USES_CBL_RXC_NO
		{
			// We avoid using the DPLL as the USC's Tx Clock source as this
			// might present a problem if the Tx is looped back to the Rx at
			// the cable interface.
			hdlc->cable.legacy.rxc	= SIO4_HDLC_CABLE_LEGACY_RXC_LOW;
			hdlc->usc.dpll.edge		= SIO4_HDLC_USC_DPLL_EDGE_RISE_EDGE;
			hdlc->usc.rx.rxc		= SIO4_HDLC_USC_RX_RXC_IN_OSC;
			hdlc->usc.rx.rxc_legacy	= SIO4_HDLC_USC_RX_RXC_LEG_IN;
			hdlc->usc.rx.clk_src	= SIO4_HDLC_USC_RX_CLK_SRC_DPLL;
			hdlc->usc.tx.clk_src	= SIO4_HDLC_USC_TX_CLK_SRC_RXC_PIN;
			hdlc->usc.tx.txc		= SIO4_HDLC_USC_TX_TXC_OUT_CLK;
			hdlc->usc.tx.txc_legacy	= SIO4_HDLC_USC_TX_TXC_LEG_OUT_CLK;
			hdlc->cable.txc			= SIO4_HDLC_CABLE_TXC_OUT_USC_TXC;
			hdlc->cable.legacy.txc	= SIO4_HDLC_CABLE_LEGACY_TXC_UP;
			cfg_rxc_ctr0_brg0_txclk(hdlc, init->tx_bit_rate);
			cfg_rxc_ctr_brg_dpll(hdlc, init->rx_bit_rate);

			switch (hdlc->rx.encoding)
			{
				default:
				case SIO4_HDLC_RX_ENCODING_NRZ:
				case SIO4_HDLC_RX_ENCODING_NRZB:
				case SIO4_HDLC_RX_ENCODING_NRZI_MARK:
				case SIO4_HDLC_RX_ENCODING_NRZI_SPACE:

						hdlc->usc.dpll.mode	= SIO4_HDLC_USC_DPLL_MODE_NRZ_NRZI;
						break;

				case SIO4_HDLC_RX_ENCODING_BI_MARK:
				case SIO4_HDLC_RX_ENCODING_BI_SPACE:

						hdlc->usc.dpll.mode	= SIO4_HDLC_USC_DPLL_MODE_BIPH_MS;
						break;

				case SIO4_HDLC_RX_ENCODING_BI_LEVEL:
				case SIO4_HDLC_RX_ENCODING_D_BI_LEVEL:

						hdlc->usc.dpll.mode	= SIO4_HDLC_USC_DPLL_MODE_BIPH_LVL;
						break;
			}
		}

		lib_dev_dec(dev);
		break;
	}

	if (err)
		err[0]	= psz;

	return(ret);
}


