// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_2.x.x_GSC_DN/driver/write.c $
// $Rev: 33968 $
// $Date: 2015-11-05 18:47:39 -0600 (Thu, 05 Nov 2015) $

#include "main.h"



/******************************************************************************
*
*	Function:	_tx_avail_csr
*
*	Purpose:
*
*		See how much DMA space is available using the Control/Status Register.
*
*	Arguments:
*
*		chan	The channel data structure.
*
*		bytes	The number of bytes desired.
*
*	Returned:
*
*		>= 0	This is the number of bytes to write.
*
******************************************************************************/

static long _tx_avail_csr(chan_data_t* chan, long bytes)
{
	dev_data_t*	dev = chan->dev;
	u32			csr;
	long		qty;

	csr	= os_reg_mem_rx_u32(dev, chan->vaddr.gsc_csr_32);

	if (csr & D11)
		qty	= bytes;
	else
		qty	= 0;

	return(qty);
}



/******************************************************************************
*
*	Function:	_tx_avail_csr_fsr
*
*	Purpose:
*
*		See how much DMA space is available using the Control/Status Register
*		and the FIFO Size Register.
*
*	Arguments:
*
*		chan	The channel data structure.
*
*		bytes	The number of bytes desired.
*
*	Returned:
*
*		>= 0	This is the number of bytes to write.
*
******************************************************************************/

static long _tx_avail_csr_fsr(chan_data_t* chan, long bytes)
{
	dev_data_t*	dev = chan->dev;
	u32			csr;
	long		qty;

	csr	= os_reg_mem_rx_u32(dev, chan->vaddr.gsc_csr_32);

	if ((csr & D11) == 0)
		qty	= 0;
	else if (bytes > (long) chan->cache.fifo_size_tx) // size)
		qty	= chan->cache.fifo_size_tx; // size;
	else
		qty	= bytes;

	return(qty);
}



/******************************************************************************
*
*	Function:	_tx_avail_csr_fsr_tar
*
*	Purpose:
*
*		See how much space is available using the Control/Status Register, the
*		FIFO Size Register and the Tx Almost Register.
*
*	Arguments:
*
*		chan	The channel data structure.
*
*	Returned:
*
*		>= 0	This is the number of spaces available.
*
******************************************************************************/

static long _tx_avail_csr_fsr_tar(chan_data_t* chan)
{
	dev_data_t*	dev	= chan->dev;
	u32			ae;
	u32			af;
	u32			csr;
	long		qty;
	u32			tar;

	for (;;)	// A convenience loop.
	{
		csr	= os_reg_mem_rx_u32(dev, chan->vaddr.gsc_csr_32);

		if ((csr & D8) == 0)
		{
			// The Tx FIFO is Empty.
			qty	= chan->cache.fifo_size_tx;
			break;
		}

		if ((csr & D9) == 0)
		{
			// The Tx FIFO is Almost Empty.
			tar	= os_reg_mem_rx_u32(dev, chan->vaddr.gsc_tar_32);
			ae	= tar & 0xFFFF;
			qty	= chan->cache.fifo_size_tx - ae;
			break;
		}

		if (csr & D10)
		{
			// The Tx FIFO is less than Almost Full.
			tar	= os_reg_mem_rx_u32(dev, chan->vaddr.gsc_tar_32);
			af	= tar >> 16;
			qty	= af;
			break;
		}

		if (csr & D11)
		{
			// The Tx FIFO is less than Full.
			qty	= 1;
			break;
		}

		// The Tx FIFO is Full.
		qty	= 0;
		break;
	}

	return(qty);
}



/******************************************************************************
*
*	Function:	_tx_avail_csr_tar
*
*	Purpose:
*
*		See how much space is available using the Control/Status Register and
*		the Tx Almost Register.
*
*	Arguments:
*
*		chan	The channel data structure.
*
*	Returned:
*
*		>= 0	This is the number of bytes to write.
*
******************************************************************************/

static long _tx_avail_csr_tar(chan_data_t* chan)
{
	dev_data_t*	dev	= chan->dev;
	u32			af;
	u32			csr;
	long		qty;
	u32			tar;

	for (;;)	// A convenience loop.
	{
		csr	= os_reg_mem_rx_u32(dev, chan->vaddr.gsc_csr_32);

		if (csr & D10)
		{
			// The Tx FIFO is less than Almost Full.
			tar	= os_reg_mem_rx_u32(dev, chan->vaddr.gsc_tar_32);
			af	= tar >> 16;
			qty	= af;
			break;
		}

		if (csr & D11)
		{
			// The Tx FIFO is less than Full.
			qty	= 1;
			break;
		}

		// The Tx FIFO is Full.
		qty	= 0;
		break;
	}

	return(qty);
}



/******************************************************************************
*
*	Function:	_tx_avail_fcr_fsr
*
*	Purpose:
*
*		See how much space is available using the FIFO Count Register and the
*		FIFO Size Register.
*
*	Arguments:
*
*		chan	The channel data structure.
*
*	Returned:
*
*		>= 0	This is the number of bytes to write.
*
******************************************************************************/

static long _tx_avail_fcr_fsr(chan_data_t* chan)
{
	dev_data_t*	dev	= chan->dev;
	u32			count;
	u32			fcr;
	long		qty;

	fcr		= os_reg_mem_rx_u32(dev, chan->vaddr.gsc_fcr_32);
	count	= fcr & 0xFFFF;
	qty		= chan->cache.fifo_size_tx - count;
	return(qty);
}



/******************************************************************************
*
*	Function:	dev_write_startup
*
*	Purpose:
*
*		Perform any work or tests needed before initiating a write operation.
*
*	Arguments:
*
*		chan	The channel data structure.
*
*	Returned:
*
*		0		All is well.
*		< 0		There was an error.
*
******************************************************************************/

int dev_write_startup(chan_data_t* chan)
{
	u32			bcr;
	u32			csr;
	dev_data_t*	dev	= chan->dev;
	long		ret	= 0;

	if ((dev->cache.tx_fifo_overrun) &&  (chan->tx.io_overrun_check))
	{
		csr	= os_reg_mem_rx_u32(chan->dev, chan->vaddr.gsc_csr_32);

		if (csr & D17)
			ret	= -EIO;
	}

	if ((chan->tx.io_mode == GSC_IO_MODE_DMDMA) || (dev->cache.dmdma_scd))
	{
		bcr	= os_reg_mem_rx_u32(chan->dev, dev->vaddr.gsc_bcr_32);

		if (((bcr & D3) == 0) || ((bcr & D7) == 0))
		{
			bcr	|= D3 | D7;
			os_reg_mem_tx_u32(dev, dev->vaddr.gsc_bcr_32, bcr);
		}
	}

	return(ret);
}



//*****************************************************************************
long pio_write_available(chan_data_t* chan, size_t count)
{
	dev_data_t*	dev	= chan->dev;
	long		qty;

	if (dev->cache.reg_fsr == 0)
		qty	= _tx_avail_csr_tar(chan);
	else if (dev->cache.reg_fcr)
		qty	= _tx_avail_fcr_fsr(chan);
	else
		qty	= _tx_avail_csr_fsr_tar(chan);

	if (qty > (long) count)
		qty	= (long) count;

	return(qty);
}



//*****************************************************************************
long dma_write_available(chan_data_t* chan, size_t count)
{
	dev_data_t*	dev	= chan->dev;
	long		qty;

	if (dev->cache.reg_fsr == 0)
		qty	= _tx_avail_csr(chan, count);
	else if (dev->cache.reg_fcr)
		qty	= _tx_avail_fcr_fsr(chan);
	else
		qty	= _tx_avail_csr_fsr(chan, count);

	if (qty > (long) count)
		qty	= (long) count;

	if (qty > chan->tx.pio_threshold)
		qty	&= ~0x3L;	// Force it to a 32-bit boundary.

	return(qty);
}



/******************************************************************************
*
*	Function:	dma_write_work
*
*	Purpose:
*
*		Perform a write of the specified number of bytes.
*
*	Arguments:
*
*		chan	The channel data structure.
*
*		mem		Put the data here.
*
*		count	What remains of the number of bytes that the application
*				requested.
*
*		st_end	The "system ticks" time at which we timeout. If this is zero,
*				then we ignore this.
*
*	Returned:
*
*		>= 0	The number of bytes written.
*		< 0		There was an error.
*
******************************************************************************/

long dma_write_work(
	chan_data_t*	chan,
	const os_mem_t*	mem,
	size_t			count,
	os_time_tick_t	st_end)
{
	u32		dpr;
	u32		mode;
	long	qty		= 0;

	if ((long) count < (long) chan->tx.pio_threshold)
	{
		qty	= gsc_write_pio_work_8_bit(chan, mem, count, st_end);
	}
	else
	{
		mode	= GSC_DMA_MODE_BLOCK_DMA
				| GSC_DMA_MODE_SIZE_8_BITS
				| GSC_DMA_MODE_INPUT_ENABLE
				| GSC_DMA_MODE_BURSTING_LOCAL
				| GSC_DMA_MODE_INTERRUPT_WHEN_DONE
				| GSC_DMA_MODE_LOCAL_ADRESS_CONSTANT
				| GSC_DMA_MODE_PCI_INTERRUPT_ENABLE;

		dpr		= GSC_DMA_DPR_HOST_TO_BOARD
				| GSC_DMA_DPR_END_OF_CHAIN
				| GSC_DMA_DPR_TERMINAL_COUNT_IRQ;

		qty		= gsc_dma_perform(	chan,
									&chan->tx,
									st_end,
									GSC_DMA_CAP_DMA_WRITE,
									mode,
									dpr,
									mem,
									count);
	}

	return(qty);
}



/******************************************************************************
*
*	Function:	dmdma_write_available
*
*	Purpose:
*
*		Check to see how many bytes are available for writing.
*
*	Arguments:
*
*		chan	The channel data structure.
*
*		count	What remains of the number of bytes that the application
*				requested.
*
*	Returned:
*
*		>= 0	The number of bytes available for writing.
*		< 0		There was an error.
*
******************************************************************************/

long dmdma_write_available(chan_data_t* chan, size_t count)
{
	if ((long) count > (long) chan->tx.pio_threshold)
		count	&= ~0x3L;	// Force it to a 32-bit boundary.

	return(count);
}



/******************************************************************************
*
*	Function:	dmdma_write_work
*
*	Purpose:
*
*		Perform a write of the specified number of bytes.
*
*	Arguments:
*
*		chan	The channel data structure.
*
*		mem		Put the data here.
*
*		count	What remains of the number of bytes that the application
*				requested.
*
*		st_end	The "system ticks" time at which we timeout. If this is zero,
*				then we ignore this.
*
*	Returned:
*
*		>= 0	The number of bytes written.
*		< 0		There was an error.
*
******************************************************************************/

long dmdma_write_work(
	chan_data_t*	chan,
	const os_mem_t*	mem,
	size_t			count,
	os_time_tick_t	st_end)
{
	u32		dpr;
	u32		mode;
	long	qty		= 0;

	if ((long) count < (long) chan->tx.pio_threshold)
	{
		qty	= gsc_write_pio_work_8_bit(chan, mem, count, st_end);
	}
	else
	{
		mode	= GSC_DMA_MODE_DM_DMA
				| GSC_DMA_MODE_SIZE_8_BITS
				| GSC_DMA_MODE_INPUT_ENABLE
				| GSC_DMA_MODE_BURSTING_LOCAL
				| GSC_DMA_MODE_INTERRUPT_WHEN_DONE
				| GSC_DMA_MODE_LOCAL_ADRESS_CONSTANT
				| GSC_DMA_MODE_PCI_INTERRUPT_ENABLE;

		dpr		= GSC_DMA_DPR_HOST_TO_BOARD
				| GSC_DMA_DPR_END_OF_CHAIN
				| GSC_DMA_DPR_TERMINAL_COUNT_IRQ;

		qty		= gsc_dma_perform(	chan,
									&chan->tx,
									st_end,
									GSC_DMA_CAP_DMDMA_WRITE,
									mode,
									dpr,
									mem,
									count);
	}

	return(qty);
}


