// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_2.x.x_GSC_DN/driver/reg.c $
// $Rev: 31815 $
// $Date: 2015-06-09 12:24:11 -0500 (Tue, 09 Jun 2015) $

#include "main.h"



// data types *****************************************************************

typedef struct
{
	u32				reg;		// all registers
	chan_data_t*	chan;		// all registers

	// Fields encoded in the register identifier.
	u32				size;		// all registers
	u32				offset;		// all registers
	u32				type;		// all registers
	int				firmware;	// 0 = USC, 1 = GSC
	int				ch_mult;	// GSC registers only
	u32				format;		// GSC registers only.
	int				shift;		// GSC registers only.

	// Fields computed to access the register/
	u32				mask;		// all registers
	u32				block;		// all registers
	VADDR_T			vaddr;		// Non PCI registers only.
	VADDR_T			vaddr_lb;	// USC registers only.
	VADDR_T			vaddr_ub;	// USC registers only.
} reg_def_t;



//*****************************************************************************
static int _reg_decode(chan_data_t* chan, u32 reg, reg_def_t* def)
{
	int	ret		= 0;

	memset(def, 0, sizeof(reg_def_t));

	def->chan		= chan;
	def->reg		= reg;
	def->size		= GSC_REG_SIZE(reg);
	def->offset		= GSC_REG_OFFSET(reg);
	def->type		= GSC_REG_TYPE(reg);
	def->firmware	= SIO4_FIRMWARE_DECODE(reg);
	def->ch_mult	= SIO4_CH_MULT_DECODE(reg);
	def->format		= SIO4_FORMAT_DECODE(reg);

	switch (def->size)
	{
		default:	ret			= -EINVAL;		break;
		case 1:		def->mask	= 0xFF;			break;
		case 2:		def->mask	= 0xFFFF;		break;
		case 3:		def->mask	= 0xFFFFFF;		break;
		case 4:		def->mask	= 0xFFFFFFFF;	break;
	}

	if (def->firmware)
	{
		// GSC firmware registers

		switch (def->format)
		{
			default:

				ret			= -EINVAL;
				def->mask	= 0;
				break;

			case 0:

				break;

			case 1:

				// Channel Bits are: xxxx xxxx xxxx xxxx 3333 2222 1111 0000
				def->shift	= chan->index * 4;
				def->mask	&= 0xF << def->shift;
				break;

			case 2:

				// Channel Bits are: 3333 2222 1111 0000 3333 2222 1111 0000
				def->shift	= chan->index * 4;
				def->mask	&= 0xF000FUL << def->shift;
				break;

			case 3:

				// Channel Bits are: 3333 3333 2222 2222 1111 1111 0000 0000
				def->shift	= chan->index * 8;
				def->mask	&= 0xFFUL << def->shift;
				break;
		}

		if ((def->size != 4) || (def->offset % 4))
			ret	= -EINVAL;

		def->block	= 256;
		def->vaddr	= (VADDR_T) ((unsigned long) chan->dev->gsc.vaddr
					+ def->offset
					+ (chan->index * def->ch_mult));
	}
	else
	{
		// USC registers

		if ((def->format) || (def->ch_mult))
			ret	= -EINVAL;

		if ((def->size != 2) || (def->offset % 2))
			ret	= -EINVAL;

		def->block		= 64;
		def->vaddr_lb	= (VADDR_T) ((unsigned long) chan->vaddr.usc_regs + def->offset);
		def->vaddr_ub	= (VADDR_T) ((unsigned long) chan->vaddr.usc_regs + def->offset + 1);
	}

	if ((def->offset + def->size) > def->block)
		ret	= -EINVAL;

	return(ret);
}



//*****************************************************************************
static void _reg_mod_alt_gsc(chan_data_t* chan, gsc_reg_t* arg, reg_def_t* def)
{
	u32	mask;
	u32	val;

	mask	= def->mask & arg->mask;
	val		= os_reg_mem_rx_u32(chan->dev, def->vaddr);
	val		= (val & ~mask) | (arg->value & mask);
	os_reg_mem_tx_u32(chan->dev, def->vaddr, val);
}



//*****************************************************************************
static void _reg_read_alt_gsc(chan_data_t* chan, gsc_reg_t* arg, reg_def_t* def)
{
	arg->value	= os_reg_mem_rx_u32(chan->dev, def->vaddr);
//	arg->value	&= def->mask;
}



//*****************************************************************************
static void _reg_write_alt_gsc(chan_data_t* chan, gsc_reg_t* arg, reg_def_t* def)
{
	u32	val;

	val	= os_reg_mem_rx_u32(chan->dev, def->vaddr);
	val	= (val & ~def->mask) | (arg->value & def->mask);
	os_reg_mem_tx_u32(chan->dev, def->vaddr, val);
}



//*****************************************************************************
static void _reg_mod_alt_usc(chan_data_t* chan, gsc_reg_t* arg, reg_def_t* def)
{
	u16	lb;
	u16	ub;
	u16	val;

	// read
	lb	= os_reg_mem_rx_u8(chan->dev, def->vaddr_lb);
	ub	= os_reg_mem_rx_u8(chan->dev, def->vaddr_ub);
	val	= lb | (ub << 8);

	// modify
	val	= (val & ~arg->mask) | (arg->value & arg->mask);

	// write
	lb	= 0xFF & val;
	ub	= 0xFF & (val >> 8);
	os_reg_mem_tx_u8(chan->dev, def->vaddr_lb, (u8) lb);
	os_reg_mem_tx_u8(chan->dev, def->vaddr_ub, (u8) ub);
}



//*****************************************************************************
static void _reg_read_alt_usc(chan_data_t* chan, gsc_reg_t* arg, reg_def_t* def)
{
	u16	lb;
	u16	ub;

	lb			= os_reg_mem_rx_u8(chan->dev, def->vaddr_lb);
	ub			= os_reg_mem_rx_u8(chan->dev, def->vaddr_ub);
	arg->value	= lb | (ub << 8);
}



//*****************************************************************************
static void _reg_write_alt_usc(chan_data_t* chan, gsc_reg_t* arg, reg_def_t* def)
{
	u8	lb;
	u8	ub;

	lb	= 0xFF & arg->value;
	ub	= 0xFF & (arg->value >> 8);

	os_reg_mem_tx_u8(chan->dev, def->vaddr_lb, lb);
	os_reg_mem_tx_u8(chan->dev, def->vaddr_ub, ub);
}



//*****************************************************************************
int dev_reg_mod_alt(chan_data_t* chan, gsc_reg_t* arg)
{
	reg_def_t	def;
	int			ret;

	ret	= _reg_decode(chan, arg->reg, &def);

	if (ret)
		;
	else if (def.firmware)
		_reg_mod_alt_gsc(chan, arg, &def);
	else
		_reg_mod_alt_usc(chan, arg, &def);

	return(ret);
}



//*****************************************************************************
int dev_reg_read_alt(chan_data_t* chan, gsc_reg_t* arg)
{
	reg_def_t	def;
	int			ret;

	ret	= _reg_decode(chan, arg->reg, &def);

	if (ret)
		;
	else if (def.firmware)
		_reg_read_alt_gsc(chan, arg, &def);
	else
		_reg_read_alt_usc(chan, arg, &def);

	return(ret);
}



//*****************************************************************************
int dev_reg_write_alt(chan_data_t* chan, gsc_reg_t* arg)
{
	reg_def_t	def;
	int			ret;

	ret	= _reg_decode(chan, arg->reg, &def);

	if (ret)
		;
	else if (def.firmware)
		_reg_write_alt_gsc(chan, arg, &def);
	else
		_reg_write_alt_usc(chan, arg, &def);

	return(ret);
}



//*****************************************************************************
void reg_mod(chan_data_t* chan, u32 reg, u32 value, u32 mask)
{
	gsc_reg_t	arg;

	arg.reg		= reg;
	arg.value	= value;
	arg.mask	= mask;
	dev_reg_mod_alt(chan, &arg);
}



//*****************************************************************************
void reg_read(chan_data_t* chan, u32 reg, u32* value)
{
	gsc_reg_t	arg;

	arg.reg		= reg;
	arg.value	= 0xDEADBEEF;
	arg.mask	= 0;
	dev_reg_read_alt(chan, &arg);
	value[0]	= arg.value;
}



//*****************************************************************************
void reg_write(chan_data_t* chan, u32 reg, u32 value)
{
	gsc_reg_t	arg;

	arg.reg		= reg;
	arg.value	= value;
	arg.mask	= 0;
	dev_reg_write_alt(chan, &arg);
}


