// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_2.x.x_GSC_DN/driver/read.c $
// $Rev: 33968 $
// $Date: 2015-11-05 18:47:39 -0600 (Thu, 05 Nov 2015) $

#include "main.h"



/******************************************************************************
*
*	Function:	_rx_avail_csr
*
*	Purpose:
*
*		See how much DMA data is available using the Control/Status Register.
*
*	Arguments:
*
*		chan	The channel data structure.
*
*		bytes	The number of bytes desired.
*
*	Returned:
*
*		>= 0	This is the number of bytes to read.
*
******************************************************************************/

static long _rx_avail_csr(chan_data_t* chan, long bytes)
{
	dev_data_t*	dev		= chan->dev;
	u32			csr;
	long		qty;

	csr	= os_reg_mem_rx_u32(dev, chan->vaddr.gsc_csr_32);

	if (csr & D12)
		qty	= bytes;
	else
		qty	= 0;

	return(qty);
}



/******************************************************************************
*
*	Function:	_rx_avail_csr_fsr
*
*	Purpose:
*
*		See how much DMA data is available using the Control/Status Register
*		and the FIFO Size Register.
*
*	Arguments:
*
*		chan	The channel data structure.
*
*		bytes	The number of bytes desired.
*
*	Returned:
*
*		>= 0	This is the number of bytes to read.
*
******************************************************************************/

static long _rx_avail_csr_fsr(chan_data_t* chan, long bytes)
{
	dev_data_t*	dev		= chan->dev;
	u32			csr;
	long		qty;

	csr	= os_reg_mem_rx_u32(dev, chan->vaddr.gsc_csr_32);

	if ((csr & D12) == 0)
		qty	= 0;
	else if (bytes > (long) chan->cache.fifo_size_rx)
		qty	= chan->cache.fifo_size_rx;
	else
		qty	= bytes;

	return(qty);
}



/******************************************************************************
*
*	Function:	_rx_avail_csr_fsr_rar
*
*	Purpose:
*
*		See how much data is available using the Control/Status Register, the
*		FIFO Size Register and the Rx Almost Register.
*
*	Arguments:
*
*		chan	The channel data structure.
*
*	Returned:
*
*		>= 0	This is the number of bytes available.
*
******************************************************************************/

static long _rx_avail_csr_fsr_rar(chan_data_t* chan)
{
	dev_data_t*	dev		= chan->dev;
	u32			ae;
	u32			af;
	u32			csr;
	long		qty;
	u32			rar;

	for (;;)	// A convenience loop.
	{
		csr	= os_reg_mem_rx_u32(dev, chan->vaddr.gsc_csr_32);

		if ((csr & D15) == 0)
		{
			// The Rx FIFO is Full.
			qty	= chan->cache.fifo_size_rx;
			break;
		}

		if ((csr & D14) == 0)
		{
			// The Rx FIFO is Almost Full.
			rar	= os_reg_mem_rx_u32(dev, chan->vaddr.gsc_rar_32);
			af	= rar >> 16;
			qty	= chan->cache.fifo_size_rx - af;
			break;
		}

		if (csr & D13)
		{
			// The Rx FIFO is more than Almost Empty.
			rar	= os_reg_mem_rx_u32(dev, chan->vaddr.gsc_rar_32);
			ae	= rar & 0xFFFF;
			qty	= ae;
			break;
		}

		if (csr & D12)
		{
			// The Rx FIFO is above Empty.
			qty	= 1;
			break;
		}

		// The Rx FIFO is empty.
		qty	= 0;
		break;
	}

	return(qty);
}



/******************************************************************************
*
*	Function:	_rx_avail_csr_rar
*
*	Purpose:
*
*		See how much data is available using the Control/Status Register and
*		the Rx Almost Register.
*
*	Arguments:
*
*		chan	The channel data structure.
*
*	Returned:
*
*		>= 0	This is the number of bytes available.
*
******************************************************************************/

static long _rx_avail_csr_rar(chan_data_t* chan)
{
	dev_data_t*	dev		= chan->dev;
	u32			ae;
	u32			csr;
	long		qty;
	u32			rar;

	for (;;)	// A convenience loop.
	{
		csr	= os_reg_mem_rx_u32(dev, chan->vaddr.gsc_csr_32);

		if (csr & D13)
		{
			// The Rx FIFO is more than Almost Empty.
			rar	= os_reg_mem_rx_u32(dev, chan->vaddr.gsc_rar_32);
			ae	= rar & 0xFFFF;
			qty	= ae;
			break;
		}

		if (csr & D12)
		{
			// The Rx FIFO is more than Empty.
			qty	= 1;
			break;
		}

		// The Rx FIFO is empty.
		qty	= 0;
		break;
	}

	return(qty);
}



/******************************************************************************
*
*	Function:	_rx_avail_fcr
*
*	Purpose:
*
*		See how much data is available using the FIFO Count Register.
*
*	Arguments:
*
*		chan	The channel data structure.
*
*	Returned:
*
*		>= 0	This is the number of bytes available.
*
******************************************************************************/

static long _rx_avail_fcr(chan_data_t* chan)
{
	dev_data_t*	dev		= chan->dev;
	u32			fcr;
	long		qty;

	fcr	= os_reg_mem_rx_u32(dev, chan->vaddr.gsc_fcr_32);
	qty	= fcr >> 16;
	return(qty);
}



//*****************************************************************************
long dev_read_startup(chan_data_t* chan)
{
	u32			bcr;
	u32			csr;
	dev_data_t*	dev		= chan->dev;
	long		ret		= 0;

	if (((chan->cache.rx_fifo_overrun) && (chan->rx.io_overrun_check))	||
		((dev->cache.rx_fifo_underrun) && (chan->rx.io_underrun_check)))
	{
		csr	= os_reg_mem_rx_u32(chan->dev, chan->vaddr.gsc_csr_32);

		if ((chan->cache.rx_fifo_overrun) && (chan->rx.io_overrun_check) && (csr & D16))
			ret	= -EIO;

		if ((dev->cache.rx_fifo_underrun) && (chan->rx.io_underrun_check) && (csr & D18))
			ret	= -EIO;
	}

	if ((chan->rx.io_mode == GSC_IO_MODE_DMDMA) || (dev->cache.dmdma_scd))
	{
		bcr	= os_reg_mem_rx_u32(chan->dev, dev->vaddr.gsc_bcr_32);

		if (((bcr & D3) == 0) || ((bcr & D7) == 0))
		{
			bcr	|= D3 | D7;
			os_reg_mem_tx_u32(dev, dev->vaddr.gsc_bcr_32, bcr);
		}
	}

	return(ret);
}



//*****************************************************************************
long pio_read_available(chan_data_t* chan, size_t count)
{
	dev_data_t*	dev	= chan->dev;
	long		qty;

	if (dev->cache.reg_fcr)
		qty	= _rx_avail_fcr(chan);
	else if (dev->cache.reg_fsr)
		qty	= _rx_avail_csr_fsr_rar(chan);
	else
		qty	= _rx_avail_csr_rar(chan);

	if (qty > (long) count)
		qty	= count;

	return(qty);
}



/******************************************************************************
*
*	Function:	dma_read_available
*
*	Purpose:
*
*		Check to see how many bytes are available for reading.
*
*	Arguments:
*
*		chan	The channel data structure.
*
*		count	What remains of the number of bytes that the application
*				requested.
*
*	Returned:
*
*		>= 0	The number of bytes available for reading.
*		< 0		There was an error.
*
******************************************************************************/

long dma_read_available(chan_data_t* chan, size_t count)
{
	dev_data_t*	dev	= chan->dev;
	long		qty;

	if (dev->cache.reg_fcr)
		qty	= _rx_avail_fcr(chan);
	else if (dev->cache.reg_fsr)
		qty	= _rx_avail_csr_fsr(chan, count);
	else
		qty	= _rx_avail_csr(chan, count);

	if (qty > (long) count)
		qty	= count;

	if (qty > chan->rx.pio_threshold)
		qty	&= ~0x3L;	// Force it to a 32-bit boundary.

	return(qty);
}



/******************************************************************************
*
*	Function:	dma_read_work
*
*	Purpose:
*
*		Perform a read of the specified number of bytes.
*
*	Arguments:
*
*		chan	The channel data structure.
*
*		mem		Put the data here.
*
*		count	What remains of the number of bytes that the application
*				requested.
*
*		st_end	The "system ticks" time at which we timeout. If this is zero,
*				then we ignore this.
*
*	Returned:
*
*		>= 0	The number of bytes read.
*		< 0		There was an error.
*
******************************************************************************/

long dma_read_work(
	chan_data_t*	chan,
	const os_mem_t*	mem,
	size_t			count,
	os_time_tick_t	st_end)
{
	u32		dpr;
	u32		mode;
	long	qty		= 0;

	if ((long) count < (long) chan->rx.pio_threshold)
	{
		qty	= gsc_read_pio_work_8_bit(chan, mem, count, st_end);
	}
	else
	{
		mode	= GSC_DMA_MODE_BLOCK_DMA
				| GSC_DMA_MODE_SIZE_8_BITS
				| GSC_DMA_MODE_INPUT_ENABLE
				| GSC_DMA_MODE_BURSTING_LOCAL
				| GSC_DMA_MODE_INTERRUPT_WHEN_DONE
				| GSC_DMA_MODE_LOCAL_ADRESS_CONSTANT
				| GSC_DMA_MODE_PCI_INTERRUPT_ENABLE;

		dpr		= GSC_DMA_DPR_BOARD_TO_HOST
				| GSC_DMA_DPR_END_OF_CHAIN
				| GSC_DMA_DPR_TERMINAL_COUNT_IRQ;

		qty		= gsc_dma_perform(	chan,
									&chan->rx,
									st_end,
									GSC_DMA_CAP_DMA_READ,
									mode,
									dpr,
									mem,
									count);
	}

	return(qty);
}



/******************************************************************************
*
*	Function:	dmdma_read_available
*
*	Purpose:
*
*		Check to see how many bytes are available for reading.
*
*	Arguments:
*
*		chan	The channel data structure.
*
*		count	What remains of the number of bytes that the application
*				requested.
*
*	Returned:
*
*		>= 0	The number of bytes available for reading.
*		< 0		There was an error.
*
******************************************************************************/

long dmdma_read_available(chan_data_t* chan, size_t count)
{
	if ((long) count > (long) chan->rx.pio_threshold)
		count	&= ~0x3L;	// Force it to a 32-bit boundary.

	return(count);
}



/******************************************************************************
*
*	Function:	dmdma_read_work
*
*	Purpose:
*
*		Perform a read of the specified number of bytes.
*
*	Arguments:
*
*		chan	The channel data structure.
*
*		mem		Put the data here.
*
*		count	What remains of the number of bytes that the application
*				requested.
*
*		st_end	The "system ticks" time at which we timeout. If this is zero,
*				then we ignore this.
*
*	Returned:
*
*		>= 0	The number of bytes read.
*		< 0		There was an error.
*
******************************************************************************/

long dmdma_read_work(
	chan_data_t*	chan,
	const os_mem_t*	mem,
	size_t			count,
	os_time_tick_t	st_end)
{
	u32		dpr;
	u32		mode;
	long	qty		= 0;

	if ((long) count < (long) chan->rx.pio_threshold)
	{
		qty	= gsc_read_pio_work_8_bit(chan, mem, count, st_end);
	}
	else
	{
		mode	= GSC_DMA_MODE_DM_DMA
				| GSC_DMA_MODE_SIZE_8_BITS
				| GSC_DMA_MODE_INPUT_ENABLE
				| GSC_DMA_MODE_BURSTING_LOCAL
				| GSC_DMA_MODE_INTERRUPT_WHEN_DONE
				| GSC_DMA_MODE_LOCAL_ADRESS_CONSTANT
				| GSC_DMA_MODE_PCI_INTERRUPT_ENABLE;

		dpr		= GSC_DMA_DPR_BOARD_TO_HOST
				| GSC_DMA_DPR_END_OF_CHAIN
				| GSC_DMA_DPR_TERMINAL_COUNT_IRQ;

		qty		= gsc_dma_perform(	chan,
									&chan->rx,
									st_end,
									GSC_DMA_CAP_DMDMA_READ,
									mode,
									dpr,
									mem,
									count);
	}

	return(qty);
}


