// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_2.x.x_GSC_DN/driver/osc_cy22393_sio4.c $
// $Rev: 33968 $
// $Date: 2015-11-05 18:47:39 -0600 (Thu, 05 Nov 2015) $

#include "main.h"



/******************************************************************************
*
*	Function:	_compute_pd_osc_program
*
*	Purpose:
*
*		Figure out what post divider and oscillator configuration best produces
*		the requested frquency, given the available oscillator resources and
*		its current usage.
*
*	Arguments:
*
*		osc		The structure for the oscillator to access.
*
*		index	The index of the channel to access.
*
*		arg		The structre we use to exchange data with the caller.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

static int _compute_pd_osc_program(
	osc_cy22393_sio4_t*	osc,
	int					index,
	osc_cy22393_arg_t*	arg)
{
	osc_cy22393_arg_t	data;
	long				delta;
	long				delta_min	= 999999999L;
	long				divider;
	long				div_max;
	long				freq_osc;
	long				freq_want	= arg->freq_want;
	int					pd;
	int					ret			= 0;
	osc_cy22393_sio4_t	test;
	osc_cy22393_sio4_t	use			= osc[0];;

	osc->chan[index].freq_want	= freq_want;

	if (freq_want)
		div_max	= (OSC_FREQ_MAX / ((freq_want + 1) / 2)) + 1;
	else
		div_max	= OSC_FREQ_MAX;

	for (pd = 15; pd >= 0; pd--)
	{
		if (pd > (int) osc->pd_max)
			continue;

		divider	= 0x1L << pd;

		if (divider > div_max)
			continue;

		freq_osc	= freq_want * divider;

		if (freq_osc > OSC_FREQ_MAX)
			continue;

		memset(&data, 0, sizeof(data));
		data.freq_want	= freq_osc;
		test			= osc[0];
		ret				= osc_cy22393_program(&test.osc, index, &data);

		if (ret)
			break;

		delta	= ((long) freq_osc / divider) - ((long) test.osc.clock[index].freq_got / divider);
		delta	= (delta < 0) ? -delta : delta;

		if ((delta < delta_min) ||
			((delta == delta_min) &&
			(test.osc.clock[index].freq_src == CY22393_FREQ_SRC_REF)))
		{
			delta_min					= delta;
			use							= test;
			use.chan[index].freq_got	= data.freq_got / divider;
			use.chan[index].pd			= pd;

			if ((delta == 0) && (test.osc.clock[index].freq_src == CY22393_FREQ_SRC_REF))
				break;
		}
	}

	osc[0]	= use;
	return(ret);
}



/******************************************************************************
*
*	Function:	_osc_reg_read
*
*	Purpose:
*
*		Read a value from an oscillator register.
*
*	Arguments:
*
*		dev		The structure for the device to access.
*
*		address	The register address to access.
*
*		value	Record the value read here.
*
*	Returned:
*
*		None.
*
******************************************************************************/

static void _osc_reg_read(dev_data_t* dev, u8 address, u8* value)
{
	// The caller should have exclusive access.
	os_reg_mem_tx_u32(NULL, dev->vaddr.gsc_porar_32, address);
	value[0]	= (u8) (os_reg_mem_rx_u32(NULL, dev->vaddr.gsc_pordr_32) & 0xFF);
}



/******************************************************************************
*
*	Function:	_osc_reg_write
*
*	Purpose:
*
*		Write a value to an oscillator register.
*
*	Arguments:
*
*		dev		The structure for the device to access.
*
*		address	The register address to access.
*
*		value	The value to write.
*
*	Returned:
*
*		None.
*
******************************************************************************/

static void _osc_reg_write(dev_data_t* dev, u8 address, u8 value)
{
	// The caller should have exclusive access.
	os_reg_mem_tx_u32(NULL, dev->vaddr.gsc_porar_32, address);
	os_reg_mem_tx_u32(NULL, dev->vaddr.gsc_pordr_32, value);

	if (dev->cache.osc_chip == SIO4_OSC_CHIP_CY22393_2)
		os_reg_mem_tx_u32(NULL, dev->vaddr.gsc_pord2r_32, value);
}



/******************************************************************************
*
*	Function:	_osc_regs_write
*
*	Purpose:
*
*		Write values to all oscillator registers.
*
*	Arguments:
*
*		dev		The structure for the device to access.
*
*		osc		The structure for the oscillator to access.
*
*	Returned:
*
*		None.
*
******************************************************************************/

static void _osc_regs_write(dev_data_t* dev, osc_cy22393_sio4_t* osc)
{
	gsc_irq_access_lock(dev);

	_osc_reg_write(dev, 0x06, osc->_06);
	_osc_reg_write(dev, 0x07, osc->_07);

	_osc_reg_write(dev, 0x08, osc->osc.data._08);
	_osc_reg_write(dev, 0x09, osc->osc.data._08);
	_osc_reg_write(dev, 0x0A, osc->osc.data._0A);
	_osc_reg_write(dev, 0x0B, osc->osc.data._0A);
	_osc_reg_write(dev, 0x0C, osc->osc.data._0C);
	_osc_reg_write(dev, 0x0D, osc->osc.data._0D);
	_osc_reg_write(dev, 0x0E, osc->osc.data._0E);
	_osc_reg_write(dev, 0x0F, osc->osc.data._0F);
	_osc_reg_write(dev, 0x10, osc->osc.data._10);
	_osc_reg_write(dev, 0x11, osc->osc.data._11);
	_osc_reg_write(dev, 0x12, osc->osc.data._12);
	_osc_reg_write(dev, 0x13, osc->osc.data._13);
	_osc_reg_write(dev, 0x14, osc->osc.data._14);
	_osc_reg_write(dev, 0x15, osc->osc.data._15);
	_osc_reg_write(dev, 0x16, osc->osc.data._16);
	_osc_reg_write(dev, 0x17, osc->osc.data._17);

	_osc_reg_write(dev, 0x18, osc->_18);
	_osc_reg_write(dev, 0x19, osc->_19);
	_osc_reg_write(dev, 0x1A, osc->_1a);
	_osc_reg_write(dev, 0x1B, osc->_1b);

	_osc_reg_write(dev, 0x40, osc->osc.data._40);
	_osc_reg_write(dev, 0x41, osc->osc.data._41);
	_osc_reg_write(dev, 0x42, osc->osc.data._42);
	_osc_reg_write(dev, 0x43, osc->osc.data._40);
	_osc_reg_write(dev, 0x44, osc->osc.data._41);
	_osc_reg_write(dev, 0x45, osc->osc.data._42);
	_osc_reg_write(dev, 0x46, osc->osc.data._40);
	_osc_reg_write(dev, 0x47, osc->osc.data._41);
	_osc_reg_write(dev, 0x48, osc->osc.data._42);
	_osc_reg_write(dev, 0x49, osc->osc.data._40);
	_osc_reg_write(dev, 0x4A, osc->osc.data._41);
	_osc_reg_write(dev, 0x4B, osc->osc.data._42);
	_osc_reg_write(dev, 0x4C, osc->osc.data._40);
	_osc_reg_write(dev, 0x4D, osc->osc.data._41);
	_osc_reg_write(dev, 0x4E, osc->osc.data._42);
	_osc_reg_write(dev, 0x4F, osc->osc.data._40);
	_osc_reg_write(dev, 0x50, osc->osc.data._41);
	_osc_reg_write(dev, 0x51, osc->osc.data._42);
	_osc_reg_write(dev, 0x52, osc->osc.data._40);
	_osc_reg_write(dev, 0x53, osc->osc.data._41);
	_osc_reg_write(dev, 0x54, osc->osc.data._42);
	_osc_reg_write(dev, 0x55, osc->osc.data._40);
	_osc_reg_write(dev, 0x56, osc->osc.data._41);
	_osc_reg_write(dev, 0x57, osc->osc.data._42);

	gsc_irq_access_unlock(dev);
}



/******************************************************************************
*
*	Function:	_osc_regs_set_constants
*
*	Purpose:
*
*		Fill in the constant data for the given oscillator.
*
*	Arguments:
*
*		osc		The oscillator data structure.
*
*	Returned:
*
*		None.
*
******************************************************************************/

static void _osc_regs_set_constants(osc_cy22393_t* osc)
{
	osc->data._0F	= 0x50;		// Xbuf_OE: 0 (disable)
								// PdnEn: 0 (OE Control)
								// Clk{A,B,D,E}_ACAdj: 1 (nominal)
								// Clk{C,X}_ACAdj: 1 (nominal)
	osc->data._10	= 0x55;		// Clk{A,B}_DCAdj: 1 (nominal)
								// ClkC_DCAdj: 1 (nominal)
								// Clk{D,E}_DCAdj: 1 (nominal)
								// ClkX_DCAdj: 1 (nominal)
	osc->data._13	&= ~0x80;	// Reserved: 0 (required)
	osc->data._16	&= ~0x80;	// Reserved: 0 (required)
	osc->data._17	= 0x00;		// OscCap: 0 (Extern Ref Clock)
	osc->data._42	&= ~0x80;	// DivSel: 0 (convention)
}



/******************************************************************************
*
*	Function:	_wait_for_ready
*
*	Purpose:
*
*		Wait for the firmware to indicate the oscillator is ready.
*
*	Arguments:
*
*		dev		The device structure to access.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

static int _wait_for_ready(dev_data_t* dev)
{
	u32	pocsr;
	int	ret		= -EIO;
	int	ticks	= os_time_tick_rate();
	int	wait;

	for (wait = 0; wait < (2 * ticks); wait++)
	{
		pocsr	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_pocsr_32);

		if (pocsr & D0)	// Programming Done
		{
			ret	= 0;
			break;
		}

		os_time_tick_sleep(1);
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	_osc_program
*
*	Purpose:
*
*		Submit the recorded Programmable Oscillator RAM data to the
*		oscillator and wait for completion.
*
*	Arguments:
*
*		dev		The structure for the device to access.
*
*		osc		The structure for the oscillator to access.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

static int _osc_program(dev_data_t* dev, osc_cy22393_sio4_t* osc)
{
	u32	pocsr;
	int	ret		= -EIO;
	int	tries;

	for (tries = 1;; tries++)
	{
		ret	= _wait_for_ready(dev);

		if (ret < 0)
			break;

		_osc_regs_write(dev, osc);
		pocsr	= D0;	// Program
		os_reg_mem_tx_u32(dev, dev->vaddr.gsc_pocsr_32, pocsr);
		os_time_us_delay(10000);	// Wait for Lock Time to pass.
		ret	= _wait_for_ready(dev);

		if (ret < 0)
			break;

		pocsr	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_pocsr_32);

		if ((pocsr & D1) == 0)	// Program error
			break;

		if (tries >= 10)
		{
			ret	= -EIO;
			break;
		}
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	_chan_fw_config_write
*
*	Purpose:
*
*		Write channel's firmware configuration to the device.
*
*	Arguments:
*
*		dev		The structure for the device to access.
*
*		osc		The structure for the oscillator to access.
*
*	Returned:
*
*		None.
*
******************************************************************************/

static void _chan_fw_config_write(dev_data_t* dev, osc_cy22393_sio4_t* osc)
{
	u32	pocsr;

	pocsr	= (((u32) osc->chan[0].pd) << 8)
			| (((u32) osc->chan[1].pd) << 12)
			| (((u32) osc->chan[2].pd) << 16)
			| (((u32) osc->chan[3].pd) << 20)
			| D7;	// Post Divider Set
	os_reg_mem_tx_u32(dev, dev->vaddr.gsc_pocsr_32, pocsr);
}



/******************************************************************************
*
*	Function:	_chan_measure_freq
*
*	Purpose:
*
*		Measure the channel's oscillator output frequency.
*
*	Arguments:
*
*		chan	The structure for the channel to access.
*
*		arg		Report the measured value here.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

static int _chan_measure_freq(chan_data_t* chan, s32* arg)
{
	dev_data_t*	dev	= chan->dev;
	int			i;
	u32			pocsr;
	int			ret;

	pocsr	= 0x2 << chan->index;
	os_reg_mem_tx_u32(dev, dev->vaddr.gsc_pocsr_32, pocsr);

	for (i = 0; i < 500; i++)
	{
		os_time_sleep_ms(1);
		pocsr	= os_reg_mem_rx_u32(dev, dev->vaddr.gsc_pocsr_32);

		if (pocsr & D2)	// Measure Done
			break;
	}

	if (pocsr & D2)	// Measure Done
		ret		= 0;
	else
		ret		= -ETIMEDOUT;

	arg[0]	= (pocsr >> 8) * 10;
	return(ret);
}



/******************************************************************************
*
*	Function:	_measure_service
*
*	Purpose:
*
*		Perform a frequency measurement test of the serial channel.
*		The hardware takes 10ms to do this. We might take longer.
*
*	Arguments:
*
*		chan	The structure for the channel to access.
*
*		arg		Report the measured value here.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

static int _measure_service(chan_data_t* chan, s32* arg)
{
	osc_cy22393_sio4_t*	osc	= &chan->dev->osc.cy22393;
	int					ret;

	for (;;)	// A convenience loop.
	{
		ret	= os_sem_lock(&osc->sem);

		if (ret)
			break;

		arg[0]	= osc->chan[chan->index].freq_got;
		ret		= _chan_measure_freq(chan, arg);

		os_sem_unlock(&osc->sem);
		break;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	_reference_service
*
*	Purpose:
*
*		Tell the oscillator code the channel's reference frequency.
*		If the specified reference frequency is zero, then we report
*		the current setting.
*
*	Arguments:
*
*		chan	The structure for the channel to access.
*
*		arg		The desired value is recorded here. The reported value is
*				also recorded here.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

static int _reference_service(chan_data_t* chan, s32* arg)
{
	osc_cy22393_sio4_t*	osc	= &chan->dev->osc.cy22393;
	int					ret;

	ret	= os_sem_lock(&osc->sem);

	if (ret)
	{
	}
	else if (arg[0] == -1)
	{
		arg[0]	= osc->osc.freq_ref;
		ret		= 0;
		os_sem_unlock(&osc->sem);
	}
	else if (arg[0] == osc->osc.freq_ref)
	{
		ret		= 0;
		os_sem_unlock(&osc->sem);
	}
	else
	{
		ret	= -EINVAL;
		os_sem_unlock(&osc->sem);
	}

	return(ret);
}



//*****************************************************************************
void osc_cy22393_sio4_close(chan_data_t* chan)
{
	// We leave the oscillator as is.
}



//*****************************************************************************
void osc_cy22393_sio4_create(dev_data_t* dev)
{
	osc_cy22393_sio4_t*	osc	= &dev->osc.cy22393;

	memset(osc, 0, sizeof(osc[0]));
	os_sem_create(&osc->sem);

	osc->pd_max			= dev->cache.osc_pd_max;
	osc->osc.freq_ref	= OSC_FREQ_REF_DEFAULT;

	gsc_irq_access_lock(dev);
	_osc_reg_read(dev, 0x06, &osc->_06);
	_osc_reg_read(dev, 0x07, &osc->_07);
	_osc_reg_read(dev, 0x18, &osc->_18);
	_osc_reg_read(dev, 0x19, &osc->_19);
	_osc_reg_read(dev, 0x1A, &osc->_1a);
	_osc_reg_read(dev, 0x1B, &osc->_1b);
	gsc_irq_access_unlock(dev);

	osc_cy22393_startup(&osc->osc, OSC_FREQ_REF_DEFAULT);
	_osc_regs_set_constants(&osc->osc);
}



//*****************************************************************************
void osc_cy22393_sio4_destroy(dev_data_t* dev)
{
	osc_cy22393_sio4_t*	osc	= &dev->osc.cy22393;

	os_sem_destroy(&osc->sem);
	memset(osc, 0, sizeof(osc[0]));
}



//*****************************************************************************
int osc_cy22393_sio4_measure_ioctl(chan_data_t* chan, s32* arg)
{
	int	ret;

	ret	= _measure_service(chan, arg);
	return(ret);
}



//*****************************************************************************
void osc_cy22393_sio4_open(chan_data_t* chan)
{
	// We leave the oscillator as is.
}



//*****************************************************************************
int osc_cy22393_sio4_program_ioctl(chan_data_t* chan, s32* arg)
{
	osc_cy22393_arg_t	data;
	dev_data_t*			dev		= chan->dev;
	int					index	= chan->index;
	s32					got;
	long				max;
	long				min;
	osc_cy22393_sio4_t*	osc		= &dev->osc.cy22393;
	int					ret		= 0;
	int					tries	= 0;
	s32					want	= arg[0];

	for (tries = 0; tries <= 20; tries++)	// A convenience loop.
	{
		if (arg[0] == -1)
		{
			// The service is supported.
			arg[0]	= 1;
			break;
		}

		if ((want < 0) || (want > _20MHZ))
		{
			ret	= -EINVAL;
			break;
		}

		ret	= os_sem_lock(&osc->sem);

		if (ret)
			break;

		data.freq_ref	= osc->osc.freq_ref;
		data.freq_want	= want;
		ret				= _compute_pd_osc_program(osc, index, &data);
		arg[0]			= osc->chan[index].freq_got;

		if (ret == 0)
			ret	= _osc_program(dev, osc);

		_chan_fw_config_write(dev, osc);

		if (ret)
		{
			os_sem_unlock(&osc->sem);
			break;
		}

		ret		= _chan_measure_freq(chan, &got);
		os_sem_unlock(&osc->sem);

		if (ret)
			break;

		// Verify the results.
		min	= (((s32) want / 10) - 1 - (((s32) want + 9999) / 10000)) * 10;
		max	= (((s32) want / 10) + 1 + (((s32) want + 9999) / 10000)) * 10;

		if (got < min)
		{
			continue;
		}
		else if (got > max)
		{
			continue;
		}
		else
		{
			break;
		}
	}

	return(ret);
}



//*****************************************************************************
int osc_cy22393_sio4_reference_ioctl(chan_data_t* chan, s32* arg)
{
	int	ret;

	ret	= _reference_service(chan, arg);
	return(ret);
}


