// $URL: http://subversion:8080/svn/gsc/trunk/drivers/gsc_common/driver/rtx/os_time.c $
// $Rev: 33965 $
// $Date: 2015-11-05 18:24:19 -0600 (Thu, 05 Nov 2015) $

// RTX driver module

#include "main.h"

// In other operating systems the system clock and the system timer may be
// very different from one another. For our purposes in this RTX driver,
// the time services code here uses the system timer as the system clock.



// #defines *******************************************************************

#define	_10M		(10L * 1000L * 1000L)



//*****************************************************************************
// hns = hundreds of ns, 1 = 100ns
static void _sleep_hns(long long hns)
{
	LARGE_INTEGER	li;

	if (hns < 1)
		hns	= 1;

	for (;;)
	{
		if (hns > _10M)
		{
			hns			-= _10M;
			li.QuadPart	= _10M;
			RtSleepFt(&li);
		}
		else
		{
			li.QuadPart	= hns;
			RtSleepFt(&li);
			break;
		}
	}
}



//*****************************************************************************
void os_time_init(void)
{
	LARGE_INTEGER	li;
	BOOL			test;

	test	= QueryPerformanceFrequency(&li);

	if (test == 0)
		li.QuadPart	= 0;

	if (li.QuadPart)
	{
		gsc_global.hr_timer		= 1;
		gsc_global.tick_rate	= li.QuadPart;
	}
	else
	{
		// This is based on CLOCK_2, which has a 100ns rate.
		gsc_global.hr_timer		= 0;
		gsc_global.tick_rate	= _10M;
	}
}



//*****************************************************************************
long os_time_delta_ms(const os_time_t* tt1, const os_time_t* tt2)
{
	os_time_tick_t	delta;
	long			ms;
	long			tpms;

	if ((tt1 == NULL) || (tt2 == NULL))
	{
		delta	= 0;
	}
	else if (tt1->count == tt2->count)
	{
		delta	= 0;
	}
	else if (tt1->count > tt2->count)
	{
		delta	= tt1->count - tt2->count;
	}
	else // if (tt1->tv_sec < tt2->tv_sec)
	{
		delta	= tt2->count - tt1->count;
	}

	tpms	= (long) (gsc_global.tick_rate / 1000);

	if (tpms < 1)
		tpms	= 1;

	ms	= (long) ((delta + tpms / 2) / tpms);
	return(ms);
}



//*****************************************************************************
void os_time_get(os_time_t* tt)
{
	LARGE_INTEGER	li;

	if (tt == NULL)
	{
	}
	else if (gsc_global.hr_timer)
	{
		QueryPerformanceCounter(&li);
		tt->count	= li.QuadPart;
	}
	else
	{
		RtGetClockTime(CLOCK_2, &li);
		tt->count	= li.QuadPart;
	}
}



//*****************************************************************************
os_time_tick_t os_time_ms_to_ticks(long ms)
{
	os_time_tick_t	ticks;

	ticks	= gsc_global.tick_rate * ms / 1000;
	return(ticks);
}



//*****************************************************************************
os_time_tick_t os_time_tick_get(void)
{
	os_time_t	tt;

	os_time_get(&tt);
	return(tt.count);
}



//*****************************************************************************
int os_time_tick_rate(void)
{
	int	tr;

	tr	= (int) gsc_global.tick_rate;
	return(tr);
}



//*****************************************************************************
void os_time_tick_sleep(int ticks)
{
	long long	hns;	// hundreds of ns, 1 = 100ns

	if (gsc_global.hr_timer)
	{
		hns	= (long long) _10M * ticks / gsc_global.tick_rate;
	}
	else
	{
		hns	= ticks;
	}

	_sleep_hns(hns);
}



//*****************************************************************************
int os_time_tick_timedout(os_time_tick_t tick_time)
{
	int			timedout	= 0;
	os_time_t	tt;

	os_time_get(&tt);
	timedout	= (tick_time >= tt.count) ? 1 : 0;
	return(timedout);
}



//*****************************************************************************
long os_time_ticks_to_ms(os_time_tick_t ticks)
{
	long	ms;
	long	tpms;	// ticks per ms

	tpms	= (long) (gsc_global.tick_rate / 1000);

	if (tpms < 1)
		tpms	= 1;

	ms	= (long) ((ticks + tpms / 2) / tpms);
	return(ms);
}



//*****************************************************************************
void os_time_us_delay(long us)
{
	long long	hns;	// hundreds of ns, 1 = 100ns

	// RTX doesn't have any delay capability.
	hns	= us * 10;
	_sleep_hns(hns);
}



//*****************************************************************************
void os_time_sleep_ms(long ms)
{
	long long	hns;	// hundreds of ns, 1 = 100ns

	hns	= ms * 10000;
	_sleep_hns(hns);
}



//*****************************************************************************
unsigned long long os_time_get_ms(void)
{
	LARGE_INTEGER		li;
	unsigned long long	ull;

	if (gsc_global.hr_timer)
		QueryPerformanceCounter(&li);
	else
		RtGetClockTime(CLOCK_2, &li);

	ull	= (unsigned long long) li.QuadPart * 1000 / gsc_global.tick_rate;
	return(ull);
}


//*****************************************************************************
unsigned long long os_time_get_us(void)
{
	LARGE_INTEGER		li;
	unsigned long long	ull;

	if (gsc_global.hr_timer)
		QueryPerformanceCounter(&li);
	else
		RtGetClockTime(CLOCK_2, &li);

	ull	= (unsigned long long) li.QuadPart * 1000000 / gsc_global.tick_rate;
	return(ull);
}


