// $URL: http://subversion:8080/svn/gsc/trunk/drivers/gsc_common/driver/rtx/os_res.c $
// $Rev: 33965 $
// $Date: 2015-11-05 18:24:19 -0600 (Thu, 05 Nov 2015) $

// RTX driver module
// This module provide support code for named semaphore and memory resources.

#include "main.h"



//*****************************************************************************
static void _append_str(char* dst, size_t size, const char* src)
{
	int	len;

	if ((dst) && (size) && (src))
	{
		len		= strlen(dst);
		dst		+= len;
		size	-= len + 1;

		for (; (size) && (src[0]); size--)
		{
			dst[0]	= src[0];
			dst++;
			src++;
		}

		dst[0]	= 0;
	}
}



//*****************************************************************************
static int _res_mem_create(
	size_t		size,
	int			read_write,
	os_mem_t*	mem,
	int			index,		// ignore if < 0,
	u32*		suffix,
	void**		vpp)
{
	char	buf[OS_RESOURCE_NAME_MAX_LEN];
	char*	name	= buf;
	void*	ptr;
	int		ret;
	u32		suf		= 0;

	if (index >= -1)
	{
		sprintf(name, "%s.%d", DEV_NAME, index);
	}
	else
	{
		name	= DEV_NAME;
		suf		= os_res_get_suffix();
	}

	ptr	= os_mem_data_alloc_named(size, mem, read_write, name, suf);

	if (ptr == NULL)
	{
		ret	= -ENOMEM;
	}
	else
	{
		ret	= 0;

		if (suffix)
			suffix[0]	= suf;

		if (vpp)
			vpp[0]	= ptr;
	}

	return(ret);
}



//*****************************************************************************
static int _res_sem_create(os_sem_t* sem, u32* suffix)
{
	int	ret;
	u32	suf;

	suf	= os_res_get_suffix();
	ret	= os_sem_create_named(sem, DEV_NAME, suf);

	if (suffix)
		suffix[0]	= suf;

	return(ret);
}



//*****************************************************************************
static int _res_main_mem_create(GSC_ALT_STRUCT_T* alt)
{
	int	ret;

	ret	= _res_mem_create(
			sizeof(os_main_mem_t),
			0,	// read only
			&alt->os.main_mem.mem,
			alt->dev_index,
			NULL,
			&alt->os.main_mem.ptr);
	return(ret);
}



//*****************************************************************************
static int _res_primary_create(GSC_ALT_STRUCT_T* alt)
{
	int	ret	= 0;
	int	tmp;

	tmp	= _res_sem_create(&alt->os.primary.xfer, &alt->os.main_mem.ptr->primary.xfer);
	ret	= ret ? ret : tmp;

	tmp	= _res_sem_create(&alt->os.primary.to_dev, &alt->os.main_mem.ptr->primary.to_dev);
	ret	= ret ? ret : tmp;

	if (tmp == 0)
		os_sem_lock(&alt->os.primary.to_dev);	// It should initially be unavailable

	tmp	= _res_sem_create(&alt->os.primary.to_app, &alt->os.main_mem.ptr->primary.to_app);
	ret	= ret ? ret : tmp;

	if (tmp == 0)
		os_sem_lock(&alt->os.primary.to_app);	// It should initially be unavailable

	tmp	= _res_mem_create(
			sizeof(os_msg_t),
			1,	// read write
			&alt->os.primary.mem,
			-99,
			&alt->os.main_mem.ptr->primary.mem,
			&alt->os.primary.msg);
	ret	= ret ? ret : tmp;

	return(ret);
}



//*****************************************************************************
static int _res_msg_mem_create(GSC_ALT_STRUCT_T* alt)
{
	int	ret	= 0;
	int	tmp;

	tmp	= _res_mem_create(
			sizeof(os_msg_t) * OS_MSG_LIST_SIZE,
			1,	// read write
			&alt->os.msg_mem.mem,
			-99,
			&alt->os.main_mem.ptr->msg_mem.mem,
			&alt->os.msg_mem.msgs);

	tmp	= os_sem_create(&alt->os.msg_mem.sem);
	ret	= ret ? ret : tmp;
	return(ret);
}



//*****************************************************************************
static int _res_msg_list_create(GSC_ALT_STRUCT_T* alt)
{
	int	i;
	int	ret	= 0;
	int	tmp;

	for (i = 0; i < OS_MSG_LIST_SIZE; i++)
	{
		tmp	= _res_sem_create(&alt->os.msg_list[i].xfer, &alt->os.main_mem.ptr->msg_list[i].xfer);
		ret	= ret ? ret : tmp;

		tmp	= _res_sem_create(&alt->os.msg_list[i].to_dev, &alt->os.main_mem.ptr->msg_list[i].to_dev);
		ret	= ret ? ret : tmp;

		if (tmp == 0)
			os_sem_lock(&alt->os.msg_list[i].to_dev);	// It should initially be unavailable

		tmp	= _res_sem_create(&alt->os.msg_list[i].to_app, &alt->os.main_mem.ptr->msg_list[i].to_app);
		ret	= ret ? ret : tmp;

		if (tmp == 0)
			os_sem_lock(&alt->os.msg_list[i].to_app);	// It should initially be unavailable

		alt->os.msg_list[i].msg	= &alt->os.msg_mem.msgs[i];
	}

	return(ret);
}



//*****************************************************************************
static void _res_rx_mem_create(GSC_ALT_STRUCT_T* alt)
{
#if defined(DEV_SUPPORTS_READ)

	alt->os.main_mem.ptr->rx.adrs	= alt->rx.mem.adrs;
	alt->os.main_mem.ptr->rx.size	= alt->rx.mem.bytes;

#else

	alt->os.main_mem.ptr->rx.adrs	= 0;
	alt->os.main_mem.ptr->rx.size	= 0;

#endif
}



//*****************************************************************************
static void _res_tx_mem_create(GSC_ALT_STRUCT_T* alt)
{
#if defined(DEV_SUPPORTS_WRITE)

	alt->os.main_mem.ptr->tx.adrs	= alt->tx.mem.adrs;
	alt->os.main_mem.ptr->tx.size	= alt->tx.mem.bytes;

#else

	alt->os.main_mem.ptr->tx.adrs	= 0;
	alt->os.main_mem.ptr->tx.size	= 0;

#endif
}



//*****************************************************************************
static int _res_create(GSC_ALT_STRUCT_T* alt)
{
	int	ret;
	int	tmp;

	if (alt->dev_index >= -1)
	{
		ret	= _res_main_mem_create(alt);

		if ((ret == 0) && (alt->os.main_mem.ptr))
		{
			tmp	= _res_primary_create(alt);
			ret	= ret ? ret : tmp;

			tmp	= _res_msg_mem_create(alt);
			ret	= ret ? ret : tmp;

			if ((ret == 0) && (alt->os.msg_mem.msgs))
			{
				tmp	= _res_msg_list_create(alt);
				ret	= ret ? ret : tmp;
			}

			_res_rx_mem_create(alt);
			_res_tx_mem_create(alt);
		}
	}
	else
	{
		// Nothing to do.
		ret	= 0;
	}

	return(ret);
}



//*****************************************************************************
void os_res_start(void)
{
	os_sem_create(&gsc_global.resources.sem);
	gsc_global.resources.count	= 1;
}



//*****************************************************************************
void os_res_stop(void)
{
	os_sem_destroy(&gsc_global.resources.sem);
}



//*****************************************************************************
u32 os_res_get_suffix(void)
{
	u32	count;
	int	ret;

	ret		= os_sem_lock(&gsc_global.resources.sem);
	count	= gsc_global.resources.count++;

	if (ret == 0)
		os_sem_unlock(&gsc_global.resources.sem);

	return(count);
}



//*****************************************************************************
void os_res_name_create(
	char*		dst,	// The name goes here.
	size_t		size,	// This is the size of the destination buffer.
	const char*	prefix,	// Should be "DEVICE" or "DEVICE.X". Ignored if NULL.
	const char*	type,	// Should be "sem", "mem" or similar.  Ignored if NULL.
						// Ignored if suffix is zero.
	u32			suffix)	// The 8 hex digit suffix. Ignored if zero.

						// The type is used only if the prefix or suffix is used.
{
	static const char*	hex	= "0123456789ABCDEF";

	char	buf[10];
	int		i;
	char*	ptr		= buf;
	int		val;

	for (;;)	// A convenience loop.
	{
		if ((dst == NULL) || (size < 1))
			break;

		dst[0]	= 0;

		if (suffix == 0)
			type	= NULL;

		if (prefix)
			_append_str(dst, size, prefix);

		if ((prefix) && ((type) || (suffix)))
			_append_str(dst, size, ".");

		if (type)
			_append_str(dst, size, type);

		if (((prefix) || (type)) && (suffix))
			_append_str(dst, size, ".");

		if (suffix)
		{
			for (i = 28; i >= 0; i-= 4)
			{
				val	= (int) ((suffix >> i) & 0xF);
				ptr[0]	= hex[val];
				ptr++;
			}

			ptr[0]	= 0;
			_append_str(dst, size, buf);
		}

		break;
	}
}



//*****************************************************************************
static void _tell_all_threads_to_quit(GSC_ALT_STRUCT_T* alt)
{
	int	i;

	for (i = 0; i < OS_MSG_LIST_SIZE; i++)
		alt->os.msg_list[i].thread.quit	= 1;

	alt->os.primary.thread.quit	= 1;
}



//*****************************************************************************
static void _unlock_all_semaphores(GSC_ALT_STRUCT_T* alt)
{
	int	i;

	for (i = 0; i < OS_MSG_LIST_SIZE; i++)
	{
		os_sem_unlock(&alt->os.msg_list[i].xfer);
		os_sem_unlock(&alt->os.msg_list[i].to_dev);
		os_sem_unlock(&alt->os.msg_list[i].to_app);
	}

	os_sem_unlock(&alt->os.msg_mem.sem);

	os_sem_unlock(&alt->os.primary.xfer);
	os_sem_unlock(&alt->os.primary.to_dev);
	os_sem_unlock(&alt->os.primary.to_app);
}



//*****************************************************************************
static void _end_all_threads(GSC_ALT_STRUCT_T* alt)
{
	int	i;

	for (i = 0; i < OS_MSG_LIST_SIZE; i++)
		os_thread_destroy(&alt->os.msg_list[i].thread);

	os_thread_destroy(&alt->os.primary.thread);
}



//*****************************************************************************
static void _destroy_all_semaphores(GSC_ALT_STRUCT_T* alt)
{
	int	i;

	for (i = 0; i < OS_MSG_LIST_SIZE; i++)
	{
		os_sem_destroy(&alt->os.msg_list[i].xfer);
		os_sem_destroy(&alt->os.msg_list[i].to_dev);
		os_sem_destroy(&alt->os.msg_list[i].to_app);
	}

	os_sem_destroy(&alt->os.msg_mem.sem);

	os_sem_destroy(&alt->os.primary.xfer);
	os_sem_destroy(&alt->os.primary.to_dev);
	os_sem_destroy(&alt->os.primary.to_app);
}



//*****************************************************************************
static void _unlink_all_data_pointers(GSC_ALT_STRUCT_T* alt)
{
	int	i;

	for (i = 0; i < OS_MSG_LIST_SIZE; i++)
		alt->os.msg_list[i].msg	= NULL;

	alt->os.msg_mem.msgs	= NULL;

	alt->os.primary.msg		= NULL;

	alt->os.main_mem.ptr	= NULL;
}



//*****************************************************************************
static void _clear_all_memory_resources(GSC_ALT_STRUCT_T* alt)
{
	if (alt->os.msg_mem.mem.ptr)
		memset(alt->os.msg_mem.mem.ptr, 0, alt->os.msg_mem.mem.bytes);

	if (alt->os.primary.mem.ptr)
		memset(alt->os.primary.mem.ptr, 0, alt->os.primary.mem.bytes);

	if (alt->os.main_mem.mem.ptr)
		memset(alt->os.main_mem.mem.ptr, 0, alt->os.main_mem.mem.bytes);
}



//*****************************************************************************
static void _free_all_memory_resources(GSC_ALT_STRUCT_T* alt)
{
	os_mem_data_free_named(&alt->os.msg_mem.mem);
	os_mem_data_free_named(&alt->os.primary.mem);
	os_mem_data_free_named(&alt->os.main_mem.mem);
}



//*****************************************************************************
static void _res_destroy(GSC_ALT_STRUCT_T* alt)
{
	_clear_all_memory_resources(alt);	// zero out shared memory
	_tell_all_threads_to_quit(alt);		// thread->quit = 1
	_unlock_all_semaphores(alt);		// most threads should resume
	_destroy_all_semaphores(alt);		// to the degree possible
	_end_all_threads(alt);				// wait for each thread to end
	_unlink_all_data_pointers(alt);		// 
	_free_all_memory_resources(alt);	// zero out shared memory
}


//*****************************************************************************
int os_res_create(dev_data_t* dev)
{
	int	ret;

#if (GSC_DEVS_PER_BOARD > 1)

	int	i;
	int	tmp;

	ret	= 0;

	for (i = 0; i < GSC_DEVS_PER_BOARD; i++)
	{
		tmp	= _res_create(&dev->channel[i]);
		ret	= ret ? ret : tmp;
	}

#else

	ret	= _res_create(dev);

#endif

	return(ret);
}



//*****************************************************************************
void os_res_destroy(dev_data_t* dev)
{
#if (GSC_DEVS_PER_BOARD > 1)

	int	i;

	for (i = 0; i < GSC_DEVS_PER_BOARD; i++)
		_res_destroy(&dev->channel[i]);

#else

	_res_destroy(dev);

#endif
}


