// $URL: http://subversion:8080/svn/gsc/trunk/drivers/gsc_common/driver/rtx/os_main.h $
// $Rev: 33965 $
// $Date: 2015-11-05 18:24:19 -0600 (Thu, 05 Nov 2015) $

// RTX driver module

#ifndef __OS_MAIN_H__
#define __OS_MAIN_H__

#include <errno.h>
#include <stdio.h>
#include <rtapi.h>

#include "os_lib.h"



// #defines *******************************************************************

// This is for the OS specific code.
#define	OS_COMMON_VERSION					"0"
// 0	Initial release.

#define	MAX_DEV_QTY							32

#define	printf								RtPrintf

#ifndef	GSC_DEVS_PER_BOARD
#define	GSC_DEVS_PER_BOARD					1
#endif

#define	OS_CAST_VOIDP_TO_U32				u32) (long) (long long
#define	GSC_RX_IO_BUF_PRESERVE				// don't overwrite during a read call
#define	GSC_TX_IO_BUF_PRESERVE				// don't overwrite during a write call



// data types *****************************************************************

typedef LONGLONG	os_time_tick_t;
typedef void*		VADDR_T;

typedef struct
{
	// These fields MUST be kept as is.
	int				index;		// BARx
	int				offset;		// Offset of BARx register in PCI space.
	u32				reg;		// Actual BARx register value.
	u32				flags;		// lower register bits
	int				io_mapped;	// Is this an I/O mapped region?
	unsigned long	phys_adrs;	// Physical address of region.
	u32				size;		// Region size in bytes.

	// The following are not order specific.

	// These are computed when mapped in.
	u32				requested;	// Is resource requested from OS?
	VADDR_T			vaddr;		// Kernel virtual address.

	// Any OS specific fields must go after this point.
	dev_data_t*		dev;
} os_bar_t;

typedef struct
{
	HANDLE			handle;		// valid if handle is non-zero
} os_event_t;

typedef struct
{
	HANDLE			handle;
	int				allotted;
	int				irq_line;	// interrupt line
} os_irq_t;

typedef struct
{
	unsigned long	adrs;		// physical address for DMA, or zero otherwise
	void*			ptr;		// pointer to memory
	u32				bytes;		// size of allocation
	HANDLE			handle;
} os_mem_t;

typedef struct
{
	// These are used to identify the device to be accessed.
	ULONG			BusNumber;	// found during enumeration
	ULONG			SlotNumber;	// found during enumeration
	dev_data_t*		dev;		// needed for init
} os_pci_t;

typedef struct
{
	int				bus_no;		// loop during enumeration
	int				dev_no;		// loop during enumeration
	int				func_no;	// loop during enumeration
	ULONG			BusNumber;	// found during enumeration
	ULONG			SlotNumber;	// found during enumeration
} os_pci_enum_t;

typedef	struct
{
	HANDLE			handle;
} os_sem_t;

typedef struct
{
	//	RTX does not support a spin loop. Consequently we use a semaphore
	//	instead and service interrupts in an Interrupt Service Thread.

	os_sem_t		sem;		// used in place of a spin lock
} os_spinlock_t;

typedef struct
{
	GSC_ALT_STRUCT_T*	alt;
	int					priority;	// run thread at this priority
	int					index;		// for service thread use
} os_thread_data_t;

typedef	struct
{
	HANDLE				handle;
	DWORD				thread_id;
	int					started;
	int					quit;
	int					done;
} os_thread_t;

typedef struct
{
	os_time_tick_t	count;
} os_time_t;

typedef struct
{
	int			in_use;
	int			quit;
	int			done;
	os_thread_t	thread;
	os_sem_t	xfer;
	os_sem_t	to_dev;
	os_sem_t	to_app;
	os_mem_t	mem;		// unused for service threads
	os_msg_t*	msg;
} os_msg_entry_t;

typedef struct
{
	struct
	{
		os_mem_t		mem;
		os_main_mem_t*	ptr;
	} main_mem;

	// The content of this structure mirrors that of the os_main_met structure.

	// corresponds to os_maim_mem_t.primary
	os_msg_entry_t		primary;

	// corresponds to os_maim_mem_t.msg_mem
	struct
	{
		os_mem_t		mem;
		os_msg_t*		msgs;
		os_sem_t		sem;
	} msg_mem;

	// corresponds to os_maim_mem_t.msg_list
	os_msg_entry_t		msg_list[OS_MSG_LIST_SIZE];

} os_data_t;

typedef struct
{
	int				dev_qty;
	int				driver_loaded;

	// The list must be locked to be accessed.
	os_sem_t		sem;
	dev_data_t*		dev_list[MAX_DEV_QTY];

	// These are for the time services.
	int				hr_timer;	// High Resolution Timer present?
	os_time_tick_t	tick_rate;	// 100ns or CLOCK_2

	// This is used to determing if a driver is already running.
	HANDLE			driver_is_running;

	// This is used for composing resource names for HANDLE objects.
	// This includes semaphores, ...
	struct
	{
		os_sem_t	sem;
		u32			count;				// get handle to incriment count.
	} resources;

} gsc_global_t;



// prototypes *****************************************************************

int				os_bar_create(dev_data_t* dev, int index, os_bar_t* bar);
void			os_bar_destroy(os_bar_t* bar);

int				os_close(int dev_id);

void			os_event_create(os_event_t* evnt);
void			os_event_destroy(os_event_t* evnt);
void			os_event_resume(os_event_t* evnt);
void			os_event_wait(os_event_t* evnt, os_time_tick_t timeout);

int				os_ioctl(int dev_id, int cmd, void* arg);
int				os_irq_acquire(dev_data_t* dev);
void			os_irq_release(dev_data_t* dev);

int				os_mem_copy_from_user_ioctl(void* dst, const void* src, long size);
int				os_mem_copy_from_user_tx(void* dst, const void* src, long size);
int				os_mem_copy_to_user_ioctl(void* dst, const void* src, long size);
int				os_mem_copy_to_user_rx(void* dst, const void* src, long size);
void*			os_mem_data_alloc(size_t size);
void*			os_mem_data_alloc_named(size_t size, os_mem_t* mem, int rw, const char* name, u32 suffix);
void			os_mem_data_free(void* ptr);
void			os_mem_data_free_named(os_mem_t* mem);
void*			os_mem_dma_alloc(size_t* size, os_mem_t* mem);
void			os_mem_dma_free(os_mem_t* mem);

int				os_open(int index, s32* dev_id);

void			os_pci_clear_master(os_pci_t* pci);
void			os_pci_dev_disable(os_pci_t* pci);
int				os_pci_dev_enable(os_pci_t* pci);
void			os_pci_enum_init(os_pci_enum_t* pcie);
int				os_pci_enum_next(os_pci_enum_t* pcie);
int				os_pci_set_master(os_pci_t* pci);
int				os_primary_thread_create(dev_data_t* dev);

int				os_read(int dev_id, void *dest, int count);

void			os_reg_mem_mx_u8	(dev_data_t* dev, VADDR_T va, u8  value, u8  mask);
void			os_reg_mem_mx_u16	(dev_data_t* dev, VADDR_T va, u16 value, u16 mask);
void			os_reg_mem_mx_u32	(dev_data_t* dev, VADDR_T va, u32 value, u32 mask);
u8				os_reg_mem_rx_u8	(dev_data_t* dev, VADDR_T va);
u16				os_reg_mem_rx_u16	(dev_data_t* dev, VADDR_T va);
u32				os_reg_mem_rx_u32	(dev_data_t* dev, VADDR_T va);
void			os_reg_mem_tx_u8	(dev_data_t* dev, VADDR_T va, u8 value);
void			os_reg_mem_tx_u16	(dev_data_t* dev, VADDR_T va, u16 value);
void			os_reg_mem_tx_u32	(dev_data_t* dev, VADDR_T va, u32 value);

void			os_reg_pci_mx_u8	(dev_data_t* dev, os_pci_t* pci, u16 offset, u8  value, u8  mask);
void			os_reg_pci_mx_u16	(dev_data_t* dev, os_pci_t* pci, u16 offset, u16 value, u16 mask);
void			os_reg_pci_mx_u32	(dev_data_t* dev, os_pci_t* pci, u16 offset, u32 value, u32 mask);
u8				os_reg_pci_rx_u8	(dev_data_t* dev, os_pci_t* pci, u16 offset);
u16				os_reg_pci_rx_u16	(dev_data_t* dev, os_pci_t* pci, u16 offset);
u32				os_reg_pci_rx_u32	(dev_data_t* dev, os_pci_t* pci, u16 offset);
void			os_reg_pci_tx_u8	(dev_data_t* dev, os_pci_t* pci, u16 offset, u8 value);
void			os_reg_pci_tx_u16	(dev_data_t* dev, os_pci_t* pci, u16 offset, u16 value);
void			os_reg_pci_tx_u32	(dev_data_t* dev, os_pci_t* pci, u16 offset, u32 value);

int				os_res_create(dev_data_t* dev);
void			os_res_destroy(dev_data_t* dev);
u32				os_res_get_suffix(void);
void			os_res_name_create(char* dst, size_t size, const char* prefix, const char* type, u32 suffix);
void			os_res_start(void);
void			os_res_stop(void);

int				os_root_close(void);
int				os_root_start(void);
void			os_root_stop(void);
int				os_root_ioctl(int cmd, void* arg);
int				os_root_open(void);
int				os_root_read(int count);
int				os_root_write(void);

int				os_sem_create(os_sem_t* sem);
int				os_sem_create_named(os_sem_t* sem, const char* name, u32 suffix);
void			os_sem_destroy(os_sem_t* sem);
int				os_sem_lock(os_sem_t* sem);
void			os_sem_unlock(os_sem_t* sem);
int				os_service_thread_create(os_thread_data_t* data);
int				os_spinlock_create(os_spinlock_t* lock);
void			os_spinlock_destroy(os_spinlock_t* lock);
void			os_spinlock_lock(os_spinlock_t* lock);
void			os_spinlock_unlock(os_spinlock_t* lock);

int				os_thread_create(os_thread_t* thread, long (*func)(os_thread_data_t* data), os_thread_data_t* data);
void			os_thread_destroy(os_thread_t* thread);

long			os_time_delta_ms(const os_time_t* tt1, const os_time_t* tt2);
void			os_time_get(os_time_t* tt);
void			os_time_init(void);
os_time_tick_t	os_time_ms_to_ticks(long ms);
os_time_tick_t	os_time_tick_get(void);
int				os_time_tick_rate(void);
void			os_time_tick_sleep(int ticks);
int				os_time_tick_timedout(os_time_tick_t tick_time);
long			os_time_ticks_to_ms(os_time_tick_t ticks);
void			os_time_us_delay(long us);
void			os_time_sleep_ms(long ms);

int				os_write(int dev_id, const void *src, int count);

unsigned long long		os_time_get_ms(void);
unsigned long long		os_time_get_us(void);



#endif
