// $URL: http://subversion:8080/svn/gsc/trunk/drivers/gsc_common/driver/rtx/os_init.c $
// $Rev: 33965 $
// $Date: 2015-11-05 18:24:19 -0600 (Thu, 05 Nov 2015) $

// RTX driver module

#include <time.h>

#include "main.h"



// variables ******************************************************************

static	int	_request_exit	= 0;



//*****************************************************************************
static int _is_already_running(void)
{
	DWORD			error;
	HANDLE			handle;
	const TCHAR*	name	= DEV_NAME ".sem.running";
	int				run_now;	// 0 = don't run, !0 = do run now

	gsc_global.driver_is_running	= 0;
	SetLastError(0);
	handle	= RtCreateSemaphore(NULL, 0, 1, name);
	error	= RtGetLastError();

	if (handle == NULL)
	{
		// This execution failed.
		run_now	= 0;
		printf("%s: unable to create 'running' handle.\n", DEV_NAME);
	}
	else if (_request_exit)
	{
		run_now	= 0;
		printf("%s: driver exit requested\n", DEV_NAME);
		RtReleaseSemaphore(handle, 1, NULL);
		RtCloseHandle(handle);
	}
	else if (error == 0)
	{
		// This is the first instanace.
		run_now	= 1;
		gsc_global.driver_is_running	= handle;
	}
	else
	{
		run_now	= 0;
		printf("%s: driver already running\n", DEV_NAME);
		RtCloseHandle(handle);
	}

	return(run_now);
}



//*****************************************************************************
static int _parse_arguments(int argc, char** argv)
{
	int			help	= 0;
	int			i;
	const char*	name;
	int			ret		= 0;	// 0 = continue, !0 = exit program
	const char*	tmp;

	for (i = 1; i < argc; i++)
	{
		if (strcmp(argv[i], "-h")== 0)
		{
			help	= 1;
			continue;
		}

		if (strcmp(argv[i], "-x")== 0)
		{
			_request_exit	= 1;
			continue;
		}

		ret	= 1;
		printf("%s: invalid argument: '%s'\n", DEV_NAME, argv[i]);
	}

	if (help)
	{
		// Get the driver's name.
		name	= argv[0];

		for (; name;)
		{
			tmp	= strchr(name, '\\');

			if (tmp)
				name	= tmp;
			else
				break;
		}

		printf("%s: command line usage\n", DEV_NAME);
		printf("%s <-h> <-x>\n", name);
		printf("  -h  Display command line usage information then exit.\n");
		printf("  -x  Request that the active running driver exit."
					" This instance exits as well.\n");
		ret	= 1;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	_cleanup
*
*	Purpose:
*
*		Clean things up when the kernel is about to unload the module.
*
*	Arguments:
*
*		None.
*
*	Returned:
*
*		None.
*
******************************************************************************/

static void _cleanup(void)
{
	dev_data_t*	dev	= NULL;
	int			i;

	printf("%s: driver unloading: version %s\n", DEV_NAME, GSC_DRIVER_VERSION);

	for (i = 0; i < (int) ARRAY_ELEMENTS(gsc_global.dev_list); i++)
	{
		if (gsc_global.dev_list[i])
		{
			dev	= gsc_global.dev_list[i];
			os_res_destroy(dev);			// devices should all be closed
			gsc_init_remove_device(dev);	// must perform after resource cleanup
			os_mem_data_free(dev);
		}
	}

	os_root_stop();
	os_res_stop();
	os_sem_destroy(&gsc_global.sem);
	os_sem_destroy(&gsc_global.resources.sem);

	if (gsc_global.driver_is_running)
	{
		RtCloseHandle(gsc_global.driver_is_running);
		gsc_global.driver_is_running	= NULL;
	}

	gsc_global.dev_qty			= 0;
	gsc_global.driver_loaded	= 0;
	printf("%s: driver unloaded: version %s\n", DEV_NAME, GSC_DRIVER_VERSION);
}



/******************************************************************************
*
*	Function:	main
*
*	Purpose:
*
*		Initialize the driver upon loading.
*
*	Arguments:
*
*		None.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

int main(int argc, char** argv)
{
	int				i;
	dev_data_t*		dev		= NULL;
	DWORD			dw;
	os_pci_t*		pci		= NULL;
	os_pci_enum_t	pci_enum;
	int				ret		= 0;
	int				run_now;	// 0 = don't run now, !0 = do run now
	int				size	= sizeof(dev_data_t);

	for (;;)	// A convenience loop.
	{
		i	= _parse_arguments(argc, argv);

		if (i)
			break;

		run_now	= _is_already_running();

		if (run_now == 0)
			break;

		// Begin initialization process.
		printf("%s: driver loading: version %s\n",
				DEV_NAME,
				GSC_DRIVER_VERSION);

		os_time_init();
		os_sem_create(&gsc_global.sem);
		os_sem_create(&gsc_global.resources.sem);
		os_res_start();

		// Locate the devices and add them to our list.
		os_pci_enum_init(&pci_enum);

		for (; os_pci_enum_next(&pci_enum);)
		{
			if (dev == NULL)
			{
				dev	= os_mem_data_alloc(size);

				if (dev == NULL)
					break;
			}

			if (pci == NULL)
			{
				pci	= os_mem_data_alloc(sizeof(os_pci_t));

				if (pci == NULL)
					break;
			}

			memset(dev, 0, size);
			memset(pci, 0, sizeof(os_pci_t));
			dev->pci		= pci;
			pci->dev		= dev;
			pci->BusNumber	= pci_enum.BusNumber;
			pci->SlotNumber	= pci_enum.SlotNumber;
			ret				= gsc_init_add_device(dev);

			if (ret <= 0)
				continue;

			ret	= os_res_create(dev);

			if (ret == 0)
				ret	= os_primary_thread_create(dev);

			if (ret == 0)
			{
				// The referenced device was added.
				printf("%s: device loaded: %s\n", DEV_NAME, dev->model);
				dev	= NULL;
				pci	= NULL;
			}
		}

		// Free memory no longer used.

		if (dev)
		{
			os_mem_data_free(dev);
			dev	= NULL;
		}

		if (pci)
		{
			os_mem_data_free(pci);
			pci	= NULL;
		}

		// Perform initialization following device discovery.

		if (gsc_global.dev_qty <= 0)
		{
			ret	= -ENODEV;
			printf(	"%s: no devices: version %s\n",
					DEV_NAME,
					GSC_DRIVER_VERSION);
			_cleanup();
			break;
		}

		ret	= os_root_start();	// must occur after board detection

		if (ret == 0)
			ret	= gsc_ioctl_init();

		if (ret)
		{
			_cleanup();
			break;
		}

		gsc_global.driver_loaded	= 1;
		printf(	"%s: driver loaded: version %s\n",
				DEV_NAME,
				GSC_DRIVER_VERSION);
		ret	= 0;

		if (gsc_global.driver_is_running)
		{
			// Continue running until told to stop.

			for (;;)
			{
				dw	= RtWaitForSingleObject(gsc_global.driver_is_running, 5000);

				if (dw == WAIT_TIMEOUT)
				{
					// We should continue waiting.
					continue;
				}

				if (dw == WAIT_OBJECT_0)
				{
					// We've been asked to exit.
					break;
				}

				// There has been a problem.
				printf("%s: idle state error ... exiting.\n", DEV_NAME);
				break;
			}
		}

		_cleanup();
		break;
	}

	return(ret);
}


