// $URL: http://subversion:8080/svn/gsc/trunk/drivers/gsc_common/driver/rtx/os_bar.c $
// $Rev: 33965 $
// $Date: 2015-11-05 18:24:19 -0600 (Thu, 05 Nov 2015) $

// RTX driver module

#include "main.h"



//*****************************************************************************
static int _bar_io_acquire(os_bar_t* bar)
{
	int		ret;
	BOOL	tst;

	tst	= RtEnablePortIo((void*) bar->phys_adrs, bar->size);

	if (tst == 0)
	{
		// This BAR region request failed.
		ret	= -EFAULT;
	}
	else
	{
		// All went well.
		ret				= 0;
		bar->requested	= 1;
		bar->vaddr		= (VADDR_T) bar->phys_adrs;
	}

	return(ret);
}



//*****************************************************************************
static int _bar_mem_acquire(os_bar_t* bar)
{
	ULONG			as		= 1;	// address space
	LARGE_INTEGER	input;
	LARGE_INTEGER	output;
	int				ret;
	BOOL			tst;
	void*			vp;

	for (;;)	// A convenience loop.
	{
		input.QuadPart	= bar->phys_adrs;
		tst	= RtTranslateBusAddress(PCIBus, bar->index, input, &as, &output);

		if (tst == 0)
		{
			// Address translation failed.
			ret	= -EFAULT;
			break;
		}

		vp	= (void*) RtMapMemory(output, bar->size, MmNonCached);

		if (vp == NULL)
		{
			// This mapping request failed.
			ret	= -EFAULT;
			break;
		}

		// All went well.
		ret				= 0;
		bar->requested	= 1;
		bar->vaddr		= vp;
		break;
	}

	return(ret);
}



//*****************************************************************************
static u32 _bar_size(dev_data_t* dev, u8 index)
{
	int	offset	= 0x10 + (4 * index);
	u32	reg;
	u32	size;

	reg	= os_reg_pci_rx_u32(dev, dev->pci, offset);
	// Use a PCI feature to get the address mask.
	os_reg_pci_tx_u32(dev, dev->pci, offset, 0xFFFFFFFF);
	size	= os_reg_pci_rx_u32(dev, dev->pci, offset);
	// Restore the register.
	os_reg_pci_tx_u32(dev, dev->pci, offset, reg);
	size	= 1 + ~(size & 0xFFFFFFF0);

	return(size);
}



//*****************************************************************************
int os_bar_create(dev_data_t* dev, int index, os_bar_t* bar)
{
	int	ret;

	memset(bar, 0,sizeof(os_bar_t));
	bar->dev		= dev;
	bar->index		= index;
	bar->offset		= 0x10 + 4 * index;
	bar->size		= _bar_size(dev, index);
	bar->reg		= os_reg_pci_rx_u32(dev, dev->pci, bar->offset);

	if (bar->reg & D0)
	{
		bar->flags		= bar->reg & 0x3;
		bar->io_mapped	= 1;
		bar->phys_adrs	= bar->reg & 0xFFFFFFFC;
		ret				= _bar_io_acquire(bar);
	}
	else
	{
		bar->flags		= bar->reg & 0xF;
		bar->io_mapped	= 0;
		bar->phys_adrs	= bar->reg & 0xFFFFFFF0;
		ret				= _bar_mem_acquire(bar);
	}

	return(ret);
}



//*****************************************************************************
void os_bar_destroy(os_bar_t* bar)
{
	if (bar->requested == 0)
	{
		// There is nothing to release.
	}
	else if (bar->io_mapped)
	{
		RtDisablePortIo((void*) bar->phys_adrs, bar->size);
	}
	else
	{
		RtUnmapMemory((void*) bar->vaddr);
	}

	memset(bar, 0,sizeof(os_bar_t));
}


