// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_2.x.x_GSC_DN/driver/irq.c $
// $Rev: 32668 $
// $Date: 2015-07-08 17:14:51 -0500 (Wed, 08 Jul 2015) $

#include "main.h"



//*****************************************************************************
static void _rcc_sw_fifo_push(chan_data_t* chan, u32 status)
{
	u8	ccsr_ub;
	int	index;
	s32	qty;
	u16	rccr;
	u16	rccr_lb;
	u16	rccr_ub;

	if (chan->rcc_fifo.enabled)
	{
		if (status == SIO4_RCC_SW_FIFO_READ_FRAME_END)
		{
			ccsr_ub	= os_reg_mem_rx_u8(NULL, chan->vaddr.usc_ccsr_ub);

			if (ccsr_ub % D6)
			{
				// An RCC entry is available. Get the frame size.
				rccr_lb	= os_reg_mem_rx_u8(NULL, chan->vaddr.usc_rccr_lb);
				rccr_ub	= os_reg_mem_rx_u8(NULL, chan->vaddr.usc_rccr_ub);
				rccr	= (rccr_ub << 8) | rccr_lb;
				status	|= rccr ^ 0xFFFF;
			}
		}

		// Add the new record.
		qty	= chan->rcc_fifo.push - chan->rcc_fifo.pull;

		if (qty == RCC_FIFO_SIZE)
		{
			// Arrrgh! There is an overrun.
			index	= (chan->rcc_fifo.push - 1) % RCC_FIFO_SIZE;
			chan->rcc_fifo.fifo[index]	|= SIO4_RCC_SW_FIFO_READ_OVER_FRAME;
		}
		else
		{
			// Add a new RCC software FIFO record.
			index	= chan->rcc_fifo.push % RCC_FIFO_SIZE;
			chan->rcc_fifo.fifo[index]	= status;
			chan->rcc_fifo.push++;
		}
	}
}



//*****************************************************************************
static int _irq_isr_usc_rx_status(dev_data_t* dev, chan_data_t* chan)
{
	u8	bit;
	u8	irq;
	u8	rcsr_lb;
	u32	resume		= 0;
	u8	ricr_lb;

	// Clear the DCCR's Rx Status IP bit.
	os_reg_mem_tx_u8(NULL, chan->vaddr.usc_dccr_lb, 0x80 | D5);

	// See which sources are asserted.
	rcsr_lb	= os_reg_mem_rx_u8(NULL, chan->vaddr.usc_rcsr_lb);
	ricr_lb	= os_reg_mem_rx_u8(NULL, chan->vaddr.usc_ricr_lb);
	irq		= ricr_lb & rcsr_lb & 0xF6;

	// Handle the highest priority source.

	if (irq & D7)
	{
		// Rx Exited Hunt
		bit		= D7;
		resume	= SIO4_WAIT_USC_RX_EXITED_HUNT;
	}
	else if (irq & D6)
	{
		// Rx Idle Received
		bit		= D6;
		resume	= SIO4_WAIT_USC_RX_IDLE_RECEIVED;
	}
	else if (irq & D5)
	{
		// Rx Break/Abort
		_rcc_sw_fifo_push(chan, SIO4_RCC_SW_FIFO_READ_BREAK_ABRT);
		bit		= D5;
		resume	= SIO4_WAIT_USC_RX_BREAK_ABORT;
	}
	else if (irq & D4)
	{
		// Rx Bound
		bit		= D4;
		resume	= SIO4_WAIT_USC_RX_BOUND;
	}
	else if (irq & D2)
	{
		// Rx Abort/Parity Error
		_rcc_sw_fifo_push(chan, SIO4_RCC_SW_FIFO_READ_ABORT_PE);
		bit		= D2;
		resume	= SIO4_WAIT_USC_RX_ABORT_PAR_ERROR;
	}
	else if (irq & D1)
	{
		// Rx Overrun
		_rcc_sw_fifo_push(chan, SIO4_RCC_SW_FIFO_READ_OVER_DATA);
		bit		= D1;
		resume	= SIO4_WAIT_USC_RX_OVERRUN;
	}
	else
	{
		// A spurious interrupt.
		bit		= 0;
		resume	= SIO4_WAIT_USC_SPURIOUS;
	}

	if (bit)
	{
		// Clear the Interrupt Pending bit.
		os_reg_mem_tx_u8(NULL, chan->vaddr.usc_rcsr_lb, bit);

		// Disable then re-enable the IA bits.
		os_reg_mem_tx_u8(NULL, chan->vaddr.usc_ricr_lb, 0);		// Disable all
		os_reg_mem_tx_u8(NULL, chan->vaddr.usc_ricr_lb, ricr_lb);	// Re-enable previous
	}

	// Resume any awaiting threads.
	gsc_wait_resume_irq_alt(chan, resume);
	return(bit);
}



//*****************************************************************************
static int _irq_isr_usc_rx_data(dev_data_t* dev, chan_data_t* chan)
{
	// Clear the DCCR's Rx Data IP bit.
	os_reg_mem_tx_u8(NULL, chan->vaddr.usc_dccr_lb, 0x80 | D4);

	// Disable the Rx Data interrupt.
	os_reg_mem_tx_u8(NULL, chan->vaddr.usc_icr_lb, 0x80 | D4);

	// Rx DATA
	gsc_wait_resume_irq_alt(chan, SIO4_WAIT_USC_RX_DATA);

	return(1);
}



//*****************************************************************************
static int _irq_isr_usc_tx_status(dev_data_t* dev, chan_data_t* chan)
{
	u8	bit;
	u8	irq;
	u32	resume		= 0;
	u8	tcsr_lb;
	u8	ticr_lb;

	// Clear the DCCR's Rx Status IP bit.
	os_reg_mem_tx_u8(NULL, chan->vaddr.usc_dccr_lb, 0x80 | D3);

	// See which sources are asserted.
	tcsr_lb	= os_reg_mem_rx_u8(NULL, chan->vaddr.usc_tcsr_lb);
	ticr_lb	= os_reg_mem_rx_u8(NULL, chan->vaddr.usc_ticr_lb);
	irq		= ticr_lb & tcsr_lb & 0xFA;

	// Handle the highest priority source.

	if (irq & D7)
	{
		// Tx Preamble Sent
		bit		= D7;
		resume	= SIO4_WAIT_USC_TX_PREAMBLE_SENT;
	}
	else if (irq & D6)
	{
		// Tx Idle Sent
		bit		= D6;
		resume	= SIO4_WAIT_USC_TX_IDLE_SENT;
	}
	else if (irq & D5)
	{
		// Tx Abort Sent
		bit		= D5;
		resume	= SIO4_WAIT_USC_TX_ABORT_SENT;
	}
	else if (irq & D4)
	{
		// Tx EOF/EOM Sent
		bit		= D4;
		resume	= SIO4_WAIT_USC_TX_END_SENT;
	}
	else if (irq & D3)
	{
		// Tx CRC Sent
		bit		= D3;
		resume	= SIO4_WAIT_USC_TX_CRC_SENT;
	}
	else if (irq & D1)
	{
		// Tx Underrun
		bit		= D1;
		resume	= SIO4_WAIT_USC_TX_UNDERRUN;
	}
	else
	{
		// A spurious interrupt.
		bit		= 0;
		resume	= SIO4_WAIT_USC_SPURIOUS;
	}

	if (bit)
	{
		// Clear the Interrupt Pending bit.
		os_reg_mem_tx_u8(NULL, chan->vaddr.usc_tcsr_lb, bit);

		// Disable then re-enable the IA bits.
		os_reg_mem_tx_u8(NULL, chan->vaddr.usc_ticr_lb, 0);		// Disable all
		os_reg_mem_tx_u8(NULL, chan->vaddr.usc_ticr_lb, ticr_lb);	// Re-enable previous
	}

	// Resume any awaiting threads.
	gsc_wait_resume_irq_alt(chan, resume);
	return(bit);
}



//*****************************************************************************
static int _irq_isr_usc_tx_data(dev_data_t* dev, chan_data_t* chan)
{
	// Clear the DCCR's Tx Data IP bit.
	os_reg_mem_tx_u8(NULL, chan->vaddr.usc_dccr_lb, 0x80 | D2);

	// Disable the Tx Data interrupt.
	os_reg_mem_tx_u8(NULL, chan->vaddr.usc_icr_lb, 0x80 | D2);

	// Tx DATA
	gsc_wait_resume_irq_alt(chan, SIO4_WAIT_USC_TX_DATA);

	return(1);
}



//*****************************************************************************
static int _irq_isr_usc_iop_status(dev_data_t* dev, chan_data_t* chan)
{
	u16	handled	= 0;
	u16	misr;
	u16	misr_lb;
	u16	misr_ub;
	u16	sicr;
	u16	sicr_lb;
	u16	sicr_ub;

	// Clear the DCCR's IOP Status IP bit.
	os_reg_mem_tx_u8(NULL, chan->vaddr.usc_dccr_lb, 0x80 | D1);

	// See which sources are asserted.
	misr_lb	= os_reg_mem_rx_u8(NULL, chan->vaddr.usc_misr_lb);
	misr_ub	= os_reg_mem_rx_u8(NULL, chan->vaddr.usc_misr_ub);
	misr	= (misr_ub << 8) | misr_lb;

	sicr_lb	= os_reg_mem_rx_u8(NULL, chan->vaddr.usc_sicr_lb);
	sicr_ub	= os_reg_mem_rx_u8(NULL, chan->vaddr.usc_sicr_ub);
	sicr	= (sicr_ub << 8) | sicr_lb;

	for (;;)	// A convenience loop.
	{
		// Handle the highest priority source.

		if ((sicr & D15) && (misr & D15) && (misr & D14))
		{
			// I/O Pin RxC Fall
			handled	= 1;

			// Disable the interrupt due to the expected high interrupt rate.
			sicr	&= ~D15;
			os_reg_mem_tx_u8(NULL, chan->vaddr.usc_sicr_ub, sicr >> 8);

			// Clear the Latch/Unlatch bit.
			os_reg_mem_tx_u8(NULL, chan->vaddr.usc_misr_ub, D15 >> 8);

			// Resume any awaiting threads.
			gsc_wait_resume_irq_alt(chan, SIO4_WAIT_USC_IOP_RXC_FALL);
			break;
		}

		if ((sicr & D14) && (misr & D15) && ((misr & D14) == 0))
		{
			// I/O Pin RxC Rise
			handled	= 1;

			// Disable the interrupt due to the expected high interrupt rate.
			sicr	&= ~D14;
			os_reg_mem_tx_u8(NULL, chan->vaddr.usc_sicr_ub, sicr >> 8);

			// Clear the Latch/Unlatch bit.
			os_reg_mem_tx_u8(NULL, chan->vaddr.usc_misr_ub, D15 >> 8);

			// Resume any awaiting threads.
			gsc_wait_resume_irq_alt(chan, SIO4_WAIT_USC_IOP_RXC_RISE);
			break;
		}

		if ((sicr & D13) && (misr & D13) && (misr & D12))
		{
			// I/O Pin TxC Fall
			handled	= 1;

			// Disable the interrupt due to the expected high interrupt rate.
			sicr	&= ~D13;
			os_reg_mem_tx_u8(NULL, chan->vaddr.usc_sicr_ub, sicr >> 8);

			// Clear the Latch/Unlatch bit.
			os_reg_mem_tx_u8(NULL, chan->vaddr.usc_misr_ub, D13 >> 8);

			// Resume any awaiting threads.
			gsc_wait_resume_irq_alt(chan, SIO4_WAIT_USC_IOP_TXC_FALL);
			break;
		}

		if ((sicr & D12) && (misr & D13) && ((misr & D12) == 0))
		{
			// I/O Pin TxC Rise
			handled	= 1;

			// Disable the interrupt due to the expected high interrupt rate.
			sicr	&= ~D12;
			os_reg_mem_tx_u8(NULL, chan->vaddr.usc_sicr_ub, sicr >> 8);

			// Clear the Latch/Unlatch bit.
			os_reg_mem_tx_u8(NULL, chan->vaddr.usc_misr_ub, D13 >> 8);

			// Resume any awaiting threads.
			gsc_wait_resume_irq_alt(chan, SIO4_WAIT_USC_IOP_TXC_RISE);
			break;
		}

		if ((sicr & D11) && (misr & D11) && (misr & D10))
		{
			// I/O Pin RxReq Fall
			handled	= 1;

			// Disable the interrupt due to the expected high interrupt rate.
			sicr	&= ~D11;
			os_reg_mem_tx_u8(NULL, chan->vaddr.usc_sicr_ub, sicr >> 8);

			// Clear the Latch/Unlatch bit.
			os_reg_mem_tx_u8(NULL, chan->vaddr.usc_misr_ub, D11 >> 8);

			// Resume any awaiting threads.
			gsc_wait_resume_irq_alt(chan, SIO4_WAIT_USC_IOP_RXREQ_FALL);
			break;
		}

		if ((sicr & D10) && (misr & D11) && ((misr & D10) == 0))
		{
			// I/O Pin RxReq Rise
			handled	= 1;

			// Disable the interrupt due to the expected high interrupt rate.
			sicr	&= ~D10;
			os_reg_mem_tx_u8(NULL, chan->vaddr.usc_sicr_ub, sicr >> 8);

			// Clear the Latch/Unlatch bit.
			os_reg_mem_tx_u8(NULL, chan->vaddr.usc_misr_ub, D11 >> 8);

			// Resume any awaiting threads.
			gsc_wait_resume_irq_alt(chan, SIO4_WAIT_USC_IOP_RXREQ_RISE);
			break;
		}

		if ((sicr & D9) && (misr & D9) && (misr & D8))
		{
			// I/O Pin TxReq Fall
			handled	= 1;

			// Disable the interrupt due to the expected high interrupt rate.
			sicr	&= ~D9;
			os_reg_mem_tx_u8(NULL, chan->vaddr.usc_sicr_ub, sicr >> 8);

			// Clear the Latch/Unlatch bit.
			os_reg_mem_tx_u8(NULL, chan->vaddr.usc_misr_ub, D9 >> 8);

			// Resume any awaiting threads.
			gsc_wait_resume_irq_alt(chan, SIO4_WAIT_USC_IOP_TXREQ_FALL);
			break;
		}

		if ((sicr & D8) && (misr & D9) && ((misr & D8) == 0))
		{
			// I/O Pin TxReq Rise
			handled	= 1;

			// Disable the interrupt due to the expected high interrupt rate.
			sicr	&= ~D8;
			os_reg_mem_tx_u8(NULL, chan->vaddr.usc_sicr_ub, sicr >> 8);

			// Clear the Latch/Unlatch bit.
			os_reg_mem_tx_u8(NULL, chan->vaddr.usc_misr_ub, D9 >> 8);

			// Resume any awaiting threads.
			gsc_wait_resume_irq_alt(chan, SIO4_WAIT_USC_IOP_TXREQ_RISE);
			break;
		}

		if ((sicr & D7) && (misr & D7) && (misr & D6))
		{
			// I/O Pin DCD Fall
			handled	= 1;

			// Clear the Latch/Unlatch bit.
			os_reg_mem_tx_u8(NULL, chan->vaddr.usc_misr_lb, D7);

			// Resume any awaiting threads.
			gsc_wait_resume_irq_alt(chan, SIO4_WAIT_USC_IOP_DCD_FALL);
			break;
		}

		if ((sicr & D6) && (misr & D7) && ((misr & D6) == 0))
		{
			// I/O Pin DCD Rise
			handled	= 1;

			// Clear the Latch/Unlatch bit.
			os_reg_mem_tx_u8(NULL, chan->vaddr.usc_misr_lb, D7);

			// Resume any awaiting threads.
			gsc_wait_resume_irq_alt(chan, SIO4_WAIT_USC_IOP_DCD_RISE);
			break;
		}

		if ((sicr & D5) && (misr & D5) && (misr & D4))
		{
			// I/O Pin CTS Fall
			handled	= 1;

			// Clear the Latch/Unlatch bit.
			os_reg_mem_tx_u8(NULL, chan->vaddr.usc_misr_lb, D5);

			// Resume any awaiting threads.
			gsc_wait_resume_irq_alt(chan, SIO4_WAIT_USC_IOP_CTS_FALL);
			break;
		}

		if ((sicr & D4) && (misr & D5) && ((misr & D4) == 0))
		{
			// I/O Pin CTS Rise
			handled	= 1;

			// Clear the Latch/Unlatch bit.
			os_reg_mem_tx_u8(NULL, chan->vaddr.usc_misr_lb, D5);

			// Resume any awaiting threads.
			gsc_wait_resume_irq_alt(chan, SIO4_WAIT_USC_IOP_CTS_RISE);
			break;
		}

		// A spurious interrupt.
		gsc_wait_resume_irq_alt(chan, SIO4_WAIT_USC_SPURIOUS);
		break;
	}

	return(handled);
}



//*****************************************************************************
static int _irq_isr_usc_misc_status(dev_data_t* dev, chan_data_t* chan)
{
	u8	bit;
	u8	irq;
	u8	misr_lb;
	u32	resume		= 0;
	u8	sicr_lb;

	// Clear the DCCR's IOP Status IP bit.
	os_reg_mem_tx_u8(NULL, chan->vaddr.usc_dccr_lb, 0x80 | D0);

	// See which sources are asserted.
	misr_lb	= os_reg_mem_rx_u8(NULL, chan->vaddr.usc_misr_lb);
	sicr_lb	= os_reg_mem_rx_u8(NULL, chan->vaddr.usc_sicr_lb);
	irq		= sicr_lb & misr_lb & 0xF;

	// Handle the highest priority source.

	if (irq & D3)
	{
		// RCC Underrun
		_rcc_sw_fifo_push(chan, SIO4_RCC_SW_FIFO_READ_FRAME_END);
		bit		= D3;
		resume	= SIO4_WAIT_USC_MISC_RCC_UNDERRUN;
	}
	else if (irq & D2)
	{
		// DPLL Desync
		_rcc_sw_fifo_push(chan, SIO4_RCC_SW_FIFO_READ_DPLL_DESYN);
		bit		= D2;
		resume	= SIO4_WAIT_USC_MISC_DPLL_DESYNC;
	}
	else if (irq & D1)
	{
		// BRG1 Zero
		bit		= D1;
		resume	= SIO4_WAIT_USC_MISC_BRG1_ZERO;
	}
	else if (irq & D0)
	{
		// BRG0 Zero
		bit		= D0;
		resume	= SIO4_WAIT_USC_MISC_BRG0_ZERO;
	}
	else
	{
		// A spurious interrupt.
		bit		= 0;
		resume	= SIO4_WAIT_USC_SPURIOUS;
	}

	if (bit)
	{
		// Clear the Interrupt Pending bit.
		os_reg_mem_tx_u8(NULL, chan->vaddr.usc_misr_lb, bit);
	}

	// Resume any awaiting threads.
	gsc_wait_resume_irq_alt(chan, resume);
	return(bit);
}



//*****************************************************************************
static int _irq_isr_usc(dev_data_t* dev, chan_data_t* chan)
{
	u8	dccr_lb;
	int	ret;

	dccr_lb	= os_reg_mem_rx_u8(NULL, chan->vaddr.usc_dccr_lb);

	if (dccr_lb & D5)
	{
		ret	= _irq_isr_usc_rx_status(dev, chan);
	}
	else if (dccr_lb & D4)
	{
		ret	= _irq_isr_usc_rx_data(dev, chan);
	}
	else if (dccr_lb & D3)
	{
		ret	= _irq_isr_usc_tx_status(dev, chan);
	}
	else if (dccr_lb & D2)
	{
		ret	= _irq_isr_usc_tx_data(dev, chan);
	}
	else if (dccr_lb & D1)
	{
		ret	= _irq_isr_usc_iop_status(dev, chan);
	}
	else if (dccr_lb & D0)
	{
		ret	= _irq_isr_usc_misc_status(dev, chan);
	}
	else
	{
		gsc_wait_resume_irq_alt(chan, SIO4_WAIT_USC_SPURIOUS);
		ret	= 0;
	}

	return(ret);
}



//*****************************************************************************
void dev_irq_isr_local_handler(dev_data_t* dev)
{
	static const struct
	{
		u32	mask;
		u32	wait;	// gsc_wait_t.gsc
	} data[32]	=
	{
		// mask			wait		chan	SIO4			SIO4-SYNC
		{ 0x00000001,	0x00001	},	// 0	Sync Detected	Rx Env
		{ 0x00000002,	0x00002	},	// 0	Tx FIFO AE		same
		{ 0x00000004,	0x00004	},	// 0	Rx FIFO AF		same
		{ 0x00000008,	0x00008	},	// 0	USC				Rx Spare
		{ 0x00010000,	0x10000	},	// 0	Tx FIFO E		same
		{ 0x00020000,	0x20000	},	// 0	Tx FIFO F		same
		{ 0x00040000,	0x40000	},	// 0	Rx FIFO E		same
		{ 0x00080000,	0x80000	},	// 0	Rx FIFO F		same

		{ 0x00000010,	0x00001	},	// 1	Sync Detected	Rx Env
		{ 0x00000020,	0x00002	},	// 1	Tx FIFO AE		same
		{ 0x00000040,	0x00004	},	// 1	Rx FIFO AF		same
		{ 0x00000080,	0x00008	},	// 1	USC				Rx Spare
		{ 0x00100000,	0x10000	},	// 1	Tx FIFO E		same
		{ 0x00200000,	0x20000	},	// 1	Tx FIFO F		same
		{ 0x00400000,	0x40000	},	// 1	Rx FIFO E		same
		{ 0x00800000,	0x80000	},	// 1	Rx FIFO F		same

		{ 0x00000100,	0x00001	},	// 2	Sync Detected	Rx Env
		{ 0x00000200,	0x00002	},	// 2	Tx FIFO AE		same
		{ 0x00000400,	0x00004	},	// 2	Rx FIFO AF		same
		{ 0x00000800,	0x00008	},	// 2	USC				Rx Spare
		{ 0x01000000,	0x10000	},	// 2	Tx FIFO E		same
		{ 0x02000000,	0x20000	},	// 2	Tx FIFO F		same
		{ 0x04000000,	0x40000	},	// 2	Rx FIFO E		same
		{ 0x08000000,	0x80000	},	// 2	Rx FIFO F		same

		{ 0x00001000,	0x00001	},	// 3	Sync Detected	Rx Env
		{ 0x00002000,	0x00002	},	// 3	Tx FIFO AE		same
		{ 0x00004000,	0x00004	},	// 3	Rx FIFO AF		same
		{ 0x00008000,	0x00008	},	// 3	USC				Rx Spare
		{ 0x10000000,	0x10000	},	// 3	Tx FIFO E		same
		{ 0x20000000,	0x20000	},	// 3	Tx FIFO F		same
		{ 0x40000000,	0x40000	},	// 3	Rx FIFO E		same
		{ 0x80000000,	0x80000	},	// 3	Rx FIFO F		same
	};

	u32				bit		= 0;
	chan_data_t*	chan	= &dev->channel[0];
	int				i;
	u32				icr;
	u32				irq;
	u32				isr;
	int				ret;

	icr	= os_reg_mem_rx_u32(NULL, dev->vaddr.gsc_icr_32);
	isr	= os_reg_mem_rx_u32(NULL, dev->vaddr.gsc_isr_32);
	irq	= icr & isr;

	if (dev->cache.irq_32 == 0)	// 16 interrupts only
		irq	&= 0xFFFF;

	for (i = 0;; i++)
	{
		bit	= data[i].mask;

		if (irq & bit)
		{
			// Clear or disable the interrupt, as appropriate.

			if ((dev->cache.model_z16c30) && (bit & 0x8888))
			{
				// Service the USC interrupt.
				os_reg_mem_tx_u32(NULL, dev->vaddr.gsc_isr_32, bit);

				for (ret = 1; ret;)
					ret	= _irq_isr_usc(dev, chan);

			}
			else
			{
				// Service the interrupt.
				os_reg_mem_tx_u32(NULL, dev->vaddr.gsc_isr_32, bit);

				// Check the status.
				isr	= os_reg_mem_rx_u32(NULL, dev->vaddr.gsc_isr_32);

				if (irq & isr)
				{
					// The interrupt is still active so we need to disable it.
					icr	&= ~bit;

					if (dev->cache.irq_32 == 0)
						icr	&= 0x0000FFFF;

					os_reg_mem_tx_u32(NULL, dev->vaddr.gsc_icr_32, icr);
				}
			}

			// Resume any threads awiting any of these interrupts.
			gsc_wait_resume_irq_gsc(chan, data[i].wait);
		}

		if (bit & 0x80000000)
			break;

		if (bit & 0x08880000)
			chan++;
	}
}



//*****************************************************************************
int dev_irq_create(dev_data_t* dev)
{
	int	ret;

	os_reg_mem_tx_u32(dev, dev->vaddr.gsc_icr_32, 0);	// Disable IRQs.
	ret	= gsc_irq_create(dev);
	return(ret);
}



//*****************************************************************************
void dev_irq_destroy(dev_data_t* dev)
{
	if (dev->vaddr.gsc_icr_32)
		os_reg_mem_tx_u32(dev, dev->vaddr.gsc_icr_32, 0);	// Disable IRQs.

	gsc_irq_destroy(dev);
}


