// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_2.x.x_GSC_DN/driver/ioctl.c $
// $Rev: 33968 $
// $Date: 2015-11-05 18:47:39 -0600 (Thu, 05 Nov 2015) $

#include "main.h"



// #defines *******************************************************************

#define	_20M		(20L * 1000 * 1000)



//*****************************************************************************
static int _arg_list_locate(s32* arg, const s32* list, int* index)
{
	int	ret;

	for (index[0] = 0; ; index[0]++)
	{
		if (list[index[0]] == -1)
		{
			ret			= -EINVAL;
			index[0]	= -1;
			break;
		}
		else if (arg[0] == list[index[0]])
		{
			ret			= 0;
			break;
		}
	}

	return(ret);
}



//*****************************************************************************
static int _arg_reg_list(
	chan_data_t*	chan,
	s32*			arg,
	const s32*		list,
	u32				reg,
	int				begin,
	int				end)
{
	int	i;
	u32	mask	= GSC_FIELD_ENCODE(0xFFFFFFFF, begin, end);
	int	ret		= -EINVAL;
	u32	v;

	if (arg[0] == -1)
	{
		ret	= 0;

		if (reg != 0)
		{
			reg_read(chan, reg, &v);
			v		&= mask;
			v		>>= end;
			arg[0]	= v;

			for (i = 0; list[i] >= 0; i++)
			{
				if (list[i] == v)
				{
					ret	= 0;
					break;
				}
			}
		}
	}
	else
	{
		for (i = 0; list[i] >= 0; i++)
		{
			if (list[i] == arg[0])
			{
				ret	= 0;

				if (reg != 0)
				{
					v	= list[i] << end;
					reg_mod(chan, reg, v, mask);
				}

				break;
			}
		}
	}

	return(ret);
}



//*****************************************************************************
static int _arg_reg_mask(
	chan_data_t*	chan,
	s32*			arg,
	const s32*		list,
	u32				reg,
	u32				mask,
	int				end)
{
	int	i;
	int	ret;
	u32	v;

	if (arg[0] == -1)
	{
		ret	= 0;

		if (reg != 0)
		{
			reg_read(chan, reg, &v);
			v	&= mask;
			v	>>= end;
			arg[0]	= v;
		}
	}
	else
	{
		ret	= -EINVAL;

		for (i = 0; list[i] >= 0; i++)
		{
			if (arg[0] == list[i])
			{
				ret	= 0;

				if (reg != 0)
				{
					v	= (arg[0] << end) & mask;
					reg_mod(chan, reg, v, mask);
				}

				break;
			}
		}
	}

	return(ret);
}



//*****************************************************************************
static int _arg_reg_range(
	chan_data_t*	chan,
	s32*			arg,
	s32				min,
	s32				max,
	u32				reg,
	int				begin,
	int				end)
{
	u32	mask;
	int	ret		= -EINVAL;
	u32	v;

	for (;;)	// A convenience loop.
	{
		if (arg[0] == -1)
		{
			mask	= GSC_FIELD_ENCODE(0xFFFFFFFF, begin, end);
			ret		= 0;

			if (reg != 0)
			{
				reg_read(chan, reg, &v);
				v		&= mask;
				v		>>= end;
				arg[0]	= v;
			}

			break;
		}

		if ((arg[0] < min) || (arg[0] > max))
			break;

		mask	= GSC_FIELD_ENCODE(0xFFFFFFFF, begin - end, 0);

		if ((arg[0] & mask) == arg[0])
		{
			mask	= mask << end;
			v		= arg[0] << end;
			ret		= 0;

			if (reg != 0)
				reg_mod(chan, reg, v, mask);
		}

		break;
	}

	return(ret);
}




//*****************************************************************************
static int _arg_var_range(s32* arg, s32 min, s32 max, s32* var)
{
	int	ret	= -EINVAL;

	if (arg[0] == -1)
	{
		ret		= 0;
		arg[0]	= var[0];
	}
	else if ((arg[0] >= min) && (arg[0] <= max))
	{
		ret		= 0;
		var[0]	= arg[0];
	}

	return(ret);
}



//*****************************************************************************
static int _irq_reg_mod(chan_data_t* chan, u32 reg, s32* arg, u32 mask)
{
	int	shift;
	u32	v;

	shift	= chan->index * 4;
	mask	<<= shift;
	mask	&= ((chan->dev->cache.irq_32) ? 0x000F000F : 0x0000000F) << shift;

	if (arg[0] == -1)
	{
		reg_read(chan, reg, &v);
		arg[0]	= (v & mask) >> shift;
	}
	else
	{
		v	= (arg[0] << shift) & mask;

		if (chan->dev->cache.irq_32 == 0)
		{
			v		&= 0x0000FFFF;
			mask	|= 0xFFFF0000;
		}

		reg_mod(chan, reg, v, mask);
	}

	return(0);
}



//*****************************************************************************
static int _query(chan_data_t* chan, s32* arg)
{
	dev_data_t*	dev	= chan->dev;
	int			ret	= 0;

	switch (arg[0])
	{
		default:								ret		= -EINVAL;								break;
		case -1:								arg[0]	= 1;									break;
		case SIO4_QUERY_BOARD_RESET:			arg[0]	= dev->cache.board_reset;				break;
		case SIO4_QUERY_BUS_SPEED:				arg[0]	= dev->cache.bus_speed;					break;
		case SIO4_QUERY_BUS_WIDTH:				arg[0]	= dev->cache.bus_width;					break;
		case SIO4_QUERY_CHANNEL_QTY:			arg[0]	= dev->cache.channel_qty;				break;
		case SIO4_QUERY_COUNT:					arg[0]	= SIO4_IOCTL_QUERY_LAST;				break;
		case SIO4_QUERY_DEVICE_QTY:				arg[0]	= dev->cache.device_qty;				break;
		case SIO4_QUERY_DEVICE_TYPE:			arg[0]	= dev->board_type;						break;
		case SIO4_QUERY_DMDMA_SCD:				arg[0]	= dev->cache.dmdma_scd;					break;
		case SIO4_QUERY_FIFO_SIZE_RX:			arg[0]	= chan->cache.fifo_size_rx;				break;
		case SIO4_QUERY_FIFO_SIZE_TOTAL:		arg[0]	= dev->cache.fifo_size_total;			break;
		case SIO4_QUERY_FIFO_SIZE_TX:			arg[0]	= chan->cache.fifo_size_tx;				break;
		case SIO4_QUERY_FIFO_SPACE_CFG:			arg[0]	= dev->cache.fifo_space_cfg;			break;
		case SIO4_QUERY_FORM_FACTOR:			arg[0]	= dev->cache.form_factor;				break;
		case SIO4_QUERY_FW_TYPE:				arg[0]	= dev->cache.fw_type;					break;
		case SIO4_QUERY_FW_TYPE_CONFIG:			arg[0]	= dev->cache.fw_type_config;			break;
		case SIO4_QUERY_INDEX_BOARD:			arg[0]	= dev->board_index;						break;
		case SIO4_QUERY_INDEX_CHANNEL:			arg[0]	= chan->index;							break;
		case SIO4_QUERY_INDEX_DEVICE:			arg[0]	= chan->dev_index;						break;
		case SIO4_QUERY_INDEX_SUBDEVICE:		arg[0]	= dev->cache.index_subdevice;			break;
		case SIO4_QUERY_IRQ_32:					arg[0]	= dev->cache.irq_32;					break;
		case SIO4_QUERY_LED_CHANNEL:			arg[0]	= dev->cache.led_channel;				break;
		case SIO4_QUERY_LED_MAIN:				arg[0]	= dev->cache.led_main;					break;
		case SIO4_QUERY_LEGACY_CABLE:			arg[0]	= dev->cache.legacy_cable;				break;
		case SIO4_QUERY_MODEL_BASE:				arg[0]	= dev->cache.model;						break;
		case SIO4_QUERY_MODEL_SYNC:				arg[0]	= dev->cache.model_sync;				break;
		case SIO4_QUERY_MODEL_Z16C30:			arg[0]	= dev->cache.model_z16c30;				break;
		case SIO4_QUERY_MP:						arg[0]	= chan->cache.mp;						break;
		case SIO4_QUERY_MP_BITMAP:				arg[0]	= chan->cache.mp_bitmap;				break;
		case SIO4_QUERY_MP_CHIP:				arg[0]	= dev->cache.mp_chip;					break;
		case SIO4_QUERY_MP_PROGRAM:				arg[0]	= chan->cache.mp_program;				break;
		case SIO4_QUERY_OSC_CHIP:				arg[0]	= dev->cache.osc_chip;					break;
		case SIO4_QUERY_OSC_MEASURE:			arg[0]	= dev->cache.osc_measure;				break;
		case SIO4_QUERY_OSC_PD_MAX:				arg[0]	= dev->cache.osc_pd_max;				break;
		case SIO4_QUERY_OSC_PER_CHANNEL:		arg[0]	= dev->cache.osc_per_chan;				break;
		case SIO4_QUERY_OSC_PROGRAM:			arg[0]	= dev->cache.osc_program;				break;
		case SIO4_QUERY_REG_BSR:				arg[0]	= dev->cache.reg_bsr;					break;
		case SIO4_QUERY_REG_CCR:				arg[0]	= chan->cache.reg_ccr;					break;
		case SIO4_QUERY_REG_FCR:				arg[0]	= dev->cache.reg_fcr;					break;
		case SIO4_QUERY_REG_FR:					arg[0]	= dev->cache.reg_fr;					break;
		case SIO4_QUERY_REG_FSR:				arg[0]	= dev->cache.reg_fsr;					break;
		case SIO4_QUERY_REG_FTR:				arg[0]	= dev->cache.reg_ftr;					break;
		case SIO4_QUERY_REG_GPIOSR:				arg[0]	= chan->cache.reg_gpiosr;				break;
		case SIO4_QUERY_REG_IELR:				arg[0]	= dev->cache.reg_ielr;					break;
		case SIO4_QUERY_REG_IHLR:				arg[0]	= dev->cache.reg_ihlr;					break;
		case SIO4_QUERY_REG_IOCR:				arg[0]	= dev->cache.reg_iocr;					break;
		case SIO4_QUERY_REG_PCR:				arg[0]	= dev->cache.reg_pcr;					break;
		case SIO4_QUERY_REG_POCSR:				arg[0]	= dev->cache.reg_pocsr;					break;
		case SIO4_QUERY_REG_PORAR:				arg[0]	= dev->cache.reg_porar;					break;
		case SIO4_QUERY_REG_PORDR:				arg[0]	= dev->cache.reg_pordr;					break;
		case SIO4_QUERY_REG_PORD2R:				arg[0]	= dev->cache.reg_pord2r;				break;
		case SIO4_QUERY_REG_PSRCR:				arg[0]	= chan->cache.reg_psrcr;				break;
		case SIO4_QUERY_REG_PSRCR_BITS:			arg[0]	= chan->cache.reg_psrcr_bits;			break;
		case SIO4_QUERY_REG_PSTSR:				arg[0]	= chan->cache.reg_pstsr;				break;
		case SIO4_QUERY_REG_PSTSR_BITS:			arg[0]	= chan->cache.reg_pstsr_bits;			break;
		case SIO4_QUERY_REG_RCR:				arg[0]	= chan->cache.reg_rcr;					break;
		case SIO4_QUERY_REG_SBR:				arg[0]	= chan->cache.reg_sbr;					break;
		case SIO4_QUERY_REG_TCR:				arg[0]	= chan->cache.reg_tcr;					break;
		case SIO4_QUERY_REG_TSR:				arg[0]	= dev->cache.reg_tsr;					break;
		case SIO4_QUERY_RX_FIFO_FULL_CFG:		arg[0]	= chan->cache.rx_fifo_full_cfg_chan;	break;
		case SIO4_QUERY_RX_FIFO_FULL_CFG_GLB:	arg[0]	= dev->cache.rx_fifo_full_cfg_glb;		break;
		case SIO4_QUERY_RX_FIFO_OVERRUN:		arg[0]	= chan->cache.rx_fifo_overrun;			break;
		case SIO4_QUERY_RX_FIFO_UNDERRUN:		arg[0]	= dev->cache.rx_fifo_underrun;			break;
		case SIO4_QUERY_RX_STATUS_WORD:			arg[0]	= chan->cache.rx_status_word;			break;
		case SIO4_QUERY_SIO4_TYPE:				arg[0]	= dev->cache.sio4_type;					break;
		case SIO4_QUERY_TIME_STAMP:				arg[0]	= dev->cache.time_stamp;				break;
		case SIO4_QUERY_TX_FIFO_EMPTY_CFG:		arg[0]	= chan->cache.tx_fifo_empty_cfg;		break;
		case SIO4_QUERY_TX_FIFO_OVERRUN:		arg[0]	= dev->cache.tx_fifo_overrun;			break;
		case SIO4_QUERY_USER_JUMPER_QTY:		arg[0]	= dev->cache.user_jumper_qty;			break;
		case SIO4_QUERY_USER_JUMPER_SENSE:		arg[0]	= dev->cache.user_jumper_sense;			break;
		case SIO4_QUERY_USER_JUMPER_VAL:		arg[0]	= dev->cache.user_jumper_val;			break;
	}

	return(ret);
}



//*****************************************************************************
int initialize_ioctl(chan_data_t* chan, s32* arg)
{
	static const struct
	{
		s32	reg;
		u32	value;
		int	wait_ms;
		s32	query;
	} list[]	=
	{
		// reg				val				wait_ms	query
		{ SIO4_GSC_CCR,		0x00000000,		0,		SIO4_QUERY_REG_CCR		},
		{ SIO4_GSC_TSR,		0x00000000,		0,		SIO4_QUERY_REG_TSR		},
		{ SIO4_GSC_TAR,		0x00070007,		0,		-1						},
		{ SIO4_GSC_RAR,		0x00070007,		0,		-1						},
		{ SIO4_GSC_CSR,		0x00000000,		0,		-1						},
		{ SIO4_GSC_SBR,		0x00000000,		0,		SIO4_QUERY_REG_SBR		},
		{ SIO4_GSC_ICR,		0x00000000,		0,		-1						},
		{ SIO4_GSC_ISR,		0x00000000,		0,		-1						},
		{ SIO4_GSC_IHLR,	0xFFFFFFFF,		0,		SIO4_QUERY_REG_IHLR		},
		{ SIO4_GSC_PSRCR,	0x00000000,		0,		SIO4_QUERY_REG_PSRCR	},
		{ SIO4_GSC_GPIOSR,	0x00000000,		0,		SIO4_QUERY_REG_GPIOSR	},
		{ SIO4_GSC_IOCR,	0x00000000,		0,		SIO4_QUERY_REG_IOCR		},
		{ SIO4_GSC_TCR,		0x00000000,		0,		SIO4_QUERY_REG_TCR		},
		{ SIO4_GSC_CSR,		0x00000001,		15,		-1						},// Reset Tx FIFO
		{ SIO4_USC_CCAR,	0x0400,			2,		SIO4_QUERY_MODEL_Z16C30	},// Reset
		{ SIO4_USC_CCAR,	0x0000,			2,		SIO4_QUERY_MODEL_Z16C30	},// Release
		{ SIO4_USC_HCR,		0x0044,			2,		SIO4_QUERY_MODEL_Z16C30	},// DMA setup
		{ SIO4_USC_IOCR,	0x0500,			2,		SIO4_QUERY_MODEL_Z16C30	},// DMA setup
		{ SIO4_USC_CCAR,	0x5000,			2,		SIO4_QUERY_MODEL_Z16C30	},// Tx FIFO Purge
		{ SIO4_USC_CCAR,	0x4800,			2,		SIO4_QUERY_MODEL_Z16C30	},// Rx FIFO Purge
		{ SIO4_USC_CCAR,	0xC800,			2,		SIO4_QUERY_MODEL_Z16C30	},// Rx Purge
		{ SIO4_GSC_CSR,		0x00000002,		15,		-1						},// Reset Rx FIFO
		{ 0 }
	};

	int			err;
	int			i;
	s32			q;
	gsc_reg_t	reg;
	int			ret		= 0;
	int			test;
	s32			tmp;

	if (arg == NULL)	// This is for convenient, internal use.
	{
		arg	= &tmp;
		tmp	= 1;
	}

	if (arg[0] == -1)
	{
		arg[0]	= 1;
	}
	else if (arg[0] == 1)
	{
		for (i = 0; list[i].reg; i++)
		{
			if (list[i].query >= 0)
			{
				q		= list[i].query;
				test	= _query(chan, &q);
				ret		= ret ? test : 0;

				if (q == 0)
					continue;
			}

			reg.reg		= list[i].reg;
			reg.value	= list[i].value;
			reg.mask	= 0;
			test		= gsc_reg_write_ioctl(chan, &reg);
			ret			= ret ? test : 0;

			if (list[i].wait_ms > 0)
				os_time_sleep_ms(list[i].wait_ms);
		}

		err	= mp_init(chan);
		ret	= ret ? ret : err;

		// I/O parameters
		chan->rx.bytes_per_sample	= 1;
		chan->rx.io_mode			= SIO4_IO_MODE_DEFAULT;
		chan->rx.pio_threshold		= SIO4_IO_PIO_THRESHOLD_DEFAULT;
		chan->rx.timeout_s			= SIO4_IO_TIMEOUT_DEFAULT;

		chan->tx.bytes_per_sample	= 1;
		chan->tx.io_mode			= GSC_IO_MODE_PIO;
		chan->tx.pio_threshold		= SIO4_IO_PIO_THRESHOLD_DEFAULT;
		chan->tx.timeout_s			= SIO4_IO_TIMEOUT_DEFAULT;
	}
	else
	{
		// The value received must be 1 or -1.
		ret	= -EINVAL;
	}

	return(ret);
}



//*****************************************************************************
static int _cbl_mode(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_CBL_MODE_DCE,
		SIO4_CBL_MODE_DTE,
		-1
	};

	int	reg	= (chan->cache.reg_psrcr_bits & D28) ? SIO4_GSC_PSRCR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 28, 28);
	return(ret);
}



//*****************************************************************************
static int _cbl_pin_status(chan_data_t* chan, s32* arg)
{
	int	ret	= 0;

	if (arg[0] != -1)
	{
		ret	= -EINVAL;
	}
	else if (chan->cache.reg_pstsr)
	{
		arg[0]	= os_reg_mem_rx_u32(chan->dev, chan->vaddr.gsc_pstsr_32);
		arg[0]	&= chan->cache.reg_pstsr_bits;
	}
	else
	{
		arg[0]	= -1;
	}

	return(ret);
}



//*****************************************************************************
static int _fifo_space_cfg(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_FIFO_SPACE_CFG_RX_2X,
		SIO4_FIFO_SPACE_CFG_TX_2X,
		-1
	};

	int	reg	= chan->dev->cache.fifo_space_cfg ? SIO4_GSC_CSR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 5, 4);
	return(ret);
}



//*****************************************************************************
static int _gpio_direction_out(chan_data_t* chan, s32* arg)
{
	int	reg	= chan->dev->cache.reg_iocr ? SIO4_GSC_IOCR : 0;
	int	ret;

	ret	= _arg_reg_range(chan, arg, 0, 0x3F, reg, 5, 0);
	return(ret);
}



//*****************************************************************************
static int _gpio_input_latching(chan_data_t* chan, s32* arg)
{
	int	reg	= chan->dev->cache.reg_iocr ? SIO4_GSC_IOCR : 0;
	int	ret;

	ret	= _arg_reg_range(chan, arg, 0, 0x3F, reg, 13, 8);
	return(ret);
}



//*****************************************************************************
static int _gpio_input_read(chan_data_t* chan, s32* arg)
{
	int	reg	= chan->dev->cache.reg_iocr ? SIO4_GSC_IOCR : 0;
	int	ret;

	ret	= _arg_reg_range(chan, arg, 0, 0x3F, reg, 21, 16);
	return(ret);
}



//*****************************************************************************
static int _gpio_output_write(chan_data_t* chan, s32* arg)
{
	int	reg	= chan->dev->cache.reg_iocr ? SIO4_GSC_IOCR : 0;
	int	ret;

	ret	= _arg_reg_range(chan, arg, 0, 0x3F, reg, 13, 8);
	return(ret);
}



//*****************************************************************************
static int _gpio_polarity(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_GPIO_POLARITY_LOW,
		SIO4_GPIO_POLARITY_HIGH,
		-1
	};

	int	reg	= chan->dev->cache.reg_iocr ? SIO4_GSC_IOCR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 30, 30);
	return(ret);
}



//*****************************************************************************
static int _gpio_sense_edge(chan_data_t* chan, s32* arg)
{
	int	reg	= chan->dev->cache.reg_iocr ? SIO4_GSC_IOCR : 0;
	int	ret;

	ret	= _arg_reg_range(chan, arg, 0, 0x3F, reg, 29, 24);
	return(ret);
}



//*****************************************************************************
static int _irq_gsc_cfg_high(chan_data_t* chan, s32* arg)
{
	u32	ihlr;
	u32	mask	= chan->dev->cache.irq_32 ? 0x000F000F : 0x000F;
	int	ret		= 0;

	if (arg[0] == -1)
	{
		reg_read(chan, SIO4_GSC_IHLR, &ihlr);
		ihlr	>>= (4 * chan->index);
		ihlr	&= mask;
		arg[0]	= ihlr;
	}
	else if (arg[0] & 0xFFF0FFF0)
	{
		ret	= -EINVAL;
	}
	else if (arg[0] & ~mask)
	{
		arg[0]	= -1;
	}
	else if (chan->dev->cache.reg_ihlr)
	{
		ret	= _irq_reg_mod(chan, SIO4_GSC_IHLR, arg, mask);
	}

	return(ret);
}



//*****************************************************************************
static int _irq_gsc_enable(chan_data_t* chan, s32* arg)
{
	u32	icr;
	u32	mask	= chan->dev->cache.irq_32 ? 0x000F000F : 0x000F;
	int	ret		= 0;

	if (arg[0] == -1)
	{
		reg_read(chan, SIO4_GSC_ICR, &icr);
		icr		>>= (4 * chan->index);
		icr		&= mask;
		arg[0]	= icr;
	}
	else if (arg[0] & 0xFFF0FFF0)
	{
		ret	= -EINVAL;
	}
	else if (arg[0] & ~mask)
	{
		arg[0]	= -1;
	}
	else
	{
		ret	= _irq_reg_mod(chan, SIO4_GSC_ICR, arg, mask);
	}

	return(ret);
}



//*****************************************************************************
static int _irq_usc_enable(chan_data_t* chan, s32* arg)
{
	static const struct
	{
		u32	irq;
		u32	reg;
		u32	bits;
	} list[]	=
	{
		// irq								reg				bits
		{ SIO4_IRQ_USC_RX_EXITED_HUNT,		SIO4_GSC_ICR,	SIO4_IRQ_USC	},
		{ SIO4_IRQ_USC_RX_EXITED_HUNT,		SIO4_USC_RICR,	D7				},
		{ SIO4_IRQ_USC_RX_EXITED_HUNT,		SIO4_USC_ICR,	D5 | D15		},

		{ SIO4_IRQ_USC_RX_IDLE_RECEIVED,	SIO4_GSC_ICR,	SIO4_IRQ_USC	},
		{ SIO4_IRQ_USC_RX_IDLE_RECEIVED,	SIO4_USC_RICR,	D6				},
		{ SIO4_IRQ_USC_RX_IDLE_RECEIVED,	SIO4_USC_ICR,	D5 | D15		},

		{ SIO4_IRQ_USC_RX_BREAK_ABORT,		SIO4_GSC_ICR,	SIO4_IRQ_USC	},
		{ SIO4_IRQ_USC_RX_BREAK_ABORT,		SIO4_USC_RICR,	D5				},
		{ SIO4_IRQ_USC_RX_BREAK_ABORT,		SIO4_USC_ICR,	D5 | D15		},

		{ SIO4_IRQ_USC_RX_BOUND,			SIO4_GSC_ICR,	SIO4_IRQ_USC	},
		{ SIO4_IRQ_USC_RX_BOUND,			SIO4_USC_RICR,	D4				},
		{ SIO4_IRQ_USC_RX_BOUND,			SIO4_USC_ICR,	D5 | D15		},

		{ SIO4_IRQ_USC_RX_ABORT_PAR_ERROR,	SIO4_GSC_ICR,	SIO4_IRQ_USC	},
		{ SIO4_IRQ_USC_RX_ABORT_PAR_ERROR,	SIO4_USC_RICR,	D2				},
		{ SIO4_IRQ_USC_RX_ABORT_PAR_ERROR,	SIO4_USC_ICR,	D5 | D15		},

		{ SIO4_IRQ_USC_RX_OVERRUN,			SIO4_GSC_ICR,	SIO4_IRQ_USC	},
		{ SIO4_IRQ_USC_RX_OVERRUN,			SIO4_USC_RICR,	D1				},
		{ SIO4_IRQ_USC_RX_OVERRUN,			SIO4_USC_ICR,	D5 | D15		},

		{ SIO4_IRQ_USC_RX_DATA,				SIO4_GSC_ICR,	SIO4_IRQ_USC	},
		{ SIO4_IRQ_USC_RX_DATA,				SIO4_USC_ICR,	D4 | D15		},

		{ SIO4_IRQ_USC_TX_PREAMBLE_SENT,	SIO4_GSC_ICR,	SIO4_IRQ_USC	},
		{ SIO4_IRQ_USC_TX_PREAMBLE_SENT,	SIO4_USC_TICR,	D7				},
		{ SIO4_IRQ_USC_TX_PREAMBLE_SENT,	SIO4_USC_ICR,	D3 | D15		},

		{ SIO4_IRQ_USC_TX_IDLE_SENT,		SIO4_GSC_ICR,	SIO4_IRQ_USC	},
		{ SIO4_IRQ_USC_TX_IDLE_SENT,		SIO4_USC_TICR,	D6				},
		{ SIO4_IRQ_USC_TX_IDLE_SENT,		SIO4_USC_ICR,	D3 | D15		},

		{ SIO4_IRQ_USC_TX_ABORT_SENT,		SIO4_GSC_ICR,	SIO4_IRQ_USC	},
		{ SIO4_IRQ_USC_TX_ABORT_SENT,		SIO4_USC_TICR,	D5				},
		{ SIO4_IRQ_USC_TX_ABORT_SENT,		SIO4_USC_ICR,	D3 | D15		},

		{ SIO4_IRQ_USC_TX_END_SENT,			SIO4_GSC_ICR,	SIO4_IRQ_USC	},
		{ SIO4_IRQ_USC_TX_END_SENT,			SIO4_USC_TICR,	D4				},
		{ SIO4_IRQ_USC_TX_END_SENT,			SIO4_USC_ICR,	D3 | D15		},

		{ SIO4_IRQ_USC_TX_CRC_SENT,			SIO4_GSC_ICR,	SIO4_IRQ_USC	},
		{ SIO4_IRQ_USC_TX_CRC_SENT,			SIO4_USC_TICR,	D3				},
		{ SIO4_IRQ_USC_TX_CRC_SENT,			SIO4_USC_ICR,	D3 | D15		},

		{ SIO4_IRQ_USC_TX_UNDERRUN,			SIO4_GSC_ICR,	SIO4_IRQ_USC	},
		{ SIO4_IRQ_USC_TX_UNDERRUN,			SIO4_USC_TICR,	D1				},
		{ SIO4_IRQ_USC_TX_UNDERRUN,			SIO4_USC_ICR,	D3 | D15		},

		{ SIO4_IRQ_USC_TX_DATA,				SIO4_GSC_ICR,	SIO4_IRQ_USC	},
		{ SIO4_IRQ_USC_TX_DATA,				SIO4_USC_ICR,	D2 | D15		},

		{ SIO4_IRQ_USC_IOP_RXC_FALL,		SIO4_GSC_ICR,	SIO4_IRQ_USC	},
		{ SIO4_IRQ_USC_IOP_RXC_FALL,		SIO4_USC_SICR,	D15				},
		{ SIO4_IRQ_USC_IOP_RXC_FALL,		SIO4_USC_ICR,	D1 | D15		},

		{ SIO4_IRQ_USC_IOP_RXC_RISE,		SIO4_GSC_ICR,	SIO4_IRQ_USC	},
		{ SIO4_IRQ_USC_IOP_RXC_RISE,		SIO4_USC_SICR,	D14				},
		{ SIO4_IRQ_USC_IOP_RXC_RISE,		SIO4_USC_ICR,	D1 | D15		},

		{ SIO4_IRQ_USC_IOP_TXC_FALL,		SIO4_GSC_ICR,	SIO4_IRQ_USC	},
		{ SIO4_IRQ_USC_IOP_TXC_FALL,		SIO4_USC_SICR,	D13				},
		{ SIO4_IRQ_USC_IOP_TXC_FALL,		SIO4_USC_ICR,	D1 | D15		},

		{ SIO4_IRQ_USC_IOP_TXC_RISE,		SIO4_GSC_ICR,	SIO4_IRQ_USC	},
		{ SIO4_IRQ_USC_IOP_TXC_RISE,		SIO4_USC_SICR,	D12				},
		{ SIO4_IRQ_USC_IOP_TXC_RISE,		SIO4_USC_ICR,	D1 | D15		},

		{ SIO4_IRQ_USC_IOP_RXREQ_FALL,		SIO4_GSC_ICR,	SIO4_IRQ_USC	},
		{ SIO4_IRQ_USC_IOP_RXREQ_FALL,		SIO4_USC_SICR,	D11				},
		{ SIO4_IRQ_USC_IOP_RXREQ_FALL,		SIO4_USC_ICR,	D1 | D15		},

		{ SIO4_IRQ_USC_IOP_RXREQ_RISE,		SIO4_GSC_ICR,	SIO4_IRQ_USC	},
		{ SIO4_IRQ_USC_IOP_RXREQ_RISE,		SIO4_USC_SICR,	D10				},
		{ SIO4_IRQ_USC_IOP_RXREQ_RISE,		SIO4_USC_ICR,	D1 | D15		},

		{ SIO4_IRQ_USC_IOP_TXREQ_FALL,		SIO4_GSC_ICR,	SIO4_IRQ_USC	},
		{ SIO4_IRQ_USC_IOP_TXREQ_FALL,		SIO4_USC_SICR,	D9				},
		{ SIO4_IRQ_USC_IOP_TXREQ_FALL,		SIO4_USC_ICR,	D1 | D15		},

		{ SIO4_IRQ_USC_IOP_TXREQ_RISE,		SIO4_GSC_ICR,	SIO4_IRQ_USC	},
		{ SIO4_IRQ_USC_IOP_TXREQ_RISE,		SIO4_USC_SICR,	D8				},
		{ SIO4_IRQ_USC_IOP_TXREQ_RISE,		SIO4_USC_ICR,	D1 | D15		},

		{ SIO4_IRQ_USC_IOP_DCD_FALL,		SIO4_GSC_ICR,	SIO4_IRQ_USC	},
		{ SIO4_IRQ_USC_IOP_DCD_FALL,		SIO4_USC_SICR,	D7				},
		{ SIO4_IRQ_USC_IOP_DCD_FALL,		SIO4_USC_ICR,	D1 | D15		},

		{ SIO4_IRQ_USC_IOP_DCD_RISE,		SIO4_GSC_ICR,	SIO4_IRQ_USC	},
		{ SIO4_IRQ_USC_IOP_DCD_RISE,		SIO4_USC_SICR,	D6				},
		{ SIO4_IRQ_USC_IOP_DCD_RISE,		SIO4_USC_ICR,	D1 | D15		},

		{ SIO4_IRQ_USC_IOP_CTS_FALL,		SIO4_GSC_ICR,	SIO4_IRQ_USC	},
		{ SIO4_IRQ_USC_IOP_CTS_FALL,		SIO4_USC_SICR,	D5				},
		{ SIO4_IRQ_USC_IOP_CTS_FALL,		SIO4_USC_ICR,	D1 | D15		},

		{ SIO4_IRQ_USC_IOP_CTS_RISE,		SIO4_GSC_ICR,	SIO4_IRQ_USC	},
		{ SIO4_IRQ_USC_IOP_CTS_RISE,		SIO4_USC_SICR,	D4				},
		{ SIO4_IRQ_USC_IOP_CTS_RISE,		SIO4_USC_ICR,	D1 | D15		},

		{ SIO4_IRQ_USC_MISC_RCC_UNDERRUN,	SIO4_GSC_ICR,	SIO4_IRQ_USC	},
		{ SIO4_IRQ_USC_MISC_RCC_UNDERRUN,	SIO4_USC_SICR,	D3				},
		{ SIO4_IRQ_USC_MISC_RCC_UNDERRUN,	SIO4_USC_ICR,	D0 | D15		},

		{ SIO4_IRQ_USC_MISC_DPLL_DESYNC,	SIO4_GSC_ICR,	SIO4_IRQ_USC	},
		{ SIO4_IRQ_USC_MISC_DPLL_DESYNC,	SIO4_USC_SICR,	D2				},
		{ SIO4_IRQ_USC_MISC_DPLL_DESYNC,	SIO4_USC_ICR,	D0 | D15		},

		{ SIO4_IRQ_USC_MISC_BRG1_ZERO,		SIO4_GSC_ICR,	SIO4_IRQ_USC	},
		{ SIO4_IRQ_USC_MISC_BRG1_ZERO,		SIO4_USC_SICR,	D1				},
		{ SIO4_IRQ_USC_MISC_BRG1_ZERO,		SIO4_USC_ICR,	D0 | D15		},

		{ SIO4_IRQ_USC_MISC_BRG0_ZERO,		SIO4_GSC_ICR,	SIO4_IRQ_USC	},
		{ SIO4_IRQ_USC_MISC_BRG0_ZERO,		SIO4_USC_SICR,	D0				},
		{ SIO4_IRQ_USC_MISC_BRG0_ZERO,		SIO4_USC_ICR,	D0 | D15		},

		{ 0,								0,				0				}
	};

	int		i;
	u32		icr_g;	// SIO4_GSC_ICR
	u32		icr_u;	// SIO4_USC_ICR
	u32*	reg;
	int		ret		= 0;
	u32		ricr;	// SIO4_USC_RICR
	u32		sicr;	// SIO4_USC_SICR
	u32		ticr;	// SIO4_USC_TICR
	s32		v;


	if (chan->dev->cache.model_z16c30 == 0)
	{
		if ((arg[0] == -1) || ((arg[0] & SIO4_IRQ_USC_ALL) == arg[0]))
			ret	= 0;
		else
			ret	= -EINVAL;
	}
	else if (arg[0] == -1)
	{
		// Report which USC interrupts are enabled.
		reg_read(chan, SIO4_GSC_ICR, &icr_g);
		reg_read(chan, SIO4_USC_ICR, &icr_u);
		reg_read(chan, SIO4_USC_RICR, &ricr);
		reg_read(chan, SIO4_USC_SICR, &sicr);
		reg_read(chan, SIO4_USC_TICR, &ticr);

		arg[0]	= 0;

		for (i = 0; list[i].bits; i++)
			arg[0]	|= list[i].irq;

		for (i = 0; list[i].bits; i++)
		{
			switch (list[i].reg)
			{
				default:

					printf(	"_irq_usc_enable_query:"
							" internal error: reg: i %d\n",
							i);
					continue;

					case SIO4_GSC_ICR:	reg	= &icr_g;	break;
					case SIO4_USC_ICR:	reg	= &icr_u;	break;
					case SIO4_USC_RICR:	reg	= &ricr;	break;
					case SIO4_USC_SICR:	reg	= &sicr;	break;
					case SIO4_USC_TICR:	reg	= &ticr;	break;
			}

			if ((reg[0] & list[i].bits) != list[i].bits)
				arg[0]	&= ~list[i].irq;
		}
	}
	else if ((arg[0] & SIO4_IRQ_USC_ALL) == arg[0])
	{
		// Enable the selected USC interrupts.
		icr_g	= 0;
		icr_u	= 0;
		ricr	= 0;
		sicr	= 0;
		ticr	= 0;
		v		= arg[0];

		for (i = 0; list[i].bits; i++)
		{
			switch (list[i].reg)
			{
				default:

					printf(	"_irq_usc_enable_query:"
							" internal error: reg: i %d\n",
							i);
					continue;

					case SIO4_GSC_ICR:	reg	= &icr_g;	break;
					case SIO4_USC_ICR:	reg	= &icr_u;	break;
					case SIO4_USC_RICR:	reg	= &ricr;	break;
					case SIO4_USC_SICR:	reg	= &sicr;	break;
					case SIO4_USC_TICR:	reg	= &ticr;	break;
			}

			if (list[i].irq & arg[0])
			{
				reg[0]	|= list[i].bits;
				v		&= ~list[i].irq;
			}
		}

		if (v)
		{
			// Invalid bits are set.
			ret	= -EINVAL;
		}
		else
		{
			v	= 0;
			ret	= _irq_reg_mod(chan, SIO4_GSC_ICR, &v, D3);
			reg_mod(chan, SIO4_USC_RICR, ricr,   0x00F6);
			reg_mod(chan, SIO4_USC_SICR, sicr,   0xFFFF);
			reg_mod(chan, SIO4_USC_TICR, ticr,   0x00FA);
			reg_mod(chan, SIO4_USC_ICR,  0x00BF, 0x00FF);
			icr_u	|= 0xC0;
			reg_mod(chan, SIO4_USC_ICR,  icr_u,  0x80FF);
			ret	= _irq_reg_mod(chan, SIO4_GSC_ICR, (void*) &icr_g, D3);
		}
	}
	else
	{
		ret	= -EINVAL;
	}

	return(ret);
}



//*****************************************************************************
static int _led_channel(chan_data_t* chan, s32* arg)
{
	u32	csr;
	int	ret		= 0;
	u32	v;

	if (chan->dev->cache.led_channel == 0)
	{
		if (arg[0] != -1)
			ret	= ((arg[0] & 0x3) == 0x3) ? 0 : -EINVAL;
	}
	else if (arg[0] == -1)
	{
		// At present there is only one implementation.
		reg_read(chan, SIO4_GSC_CSR, &csr);
		arg[0]	= 0;

		// upper LED
		v		= GSC_FIELD_DECODE(csr, 23, 22);
		arg[0]	|= (v == 3) ? 2 : 0;

		// lower LED
		v		= GSC_FIELD_DECODE(csr, 21, 20);
		arg[0]	|= (v == 3) ? 1 : 0;
	}
	else
	{
		if ((arg[0] < 0) || (arg[0] > 3))
		{
			ret	= -EINVAL;
		}
		else
		{
			// At present there is only one implementation.
			v	= ((arg[0] & 0x2) ? (0x3 << 22) : (0x2 << 22))
				| ((arg[0] & 0x1) ? (0x3 << 20) : (0x2 << 20));
			reg_mod(chan, SIO4_GSC_CSR, v, 0xF << 20);
		}
	}

	return(ret);
}



//*****************************************************************************
static int _led_main(chan_data_t* chan, s32* arg)
{
	int	reg	= chan->dev->cache.led_main ? SIO4_GSC_BCR : 0;
	int	ret;

	if (chan->dev->cache.led_main == 2)
		ret	= _arg_reg_range(chan, arg, 0, 0x3, reg, 25, 24);
	else
		ret	= _arg_reg_range(chan, arg, 0, 0x7, reg, 26, 24);

	return(ret);
}



//*****************************************************************************
static int _loop_back(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_LOOP_BACK_DISABLE,
		SIO4_LOOP_BACK_INTERNAL,
		SIO4_LOOP_BACK_EXTERNAL,
		-1
	};

	int	ret;
	s32	v	= arg[0];

	if (chan->dev->cache.model == SIO4_MODEL_SIO4A_SYNC)
	{
		if (arg[0] == SIO4_LOOP_BACK_EXTERNAL)
			arg[0]	= SIO4_LOOP_BACK_INTERNAL;

		ret	= _arg_reg_mask(chan, arg, list, SIO4_GSC_CSR, D28, 28);
	}
	else if (	(chan->cache.reg_psrcr_bits & D23)	&&
				(chan->cache.reg_psrcr_bits & D29))
	{
		ret	= _arg_reg_mask(chan, arg, list, SIO4_GSC_PSRCR, D23 | D29, 23);
	}
	else
	{
		ret	= _arg_reg_mask(chan, arg, list, 0, D23 | D29, 23);
	}

	if ((v == -1) &&
		(arg[0] != -1) &&
		(arg[0] & SIO4_LOOP_BACK_INTERNAL) &&
		(arg[0] & SIO4_LOOP_BACK_EXTERNAL))
	{
		arg[0]	= SIO4_LOOP_BACK_INTERNAL;
	}

	return(ret);
}



//*****************************************************************************
static int _osc_measure(chan_data_t* chan, s32* arg)
{
	int	ret;

	if (arg[0] == -1)
	{
		arg[0]	= chan->dev->cache.osc_measure ? 1 : -1;
		ret	= 0;
	}
	else if (arg[0] != 1)
	{
		ret	= -EINVAL;
	}
	else if (chan->dev->cache.osc_measure)
	{
		ret	= osc_measure_ioctl(chan, arg);
	}
	else
	{
		arg[0]	= -1;
		ret		= 0;
	}

	return(ret);
}



//*****************************************************************************
static int _osc_program(chan_data_t* chan, s32* arg)
{
	int	ret	= 0;

	if (arg[0] == -1)
	{
		arg[0]	= chan->dev->cache.osc_program ? 1 : -1;
	}
	else if ((arg[0] >= 0) && (arg[0] <= _20M))
	{
		if (chan->dev->cache.osc_program)
			ret		= osc_program_ioctl(chan, arg);
		else
			arg[0]	= -1;
	}
	else
	{
		ret	= -EINVAL;
	}

	return(ret);
}



//*****************************************************************************
static int _osc_reference(chan_data_t* chan, s32* arg)
{
	int	ret;

	if ((arg[0] == -1) || ((arg[0] >= _8M) && (arg[0] <= _20M)))
	{
		ret	= osc_reference_ioctl(chan, arg);
	}
	else
	{
		ret	= -EINVAL;
	}

	return(ret);
}



//*****************************************************************************
static int _rx_fifo_reset(chan_data_t* chan, s32* arg)
{
	int	ret;

	if (arg[0] == -1)
	{
		arg[0]	= 1;
		ret		= 0;
	}
	else if (arg[0] == 1)
	{
		os_reg_mem_mx_u32(chan->dev, chan->vaddr.gsc_csr_32, D1, D1);
		os_time_sleep_ms(15);
		ret	= 0;
	}
	else
	{
		ret	= -EINVAL;
	}

	return(ret);
}



//*****************************************************************************
static int _rx_fifo_ae(chan_data_t* chan, s32* arg)
{
	int	ret;
	s32	set	= 1;
	s32	v	= arg[0];

	ret	= _arg_reg_range(chan, arg, 0, 0xFFFF, SIO4_GSC_RAR, 15, 0);

	if ((ret == 0) && (v >= 0))
		ret	= _rx_fifo_reset(chan, &set);

	return(ret);
}



//*****************************************************************************
static int _rx_fifo_af(chan_data_t* chan, s32* arg)
{
	int	ret;
	s32	set	= 1;
	s32	v	= arg[0];

	ret	= _arg_reg_range(chan, arg, 0, 0xFFFF, SIO4_GSC_RAR, 31, 16);

	if ((ret == 0) && (v >= 0))
		ret	= _rx_fifo_reset(chan, &set);

	return(ret);
}



//*****************************************************************************
static int _rx_fifo_status(chan_data_t* chan, s32* arg)
{
	u32	csr;
	int	ret;

	if (arg[0] == -1)
	{
		ret	= 0;
		csr	= os_reg_mem_rx_u32(chan->dev, chan->vaddr.gsc_csr_32);

		if ((csr & D15) == 0)
			arg[0]	= SIO4_FIFO_STATUS_FULL;
		else if ((csr & D14) == 0)
			arg[0]	= SIO4_FIFO_STATUS_ALMOST_FULL;
		else if ((csr & D12) == 0)
			arg[0]	= SIO4_FIFO_STATUS_EMPTY;
		else if ((csr & D13) == 0)
			arg[0]	= SIO4_FIFO_STATUS_ALMOST_EMPTY;
		else
			arg[0]	= SIO4_FIFO_STATUS_MEDIUM;
	}
	else
	{
		ret	= -EINVAL;
	}

	return(ret);
}



//*****************************************************************************
static int _rx_fifo_fill_level(chan_data_t* chan, s32* arg)
{
	int	ret;

	if (arg[0] != -1)
	{
		ret	= -EINVAL;
	}
	else if (chan->dev->cache.reg_fcr)
	{
		ret	= _arg_reg_range(chan, arg, 0, 0xFFFF, SIO4_GSC_FCR, 31, 16);
	}
	else
	{
		// Approximate the fill level.
		ret	= _rx_fifo_status(chan, arg);

		switch (arg[0])
		{
			default:
			case SIO4_FIFO_STATUS_EMPTY:

				arg[0]	= 0;
				break;

			case SIO4_FIFO_STATUS_ALMOST_EMPTY:

				arg[0]	= 1;
				break;

			case SIO4_FIFO_STATUS_MEDIUM:

				arg[0]	= -1;
				ret		= _rx_fifo_ae(chan, arg);
				break;

			case SIO4_FIFO_STATUS_ALMOST_FULL:

				arg[0]	= -1;
				ret		= _rx_fifo_af(chan, arg);
				arg[0]	= chan->cache.fifo_size_rx - arg[0];
				break;

			case SIO4_FIFO_STATUS_FULL:

				arg[0]	= chan->cache.fifo_size_rx;
				break;
		}
	}

	return(ret);
}



//*****************************************************************************
static int _rx_fifo_full_cfg(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_RX_FIFO_FULL_CFG_OVER,
		SIO4_RX_FIFO_FULL_CFG_DISABLE,
		-1	// terminate list
	};

	int	reg	= (chan->cache.rx_fifo_full_cfg_chan) ? SIO4_GSC_CSR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 19, 19);
	return(ret);
}



//*****************************************************************************
static int _rx_fifo_full_cfg_glb(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_RX_FIFO_FULL_CFG_GLB_OVER,
		SIO4_RX_FIFO_FULL_CFG_GLB_DISABLE,
		-1	// terminate list
	};

	int	reg	= (chan->dev->cache.rx_fifo_full_cfg_glb) ? SIO4_GSC_BCR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 8, 8);
	return(ret);
}



//*****************************************************************************
static int _rx_fifo_overrun(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_FIFO_OVERRUN_NO,
		SIO4_FIFO_OVERRUN_YES,
		-1	// terminate list
	};

	int	reg;
	int	ret;

	if ((chan->cache.rx_fifo_overrun) &&
		(	(arg[0] == SIO4_FIFO_OVERRUN_CLEAR) ||
			(arg[0] == SIO4_FIFO_OVERRUN_TEST)))
	{
		reg	= SIO4_GSC_CSR;
	}
	else
	{
		reg	= 0;
	}

	ret	= _arg_reg_list(chan, arg, list, reg, 16, 16);
	return(ret);
}



//*****************************************************************************
static int _rx_fifo_underrun(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_FIFO_UNDERRUN_NO,
		SIO4_FIFO_UNDERRUN_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.rx_fifo_underrun ? SIO4_GSC_CSR : 0;
	int	ret;

	if (arg[0] == -1)
	{
		arg[0]	= chan->dev->cache.rx_fifo_underrun ? 1 : -1;
		ret		= 0;
	}
	else
	{
		ret	= _arg_reg_list(chan, arg, list, reg, 18, 18);
	}

	return(ret);
}



//*****************************************************************************
static int _rx_io_abort(chan_data_t* chan, s32* arg)
{
	int	ret	= 0;

	switch (arg[0])
	{
		default:

			ret	= -EINVAL;
			break;

		case -1:

			arg[0]	= 1;
			break;

		case SIO4_IO_ABORT_NO:

			break;

		case SIO4_IO_ABORT_YES:

			ret	= gsc_read_abort_active_xfer(chan);

			if (ret < 0)
			{
				arg[0]	= 0;
			}
			else
			{
				ret		= 0;
				arg[0]	= ret;
			}
	}

	return(ret);
}



//*****************************************************************************
static int _rx_io_mode(chan_data_t* chan, void* arg)
{
	static const s32	list[]	=
	{
		GSC_IO_MODE_PIO,
		GSC_IO_MODE_DMA,
		GSC_IO_MODE_DMDMA,
		-1
	};

	int	ret;

	ret	= gsc_s32_list_var(arg, list, &chan->rx.io_mode);
	return(ret);
}



//*****************************************************************************
static int _rx_io_overrun(chan_data_t* chan, void* arg)
{
	static const s32	options[]	=
	{
		SIO4_IO_OVERRUN_CHECK,
		SIO4_IO_OVERRUN_IGNORE,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_var(arg, options, &chan->rx.io_overrun_check);
	return(ret);
}



//*****************************************************************************
static int _rx_io_pio_threshold(chan_data_t* chan, s32* arg)
{
	int	ret;

	ret	= _arg_var_range(arg, 0, 0xFFFFFFF, &chan->rx.pio_threshold);
	return(ret);
}



//*****************************************************************************
static int _rx_io_timeout(chan_data_t* chan, s32* arg)
{
	int	ret;

	ret	= gsc_s32_range_var(
			arg,
			SIO4_IO_TIMEOUT_MIN,
			SIO4_IO_TIMEOUT_MAX,
			&chan->rx.timeout_s);
	return(ret);
}



//*****************************************************************************
static int _rx_io_underrun(chan_data_t* chan, void* arg)
{
	static const s32	options[]	=
	{
		SIO4_IO_UNDERRUN_CHECK,
		SIO4_IO_UNDERRUN_IGNORE,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_var(arg, options, &chan->rx.io_underrun_check);
	return(ret);
}



//*****************************************************************************
static int _sync_leg_rxd_cfg(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_SYNC_LEG_RXD_CFG_TRI,	// 0
		SIO4_SYNC_LEG_RXD_CFG_LOW,	// 1
		SIO4_SYNC_LEG_RXD_CFG_UP,	// 2
		SIO4_SYNC_LEG_RXD_CFG_UP,	// 3
		-1	// terminate list
	};

	int	i;
	int	ret;
	u32	v;

	if ((chan->dev->cache.model_sync == 0)	||
		(chan->dev->cache.legacy_cable == 0))
	{
		if (arg[0] == -1)
			ret	= 0;
		else
			ret	= _arg_list_locate(arg, list, &i);
	}
	else if (arg[0] == -1)
	{
		reg_read(chan, SIO4_GSC_CSR, &v);
		v		= GSC_FIELD_DECODE(v, 5, 4);
		arg[0]	= list[v];
		ret		= 0;
	}
	else
	{
		ret	= _arg_reg_list(chan, arg, list, SIO4_GSC_CSR, 5, 4);
	}

	return(ret);
}



//*****************************************************************************
static int _sync_leg_txd_cfg(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_SYNC_LEG_TXD_CFG_TRI,
		SIO4_SYNC_LEG_TXD_CFG_LOW,
		SIO4_SYNC_LEG_TXD_CFG_UP,
		SIO4_SYNC_LEG_TXD_CFG_UP_LOW,
		-1	// terminate list
	};

	int	reg;
	int	ret;

	if ((chan->dev->cache.model_sync)	&&
		(chan->dev->cache.legacy_cable))
	{
		reg	= SIO4_GSC_CSR;
	}
	else
	{
		reg	= 0;
	}

	ret	= _arg_reg_list(chan, arg, list, reg, 3, 2);
	return(ret);
}



//*****************************************************************************
static int _sync_mode(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_SYNC_MODE_NORMAL,
		SIO4_SYNC_MODE_DUAL,
		-1	// terminate list
	};

	int	reg	= (chan->dev->cache.model == SIO4_MODEL_SIO4A_SYNC) ? SIO4_GSC_CSR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 29, 29);
	return(ret);
}



//*****************************************************************************
static int _sync_rx_bit_count(chan_data_t* chan, s32* arg)
{
	int	reg	= chan->dev->cache.model_sync ? SIO4_GSC_RCR : 0;
	int	ret;

	ret	= _arg_reg_range(chan, arg, 0, 0xFFFF, reg, 15, 0);
	return(ret);
}



//*****************************************************************************
static int _sync_rx_bit_order(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_SYNC_BIT_ORDER_MSB,
		SIO4_SYNC_BIT_ORDER_LSB,
		-1	// terminate list
	};

	int	ret;

	if (chan->dev->cache.model == SIO4_MODEL_SIO4A_SYNC)
	{
		ret	= _arg_reg_list(chan, arg, list, SIO4_GSC_CSR, 24, 24);
	}
	else if (chan->dev->cache.model_sync)
	{
		ret	= _arg_reg_list(chan, arg, list, SIO4_GSC_CSR, 27, 27);
	}
	else
	{
		ret	= _arg_reg_list(chan, arg, list, 0, 27, 27);
	}

	return(ret);
}



//*****************************************************************************
static int _sync_rx_count_error(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_SYNC_RX_COUNT_ERROR_NO,
		SIO4_SYNC_RX_COUNT_ERROR_YES,
		-1	// terminate list
	};

	int	reg	= (chan->dev->cache.model == SIO4_MODEL_SIO4A_SYNC) ? SIO4_GSC_CSR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 31, 31);
	return(ret);
}



//*****************************************************************************
static int _sync_rx_count_reset(chan_data_t* chan, s32* arg)
{
	int	ret	= 0;

	if (arg[0] == -1)
	{
		arg[0]	= (chan->dev->cache.model_sync) ? 1 : -1;
	}
	else if (arg[0] == 0)
	{
		ret	= 0;
	}
	else if (arg[0] != 1)
	{
		ret	= -EINVAL;
	}
	else if (chan->dev->cache.model == SIO4_MODEL_SIO4A_SYNC)
	{
		os_reg_mem_mx_u32(chan->dev, chan->vaddr.gsc_csr_32, D23, D23);
		arg[0]	= 1;
	}
	else if (chan->dev->cache.model_sync)
	{
		os_reg_mem_mx_u32(chan->dev, chan->vaddr.gsc_csr_32, D29, D29);
		arg[0]	= 1;
	}
	else
	{
		arg[0]	= -1;
	}

	return(ret);
}



//*****************************************************************************
static int _sync_rx_enable(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_SYNC_ENABLE_NO,
		SIO4_SYNC_ENABLE_YES,
		-1	// terminate list
	};

	int	ret;

	if (chan->dev->cache.model == SIO4_MODEL_SIO4A_SYNC)
	{
		ret	= _arg_reg_list(chan, arg, list, SIO4_GSC_CSR, 16, 16);
	}
	else if (chan->dev->cache.model_sync)
	{
		ret	= _arg_reg_list(chan, arg, list, SIO4_GSC_CSR, 24, 24);
	}
	else
	{
		ret	= _arg_reg_list(chan, arg, list, 0, 24, 24);
	}

	return(ret);
}



//*****************************************************************************
static int _sync_rx_gap_enable(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_SYNC_RX_GAP_ENABLE_NO,
		SIO4_SYNC_RX_GAP_ENABLE_YES,
		-1	// terminate list
	};

	int	reg;
	int	ret;

	if ((chan->dev->cache.model_sync)	&&
		(chan->dev->cache.sio4_type >= SIO4_TYPE_SIO4BX))
	{
		reg	= SIO4_GSC_CSR;
	}
	else
	{
		reg	= 0;
	}

	ret	= _arg_reg_list(chan, arg, list, reg, 30, 30);
	return(ret);
}



//*****************************************************************************
static int _sync_rxc_cfg(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_SYNC_RXC_CFG_FALL_EDGE,
		SIO4_SYNC_RXC_CFG_RISE_EDGE,
		-1	// terminate list
	};

	int	reg;
	int	ret;

	if ((chan->dev->cache.model_sync)	&&
		(chan->cache.reg_psrcr_bits & D13))
	{
		reg	= SIO4_GSC_PSRCR;
	}
	else
	{
		reg	= 0;
	}

	ret	= _arg_reg_list(chan, arg, list, reg, 13, 13);
	return(ret);
}



//*****************************************************************************
static int _sync_rxc_pol(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_SYNC_CLOCK_POL_FALL,
		SIO4_SYNC_CLOCK_POL_RISE,
		-1	// terminate list
	};

	int	reg	= (chan->dev->cache.model == SIO4_MODEL_SIO4A_SYNC) ? SIO4_GSC_CSR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 19, 19);
	return(ret);
}



//*****************************************************************************
static int _sync_rxd_cfg(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_SYNC_RXD_CFG_ACTIVE_HI,
		SIO4_SYNC_RXD_CFG_ACTIVE_LO,
		-1	// terminate list
	};

	int	reg;
	int	ret;

	if ((chan->dev->cache.model_sync)	&&
		(chan->cache.reg_psrcr_bits & D16))
	{
		reg	= SIO4_GSC_PSRCR;
	}
	else
	{
		reg	= 0;
	}

	ret	= _arg_reg_list(chan, arg, list, reg, 16, 16);
	return(ret);
}



//*****************************************************************************
static int _sync_rxe_cfg(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_SYNC_RXE_CFG_ACTIVE_HI,
		SIO4_SYNC_RXE_CFG_ACTIVE_LO,
		SIO4_SYNC_RXE_CFG_DISABLE,
		SIO4_SYNC_RXE_CFG_DISABLE,
		-1	// terminate list
	};

	int	i;
	u32	psrcr;
	int	ret;
	s32	v;

	if ((chan->dev->cache.model_sync == 0)				||
		((chan->cache.reg_psrcr_bits & D14) == 0)	||
		((chan->cache.reg_psrcr_bits & D15) == 0))
	{
		if (arg[0] == -1)
			ret	= 0;
		else
			ret	= _arg_list_locate(arg, list, &i);
	}
	else if (arg[0] == -1)
	{
		reg_read(chan, SIO4_GSC_PSRCR, &psrcr);
		v		= GSC_FIELD_DECODE(psrcr, 15, 14);
		arg[0]	= list[v];
		ret		= 0;
	}
	else
	{
		ret	= _arg_reg_list(chan, arg, list, SIO4_GSC_PSRCR, 15, 14);
	}

	return(ret);
}



//*****************************************************************************
static int _sync_rxe_pol(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_SYNC_ENV_POL_ACTIVE_LO,
		SIO4_SYNC_ENV_POL_ACTIVE_HI,
		-1	// terminate list
	};

	int	reg	= (chan->dev->cache.model == SIO4_MODEL_SIO4A_SYNC) ? SIO4_GSC_CSR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 26, 26);
	return(ret);
}



//*****************************************************************************
static int _sync_tx_word_size(chan_data_t* chan, s32* arg)
{
	int	reg	= chan->dev->cache.model_sync ? SIO4_GSC_TCR : 0;
	int	ret;

	ret	= _arg_reg_range(chan, arg, 0, 0xFFFF, reg, 15, 0);
	return(ret);
}



//*****************************************************************************
static int _sync_tx_bit_order(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_SYNC_BIT_ORDER_MSB,
		SIO4_SYNC_BIT_ORDER_LSB,
		-1	// terminate list
	};

	int	ret;

	if (chan->dev->cache.model == SIO4_MODEL_SIO4A_SYNC)
	{
		ret	= _arg_reg_list(chan, arg, list, SIO4_GSC_CSR, 25, 25);
	}
	else if (chan->dev->cache.model_sync)
	{
		ret	= _arg_reg_list(chan, arg, list, SIO4_GSC_CSR, 28, 28);
	}
	else
	{
		ret	= _arg_reg_list(chan, arg, list, 0, 28, 28);
	}

	return(ret);
}



//*****************************************************************************
static int _sync_tx_enable(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_SYNC_ENABLE_NO,
		SIO4_SYNC_ENABLE_YES,
		-1	// terminate list
	};

	int	ret;

	if (chan->dev->cache.model == SIO4_MODEL_SIO4A_SYNC)
	{
		ret	= _arg_reg_list(chan, arg, list, SIO4_GSC_CSR, 17, 17);
	}
	else if (chan->dev->cache.model_sync)
	{
		ret	= _arg_reg_list(chan, arg, list, SIO4_GSC_CSR, 25, 25);
	}
	else
	{
		ret	= _arg_reg_list(chan, arg, list, 0, 25, 25);
	}

	return(ret);
}




//*****************************************************************************
static int _sync_tx_gap_size(chan_data_t* chan, s32* arg)
{
	int	reg	= chan->dev->cache.model_sync ? SIO4_GSC_TCR : 0;
	int	ret;

	ret	= _arg_reg_range(chan, arg, 0, 0xFFFF, reg, 31, 16);
	return(ret);
}



//*****************************************************************************
static int _sync_txauxc_cfg(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_SYNC_TXAUXC_CFG_TRI,
		SIO4_SYNC_TXAUXC_CFG_OSC_HALF,
		SIO4_SYNC_TXAUXC_CFG_OUT_0,
		SIO4_SYNC_TXAUXC_CFG_OUT_1,
		-1	// terminate list
	};

	int	reg;
	int	ret;

	if ((chan->dev->cache.model_sync)			&&
		(chan->cache.reg_psrcr_bits & D9)	&&
		(chan->cache.reg_psrcr_bits & D10))
	{
		reg	= SIO4_GSC_PSRCR;
	}
	else
	{
		reg	= 0;
	}

	ret	= _arg_reg_list(chan, arg, list, reg, 10, 9);
	return(ret);
}



//*****************************************************************************
static int _sync_txc_cfg(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_SYNC_TXC_CFG_INT,
		SIO4_SYNC_TXC_CFG_EXT,
		-1	// terminate list
	};

	int	reg	= (chan->dev->cache.model == SIO4_MODEL_SIO4A_SYNC) ? SIO4_GSC_CSR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 22, 22);
	return(ret);
}



//*****************************************************************************
static int _sync_txc_idle(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_SYNC_TXC_IDLE_YES,
		SIO4_SYNC_TXC_IDLE_NO,
		-1	// terminate list
	};

	int	reg	= (chan->dev->cache.model == SIO4_MODEL_SIO4A_SYNC) ? SIO4_GSC_CSR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 21, 21);
	return(ret);
}



//*****************************************************************************
static int _sync_txc_idle_cfg(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_SYNC_TXC_IDLE_CFG_IDLE_1,
		SIO4_SYNC_TXC_IDLE_CFG_IDLE_0,
		SIO4_SYNC_TXC_IDLE_CFG_ACTIVE,
		-1	// terminate list
	};

	int	ret;

	if ((chan->dev->cache.model_sync)			&&
		(chan->cache.reg_psrcr_bits & D3)	&&
		(chan->cache.reg_psrcr_bits & D20))
	{
		ret	= _arg_reg_mask(chan, arg, list, SIO4_GSC_PSRCR, D20 | D3, 3);
	}
	else if (	(chan->dev->cache.model_sync)	&&
				(chan->cache.reg_psrcr_bits & D3))
	{
		if (arg[0] == SIO4_SYNC_TXC_IDLE_CFG_IDLE_1)
			arg[0]	= SIO4_SYNC_TXC_IDLE_CFG_IDLE_0;

		ret	= _arg_reg_mask(chan, arg, list, SIO4_GSC_PSRCR, D20 | D3, 3);
	}
	else
	{
		ret	= _arg_reg_mask(chan, arg, list, 0, D20 | D3, 3);
	}

	return(ret);
}



//*****************************************************************************
static int _sync_txc_pol(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_SYNC_CLOCK_POL_FALL,
		SIO4_SYNC_CLOCK_POL_RISE,
		-1	// terminate list
	};

	int	reg	= (chan->dev->cache.model == SIO4_MODEL_SIO4A_SYNC) ? SIO4_GSC_CSR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 20, 20);
	return(ret);
}



//*****************************************************************************
static int _sync_txc_src(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_SYNC_TXC_SRC_OSC_HALF_RISE,
		SIO4_SYNC_TXC_SRC_OSC_HALF_FALL,
		SIO4_SYNC_TXC_SRC_EXT_RISE,
		SIO4_SYNC_TXC_SRC_EXT_FALL,
		SIO4_SYNC_TXC_SRC_0,
		SIO4_SYNC_TXC_SRC_1,
		-1	// terminate list
	};

	int	reg;
	int	ret;

	if ((chan->dev->cache.model_sync)		&&
		(chan->cache.reg_psrcr_bits & D0)	&&
		(chan->cache.reg_psrcr_bits & D1)	&&
		(chan->cache.reg_psrcr_bits & D2))
	{
		reg	= SIO4_GSC_PSRCR;
	}
	else
	{
		reg	= 0;
	}

	ret	= _arg_reg_list(chan, arg, list, reg, 2, 0);
	return(ret);
}



//*****************************************************************************
static int _sync_txd_cfg(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_SYNC_TXD_CFG_ACTIVE_HI,
		SIO4_SYNC_TXD_CFG_ACTIVE_LO,
		SIO4_SYNC_TXD_CFG_OUT_0,
		SIO4_SYNC_TXD_CFG_OUT_1,
		-1	// terminate list
	};

	int	reg;
	int	ret;

	if ((chan->dev->cache.model_sync)		&&
		(chan->cache.reg_psrcr_bits & D6)	&&
		(chan->cache.reg_psrcr_bits & D7)	&&
		(chan->cache.reg_psrcr_bits & D8))
	{
		reg	= SIO4_GSC_PSRCR;
	}
	else
	{
		reg	= 0;
	}

	ret	= _arg_reg_list(chan, arg, list, reg, 8, 6);
	return(ret);
}



//*****************************************************************************
static int _sync_txd_idle_cfg(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_SYNC_TXD_IDLE_CFG_OUT_0,
		SIO4_SYNC_TXD_IDLE_CFG_OUT_1,
		-1	// terminate list
	};

	int	reg;
	int	ret;

	if ((chan->dev->cache.model_sync)	&&
		(chan->cache.reg_psrcr_bits & D19))
	{
		reg	= SIO4_GSC_PSRCR;
	}
	else
	{
		reg	= 0;
	}

	ret	= _arg_reg_list(chan, arg, list, reg, 19, 19);
	return(ret);
}



//*****************************************************************************
static int _sync_txe_cfg(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_SYNC_TXE_CFG_ACTIVE_HI,
		SIO4_SYNC_TXE_CFG_ACTIVE_LO,
		SIO4_SYNC_TXE_CFG_OUT_0,
		SIO4_SYNC_TXE_CFG_OUT_1,
		-1	// terminate list
	};

	int	reg;
	int	ret;

	if ((chan->dev->cache.model_sync)		&&
		(chan->cache.reg_psrcr_bits & D4)	&&
		(chan->cache.reg_psrcr_bits & D5))
	{
		reg	= SIO4_GSC_PSRCR;
	}
	else
	{
		reg	= 0;
	}

	ret	= _arg_reg_list(chan, arg, list, reg, 5, 4);
	return(ret);
}



//*****************************************************************************
static int _sync_txe_pol(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_SYNC_ENV_POL_ACTIVE_LO,
		SIO4_SYNC_ENV_POL_ACTIVE_HI,
		-1	// terminate list
	};

	int	reg	= (chan->dev->cache.model == SIO4_MODEL_SIO4A_SYNC) ? SIO4_GSC_CSR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 27, 27);
	return(ret);
}



//*****************************************************************************
static int _sync_txsp_cfg(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_SYNC_TXSP_CFG_DISABLE,
		SIO4_SYNC_TXSP_CFG_INPUT,
		SIO4_SYNC_TXSP_CFG_OUT_0,
		SIO4_SYNC_TXSP_CFG_OUT_1,
		-1	// terminate list
	};

	int	reg;
	int	ret;

	if ((chan->dev->cache.model_sync)		&&
		(chan->cache.reg_psrcr_bits & D11)	&&
		(chan->cache.reg_psrcr_bits & D12))
	{
		reg	= SIO4_GSC_PSRCR;
	}
	else
	{
		reg	= 0;
	}

	ret	= _arg_reg_list(chan, arg, list, reg, 12, 11);
	return(ret);
}



//*****************************************************************************
static int _time_stamp_count(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_TIME_STAMP_COUNT_CONTINUE,
		SIO4_TIME_STAMP_COUNT_CLEAR,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.time_stamp ? SIO4_GSC_BCR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 23, 23);
	return(ret);
}



//*****************************************************************************
static int _time_stamp_enable(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_TIME_STAMP_ENABLE_NO,
		SIO4_TIME_STAMP_ENABLE_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.time_stamp ? SIO4_GSC_CSR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 2, 2);
	return(ret);
}



//*****************************************************************************
static int _time_stamp_src(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_TIME_STAMP_SRC_INT,
		SIO4_TIME_STAMP_SRC_EXT,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.time_stamp ? SIO4_GSC_BCR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 22, 22);
	return(ret);
}



//*****************************************************************************
static int _time_stamp_val(chan_data_t* chan, s32* arg)
{
	int	reg	= chan->dev->cache.time_stamp ? SIO4_GSC_TSR : 0;
	int	ret;

	ret	= _arg_reg_range(chan, arg, 0, 0xFFFFFF, reg, 23, 0);
	return(ret);
}



//*****************************************************************************
static int _tx_fifo_reset(chan_data_t* chan, s32* arg)
{
	int	ret	= 0;

	if (arg[0] == -1)
	{
		arg[0]	= 1;
	}
	else if (arg[0] == 1)
	{
		os_reg_mem_mx_u32(chan->dev, chan->vaddr.gsc_csr_32, D0, D0);
		os_time_sleep_ms(15);
	}
	else
	{
		ret	= -EINVAL;
	}

	return(ret);
}



//*****************************************************************************
static int _tx_fifo_ae(chan_data_t* chan, s32* arg)
{
	int	ret;
	s32	set	= 1;
	s32	v	= arg[0];

	ret	= _arg_reg_range(chan, arg, 0, 0xFFFF, SIO4_GSC_TAR, 15, 0);

	if ((ret == 0) && (v >= 0))
		ret	= _tx_fifo_reset(chan, &set);

	return(ret);
}



//*****************************************************************************
static int _tx_fifo_af(chan_data_t* chan, s32* arg)
{
	int	ret;
	s32	set	= 1;
	s32	v	= arg[0];

	ret	= _arg_reg_range(chan, arg, 0, 0xFFFF, SIO4_GSC_TAR, 31, 16);

	if ((ret == 0) && (v >= 0))
		ret	= _tx_fifo_reset(chan, &set);

	return(ret);
}



//*****************************************************************************
static int _tx_fifo_empty_cfg(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_TX_FIFO_EMPTY_CFG_IGNORE,
		SIO4_TX_FIFO_EMPTY_CFG_TX_OFF,
		-1	// terminate list
	};

	int	ret;

	if (chan->cache.tx_fifo_empty_cfg == 0)
	{
		ret	= _arg_reg_list(chan, arg, list, 0, 26, 26);
	}
	else if (chan->dev->cache.model == SIO4_MODEL_SIO4A_SYNC)
	{
		ret	= _arg_reg_list(chan, arg, list, SIO4_GSC_CSR, 18, 18);
	}
	else if (chan->dev->cache.model_sync)
	{
		ret	= _arg_reg_list(chan, arg, list, SIO4_GSC_CSR, 26, 26);
	}
	else
	{
		ret	= _arg_reg_list(chan, arg, list, 0, 26, 26);
	}

	return(ret);
}



//*****************************************************************************
static int _tx_fifo_status(chan_data_t* chan, s32* arg)
{
	u32	csr;
	int	ret		= 0;

	if (arg[0] != -1)
	{
		ret	= -EINVAL;
	}
	else
	{
		csr	= os_reg_mem_rx_u32(chan->dev, chan->vaddr.gsc_csr_32);

		if ((csr & D11) == 0)
			arg[0]	= SIO4_FIFO_STATUS_FULL;
		else if ((csr & D10) == 0)
			arg[0]	= SIO4_FIFO_STATUS_ALMOST_FULL;
		else if ((csr & D8) == 0)
			arg[0]	= SIO4_FIFO_STATUS_EMPTY;
		else if ((csr & D9) == 0)
			arg[0]	= SIO4_FIFO_STATUS_ALMOST_EMPTY;
		else
			arg[0]	= SIO4_FIFO_STATUS_MEDIUM;
	}

	return(ret);
}



//*****************************************************************************
static int _tx_fifo_fill_level(chan_data_t* chan, s32* arg)
{
	int	ret;

	if (arg[0] != -1)
	{
		ret	= -EINVAL;
	}
	else if (chan->dev->cache.reg_fcr)
	{
		ret	= _arg_reg_range(chan, arg, 0, 0xFFFF, SIO4_GSC_FCR, 15, 0);
	}
	else
	{
		// Approximate the fill level.
		ret	= _tx_fifo_status(chan, arg);

		switch (arg[0])
		{
			default:
			case SIO4_FIFO_STATUS_EMPTY:

				arg[0]	= 0;
				break;

			case SIO4_FIFO_STATUS_ALMOST_EMPTY:

				arg[0]	= 1;
				break;

			case SIO4_FIFO_STATUS_MEDIUM:

				arg[0]	= -1;
				ret		= _tx_fifo_ae(chan, arg);
				break;

			case SIO4_FIFO_STATUS_ALMOST_FULL:

				arg[0]	= -1;
				ret		= _tx_fifo_af(chan, arg);
				arg[0]	= chan->cache.fifo_size_tx - arg[0];
				break;

			case SIO4_FIFO_STATUS_FULL:

				arg[0]	= chan->cache.fifo_size_tx;
				break;
		}
	}

	return(ret);
}



//*****************************************************************************
static int _tx_fifo_overrun(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_FIFO_OVERRUN_NO,
		SIO4_FIFO_OVERRUN_YES,
		-1	// terminate list
	};

	int	reg;
	int	ret;

	if ((chan->dev->cache.tx_fifo_overrun)	&&
		(	(arg[0] == SIO4_FIFO_OVERRUN_CLEAR) ||
			(arg[0] == SIO4_FIFO_OVERRUN_TEST)))
	{
		reg	= SIO4_GSC_CSR;
	}
	else
	{
		reg	= 0;
	}

	ret	= _arg_reg_list(chan, arg, list, reg, 17, 17);
	return(ret);
}



//*****************************************************************************
static int _tx_io_abort(chan_data_t* chan, s32* arg)
{
	int	ret	= 0;

	switch (arg[0])
	{
		default:

			ret	= -EINVAL;
			break;

		case -1:

			arg[0]	= 1;
			break;

		case SIO4_IO_ABORT_NO:

			break;

		case SIO4_IO_ABORT_YES:

			ret	= gsc_write_abort_active_xfer(chan);

			if (ret < 0)
			{
				arg[0]	= 0;
			}
			else
			{
				ret		= 0;
				arg[0]	= ret;
			}
	}

	return(ret);
}



//*****************************************************************************
static int _tx_io_mode(chan_data_t* chan, void* arg)
{
	static const s32	list[]	=
	{
		GSC_IO_MODE_PIO,
		GSC_IO_MODE_DMA,
		GSC_IO_MODE_DMDMA,
		-1
	};

	int	ret;

	ret	= gsc_s32_list_var(arg, list, &chan->tx.io_mode);
	return(ret);
}



//*****************************************************************************
static int _tx_io_overrun(chan_data_t* chan, void* arg)
{
	static const s32	options[]	=
	{
		SIO4_IO_OVERRUN_CHECK,
		SIO4_IO_OVERRUN_IGNORE,
		-1	// terminate list
	};

	int	ret;

	ret	= gsc_s32_list_var(arg, options, &chan->tx.io_overrun_check);
	return(ret);
}



//*****************************************************************************
static int _tx_io_pio_threshold(chan_data_t* chan, s32* arg)
{
	int	ret;

	ret	= _arg_var_range(arg, 0, 0xFFFFFFF, &chan->tx.pio_threshold);
	return(ret);
}



//*****************************************************************************
static int _tx_io_timeout(chan_data_t* chan, s32* arg)
{
	int	ret;

	ret	= gsc_s32_range_var(
			arg,
			SIO4_IO_TIMEOUT_MIN,
			SIO4_IO_TIMEOUT_MAX,
			&chan->tx.timeout_s);
	return(ret);
}



//*****************************************************************************
static int _xcvr_enable(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_XCVR_ENABLE_NO,
		SIO4_XCVR_ENABLE_YES,
		-1	// terminate list
	};

	int	reg	= (chan->cache.reg_psrcr_bits & D31) ? SIO4_GSC_PSRCR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 31, 31);
	return(ret);
}



//*****************************************************************************
static int _xcvr_protocol(chan_data_t* chan, s32* arg)
{
	int	ret;

	if ((chan->cache.reg_psrcr_bits & D24)	&&
		(chan->cache.reg_psrcr_bits & D25)	&&
		(chan->cache.reg_psrcr_bits & D26)	&&
		(chan->cache.reg_psrcr_bits & D27))
	{
		ret	= mp_select_ioctl(chan, arg);
	}
	else if (arg[0] == -1)
	{
		ret	= 0;
	}
	else if ((arg[0] & 0xF) == arg[0])
	{
		ret	= 0;
	}
	else
	{
		ret	= -EINVAL;
	}

	return(ret);
}



//*****************************************************************************
static int _xcvr_term(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_XCVR_TERM_ENABLE,
		SIO4_XCVR_TERM_DISABLE,
		-1	// terminate list
	};

	int	reg	= (chan->cache.reg_psrcr_bits & D30) ? SIO4_GSC_PSRCR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 30, 30);
	return(ret);
}



//*****************************************************************************
static int _z16_cbl_dcd_cfg(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_Z16_CBL_DCD_CFG_OUT_USC_DCD,
		SIO4_Z16_CBL_DCD_CFG_OUT_RTS,
		SIO4_Z16_CBL_DCD_CFG_OUT_0,
		SIO4_Z16_CBL_DCD_CFG_OUT_1,
		-1	// terminate list
	};

	int	reg;
	int	ret;

	if ((chan->cache.reg_psrcr_bits & D15)	&&
		(chan->cache.reg_psrcr_bits & D16))
	{
		reg	= SIO4_GSC_PSRCR;
	}
	else
	{
		reg	= 0;
	}

	ret	= _arg_reg_list(chan, arg, list, reg, 16, 15);
	return(ret);
}



//*****************************************************************************
static int _z16_cbl_dtr_dsr_cfg(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_Z16_CBL_DTR_DSR_CFG_TRI,
		SIO4_Z16_CBL_DTR_DSR_CFG_IN,
		SIO4_Z16_CBL_DTR_DSR_CFG_OUT_0,
		SIO4_Z16_CBL_DTR_DSR_CFG_OUT_1,
		-1	// terminate list
	};

	int	reg;
	int	ret;

	if ((chan->cache.reg_psrcr_bits & D21)	&&
		(chan->cache.reg_psrcr_bits & D22))
	{
		reg	= SIO4_GSC_PSRCR;
	}
	else
	{
		reg	= 0;
	}

	ret	= _arg_reg_list(chan, arg, list, reg, 22, 21);
	return(ret);
}



//*****************************************************************************
static int _z16_cbl_rts_cfg(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_Z16_CBL_RTS_CFG_OUT_USC_CTS,
		SIO4_Z16_CBL_RTS_CFG_OUT_RTS,
		SIO4_Z16_CBL_RTS_CFG_OUT_0,
		SIO4_Z16_CBL_RTS_CFG_OUT_1,
		-1	// terminate list
	};

	int	reg;
	int	ret;

	if ((chan->cache.reg_psrcr_bits & D13)	&&
		(chan->cache.reg_psrcr_bits & D14))
	{
		reg	= SIO4_GSC_PSRCR;
	}
	else
	{
		reg	= 0;
	}

	ret	= _arg_reg_list(chan, arg, list, reg, 14, 13);
	return(ret);
}



//*****************************************************************************
static int _z16_cbl_txauxc_cfg(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_Z16_CBL_TXAUXC_CFG_TRI,
		SIO4_Z16_CBL_TXAUXC_CFG_OUT_OSC,
		SIO4_Z16_CBL_TXAUXC_CFG_OUT_0,
		SIO4_Z16_CBL_TXAUXC_CFG_OUT_1,
		-1	// terminate list
	};

	int	reg;
	int	ret;

	if ((chan->cache.reg_psrcr_bits & D17)	&&
		(chan->cache.reg_psrcr_bits & D18))
	{
		reg	= SIO4_GSC_PSRCR;
	}
	else
	{
		reg	= 0;
	}

	ret	= _arg_reg_list(chan, arg, list, reg, 18, 17);
	return(ret);
}



//*****************************************************************************
static int _z16_cbl_txc_cfg(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_Z16_CBL_TXC_CFG_OUT_OSC,
		SIO4_Z16_CBL_TXC_CFG_OUT_OSC_INV,
		SIO4_Z16_CBL_TXC_CFG_OUT_0,
		SIO4_Z16_CBL_TXC_CFG_OUT_1,
		SIO4_Z16_CBL_TXC_CFG_OUT_USC_TXC,
		SIO4_Z16_CBL_TXC_CFG_OUT_USC_RXC,
		SIO4_Z16_CBL_TXC_CFG_OUT_CBL_RXC,
		SIO4_Z16_CBL_TXC_CFG_OUT_CBL_RXA,
		-1	// terminate list
	};

	int	reg;
	int	ret;

	if ((chan->cache.reg_psrcr_bits & D6)	&&
		(chan->cache.reg_psrcr_bits & D7)	&&
		(chan->cache.reg_psrcr_bits & D8))
	{
		reg	= SIO4_GSC_PSRCR;
	}
	else
	{
		reg	= 0;
	}

	ret	= _arg_reg_list(chan, arg, list, reg, 8, 6);
	return(ret);
}



//*****************************************************************************
static int _z16_cbl_txd_cfg(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_Z16_CBL_TXD_CFG_OUT_USC_TXD,	// 0
		SIO4_Z16_CBL_TXD_CFG_OUT_USC_TXD,	// 1
		SIO4_Z16_CBL_TXD_CFG_OUT_0,			// 2
		SIO4_Z16_CBL_TXD_CFG_OUT_1,			// 3
		-1	// terminate list
	};

	int	i;
	int	ret;
	u32	v;

	if (((chan->cache.reg_psrcr_bits & D19) == 0)	||
		((chan->cache.reg_psrcr_bits & D20) == 0))
	{
		if (arg[0] == -1)
			ret	= 0;
		else
			ret	= _arg_list_locate(arg, list, &i);
	}
	else if (arg[0] == -1)
	{
		reg_read(chan, SIO4_GSC_PSRCR, &v);
		v		= GSC_FIELD_DECODE(v, 20, 19);
		arg[0]	= list[v];
		ret		= 0;
	}
	else
	{
		ret	= _arg_reg_list(chan, arg, list, SIO4_GSC_PSRCR, 20, 19);
	}

	return(ret);
}



//*****************************************************************************
static int _z16_leg_rxc(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_Z16_LEG_RXC_DISABLE,	// 0
		SIO4_Z16_LEG_RXC_UPPER,		// 1
		SIO4_Z16_LEG_RXC_LOWER,		// 2
		SIO4_Z16_LEG_RXC_UPPER,		// 3
		-1
	};

	int	begin	= (chan->index * 4) + 3;
	int	end		= begin - 1;
	int	i;
	int	ret;
	u32	v;

	if (chan->cache.reg_ccr == 0)
	{
		if (arg[0] == -1)
			ret	= 0;
		else
			ret	= _arg_list_locate(arg, list, &i);
	}
	else if (arg[0] == -1)
	{
		reg_read(chan, SIO4_GSC_CCR, &v);
		v		= GSC_FIELD_DECODE(v, begin, end);
		arg[0]	= list[v];
		ret		= 0;
	}
	else
	{
		ret	= _arg_reg_list(chan, arg, list, SIO4_GSC_CCR, begin, end);
	}

	return(ret);
}



//*****************************************************************************
static int _z16_leg_rxd_dcd_cfg(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_Z16_LEG_RXD_DCD_CFG_TRI,	// 0
		SIO4_Z16_LEG_RXD_DCD_CFG_UP,	// 1
		SIO4_Z16_LEG_RXD_DCD_CFG_LOW,	// 2
		SIO4_Z16_LEG_RXD_DCD_CFG_UP,	// 3
		-1	// terminate list
	};

	int	i;
	int	ret;
	u32	v;

	if ((chan->dev->cache.model_z16c30 == 0)	||
		(chan->dev->cache.legacy_cable == 0))
	{
		if (arg[0] == -1)
			ret	= 0;
		else
			ret	= _arg_list_locate(arg, list, &i);
	}
	else if (arg[0] == -1)
	{
		reg_read(chan, SIO4_GSC_CSR, &v);
		v		= GSC_FIELD_DECODE(v, 5, 4);
		arg[0]	= list[v];
		ret		= 0;
	}
	else
	{
		ret	= _arg_reg_list(chan, arg, list, SIO4_GSC_CSR, 5, 4);
	}

	return(ret);
}



//*****************************************************************************
static int _z16_leg_txc(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_Z16_LEG_TXC_DISABLE,
		SIO4_Z16_LEG_TXC_UPPER,
		SIO4_Z16_LEG_TXC_LOWER,
		SIO4_Z16_LEG_TXC_BOTH,
		-1
	};

	int	begin	= chan->index * 4 + 1;
	int	end		= begin - 1;
	int	reg		= chan->cache.reg_ccr ? SIO4_GSC_CCR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, begin, end);
	return(ret);
}



//*****************************************************************************
static int _z16_leg_txd_cts_cfg(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_Z16_LEG_TXD_CTS_CFG_TRI,
		SIO4_Z16_LEG_TXD_CTS_CFG_UP,
		SIO4_Z16_LEG_TXD_CTS_CFG_LOW,
		SIO4_Z16_LEG_TXD_CTS_CFG_UP_LOW,
		-1	// terminate list
	};

	int	reg;
	int	ret;

	if ((chan->dev->cache.model_z16c30)	&& (chan->dev->cache.legacy_cable))
	{
		reg	= SIO4_GSC_CSR;
	}
	else
	{
		reg	= 0;
	}

	ret	= _arg_reg_list(chan, arg, list, reg, 3, 2);
	return(ret);
}



//*****************************************************************************
static int _z16_rx_sts_wrd_enable(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_Z16_RX_STS_WRD_ENABLE_NO,
		SIO4_Z16_RX_STS_WRD_ENABLE_YES,
		-1	// terminate list
	};

	int	reg	= chan->cache.rx_status_word ? SIO4_GSC_CSR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 3, 3);
	return(ret);
}



//*****************************************************************************
static int _z16_sync_byte(chan_data_t* chan, s32* arg)
{
	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_GSC_SBR : 0;
	int	ret;

	ret	= _arg_reg_range(chan, arg, 0, 0xFF, reg, 7, 0);
	return(ret);
}




//*****************************************************************************
static int _usc_8023_rx_adrs_srch(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_8023_RX_ADRS_SRCH_NO,
		SIO4_USC_8023_RX_ADRS_SRCH_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMR : 0;
	int	ret;

	// 802.3
	ret	= _arg_reg_list(chan, arg, list, reg, 4, 4);
	return(ret);
}



//*****************************************************************************
static int _usc_8023_tx_underrun(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_8023_TX_UNDERRUN_CRC,
		SIO4_USC_8023_TX_UNDERRUN_NONE,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMR : 0;
	int	ret;

	// 802.3
	ret	= _arg_reg_list(chan, arg, list, reg, 15, 15);
	return(ret);
}



//*****************************************************************************
static int _usc_accept_cv(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_ACCEPT_CV_NO,
		SIO4_USC_ACCEPT_CV_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_HCR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 12, 12);
	return(ret);
}



//*****************************************************************************
static int _usc_acv_rx_ext_w(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_ACV_EXT_W_NO,
		SIO4_USC_ACV_EXT_W_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMR : 0;
	int	ret;

	// Asynchronous w/ CV
	ret	= _arg_reg_list(chan, arg, list, reg, 4, 4);
	return(ret);
}



//*****************************************************************************
static int _usc_acv_tx_cv_pol(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_ACV_TX_CV_POL_0,
		SIO4_USC_ACV_TX_CV_POL_1,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMR : 0;
	int	ret;

	// Asynchronous w/ CV
	ret	= _arg_reg_list(chan, arg, list, reg, 12, 12);
	return(ret);
}



//*****************************************************************************
static int _usc_acv_tx_ext_w(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_ACV_EXT_W_NO,
		SIO4_USC_ACV_EXT_W_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMR : 0;
	int	ret;

	// Asynchronous w/ CV
	ret	= _arg_reg_list(chan, arg, list, reg, 13, 13);
	return(ret);
}



//*****************************************************************************
static int _usc_acv_tx_stop_bit(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_ACV_TX_STOP_BIT_1,
		SIO4_USC_ACV_TX_STOP_BIT_2,
		SIO4_USC_ACV_TX_STOP_BIT_NONE,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMR : 0;
	int	ret;

	// Asynchronous w/ CV
	ret	= _arg_reg_list(chan, arg, list, reg, 15, 14);
	return(ret);
}



//*****************************************************************************
static int _usc_async_rx_clk_rate(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_ASYNC_CLK_RATE_16X,
		SIO4_USC_ASYNC_CLK_RATE_32X,
		SIO4_USC_ASYNC_CLK_RATE_64X,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMR : 0;
	int	ret;

	// Asynchronous
	ret	= _arg_reg_list(chan, arg, list, reg, 5, 4);
	return(ret);
}



//*****************************************************************************
static int _usc_async_tx_clk_rate(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_ASYNC_CLK_RATE_16X,
		SIO4_USC_ASYNC_CLK_RATE_32X,
		SIO4_USC_ASYNC_CLK_RATE_64X,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMR : 0;
	int	ret;

	// Asynchronous
	ret	= _arg_reg_list(chan, arg, list, reg, 13, 12);
	return(ret);
}



//*****************************************************************************
static int _usc_async_tx_stop_bit(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_ASYNC_TX_STOP_BIT_1,		// 0	USC CMR D14-D15 = 0, USC CCR D8-D11 = 0	// upper nibble: CMR D14-D15
		SIO4_USC_ASYNC_TX_STOP_BIT_1,		// 1	USC CMR D14-D15 = 0, USC CCR D8-D11 = 1	// lower nibble: CCR D8-D11, if used
		SIO4_USC_ASYNC_TX_STOP_BIT_1,		// 2	USC CMR D14-D15 = 0, USC CCR D8-D11 = 2
		SIO4_USC_ASYNC_TX_STOP_BIT_1,		// 3	USC CMR D14-D15 = 0, USC CCR D8-D11 = 3
		SIO4_USC_ASYNC_TX_STOP_BIT_1,		// 4	USC CMR D14-D15 = 0, USC CCR D8-D11 = 4
		SIO4_USC_ASYNC_TX_STOP_BIT_1,		// 5	USC CMR D14-D15 = 0, USC CCR D8-D11 = 5
		SIO4_USC_ASYNC_TX_STOP_BIT_1,		// 6	USC CMR D14-D15 = 0, USC CCR D8-D11 = 6
		SIO4_USC_ASYNC_TX_STOP_BIT_1,		// 7	USC CMR D14-D15 = 0, USC CCR D8-D11 = 7
		SIO4_USC_ASYNC_TX_STOP_BIT_1,		// 8	USC CMR D14-D15 = 0, USC CCR D8-D11 = 8
		SIO4_USC_ASYNC_TX_STOP_BIT_1,		// 9	USC CMR D14-D15 = 0, USC CCR D8-D11 = 9
		SIO4_USC_ASYNC_TX_STOP_BIT_1,		// 10	USC CMR D14-D15 = 0, USC CCR D8-D11 = 10
		SIO4_USC_ASYNC_TX_STOP_BIT_1,		// 11	USC CMR D14-D15 = 0, USC CCR D8-D11 = 11
		SIO4_USC_ASYNC_TX_STOP_BIT_1,		// 12	USC CMR D14-D15 = 0, USC CCR D8-D11 = 12
		SIO4_USC_ASYNC_TX_STOP_BIT_1,		// 13	USC CMR D14-D15 = 0, USC CCR D8-D11 = 13
		SIO4_USC_ASYNC_TX_STOP_BIT_1,		// 14	USC CMR D14-D15 = 0, USC CCR D8-D11 = 14
		SIO4_USC_ASYNC_TX_STOP_BIT_1,		// 15	USC CMR D14-D15 = 0, USC CCR D8-D11 = 15

		SIO4_USC_ASYNC_TX_STOP_BIT_2,		// 16	USC CMR D14-D15 = 1, USC CCR D8-D11 = 0
		SIO4_USC_ASYNC_TX_STOP_BIT_2,		// 17	USC CMR D14-D15 = 1, USC CCR D8-D11 = 1
		SIO4_USC_ASYNC_TX_STOP_BIT_2,		// 18	USC CMR D14-D15 = 1, USC CCR D8-D11 = 2
		SIO4_USC_ASYNC_TX_STOP_BIT_2,		// 19	USC CMR D14-D15 = 1, USC CCR D8-D11 = 3
		SIO4_USC_ASYNC_TX_STOP_BIT_2,		// 20	USC CMR D14-D15 = 1, USC CCR D8-D11 = 4
		SIO4_USC_ASYNC_TX_STOP_BIT_2,		// 21	USC CMR D14-D15 = 1, USC CCR D8-D11 = 5
		SIO4_USC_ASYNC_TX_STOP_BIT_2,		// 22	USC CMR D14-D15 = 1, USC CCR D8-D11 = 6
		SIO4_USC_ASYNC_TX_STOP_BIT_2,		// 23	USC CMR D14-D15 = 1, USC CCR D8-D11 = 7
		SIO4_USC_ASYNC_TX_STOP_BIT_2,		// 24	USC CMR D14-D15 = 1, USC CCR D8-D11 = 8
		SIO4_USC_ASYNC_TX_STOP_BIT_2,		// 25	USC CMR D14-D15 = 1, USC CCR D8-D11 = 9
		SIO4_USC_ASYNC_TX_STOP_BIT_2,		// 26	USC CMR D14-D15 = 1, USC CCR D8-D11 = 10
		SIO4_USC_ASYNC_TX_STOP_BIT_2,		// 27	USC CMR D14-D15 = 1, USC CCR D8-D11 = 11
		SIO4_USC_ASYNC_TX_STOP_BIT_2,		// 28	USC CMR D14-D15 = 1, USC CCR D8-D11 = 12
		SIO4_USC_ASYNC_TX_STOP_BIT_2,		// 29	USC CMR D14-D15 = 1, USC CCR D8-D11 = 13
		SIO4_USC_ASYNC_TX_STOP_BIT_2,		// 30	USC CMR D14-D15 = 1, USC CCR D8-D11 = 14
		SIO4_USC_ASYNC_TX_STOP_BIT_2,		// 31	USC CMR D14-D15 = 1, USC CCR D8-D11 = 15

		0x102,								// 33	USC CMR D14-D15 = 2, USC CCR D8-D11 = 0	DO NOT USE
		0x112,								// 34	USC CMR D14-D15 = 2, USC CCR D8-D11 = 1	DO NOT USE
		0x122,								// 35	USC CMR D14-D15 = 2, USC CCR D8-D11 = 2	DO NOT USE
		0x132,								// 36	USC CMR D14-D15 = 2, USC CCR D8-D11 = 3	DO NOT USE
		0x142,								// 37	USC CMR D14-D15 = 2, USC CCR D8-D11 = 4	DO NOT USE
		0x152,								// 38	USC CMR D14-D15 = 2, USC CCR D8-D11 = 5	DO NOT USE
		0x162,								// 39	USC CMR D14-D15 = 2, USC CCR D8-D11 = 6	DO NOT USE
		0x172,								// 40	USC CMR D14-D15 = 2, USC CCR D8-D11 = 7	DO NOT USE
		SIO4_USC_ASYNC_TX_STOP_BIT_0__9_16,	// 41	USC CMR D14-D15 = 2, USC CCR D8-D11 = 8
		SIO4_USC_ASYNC_TX_STOP_BIT_0_10_16,	// 42	USC CMR D14-D15 = 2, USC CCR D8-D11 = 9
		SIO4_USC_ASYNC_TX_STOP_BIT_0_11_16,	// 43	USC CMR D14-D15 = 2, USC CCR D8-D11 = 10,
		SIO4_USC_ASYNC_TX_STOP_BIT_0_12_16,	// 44	USC CMR D14-D15 = 2, USC CCR D8-D11 = 11,
		SIO4_USC_ASYNC_TX_STOP_BIT_0_13_16,	// 45	USC CMR D14-D15 = 2, USC CCR D8-D11 = 12,
		SIO4_USC_ASYNC_TX_STOP_BIT_0_14_16,	// 46	USC CMR D14-D15 = 2, USC CCR D8-D11 = 13,
		SIO4_USC_ASYNC_TX_STOP_BIT_0_15_16,	// 47	USC CMR D14-D15 = 2, USC CCR D8-D11 = 14,
		SIO4_USC_ASYNC_TX_STOP_BIT_1,		// 47	USC CMR D14-D15 = 2, USC CCR D8-D11 = 15,

		SIO4_USC_ASYNC_TX_STOP_BIT_1__1_16,	// 48	USC CMR D14-D15 = 3, USC CCR D8-D11 = 0,
		SIO4_USC_ASYNC_TX_STOP_BIT_1__2_16,	// 49	USC CMR D14-D15 = 3, USC CCR D8-D11 = 1,
		SIO4_USC_ASYNC_TX_STOP_BIT_1__3_16,	// 50	USC CMR D14-D15 = 3, USC CCR D8-D11 = 2,
		SIO4_USC_ASYNC_TX_STOP_BIT_1__4_16,	// 51	USC CMR D14-D15 = 3, USC CCR D8-D11 = 3,
		SIO4_USC_ASYNC_TX_STOP_BIT_1__5_16,	// 52	USC CMR D14-D15 = 3, USC CCR D8-D11 = 4,
		SIO4_USC_ASYNC_TX_STOP_BIT_1__6_16,	// 53	USC CMR D14-D15 = 3, USC CCR D8-D11 = 5,
		SIO4_USC_ASYNC_TX_STOP_BIT_1__7_16,	// 54	USC CMR D14-D15 = 3, USC CCR D8-D11 = 6,
		SIO4_USC_ASYNC_TX_STOP_BIT_1__8_16,	// 55	USC CMR D14-D15 = 3, USC CCR D8-D11 = 7,
		SIO4_USC_ASYNC_TX_STOP_BIT_1__9_16,	// 56	USC CMR D14-D15 = 3, USC CCR D8-D11 = 8,
		SIO4_USC_ASYNC_TX_STOP_BIT_1_10_16,	// 57	USC CMR D14-D15 = 3, USC CCR D8-D11 = 9,
		SIO4_USC_ASYNC_TX_STOP_BIT_1_11_16,	// 58	USC CMR D14-D15 = 3, USC CCR D8-D11 = 10,
		SIO4_USC_ASYNC_TX_STOP_BIT_1_12_16,	// 59	USC CMR D14-D15 = 3, USC CCR D8-D11 = 11,
		SIO4_USC_ASYNC_TX_STOP_BIT_1_13_16,	// 60	USC CMR D14-D15 = 3, USC CCR D8-D11 = 12,
		SIO4_USC_ASYNC_TX_STOP_BIT_1_14_16,	// 61	USC CMR D14-D15 = 3, USC CCR D8-D11 = 13,
		SIO4_USC_ASYNC_TX_STOP_BIT_1_15_16,	// 62	USC CMR D14-D15 = 3, USC CCR D8-D11 = 14,
		SIO4_USC_ASYNC_TX_STOP_BIT_2,		// 63	USC CMR D14-D15 = 3, USC CCR D8-D11 = 15,
		-1	// terminate list
	};

	u32	ccr;
	u32	cmr;
	int	i;
	u32	mask;
	int	ret;
	s32	v;

	if (chan->dev->cache.model_z16c30 == 0)
	{
		if (arg[0] == -1)
			ret	= 0;
		else
			ret	= _arg_list_locate(arg, list, &i);
	}
	else if (arg[0] == -1)
	{
		// Asynchronous
		reg_read(chan, SIO4_USC_CCR, &ccr);
		ccr	= GSC_FIELD_DECODE(ccr, 11, 8);

		reg_read(chan, SIO4_USC_CMR, &cmr);
		cmr	= GSC_FIELD_DECODE(cmr, 15, 14);

		v		= (cmr << 4) | ccr;
		arg[0]	= list[v] & 0x3F;
		ret		= 0;
	}
	else
	{
		// Asynchronous

		if (arg[0] > 0xFF)
			ret	= -EINVAL;
		else
			ret	= _arg_list_locate(arg, list, &i);

		if (ret == 0)
		{
			cmr		= GSC_FIELD_ENCODE(arg[0] >> 4,	15, 14);
			mask	= GSC_FIELD_ENCODE(0x3,			15, 14);
			reg_mod(chan, SIO4_USC_CMR, cmr, mask);

			if (arg[0] & 0x20)
			{
				ccr		= GSC_FIELD_ENCODE(arg[0],	11, 8);
				mask	= GSC_FIELD_ENCODE(0xF,		11, 8);
				reg_mod(chan, SIO4_USC_CCR, ccr, mask);
			}
		}
	}

	return(ret);
}



//*****************************************************************************
static int _usc_brg0_clk_src(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_BRG_CLK_SRC_CTR0,
		SIO4_USC_BRG_CLK_SRC_CTR1,
		SIO4_USC_BRG_CLK_SRC_RXC_PIN,
		SIO4_USC_BRG_CLK_SRC_TXC_PIN,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMCR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 9, 8);
	return(ret);
}



//*****************************************************************************
static int _usc_brg0_enable(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_BRG_ENABLE_NO,
		SIO4_USC_BRG_ENABLE_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_HCR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 0, 0);
	return(ret);
}



//*****************************************************************************
static int _usc_brg0_mode(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_BRG_MODE_CONT,
		SIO4_USC_BRG_MODE_SINGLE,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_HCR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 1, 1);
	return(ret);
}



//*****************************************************************************
static int _usc_brg1_clk_src(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_BRG_CLK_SRC_CTR0,
		SIO4_USC_BRG_CLK_SRC_CTR1,
		SIO4_USC_BRG_CLK_SRC_RXC_PIN,
		SIO4_USC_BRG_CLK_SRC_TXC_PIN,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMCR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 11, 10);
	return(ret);
}



//*****************************************************************************
static int _usc_brg1_enable(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_BRG_ENABLE_NO,
		SIO4_USC_BRG_ENABLE_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_HCR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 4, 4);
	return(ret);
}



//*****************************************************************************
static int _usc_brg1_mode(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_BRG_MODE_CONT,
		SIO4_USC_BRG_MODE_SINGLE,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_HCR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 5, 5);
	return(ret);
}



//*****************************************************************************
static int _usc_bsc_rx_short(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_BSC_SHORT_NO,
		SIO4_USC_BSC_SHORT_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMR : 0;
	int	ret;

	// Bisynchronous
	ret	= _arg_reg_list(chan, arg, list, reg, 4, 4);
	return(ret);
}



//*****************************************************************************
static int _usc_bsc_rx_strip(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_BSC_RX_STRIP_NO,
		SIO4_USC_BSC_RX_STRIP_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMR : 0;
	int	ret;

	// Bisynchronous
	ret	= _arg_reg_list(chan, arg, list, reg, 5, 5);
	return(ret);
}



//*****************************************************************************
static int _usc_bsc_rx_syn0(chan_data_t* chan, s32* arg)
{
	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_RSR : 0;
	int	ret;

	// Bisynchronous
	ret	= _arg_reg_range(chan, arg, 0, 0xFF, reg, 7, 0);
	return(ret);
}



//*****************************************************************************
static int _usc_bsc_rx_syn1(chan_data_t* chan, s32* arg)
{
	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_RSR : 0;
	int	ret;

	// Bisynchronous
	ret	= _arg_reg_range(chan, arg, 0, 0xFF, reg, 15, 8);
	return(ret);
}



//*****************************************************************************
static int _usc_bsc_tx_preamble(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_BSC_TX_PREAMBLE_NO,
		SIO4_USC_BSC_TX_PREAMBLE_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMR : 0;
	int	ret;

	// Bisynchronous
	ret	= _arg_reg_list(chan, arg, list, reg, 13, 13);
	return(ret);
}



//*****************************************************************************
static int _usc_bsc_tx_short(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_BSC_SHORT_NO,
		SIO4_USC_BSC_SHORT_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMR : 0;
	int	ret;

	// Bisynchronous
	ret	= _arg_reg_list(chan, arg, list, reg, 12, 12);
	return(ret);
}



//*****************************************************************************
static int _usc_bsc_tx_syn0(chan_data_t* chan, s32* arg)
{
	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_TSR : 0;
	int	ret;

	// Bisynchronous
	ret	= _arg_reg_range(chan, arg, 0, 0xFF, reg, 7, 0);
	return(ret);
}



//*****************************************************************************
static int _usc_bsc_tx_syn1(chan_data_t* chan, s32* arg)
{
	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_TSR : 0;
	int	ret;

	// Bisynchronous
	ret	= _arg_reg_range(chan, arg, 0, 0xFF, reg, 15, 8);
	return(ret);
}



//*****************************************************************************
static int _usc_bsc_tx_underrun(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_BSC_TX_UNDERRUN_S1,
		SIO4_USC_BSC_TX_UNDERRUN_S01,
		SIO4_USC_BSC_TX_UNDERRUN_CRC_S1,
		SIO4_USC_BSC_TX_UNDERRUN_CRC_S01,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMR : 0;
	int	ret;

	// Bisynchronous
	ret	= _arg_reg_list(chan, arg, list, reg, 15, 14);
	return(ret);
}



//*****************************************************************************
static int _usc_ctr0_clk_src(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_CTR_CLK_SRC_DISABLE,
		SIO4_USC_CTR_CLK_SRC_RXC_PIN,
		SIO4_USC_CTR_CLK_SRC_TXC_PIN,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMCR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 13, 12);
	return(ret);
}



//*****************************************************************************
static int _usc_ctr0_rate(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_CTR0_RATE_32X,
		SIO4_USC_CTR0_RATE_16X,
		SIO4_USC_CTR0_RATE_8X,
		SIO4_USC_CTR0_RATE_4X,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_HCR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 15, 14);
	return(ret);
}



//*****************************************************************************
static int _usc_ctr1_clk_src(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_CTR_CLK_SRC_DISABLE,	// 0
		SIO4_USC_CTR_CLK_SRC_DISABLE,	// 1
		SIO4_USC_CTR_CLK_SRC_RXC_PIN,	// 2
		SIO4_USC_CTR_CLK_SRC_TXC_PIN,	// 3
		-1	// terminate list
	};

	int	i;
	int	ret;
	u32	v;

	if (chan->dev->cache.model_z16c30 == 0)
	{
		if (arg[0] == -1)
			ret	= 0;
		else
			ret	= _arg_list_locate(arg, list, &i);
	}
	else if (arg[0] == -1)
	{
		reg_read(chan, SIO4_USC_CMCR, &v);
		v		= GSC_FIELD_DECODE(v, 15, 14);
		arg[0]	= list[v];
		ret		= 0;
	}
	else
	{
		ret	= _arg_reg_list(chan, arg, list, SIO4_USC_CMCR, 15, 14);
	}

	return(ret);
}



//*****************************************************************************
static int _usc_ctr1_rate(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_CTR1_RATE_CTR0,
		SIO4_USC_CTR1_RATE_DPLL,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_HCR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 13, 13);
	return(ret);
}



//*****************************************************************************
static int _usc_cts_cfg(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_CTS_CFG_TRI,			// 0	PSRCR D9-D10 = 0, USC IOCR D14-D15 = 0
		SIO4_USC_CTS_CFG_TRI,			// 1	PSRCR D9-D10 = 0, USC IOCR D14-D15 = 1
		SIO4_USC_CTS_CFG_TRI,			// 2	PSRCR D9-D10 = 0, USC IOCR D14-D15 = 2
		SIO4_USC_CTS_CFG_TRI,			// 3	PSRCR D9-D10 = 0, USC IOCR D14-D15 = 3

		SIO4_USC_CTS_CFG_IN_CBL_CTS,	// 4	PSRCR D9-D10 = 1, USC IOCR D14-D15 = 0
		SIO4_USC_CTS_CFG_IN_CBL_CTS,	// 5	PSRCR D9-D10 = 1, USC IOCR D14-D15 = 1
		SIO4_USC_CTS_CFG_TRI,			// 6	PSRCR D9-D10 = 1, USC IOCR D14-D15 = 2	RESERVED
		SIO4_USC_CTS_CFG_TRI,			// 7	PSRCR D9-D10 = 1, USC IOCR D14-D15 = 3	RESERVED

		SIO4_USC_CTS_CFG_TRI,			// 8	PSRCR D9-D10 = 2, USC IOCR D14-D15 = 0	RESERVED
		SIO4_USC_CTS_CFG_TRI,			// 9	PSRCR D9-D10 = 2, USC IOCR D14-D15 = 1	RESERVED
		SIO4_USC_CTS_CFG_TRI,			// 10	PSRCR D9-D10 = 2, USC IOCR D14-D15 = 2	RESERVED
		SIO4_USC_CTS_CFG_TRI,			// 11	PSRCR D9-D10 = 2, USC IOCR D14-D15 = 3	RESERVED

		SIO4_USC_CTS_CFG_TRI,			// 12	PSRCR D9-D10 = 3, USC IOCR D14-D15 = 0	RESERVED
		SIO4_USC_CTS_CFG_TRI,			// 13	PSRCR D9-D10 = 3, USC IOCR D14-D15 = 1	RESERVED
		SIO4_USC_CTS_CFG_OUT_0,			// 14	PSRCR D9-D10 = 3, USC IOCR D14-D15 = 2
		SIO4_USC_CTS_CFG_OUT_1,			// 15	PSRCR D9-D10 = 3, USC IOCR D14-D15 = 3
		-1	// terminate list
	};

	int	i;
	u32	iocr;
	u32	mask;
	u32	psrcr;
	int	ret;
	u32	v;

	if ((chan->dev->cache.model_z16c30 == 0)		||
		((chan->cache.reg_psrcr_bits & D9) == 0)	||
		((chan->cache.reg_psrcr_bits & D10) == 0))
	{
		if (arg[0] == -1)
			ret	= 0;
		else
			ret	= _arg_list_locate(arg, list, &i);
	}
	else if (arg[0] == -1)
	{
		reg_read(chan, SIO4_GSC_PSRCR, &psrcr);
		psrcr	= GSC_FIELD_DECODE(psrcr, 10, 9);

		reg_read(chan, SIO4_USC_IOCR, &iocr);
		iocr	= GSC_FIELD_DECODE(iocr, 15, 14);

		v		= (psrcr << 2) | iocr;
		arg[0]	= list[v];
		ret		= 0;
	}
	else
	{
		ret	= _arg_list_locate(arg, list, &i);

		if (ret == 0)
		{
			mask	= GSC_FIELD_ENCODE(0x3,			10, 9);
			psrcr	= GSC_FIELD_ENCODE(arg[0] >> 4,	10, 9);
			reg_mod(chan, SIO4_GSC_PSRCR, psrcr, mask);

			mask	= GSC_FIELD_ENCODE(0x3,		15, 14);
			iocr	= GSC_FIELD_ENCODE(arg[0],	15, 14);
			reg_mod(chan, SIO4_USC_IOCR, iocr, mask);
		}
	}

	return(ret);
}



//*****************************************************************************
static int _usc_dcd_cfg(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_DCD_CFG_DISABLE,			// 0	PSRCR D12-D11 = 0, USC IOCR D13-D12 = 0
		SIO4_USC_DCD_CFG_DISABLE,			// 1	PSRCR D12-D11 = 0, USC IOCR D13-D12 = 1
		SIO4_USC_DCD_CFG_DISABLE,			// 2	PSRCR D12-D11 = 0, USC IOCR D13-D12 = 2
		SIO4_USC_DCD_CFG_DISABLE,			// 3	PSRCR D12-D11 = 0, USC IOCR D13-D12 = 3

		SIO4_USC_DCD_CFG_IN_DCD_CBL_DCD,	// 4	PSRCR D12-D11 = 1, USC IOCR D13-D12 = 0
		SIO4_USC_DCD_CFG_IN_SYNC_CBL_DCD,	// 5	PSRCR D12-D11 = 1, USC IOCR D13-D12 = 1
		SIO4_USC_DCD_CFG_DISABLE,			// 6	PSRCR D12-D11 = 1, USC IOCR D13-D12 = 2	RESERVED
		SIO4_USC_DCD_CFG_DISABLE,			// 7	PSRCR D12-D11 = 1, USC IOCR D13-D12 = 3	RESERVED

		SIO4_USC_DCD_CFG_DISABLE,			// 8	PSRCR D12-D11 = 2, USC IOCR D13-D12 = 0	RESERVED
		SIO4_USC_DCD_CFG_DISABLE,			// 9	PSRCR D12-D11 = 2, USC IOCR D13-D12 = 1	RESERVED
		SIO4_USC_DCD_CFG_DISABLE,			// 10	PSRCR D12-D11 = 2, USC IOCR D13-D12 = 2	RESERVED
		SIO4_USC_DCD_CFG_DISABLE,			// 11	PSRCR D12-D11 = 2, USC IOCR D13-D12 = 3	RESERVED

		SIO4_USC_DCD_CFG_DISABLE,			// 12	PSRCR D12-D11 = 3, USC IOCR D13-D12 = 0	RESERVED
		SIO4_USC_DCD_CFG_DISABLE,			// 13	PSRCR D12-D11 = 3, USC IOCR D13-D12 = 1	RESERVED
		SIO4_USC_DCD_CFG_OUT_0,				// 14	PSRCR D12-D11 = 3, USC IOCR D13-D12 = 2
		SIO4_USC_DCD_CFG_OUT_1,				// 15	PSRCR D12-D11 = 3, USC IOCR D13-D12 = 3
		-1	// terminate list
	};

	int	i;
	u32	iocr;
	u32	mask;
	u32	psrcr;
	int	ret;
	u32	v;

	if ((chan->dev->cache.model_z16c30 == 0)		||
		((chan->cache.reg_psrcr_bits & D11) == 0)	||
		((chan->cache.reg_psrcr_bits & D12) == 0))
	{
		if (arg[0] == -1)
			ret	= 0;
		else
			ret	= _arg_list_locate(arg, list, &i);
	}
	else if (arg[0] == -1)
	{
		reg_read(chan, SIO4_GSC_PSRCR, &psrcr);
		psrcr	= GSC_FIELD_DECODE(psrcr, 12, 11);

		reg_read(chan, SIO4_USC_IOCR, &iocr);
		iocr	= GSC_FIELD_DECODE(iocr, 13, 12);

		v		= (psrcr << 2) | iocr;
		arg[0]	= list[v];
		ret		= 0;

	}
	else
	{
		ret	= _arg_list_locate(arg, list, &i);

		if (ret == 0)
		{
			mask	= GSC_FIELD_ENCODE(0x3,			12, 11);
			psrcr	= GSC_FIELD_ENCODE(arg[0] >> 4,	12, 11);
			reg_mod(chan, SIO4_GSC_PSRCR, psrcr, mask);

			mask	= GSC_FIELD_ENCODE(0x3,		13, 12);
			iocr	= GSC_FIELD_ENCODE(arg[0],	13, 12);
			reg_mod(chan, SIO4_USC_IOCR, iocr, mask);
		}
	}

	return(ret);
}



//*****************************************************************************
static int _usc_dpll_adj_sync(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_DPLL_ADJ_SYNC_BOTH_EDGE,
		SIO4_USC_DPLL_ADJ_SYNC_RISE_EDGE,
		SIO4_USC_DPLL_ADJ_SYNC_FALL_EDGE,
		SIO4_USC_DPLL_ADJ_SYNC_INHIBIT,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CCSR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 9, 8);
	return(ret);
}



//*****************************************************************************
static int _usc_dpll_clk_src(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_DPLL_CLK_SRC_BRG0,
		SIO4_USC_DPLL_CLK_SRC_BRG1,
		SIO4_USC_DPLL_CLK_SRC_RXC_PIN,
		SIO4_USC_DPLL_CLK_SRC_TXC_PIN,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMCR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 7, 6);
	return(ret);
}



//*****************************************************************************
static int _usc_dpll_miss_1(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_DPLL_MISS_1_NO,
		SIO4_USC_DPLL_MISS_1_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CCSR : 0;
	int	ret;

	if ((arg[0] == -1) || (arg[0] == SIO4_USC_DPLL_MISS_1_CLEAR))
		ret	= _arg_reg_list(chan, arg, list, reg, 10, 10);
	else
		ret	= -EINVAL;

	return(ret);
}



//*****************************************************************************
static int _usc_dpll_miss_2(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_DPLL_MISS_2_NO,
		SIO4_USC_DPLL_MISS_2_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CCSR : 0;
	int	ret;

	if ((arg[0] == -1) || (arg[0] == SIO4_USC_DPLL_MISS_2_CLEAR))
		ret	= _arg_reg_list(chan, arg, list, reg, 11, 11);
	else
		ret	= -EINVAL;

	return(ret);
}



//*****************************************************************************
static int _usc_dpll_mode(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_DPLL_MODE_DISABLE,
		SIO4_USC_DPLL_MODE_NRZ_NRZI,
		SIO4_USC_DPLL_MODE_BIPH_MS,
		SIO4_USC_DPLL_MODE_BIPH_LVL,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_HCR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 9, 8);
	return(ret);
}



//*****************************************************************************
static int _usc_dpll_rate(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_DPLL_RATE_32X,
		SIO4_USC_DPLL_RATE_16X,
		SIO4_USC_DPLL_RATE_8X,
		SIO4_USC_DPLL_RATE_CTR1_4X,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_HCR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 11, 10);
	return(ret);
}



//*****************************************************************************
static int _usc_dpll_sync(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_DPLL_SYNC_NO,
		SIO4_USC_DPLL_SYNC_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CCSR : 0;
	int	ret;

	if ((arg[0] == -1) || (arg[0] == SIO4_USC_DPLL_SYNC_START))
		ret	= _arg_reg_list(chan, arg, list, reg, 12, 12);
	else
		ret	= -EINVAL;

	return(ret);
}



//*****************************************************************************
static int _usc_hdlc_rx_adrs_ctrl(chan_data_t* chan, s32* arg)
{
	static const  s32	list[]	=
	{
		SIO4_USC_HDLC_RX_ADRS_CTRL_OFF,		// 0	// The upper nibble is the mask.
		SIO4_USC_HDLC_RX_ADRS_CTRL_16,		// 1	// The lower nibble is the value.
		SIO4_USC_HDLC_RX_ADRS_CTRL_24,		// 2
		SIO4_USC_HDLC_RX_ADRS_CTRL_EA_16,	// 3
		SIO4_USC_HDLC_RX_ADRS_CTRL_OFF,		// 4
		SIO4_USC_HDLC_RX_ADRS_CTRL_16,		// 5
		SIO4_USC_HDLC_RX_ADRS_CTRL_32,		// 6
		SIO4_USC_HDLC_RX_ADRS_CTRL_EA_24,	// 7
		SIO4_USC_HDLC_RX_ADRS_CTRL_OFF,		// 8
		SIO4_USC_HDLC_RX_ADRS_CTRL_16,		// 9
		SIO4_USC_HDLC_RX_ADRS_CTRL_24,		// 10
		SIO4_USC_HDLC_RX_ADRS_CTRL_EAC8,	// 11
		SIO4_USC_HDLC_RX_ADRS_CTRL_OFF,		// 12
		SIO4_USC_HDLC_RX_ADRS_CTRL_16,		// 13
		SIO4_USC_HDLC_RX_ADRS_CTRL_32,		// 14
		SIO4_USC_HDLC_RX_ADRS_CTRL_EAC16,	// 15
		-1	// terminate list
	};

	int	i;
	int	ret;
	u32	v;

	if (chan->dev->cache.model_z16c30 == 0)
	{
		if (arg[0] == -1)
			ret	= 0;
		else
			ret	= _arg_list_locate(arg, list, &i);
	}
	else if (arg[0] == -1)
	{
		// HDLC
		reg_read(chan, SIO4_USC_CMR, &v);
		v		= GSC_FIELD_DECODE(v, 7, 4);
		arg[0]	= list[v];
		ret		= 0;
	}
	else
	{
		// HDLC
		ret	= _arg_list_locate(arg, list, &i);

		if (ret == 0)
			reg_mod(chan, SIO4_USC_CMR, arg[0] << 4, arg[0] & 0xF0);
	}

	return(ret);
}



//*****************************************************************************
static int _usc_hdlc_tx_l_chr_len(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_HDLC_TX_L_CHR_LEN_1,
		SIO4_USC_HDLC_TX_L_CHR_LEN_2,
		SIO4_USC_HDLC_TX_L_CHR_LEN_3,
		SIO4_USC_HDLC_TX_L_CHR_LEN_4,
		SIO4_USC_HDLC_TX_L_CHR_LEN_5,
		SIO4_USC_HDLC_TX_L_CHR_LEN_6,
		SIO4_USC_HDLC_TX_L_CHR_LEN_7,
		SIO4_USC_HDLC_TX_L_CHR_LEN_8,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CCSR : 0;
	int	ret;

	// HDLC
	ret	= _arg_reg_list(chan, arg, list, reg, 4, 2);
	return(ret);
}



//*****************************************************************************
static int _usc_hdlc_tx_preamble(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_HDLC_TX_PREAMBLE_NO,
		SIO4_USC_HDLC_TX_PREAMBLE_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMR : 0;
	int	ret;

	// HDLC
	ret	= _arg_reg_list(chan, arg, list, reg, 13, 13);
	return(ret);
}



//*****************************************************************************
static int _usc_hdlc_tx_share_0(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_HDLC_TX_SHARE_0_NO,
		SIO4_USC_HDLC_TX_SHARE_0_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMR : 0;
	int	ret;

	// HDLC
	ret	= _arg_reg_list(chan, arg, list, reg, 12, 12);
	return(ret);
}



//*****************************************************************************
static int _usc_hdlc_tx_underrun(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_HDLC_TX_UNDERRUN_ABORT,
		SIO4_USC_HDLC_TX_UNDERRUN_EXT_A,
		SIO4_USC_HDLC_TX_UNDERRUN_FLAG,
		SIO4_USC_HDLC_TX_UNDERRUN_CRC_F,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMR : 0;
	int	ret;

	// HDLC
	ret	= _arg_reg_list(chan, arg, list, reg, 15, 14);
	return(ret);
}



//*****************************************************************************
static int _usc_hdlcl_tx_active(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_HDLCL_TX_ACTIVE_NONE,
		SIO4_USC_HDLCL_TX_ACTIVE_POLL,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMR : 0;
	int	ret;

	// HDLC Loop
	ret	= _arg_reg_list(chan, arg, list, reg, 13, 13);
	return(ret);
}



//*****************************************************************************
static int _usc_hdlcl_tx_share_0(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_HDLCL_TX_SHARE_0_NO,
		SIO4_USC_HDLCL_TX_SHARE_0_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMR : 0;
	int	ret;

	// HDLC Loop
	ret	= _arg_reg_list(chan, arg, list, reg, 12, 12);
	return(ret);
}



//*****************************************************************************
static int _usc_hdlcl_tx_underrun(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_HDLCL_TX_UNDERRUN_ABORT,
		SIO4_USC_HDLCL_TX_UNDERRUN_E_ABT,
		SIO4_USC_HDLCL_TX_UNDERRUN_FLAG,
		SIO4_USC_HDLCL_TX_UNDERRUN_CRC_F,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMR : 0;
	int	ret;

	// HDLC Loop
	ret	= _arg_reg_list(chan, arg, list, reg, 15, 14);
	return(ret);
}



//*****************************************************************************
static int _usc_isoc_tx_stop_bit(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_ISOC_TX_STOP_BIT_1,
		SIO4_USC_ISOC_TX_STOP_BIT_2,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMR : 0;
	int	ret;

	// Isochronous
	ret	= _arg_reg_list(chan, arg, list, reg, 14, 14);
	return(ret);
}



//*****************************************************************************
static int _usc_loop_sending(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_LOOP_SENDING_NO,
		SIO4_USC_LOOP_SENDING_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CCSR : 0;
	int	ret;

	// Isochronous
	ret	= _arg_reg_list(chan, arg, list, reg, 6, 6);
	return(ret);
}



//*****************************************************************************
static int _usc_mono_rx_short(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_MONO_SHORT_NO,
		SIO4_USC_MONO_SHORT_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMR : 0;
	int	ret;

	// Monosync
	ret	= _arg_reg_list(chan, arg, list, reg, 4, 4);
	return(ret);
}



//*****************************************************************************
static int _usc_mono_rx_strip(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_MONO_RX_STRIP_NO,
		SIO4_USC_MONO_RX_STRIP_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMR : 0;
	int	ret;

	// Monosync
	ret	= _arg_reg_list(chan, arg, list, reg, 5, 5);
	return(ret);
}



//*****************************************************************************
static int _usc_mono_rx_sync(chan_data_t* chan, s32* arg)
{
	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_RSR : 0;
	int	ret;

	// Monosync
	ret	= _arg_reg_range(chan, arg, 0, 0xFF, reg, 15, 8);
	return(ret);
}



//*****************************************************************************
static int _usc_mono_tx_crc_under(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_MONO_TX_CRC_UNDER_NO,
		SIO4_USC_MONO_TX_CRC_UNDER_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMR : 0;
	int	ret;

	// Monosync
	ret	= _arg_reg_list(chan, arg, list, reg, 15, 15);
	return(ret);
}



//*****************************************************************************
static int _usc_mono_tx_preamble(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_MONO_TX_PREAMBLE_NO,
		SIO4_USC_MONO_TX_PREAMBLE_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMR : 0;
	int	ret;

	// Monosync
	ret	= _arg_reg_list(chan, arg, list, reg, 13, 13);
	return(ret);
}



//*****************************************************************************
static int _usc_mono_tx_short(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_MONO_SHORT_NO,
		SIO4_USC_MONO_SHORT_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMR : 0;
	int	ret;

	// Monosync
	ret	= _arg_reg_list(chan, arg, list, reg, 12, 12);
	return(ret);
}



//*****************************************************************************
static int _usc_mono_tx_sync(chan_data_t* chan, s32* arg)
{
	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_TSR : 0;
	int	ret;

	// Monosync
	ret	= _arg_reg_range(chan, arg, 0, 0xFF, reg, 7, 0);
	return(ret);
}



//*****************************************************************************
static int _usc_nbip_rx_clk_rate(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_NBIP_CLK_RATE_16X,
		SIO4_USC_NBIP_CLK_RATE_32X,
		SIO4_USC_NBIP_CLK_RATE_64X,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMR : 0;
	int	ret;

	// Nine-Bit Mode
	ret	= _arg_reg_list(chan, arg, list, reg, 5, 4);
	return(ret);
}



//*****************************************************************************
static int _usc_nbip_rx_parity(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_NBIP_PARITY_NO,
		SIO4_USC_NBIP_PARITY_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMR : 0;
	int	ret;

	// Nine-Bit Mode
	ret	= _arg_reg_list(chan, arg, list, reg, 6, 6);
	return(ret);
}



//*****************************************************************************
static int _usc_nbip_tx_adrs_bit(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_NBIP_TX_ADRS_BIT_NO,
		SIO4_USC_NBIP_TX_ADRS_BIT_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMR : 0;
	int	ret;

	// Nine-Bit Mode
	ret	= _arg_reg_list(chan, arg, list, reg, 15, 15);
	return(ret);
}



//*****************************************************************************
static int _usc_nbip_tx_clk_rate(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_NBIP_CLK_RATE_16X,
		SIO4_USC_NBIP_CLK_RATE_32X,
		SIO4_USC_NBIP_CLK_RATE_64X,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMR : 0;
	int	ret;

	// Nine-Bit Mode
	ret	= _arg_reg_list(chan, arg, list, reg, 13, 12);
	return(ret);
}



//*****************************************************************************
static int _usc_nbip_tx_parity(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_NBIP_PARITY_NO,
		SIO4_USC_NBIP_PARITY_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMR : 0;
	int	ret;

	// Nine-Bit Mode
	ret	= _arg_reg_list(chan, arg, list, reg, 14, 14);
	return(ret);
}



//*****************************************************************************
static int _usc_on_loop(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_ON_LOOP_NO,
		SIO4_USC_ON_LOOP_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CCSR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 7, 7);
	return(ret);
}



//*****************************************************************************
static int _usc_oper_mode(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_OPER_MODE_NORMAL,
		SIO4_USC_OPER_MODE_AUTO_ECHO,
		SIO4_USC_OPER_MODE_EXT_LOOPBACK,
		SIO4_USC_OPER_MODE_INT_LOOPBACK,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CCAR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 9, 8);
	return(ret);
}



//*****************************************************************************
static int _usc_rcc_fifo_clear(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_RCC_FIFO_CLEAR_NO,
		SIO4_USC_RCC_FIFO_CLEAR_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CCSR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 13, 13);
	return(ret);
}



//*****************************************************************************
static int _usc_rcc_fifo_overrun(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_RCC_FIFO_OVERRUN_NO,
		SIO4_USC_RCC_FIFO_OVERRUN_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CCSR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 15, 15);
	return(ret);
}



//*****************************************************************************
static int _usc_rcc_fifo_valid(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_RCC_FIFO_VALID_NO,
		SIO4_USC_RCC_FIFO_VALID_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CCSR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 14, 14);
	return(ret);
}



//*****************************************************************************
static int _usc_reset(chan_data_t* chan, s32* arg)
{
	int	ret	= 0;

	if (arg[0] == -1)
	{
		arg[0]	= chan->dev->cache.model_z16c30 ? 0 : -1;
	}
	else if (arg[0] != 1)
	{
		ret	= -EINVAL;
	}
	else if (chan->dev->cache.model_z16c30)
	{
		os_reg_mem_tx_u8(chan->dev, chan->vaddr.usc_ccar_ub, D2);
		os_reg_mem_tx_u8(chan->dev, chan->vaddr.usc_ccar_lb, 0);
		os_reg_mem_tx_u8(chan->dev, chan->vaddr.usc_ccar_ub, 0);
	}
	else
	{
		arg[0]	= -1;
	}

	return(ret);
}



//*****************************************************************************
static int _usc_rx_char_len(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_CHAR_LEN_1,
		SIO4_USC_CHAR_LEN_2,
		SIO4_USC_CHAR_LEN_3,
		SIO4_USC_CHAR_LEN_4,
		SIO4_USC_CHAR_LEN_5,
		SIO4_USC_CHAR_LEN_6,
		SIO4_USC_CHAR_LEN_7,
		SIO4_USC_CHAR_LEN_8,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_RMR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 4, 2);
	return(ret);
}



//*****************************************************************************
static int _usc_rx_char_cnt(chan_data_t* chan, s32* arg)
{
	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_RCCR : 0;
	int	ret	= 0;

	if (arg[0] == -1)
		ret	= _arg_reg_range(chan, arg, 0, 0xFFFF, reg, 15, 0);
	else
		ret	= -EINVAL;

	return(ret);
}



//*****************************************************************************
static int _usc_rx_char_cnt_lim(chan_data_t* chan, s32* arg)
{
	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_RCLR : 0;
	int	ret;

	ret	= _arg_reg_range(chan, arg, 0, 0xFFFF, reg, 15, 0);
	return(ret);
}



//*****************************************************************************
static int _usc_rx_clk_src(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_CLK_SRC_DISABLE,
		SIO4_USC_CLK_SRC_RXC_PIN,
		SIO4_USC_CLK_SRC_TXC_PIN,
		SIO4_USC_CLK_SRC_DPLL,
		SIO4_USC_CLK_SRC_BRG0,
		SIO4_USC_CLK_SRC_BRG1,
		SIO4_USC_CLK_SRC_CTR0,
		SIO4_USC_CLK_SRC_CTR1,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMCR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 2, 0);
	return(ret);
}



//*****************************************************************************
static int _usc_rx_cmd(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_RX_CMD_NULL,
		SIO4_USC_RX_CMD_PRESET_CRC,
		SIO4_USC_RX_CMD_ENTER_HUNT_MODE,
		SIO4_USC_RX_CMD_SEL_FIFO_STATUS,
		SIO4_USC_RX_CMD_SEL_FIFO_INT_LVL,
		SIO4_USC_RX_CMD_SEL_FIFO_STS_LVL,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_RCSR : 0;
	int	ret;

	if (arg[0] == -1)
	{
		arg[0]	= chan->dev->cache.model_z16c30 ? 1 : -1;
		ret		= 0;
	}
	else
	{
		ret		= _arg_reg_list(chan, arg, list, reg, 15, 12);
	}

	return(ret);
}



//*****************************************************************************
static int _usc_rx_crc_enable(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_CRC_ENABLE_NO,
		SIO4_USC_CRC_ENABLE_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_RMR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 9, 9);
	return(ret);
}



//*****************************************************************************
static int _usc_rx_crc_preset(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_CRC_PRESET_ALL_0,
		SIO4_USC_CRC_PRESET_ALL_1,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_RMR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 10, 10);
	return(ret);
}



//*****************************************************************************
static int _usc_rx_crc_type(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_CRC_TYPE_CCITT,
		SIO4_USC_CRC_TYPE_16,
		SIO4_USC_CRC_TYPE_32,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_RMR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 12, 11);
	return(ret);
}



//*****************************************************************************
static int _usc_rx_data_encode(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_DATA_ENCODE_NRZ,
		SIO4_USC_DATA_ENCODE_NRZB,
		SIO4_USC_DATA_ENCODE_NRZI_MARK,
		SIO4_USC_DATA_ENCODE_NRZI_SPACE,
		SIO4_USC_DATA_ENCODE_BI_MARK,
		SIO4_USC_DATA_ENCODE_BI_SPACE,
		SIO4_USC_DATA_ENCODE_BI_LEVEL,
		SIO4_USC_DATA_ENCODE_D_BI_LEVEL,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_RMR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 15, 13);
	return(ret);
}



//*****************************************************************************
static int _usc_rx_enable(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_ENABLE_NO_NOW,
		SIO4_USC_ENABLE_NO_AFTER,
		SIO4_USC_ENABLE_YES_NOW,
		SIO4_USC_ENABLE_YES_W_AE,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_RMR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 1, 0);
	return(ret);
}



//*****************************************************************************
static int _usc_rx_mode(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_MODE_ASYNC,	// Asynchronous
		SIO4_USC_MODE_E_SYNC,	// External Sync
		SIO4_USC_MODE_ISOC,		// Isochronous
		SIO4_USC_MODE_ASY_CV,	// Async w/ Code Violations
		SIO4_USC_MODE_MONO,		// Monosync
		SIO4_USC_MODE_BSC,		// Bisync Serial Communications
		SIO4_USC_MODE_HDLC,		// HDLC
		SIO4_USC_MODE_TBSC,		// Transparent Bisync
		SIO4_USC_MODE_NBIP,		// 9-Bit Intercomm Protocol
		SIO4_USC_MODE_8023,		// 802.3
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 3, 0);
	return(ret);
}



//*****************************************************************************
static int _usc_rx_par_enable(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_PAR_ENABLE_NO,
		SIO4_USC_PAR_ENABLE_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_RMR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 5, 5);
	return(ret);
}



//*****************************************************************************
static int _usc_rx_par_type(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_PAR_TYPE_EVEN,
		SIO4_USC_PAR_TYPE_ODD,
		SIO4_USC_PAR_TYPE_ZERO,
		SIO4_USC_PAR_TYPE_ONE,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_RMR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 7, 6);
	return(ret);
}



//*****************************************************************************
static int _usc_rx_queue_abort(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_RX_QUEUE_ABORT_NO,
		SIO4_USC_RX_QUEUE_ABORT_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_RMR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 8, 8);
	return(ret);
}



//*****************************************************************************
static int _usc_rx_status(chan_data_t* chan, s32* arg)
{
	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_RCSR : 0;
	int	ret	= 0;

	ret	= _arg_reg_range(chan, arg, 0, 0xFFFF, reg, 15, 0);

	if ((ret == 0) && (reg != 0))
		arg[0]	&= 0xCFFF;

	return(ret);
}



//*****************************************************************************
static int _usc_rx_status_block(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_STATUS_BLOCK_NO,
		SIO4_USC_STATUS_BLOCK_1_WORD,
		SIO4_USC_STATUS_BLOCK_2_WORD,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CCR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 7, 6);
	return(ret);
}



//*****************************************************************************
static int _usc_rx_wait_dma_trig(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_WAIT_DMA_TRIG_NO,
		SIO4_USC_WAIT_DMA_TRIG_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CCR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 5, 5);
	return(ret);
}



//*****************************************************************************
static int _usc_rxc_cfg(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_RXC_CFG_IN_OSC,		// lower nibble: PSRCR D3-D5
		SIO4_USC_RXC_CFG_IN_OSC_INV,	// upper nibble: USC IOCR D0-D2
		SIO4_USC_RXC_CFG_IN_0,
		SIO4_USC_RXC_CFG_IN_1,
		SIO4_USC_RXC_CFG_IN_CBL_RXC,
		SIO4_USC_RXC_CFG_IN_CBL_RXAUXC,
		SIO4_USC_RXC_CFG_OUT_RX_CLK,
		SIO4_USC_RXC_CFG_OUT_RX_BYTE_CLK,
		SIO4_USC_RXC_CFG_OUT_SYNC,
		SIO4_USC_RXC_CFG_OUT_BRG0,
		SIO4_USC_RXC_CFG_OUT_BRG1,
		SIO4_USC_RXC_CFG_OUT_CTR0,
		SIO4_USC_RXC_CFG_OUT_DPLL_RX,
		-1	// terminate list
	};

	int	i;
	u32	iocr;
	u32	mask;
	u32	psrcr;
	int	ret;

	if ((chan->dev->cache.model_z16c30)		&&
		(chan->cache.reg_psrcr_bits & D3)	&&
		(chan->cache.reg_psrcr_bits & D4)	&&
		(chan->cache.reg_psrcr_bits & D5))
	{

		if (arg[0] == -1)
		{
			reg_read(chan, SIO4_GSC_PSRCR, &psrcr);
			psrcr	= GSC_FIELD_DECODE(psrcr, 5, 3);

			reg_read(chan, SIO4_USC_IOCR, &iocr);
			iocr	= GSC_FIELD_DECODE(iocr, 2, 0);

			arg[0]	= (iocr << 4) | psrcr;
			ret		= _arg_list_locate(arg, list, &i);
		}
		else
		{
			ret	= _arg_list_locate(arg, list, &i);

			if (ret == 0)
			{
				mask	= GSC_FIELD_ENCODE(0x7,		5, 3);
				psrcr	= GSC_FIELD_ENCODE(arg[0],	5, 3);
				reg_mod(chan, SIO4_GSC_PSRCR, psrcr, mask);

				mask	= GSC_FIELD_ENCODE(0x7,			2, 0);
				iocr	= GSC_FIELD_ENCODE(arg[0] >> 4,	2, 0);
				reg_mod(chan, SIO4_USC_IOCR, iocr, mask);
			}
		}
	}
	else
	{
		if (arg[0] == -1)
			ret	= 0;
		else
			ret	= _arg_list_locate(arg, list, &i);
	}

	return(ret);
}



//*****************************************************************************
static int _usc_send_command(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_SEND_CMD_NONE,
		SIO4_USC_SEND_CMD_RESET_H_IUS,
		SIO4_USC_SEND_CMD_TRIG_LD_DMA,
		SIO4_USC_SEND_CMD_TRIG_RX_DMA,
		SIO4_USC_SEND_CMD_TRIG_TX_DMA,
		SIO4_USC_SEND_CMD_TRIG_TX_RX_DMA,
		SIO4_USC_SEND_CMD_RX_FIFO_PURGE,
		SIO4_USC_SEND_CMD_TX_FIFO_PURGE,
		SIO4_USC_SEND_CMD_FIFOS_PURGE,
		SIO4_USC_SEND_CMD_LD_RX_CHAR_CNT,
		SIO4_USC_SEND_CMD_LD_TX_CHAR_CNT,
		SIO4_USC_SEND_CMD_LD_CHAR_CNTS,
		SIO4_USC_SEND_CMD_LD_TC0,
		SIO4_USC_SEND_CMD_LD_TC1,
		SIO4_USC_SEND_CMD_LD_TC0_TC1,
		SIO4_USC_SEND_CMD_SEL_LSB_FIRST,
		SIO4_USC_SEND_CMD_SEL_MSB_FIRST,
		SIO4_USC_SEND_CMD_SEL_STRAIT_MEM,
		SIO4_USC_SEND_CMD_SEL_SWAP_MEM,
		SIO4_USC_SEND_CMD_RX_PURGE,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CCAR : 0;
	int	ret;

	if (arg[0] == -1)
	{
		arg[0]	= chan->dev->cache.model_z16c30 ? 1 : -1;
		ret		= 0;
	}
	else
	{
		ret	= _arg_reg_list(chan, arg, list, reg, 15, 11);
	}

	return(ret);
}



//*****************************************************************************
static int _usc_smono_rx_sync(chan_data_t* chan, s32* arg)
{
	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_RSR : 0;
	int	ret;

	// Slaved Monosync
	ret	= _arg_reg_range(chan, arg, 0, 0xFF, reg, 15, 8);
	return(ret);
}



//*****************************************************************************
static int _usc_smono_tx_active(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_SMONO_TX_ACTIVE_WAIT,
		SIO4_USC_SMONO_TX_ACTIVE_ON_SYNC,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMR : 0;
	int	ret;

	// Slaved Monosync
	ret	= _arg_reg_list(chan, arg, list, reg, 13, 13);
	return(ret);
}



//*****************************************************************************
static int _usc_smono_tx_short(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_SMONO_TX_SHORT_NO,
		SIO4_USC_SMONO_TX_SHORT_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMR : 0;
	int	ret;

	// Slaved Monosync
	ret	= _arg_reg_list(chan, arg, list, reg, 12, 12);
	return(ret);
}



//*****************************************************************************
static int _usc_smono_tx_sync(chan_data_t* chan, s32* arg)
{
	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_TSR : 0;
	int	ret;

	// Slaved Monosync
	ret	= _arg_reg_range(chan, arg, 0, 0xFF, reg, 7, 0);
	return(ret);
}



//*****************************************************************************
static int _usc_smono_tx_underrun(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_SMONO_TX_UNDERRUN_NONE,
		SIO4_USC_SMONO_TX_UNDERRUN_CRC,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMR : 0;
	int	ret;

	// Slaved Monosync
	ret	= _arg_reg_list(chan, arg, list, reg, 15, 15);
	return(ret);
}



//*****************************************************************************
static int _usc_tbsc_rx_encoding(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_TBSC_ENCODING_ASCII,
		SIO4_USC_TBSC_ENCODING_EBCDIC,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMR : 0;
	int	ret;

	// Transparent Bisync
	ret	= _arg_reg_list(chan, arg, list, reg, 4, 4);
	return(ret);
}



//*****************************************************************************
static int _usc_tbsc_tx_encoding(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_TBSC_ENCODING_ASCII,
		SIO4_USC_TBSC_ENCODING_EBCDIC,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMR : 0;
	int	ret;

	// Transparent Bisync
	ret	= _arg_reg_list(chan, arg, list, reg, 12, 12);
	return(ret);
}



//*****************************************************************************
static int _usc_tbsc_tx_preamble(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_TBSC_TX_PREAMBLE_NO,
		SIO4_USC_TBSC_TX_PREAMBLE_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMR : 0;
	int	ret;

	// Transparent Bisync
	ret	= _arg_reg_list(chan, arg, list, reg, 13, 13);
	return(ret);
}



//*****************************************************************************
static int _usc_tbsc_tx_underrun(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_TBSC_TX_UNDERRUN_S,
		SIO4_USC_TBSC_TX_UNDERRUN_D_S,
		SIO4_USC_TBSC_TX_UNDERRUN_C_S,
		SIO4_USC_TBSC_TX_UNDERRUN_C_D_S,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMR : 0;
	int	ret;

	// Transparent Bisync
	ret	= _arg_reg_list(chan, arg, list, reg, 15, 14);
	return(ret);
}



//*****************************************************************************
static int _usc_tc0(chan_data_t* chan, s32* arg)
{
	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_TC0R : 0;
	int	ret;

	ret	= _arg_reg_range(chan, arg, 0, 0xFFFF, reg, 15, 0);
	return(ret);
}



//*****************************************************************************
static int _usc_tc1(chan_data_t* chan, s32* arg)
{
	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_TC1R : 0;
	int	ret;

	ret	= _arg_reg_range(chan, arg, 0, 0xFFFF, reg, 15, 0);
	return(ret);
}



//*****************************************************************************
static int _usc_tx_char_cnt(chan_data_t* chan, s32* arg)
{
	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_TCCR : 0;
	int	ret;

	ret	= _arg_reg_range(chan, arg, 0, 0xFFFF, reg, 15, 0);
	return(ret);
}



//*****************************************************************************
static int _usc_tx_char_len(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_CHAR_LEN_1,
		SIO4_USC_CHAR_LEN_2,
		SIO4_USC_CHAR_LEN_3,
		SIO4_USC_CHAR_LEN_4,
		SIO4_USC_CHAR_LEN_5,
		SIO4_USC_CHAR_LEN_6,
		SIO4_USC_CHAR_LEN_7,
		SIO4_USC_CHAR_LEN_8,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_TMR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 4, 2);
	return(ret);
}



//*****************************************************************************
static int _usc_tx_char_cnt_lim(chan_data_t* chan, s32* arg)
{
	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_TCLR : 0;
	int	ret;

	ret	= _arg_reg_range(chan, arg, 0, 0xFFFF, reg, 15, 0);
	return(ret);
}



//*****************************************************************************
static int _usc_tx_clk_src(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_CLK_SRC_DISABLE,
		SIO4_USC_CLK_SRC_RXC_PIN,
		SIO4_USC_CLK_SRC_TXC_PIN,
		SIO4_USC_CLK_SRC_DPLL,
		SIO4_USC_CLK_SRC_BRG0,
		SIO4_USC_CLK_SRC_BRG1,
		SIO4_USC_CLK_SRC_CTR0,
		SIO4_USC_CLK_SRC_CTR1,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMCR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 5, 3);
	return(ret);
}



//*****************************************************************************
static int _usc_tx_cmd(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_TX_CMD_NULL,
		SIO4_USC_TX_CMD_PRESET_CRC,
		SIO4_USC_TX_CMD_ENTER_HUNT_MODE,
		SIO4_USC_TX_CMD_SEL_FIFO_STATUS,
		SIO4_USC_TX_CMD_SEL_FIFO_INT_LVL,
		SIO4_USC_TX_CMD_SEL_FIFO_STS_LVL,
		SIO4_USC_TX_CMD_SEND_FRM_MSG,
		SIO4_USC_TX_CMD_SEND_ABORT,
		SIO4_USC_TX_CMD_RST_DLE_INHIBIT,
		SIO4_USC_TX_CMD_SET_DLE_INHIBIT,
		SIO4_USC_TX_CMD_RST_EOF_EOM,
		SIO4_USC_TX_CMD_SET_EOF_EOM,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_TCSR : 0;
	int	ret;

	if (arg[0] == -1)
	{
		arg[0]	= chan->dev->cache.model_z16c30 ? 1 : -1;
		ret		= 0;
	}
	else
	{
		ret		= _arg_reg_list(chan, arg, list, reg, 15, 12);
	}

	return(ret);
}



//*****************************************************************************
static int _usc_tx_crc_enable(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_CRC_ENABLE_NO,
		SIO4_USC_CRC_ENABLE_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_TMR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 9, 9);
	return(ret);
}



//*****************************************************************************
static int _usc_tx_crc_on_end(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_TX_CRC_ON_END_NO,
		SIO4_USC_TX_CRC_ON_END_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_TMR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 8, 8);
	return(ret);
}



//*****************************************************************************
static int _usc_tx_crc_preset(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_CRC_PRESET_ALL_0,
		SIO4_USC_CRC_PRESET_ALL_1,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_TMR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 10, 10);
	return(ret);
}



//*****************************************************************************
static int _usc_tx_crc_type(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_CRC_TYPE_CCITT,
		SIO4_USC_CRC_TYPE_16,
		SIO4_USC_CRC_TYPE_32,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_TMR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 12, 11);
	return(ret);
}



//*****************************************************************************
static int _usc_tx_data_encode(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_DATA_ENCODE_NRZ,
		SIO4_USC_DATA_ENCODE_NRZB,
		SIO4_USC_DATA_ENCODE_NRZI_MARK,
		SIO4_USC_DATA_ENCODE_NRZI_SPACE,
		SIO4_USC_DATA_ENCODE_BI_MARK,
		SIO4_USC_DATA_ENCODE_BI_SPACE,
		SIO4_USC_DATA_ENCODE_BI_LEVEL,
		SIO4_USC_DATA_ENCODE_D_BI_LEVEL,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_TMR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 15, 13);
	return(ret);
}



//*****************************************************************************
static int _usc_tx_enable(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_ENABLE_NO_NOW,
		SIO4_USC_ENABLE_NO_AFTER,
		SIO4_USC_ENABLE_YES_NOW,
		SIO4_USC_ENABLE_YES_W_AE,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_TMR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 1, 0);
	return(ret);
}



//*****************************************************************************
static int _usc_tx_idle_cond(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_TX_IDLE_COND_DEFAULT,
		SIO4_USC_TX_IDLE_COND_0_1,
		SIO4_USC_TX_IDLE_COND_0,
		SIO4_USC_TX_IDLE_COND_1,
		SIO4_USC_TX_IDLE_COND_MARK_SPACE,
		SIO4_USC_TX_IDLE_COND_SPACE,
		SIO4_USC_TX_IDLE_COND_MARK,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_TCSR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 10, 8);
	return(ret);
}



//*****************************************************************************
static int _usc_tx_mode(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_MODE_ASYNC,	// Asynchronous
		SIO4_USC_MODE_ISOC,		// Isochronous
		SIO4_USC_MODE_ASY_CV,	// Async w/ Code Violations
		SIO4_USC_MODE_MONO,		// Monosync
		SIO4_USC_MODE_BSC,		// Bisync Serial Communications
		SIO4_USC_MODE_HDLC,		// HDLC
		SIO4_USC_MODE_TBSC,		// Trabsparent Bisync
		SIO4_USC_MODE_NBIP,		// 9-Bit Intercomm Protocol
		SIO4_USC_MODE_8023,		// 802.3
		SIO4_USC_MODE_S_MONO,	// Slaved Monosync
		SIO4_USC_MODE_HDLC_L,	// HDLC Loop
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CMR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 11, 8);
	return(ret);
}



//*****************************************************************************
static int _usc_tx_par_enable(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_PAR_ENABLE_NO,
		SIO4_USC_PAR_ENABLE_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_TMR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 5, 5);
	return(ret);
}



//*****************************************************************************
static int _usc_tx_par_type(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_PAR_TYPE_EVEN,
		SIO4_USC_PAR_TYPE_ODD,
		SIO4_USC_PAR_TYPE_ZERO,
		SIO4_USC_PAR_TYPE_ONE,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_TMR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 7, 6);
	return(ret);
}



//*****************************************************************************
static int _usc_tx_preamble_flag(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_TX_PREAMBLE_FLAG_NO,
		SIO4_USC_TX_PREAMBLE_FLAG_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CCR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 12, 12);
	return(ret);
}



//*****************************************************************************
static int _usc_tx_preamble_len(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_TX_PREAMBLE_LEN_8_BITS,
		SIO4_USC_TX_PREAMBLE_LEN_16_BITS,
		SIO4_USC_TX_PREAMBLE_LEN_32_BITS,
		SIO4_USC_TX_PREAMBLE_LEN_64_BITS,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CCR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 11, 10);
	return(ret);
}



//*****************************************************************************
static int _usc_tx_preamble_pat(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_TX_PREAMBLE_PAT_0,
		SIO4_USC_TX_PREAMBLE_PAT_1,
		SIO4_USC_TX_PREAMBLE_PAT_10,
		SIO4_USC_TX_PREAMBLE_PAT_01,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CCR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 9, 8);
	return(ret);
}



//*****************************************************************************
static int _usc_tx_status(chan_data_t* chan, s32* arg)
{
	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_TCSR : 0;
	int	ret;

	if (arg[0] != -1)
	{
		ret	= -EINVAL;;
	}
	else if (chan->dev->cache.model_z16c30)
	{
		ret	= _arg_reg_range(chan, arg, 0, 0xFF, reg, 7, 0);
	}
	else
	{
		arg[0]	= -1;
		ret		= 0;
	}

	return(ret);
}



//*****************************************************************************
static int _usc_tx_ctrl_block(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_CTRL_BLOCK_NO,
		SIO4_USC_CTRL_BLOCK_1_WORD,
		SIO4_USC_CTRL_BLOCK_2_WORD,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CCR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 15, 14);
	return(ret);
}



//*****************************************************************************
static int _usc_tx_wait_dma_trig(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_WAIT_DMA_TRIG_NO,
		SIO4_USC_WAIT_DMA_TRIG_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_CCR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 13, 13);
	return(ret);
}



//*****************************************************************************
static int _usc_tx_wait_underrun(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_TX_WAIT_UNDERRUN_NO,
		SIO4_USC_TX_WAIT_UNDERRUN_YES,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_TCSR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 11, 11);
	return(ret);
}



//*****************************************************************************
static int _usc_txc_cfg(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_TXC_CFG_IN_OSC,		// Lower nibble: PSRCR D0-D2
		SIO4_USC_TXC_CFG_IN_OSC_INV,	// Upper nibble: USC IOCR D3-D5
		SIO4_USC_TXC_CFG_IN_0,
		SIO4_USC_TXC_CFG_IN_1,
		SIO4_USC_TXC_CFG_IN_CBL_RXC,
		SIO4_USC_TXC_CFG_IN_CBL_RXAUXC,
		SIO4_USC_TXC_CFG_OUT_TX_CLK,
		SIO4_USC_TXC_CFG_OUT_TX_BYTE_CLK,
		SIO4_USC_TXC_CFG_OUT_TX_COMP,
		SIO4_USC_TXC_CFG_OUT_BRG0,
		SIO4_USC_TXC_CFG_OUT_BRG1,
		SIO4_USC_TXC_CFG_OUT_CTR1,
		SIO4_USC_TXC_CFG_OUT_DPLL_TX,
		-1	// terminate list
	};

	int	i;
	u32	iocr;
	u32	mask;
	u32	psrcr;
	int	ret;

	if ((chan->dev->cache.model_z16c30)		&&
		(chan->cache.reg_psrcr_bits & D0)	&&
		(chan->cache.reg_psrcr_bits & D1)	&&
		(chan->cache.reg_psrcr_bits & D2))
	{

		if (arg[0] == -1)
		{
			reg_read(chan, SIO4_GSC_PSRCR, &psrcr);
			psrcr	= GSC_FIELD_DECODE(psrcr, 2, 0);

			reg_read(chan, SIO4_USC_IOCR, &iocr);
			iocr	= GSC_FIELD_DECODE(iocr, 5, 3);

			arg[0]	= (iocr << 4) | psrcr;
			ret		= _arg_list_locate(arg, list, &i);
		}
		else
		{
			ret	= _arg_list_locate(arg, list, &i);

			if (ret == 0)
			{
				mask	= GSC_FIELD_ENCODE(0x7,		2, 0);
				psrcr	= GSC_FIELD_ENCODE(arg[0],	2, 0);
				reg_mod(chan, SIO4_GSC_PSRCR, psrcr, mask);

				mask	= GSC_FIELD_ENCODE(0x7,			5, 3);
				iocr	= GSC_FIELD_ENCODE(arg[0] >> 4,	5, 3);
				reg_mod(chan, SIO4_USC_IOCR, iocr, mask);
			}
		}
	}
	else
	{
		if (arg[0] == -1)
			ret	= 0;
		else
			ret	= _arg_list_locate(arg, list, &i);
	}

	return(ret);
}



//*****************************************************************************
static int _usc_txd_cfg(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_TXD_CFG_OUT_TXD,
		SIO4_USC_TXD_CFG_TRI,
		SIO4_USC_TXD_CFG_OUT_0,
		SIO4_USC_TXD_CFG_OUT_1,
		-1	// terminate list
	};

	int	reg	= chan->dev->cache.model_z16c30 ? SIO4_USC_IOCR : 0;
	int	ret;

	ret	= _arg_reg_list(chan, arg, list, reg, 7, 6);
	return(ret);
}



//*****************************************************************************
static int _usc_txc_leg(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_TXC_LEG_IN,
		SIO4_USC_TXC_LEG_OUT_TX_CLK,
		SIO4_USC_TXC_LEG_OUT_TX_BYTE_CLK,
		SIO4_USC_TXC_LEG_OUT_TX_COMP,
		SIO4_USC_TXC_LEG_OUT_BRG0,
		SIO4_USC_TXC_LEG_OUT_BRG1,
		SIO4_USC_TXC_LEG_OUT_CTR1,
		SIO4_USC_TXC_LEG_OUT_DPLL_TX,
		-1	// terminate list
	};

	int	reg;
	int	ret;

	if ((chan->dev->cache.model_z16c30) && (chan->dev->cache.legacy_cable))
		reg	= SIO4_USC_IOCR;
	else
		reg	= 0;

	ret	= _arg_reg_list(chan, arg, list, reg, 5, 3);
	return(ret);
}



//*****************************************************************************
static int _usc_rxc_leg(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_RXC_LEG_IN,
		SIO4_USC_RXC_LEG_OUT_RX_CLK,
		SIO4_USC_RXC_LEG_OUT_RX_BYTE_CLK,
		SIO4_USC_RXC_LEG_OUT_SYNC,
		SIO4_USC_RXC_LEG_OUT_BRG0,
		SIO4_USC_RXC_LEG_OUT_BRG1,
		SIO4_USC_RXC_LEG_OUT_CTR0,
		SIO4_USC_RXC_LEG_OUT_DPLL_RX,
		-1	// terminate list
	};

	int	reg;
	int	ret;

	if ((chan->dev->cache.model_z16c30) && (chan->dev->cache.legacy_cable))
		reg	= SIO4_USC_IOCR;
	else
		reg	= 0;

	ret	= _arg_reg_list(chan, arg, list, reg, 2, 0);
	return(ret);
}



//*****************************************************************************
static int _usc_cts_leg(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_CTS_LEG_IN,	// 0
		SIO4_USC_CTS_LEG_IN,	// 1
		SIO4_USC_CTS_LEG_OUT_0,	// 2
		SIO4_USC_CTS_LEG_OUT_1,	// 3
		-1	// terminate list
	};

	int	i;
	int	ret;
	u32	v;

	if ((chan->dev->cache.model_z16c30 == 0)	||
		(chan->dev->cache.legacy_cable == 0))
	{
		if (arg[0] == -1)
			ret	= 0;
		else
			ret	= _arg_list_locate(arg, list, &i);
	}
	else if (arg[0] == -1)
	{
		reg_read(chan, SIO4_USC_IOCR, &v);
		v		= GSC_FIELD_DECODE(v, 15, 14);
		arg[0]	= list[v];
		ret		= 0;
	}
	else
	{
		ret	= _arg_reg_list(chan, arg, list, SIO4_USC_IOCR, 15, 14);
	}

	return(ret);
}



//*****************************************************************************
static int _usc_dcd_leg(chan_data_t* chan, s32* arg)
{
	static const s32	list[]	=
	{
		SIO4_USC_DCD_LEG_IN_DCD,
		SIO4_USC_DCD_LEG_IN_SYNC,
		SIO4_USC_DCD_LEG_OUT_0,
		SIO4_USC_DCD_LEG_OUT_1,
		-1	// terminate list
	};

	int	reg;
	int	ret;

	if ((chan->dev->cache.model_z16c30) && (chan->dev->cache.legacy_cable))
		reg	= SIO4_USC_IOCR;
	else
		reg	= 0;

	ret	= _arg_reg_list(chan, arg, list, reg, 13, 12);
	return(ret);
}



//*****************************************************************************
static int _reg_read_raw(chan_data_t* chan, gsc_reg_t* arg)
{
	gsc_reg_t	reg;
	int			ret;

	if ((arg->reg < 0) || (arg->reg > 63))
	{
		ret	= -EINVAL;
	}
	else
	{
		reg.reg		= SIO4_GSC_ENCODE(GSC_REG_GSC, 0, 0, 4, 4 * arg->reg);
		reg.value	= 0xDEADBEEF;
		reg.mask	= 0;
		ret			= gsc_reg_read_ioctl(chan, &reg);
		arg->value	= reg.value;
		arg->mask	= 0;
	}

	return(ret);
}



//*****************************************************************************
static int _rcc_sw_fifo_enable_no(chan_data_t* chan)
{
	chan->rcc_fifo.enabled	= 0;
	return(0);
}



//*****************************************************************************
static int _rcc_sw_fifo_flush(chan_data_t* chan, s32* arg)
{
	int	ret	= 0;

	if (chan->dev->cache.model_z16c30 == 0)
	{
		if (arg[0] != -1)
			ret	= -EINVAL;
	}
	else
	{
		switch (arg[0])
		{
			default:

				ret	= -EINVAL;
				break;

			case -1:

				arg[0]	= 1;
				break;

			case SIO4_RCC_SW_FIFO_FLUSH_NO:

				break;

			case SIO4_RCC_SW_FIFO_FLUSH_YES:

				ret	= _rcc_sw_fifo_enable_no(chan);
				chan->rcc_fifo.push	= 0;
				chan->rcc_fifo.pull	= 0;
				reg_mod(chan, SIO4_USC_CCSR, D13 | D15, D13 | D15);
				break;
		}
	}

	return(ret);

}



//*****************************************************************************
static int _rcc_sw_fifo_enable_yes(chan_data_t* chan)
{
	s32	arg;
	int	ret;

	for (;;)	// A convenience loop.
	{
		// Start out disabled.
		ret	= _rcc_sw_fifo_enable_no(chan);

		if (ret)
			break;

		// Flush the current contents.
		arg	= SIO4_RCC_SW_FIFO_FLUSH_YES;
		ret	= _rcc_sw_fifo_flush(chan, &arg);

		if (ret)
			break;

		// Enable the RCC feature.
		reg_write(chan, SIO4_USC_RCCR, 0xFFFF);

		// We're done.
		chan->rcc_fifo.enabled	= 1;

		// The application level code is responsible for enabling appropriate
		// USC interrupts, as the interrupts required are protocol specific.
		break;
	}

	return(ret);
}



//*****************************************************************************
static int _rcc_sw_fifo_enable(chan_data_t* chan, s32* arg)
{
	int	ret	= 0;

	if (chan->dev->cache.model_z16c30 == 0)
	{
		if (arg[0] != -1)
			ret	= -EINVAL;
	}
	else
	{
		switch (arg[0])
		{
			default:

				ret	= -EINVAL;
				break;

			case -1:

				arg[0]	= chan->rcc_fifo.enabled;
				break;

			case SIO4_RCC_SW_FIFO_ENABLE_NO:

				ret	= _rcc_sw_fifo_enable_no(chan);
				break;

			case SIO4_RCC_SW_FIFO_ENABLE_YES:

				ret	= _rcc_sw_fifo_enable_yes(chan);
				break;
		}
	}

	return(ret);
}



//*****************************************************************************
static s32 _rcc_sw_fifo_pull(chan_data_t* chan, s32* arg)
{
	s32	index;
	s32	qty;

	qty	= chan->rcc_fifo.push - chan->rcc_fifo.pull;

	if (qty > 0)
	{
		index	= chan->rcc_fifo.pull % RCC_FIFO_SIZE;
		arg[0]	= chan->rcc_fifo.fifo[index];
		chan->rcc_fifo.pull++;
	}
	else
	{
		arg[0]	= 0;
	}

	return(qty);
}



//*****************************************************************************
static int _rcc_sw_fifo_read_start(chan_data_t* chan, void* arg)
{
	return(-1);
}



//*****************************************************************************
static int _rcc_sw_fifo_read(chan_data_t* chan, s32* arg)
{
	s32			ms;
	s32			qty;
	int			ret	= 0;
	gsc_wait_t	wt;

	if (chan->dev->cache.model_z16c30 == 0)
	{
		if (arg[0] != -1)
			ret	= -EINVAL;
	}
	else if ((arg[0] < -1) || (arg[0] > 60000L))
	{
		ret	= -EINVAL;
	}
	else if (arg[0] == -1)
	{
		arg[0]	= chan->rcc_fifo.enabled;
	}
	else if (chan->rcc_fifo.enabled == 0)
	{
		arg[0]	= 0;
	}
	else if (arg[0] == 0)
	{
		_rcc_sw_fifo_pull(chan, arg);
	}
	else
	{
		ms	= arg[0];
		qty	= _rcc_sw_fifo_pull(chan, arg);

		if (qty == 0)
		{
			memset(&wt, 0, sizeof(wt));
			wt.flags		= GSC_WAIT_FLAG_INTERNAL;
			wt.alt			= SIO4_IRQ_USC_MISC_RCC_UNDERRUN;
			wt.timeout_ms	= ms;
			ret				= gsc_wait_event(
								chan,
								&wt,
								_rcc_sw_fifo_read_start,
								NULL,
								&chan->sem);
			_rcc_sw_fifo_pull(chan, arg);
		}
	}

	return(ret);
}



// variables ******************************************************************

const gsc_ioctl_t	dev_ioctl_list[]	=
{
	// 0
	{ SIO4_IOCTL_REG_READ,				(void*) gsc_reg_read_ioctl		},
	{ SIO4_IOCTL_REG_WRITE,				(void*) gsc_reg_write_ioctl		},
	{ SIO4_IOCTL_REG_MOD,				(void*) gsc_reg_mod_ioctl		},
	{ SIO4_IOCTL_QUERY,					(void*) _query					},
	{ SIO4_IOCTL_INITIALIZE,			(void*) initialize_ioctl		},
	{ SIO4_IOCTL_CBL_MODE,				(void*) _cbl_mode				},
	{ SIO4_IOCTL_CBL_PIN_STATUS,		(void*) _cbl_pin_status			},
	{ SIO4_IOCTL_FIFO_SPACE_CFG,		(void*) _fifo_space_cfg			},
	{ SIO4_IOCTL_GPIO_DIRECTION_OUT,	(void*) _gpio_direction_out		},
	{ SIO4_IOCTL_GPIO_INPUT_LATCHING,	(void*) _gpio_input_latching	},
	// 10
	{ SIO4_IOCTL_GPIO_INPUT_READ,		(void*) _gpio_input_read		},
	{ SIO4_IOCTL_GPIO_OUTPUT_WRITE,		(void*) _gpio_output_write		},
	{ SIO4_IOCTL_GPIO_POLARITY,			(void*) _gpio_polarity			},
	{ SIO4_IOCTL_GPIO_SENSE_EDGE,		(void*) _gpio_sense_edge		},
	{ SIO4_IOCTL_IRQ_GSC_CFG_HIGH,		(void*) _irq_gsc_cfg_high		},
	{ SIO4_IOCTL_IRQ_GSC_ENABLE,		(void*) _irq_gsc_enable			},
	{ SIO4_IOCTL_IRQ_USC_ENABLE,		(void*) _irq_usc_enable			},
	{ SIO4_IOCTL_LED_CHANNEL,			(void*) _led_channel			},
	{ SIO4_IOCTL_LED_MAIN,				(void*) _led_main				},
	{ SIO4_IOCTL_LOOP_BACK,				(void*) _loop_back				},
	// 20
	{ SIO4_IOCTL_OSC_MEASURE,			(void*) _osc_measure			},
	{ SIO4_IOCTL_OSC_PROGRAM,			(void*) _osc_program			},
	{ SIO4_IOCTL_OSC_REFERENCE,			(void*) _osc_reference			},
	{ SIO4_IOCTL_RX_FIFO_AE,			(void*) _rx_fifo_ae				},
	{ SIO4_IOCTL_RX_FIFO_AF,			(void*) _rx_fifo_af				},
	{ SIO4_IOCTL_RX_FIFO_FILL_LEVEL,	(void*) _rx_fifo_fill_level		},
	{ SIO4_IOCTL_RX_FIFO_FULL_CFG,		(void*) _rx_fifo_full_cfg		},
	{ SIO4_IOCTL_RX_FIFO_OVERRUN,		(void*) _rx_fifo_overrun		},
	{ SIO4_IOCTL_RX_FIFO_RESET,			(void*) _rx_fifo_reset			},
	{ SIO4_IOCTL_RX_FIFO_STATUS,		(void*) _rx_fifo_status			},
	// 30
	{ SIO4_IOCTL_RX_FIFO_UNDERRUN,		(void*) _rx_fifo_underrun		},
	{ SIO4_IOCTL_RX_IO_MODE,			(void*) _rx_io_mode				},
	{ SIO4_IOCTL_RX_IO_OVERRUN,			(void*) _rx_io_overrun			},
	{ SIO4_IOCTL_RX_IO_PIO_THRESHOLD,	(void*) _rx_io_pio_threshold	},
	{ SIO4_IOCTL_RX_IO_TIMEOUT,			(void*) _rx_io_timeout			},
	{ SIO4_IOCTL_RX_IO_UNDERRUN,		(void*) _rx_io_underrun			},
	{ SIO4_IOCTL_SYNC_LEG_RXD_CFG,		(void*) _sync_leg_rxd_cfg		},
	{ SIO4_IOCTL_SYNC_LEG_TXD_CFG,		(void*) _sync_leg_txd_cfg		},
	{ SIO4_IOCTL_SYNC_MODE,				(void*) _sync_mode				},
	{ SIO4_IOCTL_SYNC_RX_BIT_COUNT,		(void*) _sync_rx_bit_count		},
	// 40
	{ SIO4_IOCTL_SYNC_RX_BIT_ORDER,		(void*) _sync_rx_bit_order		},
	{ SIO4_IOCTL_SYNC_RX_COUNT_ERROR,	(void*) _sync_rx_count_error	},
	{ SIO4_IOCTL_SYNC_RX_COUNT_RESET,	(void*) _sync_rx_count_reset	},
	{ SIO4_IOCTL_SYNC_RX_ENABLE,		(void*) _sync_rx_enable			},
	{ SIO4_IOCTL_SYNC_RX_GAP_ENABLE,	(void*) _sync_rx_gap_enable		},
	{ SIO4_IOCTL_SYNC_RXC_CFG,			(void*) _sync_rxc_cfg			},
	{ SIO4_IOCTL_SYNC_RXC_POL,			(void*) _sync_rxc_pol			},
	{ SIO4_IOCTL_SYNC_RXD_CFG,			(void*) _sync_rxd_cfg			},
	{ SIO4_IOCTL_SYNC_RXE_CFG,			(void*) _sync_rxe_cfg			},
	// 50
	{ SIO4_IOCTL_SYNC_RXE_POL,			(void*) _sync_rxe_pol			},
	{ SIO4_IOCTL_SYNC_TX_WORD_SIZE,		(void*) _sync_tx_word_size		},
	{ SIO4_IOCTL_SYNC_TX_BIT_ORDER,		(void*) _sync_tx_bit_order		},
	{ SIO4_IOCTL_SYNC_TX_ENABLE,		(void*) _sync_tx_enable			},
	{ SIO4_IOCTL_SYNC_TX_GAP_SIZE,		(void*) _sync_tx_gap_size		},
	{ SIO4_IOCTL_SYNC_TXAUXC_CFG,		(void*) _sync_txauxc_cfg		},
	{ SIO4_IOCTL_SYNC_TXC_CFG,			(void*) _sync_txc_cfg			},
	{ SIO4_IOCTL_SYNC_TXC_IDLE,			(void*) _sync_txc_idle			},
	{ SIO4_IOCTL_SYNC_TXC_IDLE_CFG,		(void*) _sync_txc_idle_cfg		},
	{ SIO4_IOCTL_SYNC_TXC_POL,			(void*) _sync_txc_pol			},
	{ SIO4_IOCTL_SYNC_TXC_SRC,			(void*) _sync_txc_src			},
	// 60
	{ SIO4_IOCTL_SYNC_TXD_CFG,			(void*) _sync_txd_cfg			},
	{ SIO4_IOCTL_SYNC_TXD_IDLE_CFG,		(void*) _sync_txd_idle_cfg		},
	{ SIO4_IOCTL_SYNC_TXE_CFG,			(void*) _sync_txe_cfg			},
	{ SIO4_IOCTL_SYNC_TXE_POL,			(void*) _sync_txe_pol			},
	{ SIO4_IOCTL_SYNC_TXSP_CFG,			(void*) _sync_txsp_cfg			},
	{ SIO4_IOCTL_TIME_STAMP_COUNT,		(void*) _time_stamp_count		},
	{ SIO4_IOCTL_TIME_STAMP_ENABLE,		(void*) _time_stamp_enable		},
	{ SIO4_IOCTL_TIME_STAMP_SRC,		(void*) _time_stamp_src			},
	{ SIO4_IOCTL_TIME_STAMP_VAL,		(void*) _time_stamp_val			},
	{ SIO4_IOCTL_TX_FIFO_AE,			(void*) _tx_fifo_ae				},
	// 70
	{ SIO4_IOCTL_TX_FIFO_AF,			(void*) _tx_fifo_af				},
	{ SIO4_IOCTL_TX_FIFO_EMPTY_CFG,		(void*) _tx_fifo_empty_cfg		},
	{ SIO4_IOCTL_TX_FIFO_FILL_LEVEL,	(void*) _tx_fifo_fill_level		},
	{ SIO4_IOCTL_TX_FIFO_OVERRUN,		(void*) _tx_fifo_overrun		},
	{ SIO4_IOCTL_TX_FIFO_RESET,			(void*) _tx_fifo_reset			},
	{ SIO4_IOCTL_TX_FIFO_STATUS,		(void*) _tx_fifo_status			},
	{ SIO4_IOCTL_TX_IO_MODE,			(void*) _tx_io_mode				},
	{ SIO4_IOCTL_TX_IO_OVERRUN,			(void*) _tx_io_overrun			},
	{ SIO4_IOCTL_TX_IO_PIO_THRESHOLD,	(void*) _tx_io_pio_threshold	},
	{ SIO4_IOCTL_TX_IO_TIMEOUT,			(void*) _tx_io_timeout			},
	// 80
	{ SIO4_IOCTL_XCVR_ENABLE,			(void*) _xcvr_enable			},
	{ SIO4_IOCTL_XCVR_PROTOCOL,			(void*) _xcvr_protocol			},
	{ SIO4_IOCTL_XCVR_TERM,				(void*) _xcvr_term				},
	{ SIO4_IOCTL_Z16_CBL_DCD_CFG,		(void*) _z16_cbl_dcd_cfg		},
	{ SIO4_IOCTL_Z16_CBL_DTR_DSR_CFG,	(void*) _z16_cbl_dtr_dsr_cfg	},
	{ SIO4_IOCTL_Z16_CBL_RTS_CFG,		(void*) _z16_cbl_rts_cfg		},
	{ SIO4_IOCTL_Z16_CBL_TXAUXC_CFG,	(void*) _z16_cbl_txauxc_cfg		},
	{ SIO4_IOCTL_Z16_CBL_TXC_CFG,		(void*) _z16_cbl_txc_cfg		},
	{ SIO4_IOCTL_Z16_CBL_TXD_CFG,		(void*) _z16_cbl_txd_cfg		},
	{ SIO4_IOCTL_Z16_LEG_RXC,			(void*) _z16_leg_rxc			},
	// 90
	{ SIO4_IOCTL_Z16_LEG_RXD_DCD_CFG,	(void*) _z16_leg_rxd_dcd_cfg	},
	{ SIO4_IOCTL_Z16_LEG_TXC,			(void*) _z16_leg_txc			},
	{ SIO4_IOCTL_Z16_LEG_TXD_CTS_CFG,	(void*) _z16_leg_txd_cts_cfg	},
	{ SIO4_IOCTL_Z16_RX_STS_WRD_ENABLE,	(void*) _z16_rx_sts_wrd_enable	},
	{ SIO4_IOCTL_Z16_SYNC_BYTE,			(void*) _z16_sync_byte			},
	{ SIO4_IOCTL_WAIT_EVENT,			(void*) gsc_wait_event_ioctl	},
	{ SIO4_IOCTL_WAIT_CANCEL,			(void*) gsc_wait_cancel_ioctl	},
	{ SIO4_IOCTL_WAIT_STATUS,			(void*) gsc_wait_status_ioctl	},
	{ SIO4_IOCTL_RX_IO_ABORT,			(void*) _rx_io_abort			},
	{ SIO4_IOCTL_TX_IO_ABORT,			(void*) _tx_io_abort			},
	// 100
	{ SIO4_IOCTL_USC_8023_RX_ADRS_SRCH,	(void*) _usc_8023_rx_adrs_srch	},
	{ SIO4_IOCTL_USC_8023_TX_UNDERRUN,	(void*) _usc_8023_tx_underrun	},
	{ SIO4_IOCTL_USC_ACCEPT_CV,			(void*) _usc_accept_cv			},
	{ SIO4_IOCTL_USC_ACV_RX_EXT_W,		(void*) _usc_acv_rx_ext_w		},
	{ SIO4_IOCTL_USC_ACV_TX_CV_POL,		(void*) _usc_acv_tx_cv_pol		},
	{ SIO4_IOCTL_USC_ACV_TX_EXT_W,		(void*) _usc_acv_tx_ext_w		},
	{ SIO4_IOCTL_USC_ACV_TX_STOP_BIT,	(void*) _usc_acv_tx_stop_bit	},
	{ SIO4_IOCTL_USC_ASYNC_RX_CLK_RATE,	(void*) _usc_async_rx_clk_rate	},
	{ SIO4_IOCTL_USC_ASYNC_TX_CLK_RATE,	(void*) _usc_async_tx_clk_rate	},
	{ SIO4_IOCTL_USC_ASYNC_TX_STOP_BIT,	(void*) _usc_async_tx_stop_bit	},
	// 110
	{ SIO4_IOCTL_USC_BRG0_CLK_SRC,		(void*) _usc_brg0_clk_src		},
	{ SIO4_IOCTL_USC_BRG0_ENABLE,		(void*) _usc_brg0_enable		},
	{ SIO4_IOCTL_USC_BRG0_MODE,			(void*) _usc_brg0_mode			},
	{ SIO4_IOCTL_USC_BRG1_CLK_SRC,		(void*) _usc_brg1_clk_src		},
	{ SIO4_IOCTL_USC_BRG1_ENABLE,		(void*) _usc_brg1_enable		},
	{ SIO4_IOCTL_USC_BRG1_MODE,			(void*) _usc_brg1_mode			},
	{ SIO4_IOCTL_USC_BSC_RX_SHORT,		(void*) _usc_bsc_rx_short		},
	{ SIO4_IOCTL_USC_BSC_RX_STRIP,		(void*) _usc_bsc_rx_strip		},
	{ SIO4_IOCTL_USC_BSC_RX_SYN0,		(void*) _usc_bsc_rx_syn0		},
	{ SIO4_IOCTL_USC_BSC_RX_SYN1,		(void*) _usc_bsc_rx_syn1		},
	// 120
	{ SIO4_IOCTL_USC_BSC_TX_PREAMBLE,	(void*) _usc_bsc_tx_preamble	},
	{ SIO4_IOCTL_USC_BSC_TX_SHORT,		(void*) _usc_bsc_tx_short		},
	{ SIO4_IOCTL_USC_BSC_TX_SYN0,		(void*) _usc_bsc_tx_syn0		},
	{ SIO4_IOCTL_USC_BSC_TX_SYN1,		(void*) _usc_bsc_tx_syn1		},
	{ SIO4_IOCTL_USC_BSC_TX_UNDERRUN,	(void*) _usc_bsc_tx_underrun	},
	{ SIO4_IOCTL_USC_CTR0_CLK_SRC,		(void*) _usc_ctr0_clk_src		},
	{ SIO4_IOCTL_USC_CTR0_RATE,			(void*) _usc_ctr0_rate			},
	{ SIO4_IOCTL_USC_CTR1_CLK_SRC,		(void*) _usc_ctr1_clk_src		},
	{ SIO4_IOCTL_USC_CTR1_RATE,			(void*) _usc_ctr1_rate			},
	{ SIO4_IOCTL_USC_CTS_CFG,			(void*) _usc_cts_cfg			},
	// 130
	{ SIO4_IOCTL_USC_DCD_CFG,			(void*) _usc_dcd_cfg			},
	{ SIO4_IOCTL_USC_DPLL_ADJ_SYNC,		(void*) _usc_dpll_adj_sync		},
	{ SIO4_IOCTL_USC_DPLL_CLK_SRC,		(void*) _usc_dpll_clk_src		},
	{ SIO4_IOCTL_USC_DPLL_MISS_1,		(void*) _usc_dpll_miss_1		},
	{ SIO4_IOCTL_USC_DPLL_MISS_2,		(void*) _usc_dpll_miss_2		},
	{ SIO4_IOCTL_USC_DPLL_MODE,			(void*) _usc_dpll_mode			},
	{ SIO4_IOCTL_USC_DPLL_RATE,			(void*) _usc_dpll_rate			},
	{ SIO4_IOCTL_USC_DPLL_SYNC,			(void*) _usc_dpll_sync			},
	{ SIO4_IOCTL_USC_HDLC_RX_ADRS_CTRL,	(void*) _usc_hdlc_rx_adrs_ctrl	},
	{ SIO4_IOCTL_USC_HDLC_TX_L_CHR_LEN,	(void*) _usc_hdlc_tx_l_chr_len	},
	// 140
	{ SIO4_IOCTL_USC_HDLC_TX_PREAMBLE,	(void*) _usc_hdlc_tx_preamble	},
	{ SIO4_IOCTL_USC_HDLC_TX_SHARE_0,	(void*) _usc_hdlc_tx_share_0	},
	{ SIO4_IOCTL_USC_HDLC_TX_UNDERRUN,	(void*) _usc_hdlc_tx_underrun	},
	{ SIO4_IOCTL_USC_HDLCL_TX_ACTIVE,	(void*) _usc_hdlcl_tx_active	},
	{ SIO4_IOCTL_USC_HDLCL_TX_SHARE_0,	(void*) _usc_hdlcl_tx_share_0	},
	{ SIO4_IOCTL_USC_HDLCL_TX_UNDERRUN,	(void*) _usc_hdlcl_tx_underrun	},
	{ SIO4_IOCTL_USC_ISOC_TX_STOP_BIT,	(void*) _usc_isoc_tx_stop_bit	},
	{ SIO4_IOCTL_USC_LOOP_SENDING,		(void*) _usc_loop_sending		},
	{ SIO4_IOCTL_USC_MONO_RX_SHORT,		(void*) _usc_mono_rx_short		},
	{ SIO4_IOCTL_USC_MONO_RX_STRIP,		(void*) _usc_mono_rx_strip		},
	// 150
	{ SIO4_IOCTL_USC_MONO_RX_SYNC,		(void*) _usc_mono_rx_sync		},
	{ SIO4_IOCTL_USC_MONO_TX_CRC_UNDER,	(void*) _usc_mono_tx_crc_under	},
	{ SIO4_IOCTL_USC_MONO_TX_PREAMBLE,	(void*) _usc_mono_tx_preamble	},
	{ SIO4_IOCTL_USC_MONO_TX_SHORT,		(void*) _usc_mono_tx_short		},
	{ SIO4_IOCTL_USC_MONO_TX_SYNC,		(void*) _usc_mono_tx_sync		},
	{ SIO4_IOCTL_USC_NBIP_RX_CLK_RATE,	(void*) _usc_nbip_rx_clk_rate	},
	{ SIO4_IOCTL_USC_NBIP_RX_PARITY,	(void*) _usc_nbip_rx_parity		},
	{ SIO4_IOCTL_USC_NBIP_TX_ADRS_BIT,	(void*) _usc_nbip_tx_adrs_bit	},
	{ SIO4_IOCTL_USC_NBIP_TX_CLK_RATE,	(void*) _usc_nbip_tx_clk_rate	},
	{ SIO4_IOCTL_USC_NBIP_TX_PARITY,	(void*) _usc_nbip_tx_parity		},
	// 160
	{ SIO4_IOCTL_USC_ON_LOOP,			(void*) _usc_on_loop			},
	{ SIO4_IOCTL_USC_OPER_MODE,			(void*) _usc_oper_mode			},
	{ SIO4_IOCTL_USC_RCC_FIFO_CLEAR,	(void*) _usc_rcc_fifo_clear		},
	{ SIO4_IOCTL_USC_RCC_FIFO_OVERRUN,	(void*) _usc_rcc_fifo_overrun	},
	{ SIO4_IOCTL_USC_RCC_FIFO_VALID,	(void*) _usc_rcc_fifo_valid		},
	{ SIO4_IOCTL_USC_RESET,				(void*) _usc_reset				},
	{ SIO4_IOCTL_USC_RX_CHAR_CNT,		(void*) _usc_rx_char_cnt		},
	{ SIO4_IOCTL_USC_RX_CHAR_CNT_LIM,	(void*) _usc_rx_char_cnt_lim	},
	{ SIO4_IOCTL_USC_RX_CHAR_LEN,		(void*) _usc_rx_char_len		},
	{ SIO4_IOCTL_USC_RX_CLK_SRC,		(void*) _usc_rx_clk_src			},
	// 170
	{ SIO4_IOCTL_USC_RX_CMD,			(void*) _usc_rx_cmd				},
	{ SIO4_IOCTL_USC_RX_CRC_ENABLE,		(void*) _usc_rx_crc_enable		},
	{ SIO4_IOCTL_USC_RX_CRC_PRESET,		(void*) _usc_rx_crc_preset		},
	{ SIO4_IOCTL_USC_RX_CRC_TYPE,		(void*) _usc_rx_crc_type		},
	{ SIO4_IOCTL_USC_RX_DATA_ENCODE,	(void*) _usc_rx_data_encode		},
	{ SIO4_IOCTL_USC_RX_ENABLE,			(void*) _usc_rx_enable			},
	{ SIO4_IOCTL_USC_RX_MODE,			(void*) _usc_rx_mode			},
	{ SIO4_IOCTL_USC_RX_PAR_ENABLE,		(void*) _usc_rx_par_enable		},
	{ SIO4_IOCTL_USC_RX_PAR_TYPE,		(void*) _usc_rx_par_type		},
	{ SIO4_IOCTL_USC_RX_QUEUE_ABORT,	(void*) _usc_rx_queue_abort		},
	// 180
	{ SIO4_IOCTL_USC_RX_STATUS,			(void*) _usc_rx_status			},
	{ SIO4_IOCTL_USC_RX_STATUS_BLOCK,	(void*) _usc_rx_status_block	},
	{ SIO4_IOCTL_USC_RX_WAIT_DMA_TRIG,	(void*) _usc_rx_wait_dma_trig	},
	{ SIO4_IOCTL_USC_RXC_CFG,			(void*) _usc_rxc_cfg			},
	{ SIO4_IOCTL_USC_SEND_COMMAND,		(void*) _usc_send_command		},
	{ SIO4_IOCTL_USC_SMONO_RX_SYNC,		(void*) _usc_smono_rx_sync		},
	{ SIO4_IOCTL_USC_SMONO_TX_ACTIVE,	(void*) _usc_smono_tx_active	},
	{ SIO4_IOCTL_USC_SMONO_TX_SHORT,	(void*) _usc_smono_tx_short		},
	{ SIO4_IOCTL_USC_SMONO_TX_SYNC,		(void*) _usc_smono_tx_sync		},
	{ SIO4_IOCTL_USC_SMONO_TX_UNDERRUN,	(void*) _usc_smono_tx_underrun	},
	// 190
	{ SIO4_IOCTL_USC_TBSC_RX_ENCODING,	(void*) _usc_tbsc_rx_encoding	},
	{ SIO4_IOCTL_USC_TBSC_TX_ENCODING,	(void*) _usc_tbsc_tx_encoding	},
	{ SIO4_IOCTL_USC_TBSC_TX_PREAMBLE,	(void*) _usc_tbsc_tx_preamble	},
	{ SIO4_IOCTL_USC_TBSC_TX_UNDERRUN,	(void*) _usc_tbsc_tx_underrun	},
	{ SIO4_IOCTL_USC_TC0,				(void*) _usc_tc0				},
	{ SIO4_IOCTL_USC_TC1,				(void*) _usc_tc1				},
	{ SIO4_IOCTL_USC_TX_CHAR_CNT,		(void*) _usc_tx_char_cnt		},
	{ SIO4_IOCTL_USC_TX_CHAR_CNT_LIM,	(void*) _usc_tx_char_cnt_lim	},
	{ SIO4_IOCTL_USC_TX_CHAR_LEN,		(void*) _usc_tx_char_len		},
	{ SIO4_IOCTL_USC_TX_CLK_SRC,		(void*) _usc_tx_clk_src			},
	// 200
	{ SIO4_IOCTL_USC_TX_CMD,			(void*) _usc_tx_cmd				},
	{ SIO4_IOCTL_USC_TX_CRC_ENABLE,		(void*) _usc_tx_crc_enable		},
	{ SIO4_IOCTL_USC_TX_CRC_ON_END,		(void*) _usc_tx_crc_on_end		},
	{ SIO4_IOCTL_USC_TX_CRC_PRESET,		(void*) _usc_tx_crc_preset		},
	{ SIO4_IOCTL_USC_TX_CRC_TYPE,		(void*) _usc_tx_crc_type		},
	{ SIO4_IOCTL_USC_TX_DATA_ENCODE,	(void*) _usc_tx_data_encode		},
	{ SIO4_IOCTL_USC_TX_ENABLE,			(void*) _usc_tx_enable			},
	{ SIO4_IOCTL_USC_TX_IDLE_COND,		(void*) _usc_tx_idle_cond		},
	{ SIO4_IOCTL_USC_TX_MODE,			(void*) _usc_tx_mode			},
	{ SIO4_IOCTL_USC_TX_PAR_ENABLE,		(void*) _usc_tx_par_enable		},
	// 210
	{ SIO4_IOCTL_USC_TX_PAR_TYPE,		(void*) _usc_tx_par_type		},
	{ SIO4_IOCTL_USC_TX_PREAMBLE_FLAG,	(void*) _usc_tx_preamble_flag	},
	{ SIO4_IOCTL_USC_TX_PREAMBLE_LEN,	(void*) _usc_tx_preamble_len	},
	{ SIO4_IOCTL_USC_TX_PREAMBLE_PAT,	(void*) _usc_tx_preamble_pat	},
	{ SIO4_IOCTL_USC_TX_STATUS,			(void*) _usc_tx_status			},
	{ SIO4_IOCTL_USC_TX_CTRL_BLOCK,		(void*) _usc_tx_ctrl_block		},
	{ SIO4_IOCTL_USC_TX_WAIT_DMA_TRIG,	(void*) _usc_tx_wait_dma_trig	},
	{ SIO4_IOCTL_USC_TX_WAIT_UNDERRUN,	(void*) _usc_tx_wait_underrun	},
	{ SIO4_IOCTL_USC_TXC_CFG,			(void*) _usc_txc_cfg			},
	{ SIO4_IOCTL_USC_TXD_CFG,			(void*) _usc_txd_cfg			},
	// 220
	{ SIO4_IOCTL_USC_TXC_LEG,			(void*) _usc_txc_leg			},
	{ SIO4_IOCTL_USC_RXC_LEG,			(void*) _usc_rxc_leg			},
	{ SIO4_IOCTL_USC_CTS_LEG,			(void*) _usc_cts_leg			},
	{ SIO4_IOCTL_USC_DCD_LEG,			(void*) _usc_dcd_leg			},
	{ SIO4_IOCTL_REG_READ_RAW,			(void*) _reg_read_raw			},
	{ SIO4_IOCTL_RCC_SW_FIFO_ENABLE,	(void*) _rcc_sw_fifo_enable		},
	{ SIO4_IOCTL_RCC_SW_FIFO_FLUSH,		(void*) _rcc_sw_fifo_flush		},
	{ SIO4_IOCTL_RCC_SW_FIFO_READ,		(void*) _rcc_sw_fifo_read		},
	{ SIO4_IOCTL_RX_FIFO_FULL_CFG_GLB,	(void*) _rx_fifo_full_cfg_glb	},

	{ -1, NULL }
};



//*****************************************************************************
int fw_type_config_get(chan_data_t* chan, s32* get)
{
	dev_data_t*	dev		= chan->dev;
	int			ret		= 0;
	s32			set		= -1;
	int			shift	= chan->index * 8;

	if ((dev->cache.reg_ftr == 0) ||
		(dev->cache.fw_type_config == 0))
	{
		set	= dev->cache.fw_type;
	}
	else
	{
		set	= os_reg_mem_rx_u32(chan->dev, dev->vaddr.gsc_ftr_32);
		set	= (set >> shift) & 0xFF;
	}

	if (get)
		get[0]	= set;

	return(ret);
}



