// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_2.x.x_GSC_DN/driver/io.c $
// $Rev: 32705 $
// $Date: 2015-07-09 17:08:54 -0500 (Thu, 09 Jul 2015) $

#include "main.h"



/******************************************************************************
*
*	Function:	dev_io_close
*
*	Purpose:
*
*		Cleanup the I/O stuff for the channel as it is being closed.
*
*	Arguments:
*
*		chan	The channel data structure.
*
*	Returned:
*
*		None.
*
******************************************************************************/

void dev_io_close(chan_data_t* chan)
{
	chan->rx.dma_channel	= NULL;
	chan->tx.dma_channel	= NULL;
}



/******************************************************************************
*
*	Function:	dev_io_create
*
*	Purpose:
*
*		Perform I/O based initialization as the driver is being loaded.
*
*	Arguments:
*
*		dev		The data for the device of interest.
*
*		index	The index of the channel to access.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error code.
*
******************************************************************************/

int dev_io_create(dev_data_t* dev, int index)
{
	static const u32	rx_dmdma_dir[4]	= { 0x0, 0x2, 0x1, 0x3 };
	static const u32	tx_dmdma_dir[4]	= { 0x4, 0x6, 0x5, 0x7 };

	chan_data_t*		chan	= &dev->channel[index];
	int					ret		= 0;
	int					rx;
	int					tx;

	rx							= gsc_io_create(chan, &chan->rx, _32K);
	chan->rx.io_reg_offset		= 0x18 + (index * 16);
	chan->rx.io_reg_vaddr		= chan->vaddr.gsc_fdr_8;
	chan->rx.dmdma_dir			= rx_dmdma_dir[index];

	tx							= gsc_io_create(chan, &chan->tx, _32K);
	chan->tx.io_reg_offset		= chan->rx.io_reg_offset;
	chan->tx.io_reg_vaddr		= chan->rx.io_reg_vaddr;
	chan->rx.dmdma_dir			= tx_dmdma_dir[index];

	ret	= rx ? rx : tx;
	return(ret);
}



/******************************************************************************
*
*	Function:	dev_io_destroy
*
*	Purpose:
*
*		Perform I/O based cleanup as the driver is being unloaded.
*
*	Arguments:
*
*		dev		The data for the device of interest.
*
*		index	The index of the channel to access.
*
*	Returned:
*
*		None.
*
******************************************************************************/

void dev_io_destroy(dev_data_t* dev, int index)
{
	chan_data_t*	chan	= &dev->channel[index];

	dev_io_close(chan);	// Just in case.
	gsc_io_destroy(chan, &chan->rx);
	gsc_io_destroy(chan, &chan->tx);
	memset(chan, 0, sizeof(chan_data_t));
}



/******************************************************************************
*
*	Function:	dev_io_open
*
*	Purpose:
*
*		Perform initialization work for a device that is being opened.
*
*	Arguments.
*
*		chan	The channel data structure.
*
*	Returned:
*
*		0	All went well.
*		< 0	An appropriate error code.
*
******************************************************************************/

int dev_io_open(chan_data_t* chan)
{
	dev_io_close(chan);	// Just in case.
	return(0);
}



//*****************************************************************************
int dev_dma_chan_setup(chan_data_t* chan, gsc_dma_ch_t* dma, u32 dpr)
{
	static const int	rx[4]	= { 0x0, 0x2, 0x1, 0x3 };
	static const int	tx[4]	= { 0x4, 0x6, 0x5, 0x7 };

	u32	bcr;
	u32	mask;
	u32	shift;
	int	v		= -1;

	if (dpr & GSC_DMA_DPR_BOARD_TO_HOST)
	{
		// Rx operation

		if (chan->rx.io_mode == GSC_IO_MODE_DMDMA)
			v	= rx[chan->index];
	}
	else
	{
		// Tx Operation

		if (chan->tx.io_mode == GSC_IO_MODE_DMDMA)
			v	= tx[chan->index];
	}

	if (v >= 0)
	{
		shift	= 4 * dma->index;
		mask	= 0x7 << shift;
		v		<<= shift;
		bcr		= os_reg_mem_rx_u32(chan->dev, chan->dev->vaddr.gsc_bcr_32);

		if ((bcr & mask) != v)
		{
			bcr	&= ~mask;
			bcr	|= v;
			os_reg_mem_tx_u32(chan->dev, chan->dev->vaddr.gsc_bcr_32, bcr);
		}
	}

	return(0);
}


