// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_2.x.x_GSC_DN/async/lib/init.c $
// $Rev: 31500 $
// $Date: 2014-12-09 18:30:36 -0600 (Tue, 09 Dec 2014) $

#include "main.h"



// #defines *******************************************************************

//#define	ARRAY_ELEMENTS(a)	(sizeof((a)) / sizeof((a)[0]))



// data types *****************************************************************

typedef struct
{
	s32	divisor;	// 0 = don't use, !0 = use
	s32	enable;		// SIO4_ASYNC_USC_BRG0/1_ENABLE_xxx
} brg_t;

typedef struct
{
	s32	divisor;	// divide by this amount
	s32	clk_src;	// SIO4_ASYNC_USC_CTR0_CLK_SRC_xxx
	s32	rate;		// SIO4_ASYNC_USC_CTR0_RATE_xxx
} ctr0_t;

typedef struct
{
	s32	divisor;	// divide by this amount
	s32	clk_src;	// SIO4_ASYNC_USC_CTR1_CLK_SRC_xxx
	s32	dpll_rate;	// SIO4_ASYNC_USC_DPLL_RATE_xxx
} ctr1_t;

typedef struct
{
	const ctr0_t*	ctr0;
	s32				brg0_clk_src;	// SIO4_ASYNC_USC_BRG0_CLK_SRC_xxx
	const brg_t*	brg0;
	s32				tx_clk_src;		// SIO4_ASYNC_USC_TX_CLK_SRC_xxx
} tx_t;

typedef struct
{
	const ctr1_t*	ctr1;
	s32				brg1_clk_src;	// SIO4_ASYNC_USC_BRG1_CLK_SRC_xxx
	const brg_t*	brg1;
	s32				rx_clk_src;		// SIO4_ASYNC_USC_RX_CLK_SRC_xxx
} rx_t;

typedef struct
{
	s32	divisor;		// divide by this amount
	s32	rx_clk_rate;	// SIO4_ASYNC_USC_TX_CLK_RATE_xxx
} rx_div_t;

typedef struct
{
	s32	divisor;		// divide by this amount
	s32	tx_clk_rate;	// SIO4_ASYNC_USC_TX_CLK_RATE_xxx
} tx_div_t;



// variables ******************************************************************

//									divisor	enable
static const brg_t	_brg0_off	= { 0,		SIO4_ASYNC_USC_BRG0_ENABLE_NO	};
static const brg_t	_brg0_on	= { 1,		SIO4_ASYNC_USC_BRG0_ENABLE_YES	};

//									divisor	enable
static const brg_t	_brg1_off	= { 0,		SIO4_ASYNC_USC_BRG1_ENABLE_NO	};
static const brg_t	_brg1_on	= { 1,		SIO4_ASYNC_USC_BRG1_ENABLE_YES	};

//									divisor	clk_src								rate
static const ctr0_t	_ctr0_off	= { 1,		SIO4_ASYNC_USC_CTR0_CLK_SRC_DISABLE, SIO4_ASYNC_USC_CTR0_RATE_32X	};
static const ctr0_t	_ctr0_4		= { 4,		SIO4_ASYNC_USC_CTR0_CLK_SRC_RXC_PIN, SIO4_ASYNC_USC_CTR0_RATE_4X	};
static const ctr0_t	_ctr0_8		= { 8,		SIO4_ASYNC_USC_CTR0_CLK_SRC_RXC_PIN, SIO4_ASYNC_USC_CTR0_RATE_8X	};
static const ctr0_t	_ctr0_16	= { 16,		SIO4_ASYNC_USC_CTR0_CLK_SRC_RXC_PIN, SIO4_ASYNC_USC_CTR0_RATE_16X	};
static const ctr0_t	_ctr0_32	= { 32,		SIO4_ASYNC_USC_CTR0_CLK_SRC_RXC_PIN, SIO4_ASYNC_USC_CTR0_RATE_32X	};


//									divisor	clk_src								dpll_rate
static const ctr1_t	_ctr1_off	= { 1,		SIO4_ASYNC_USC_CTR1_CLK_SRC_DISABLE, SIO4_ASYNC_USC_DPLL_RATE_32X	};
static const ctr1_t	_ctr1_4		= { 4,		SIO4_ASYNC_USC_CTR1_CLK_SRC_RXC_PIN, SIO4_ASYNC_USC_DPLL_RATE_4X	};
static const ctr1_t	_ctr1_8		= { 8,		SIO4_ASYNC_USC_CTR1_CLK_SRC_RXC_PIN, SIO4_ASYNC_USC_DPLL_RATE_8X	};
static const ctr1_t	_ctr1_16	= { 16,		SIO4_ASYNC_USC_CTR1_CLK_SRC_RXC_PIN, SIO4_ASYNC_USC_DPLL_RATE_16X	};
static const ctr1_t	_ctr1_32	= { 32,		SIO4_ASYNC_USC_CTR1_CLK_SRC_RXC_PIN, SIO4_ASYNC_USC_DPLL_RATE_32X	};

static const rx_t	_rx_list[]	=
{
	// ctr1			brg1_clk_src							brg1		rx_clk_src

	// RxC -> RxClk
	{ &_ctr1_off,	SIO4_ASYNC_USC_BRG1_CLK_SRC_CTR1,		&_brg1_off,	SIO4_ASYNC_USC_RX_CLK_SRC_RXC_PIN	},

	// RxC -> BRG1 -> RxClk
	{ &_ctr1_off,	SIO4_ASYNC_USC_BRG1_CLK_SRC_RXC_PIN,	&_brg1_on,	SIO4_ASYNC_USC_RX_CLK_SRC_BRG1		},

	// RxC -> CTR1 -> RxClk
	{ &_ctr1_32,	SIO4_ASYNC_USC_BRG1_CLK_SRC_CTR1,		&_brg1_off,	SIO4_ASYNC_USC_RX_CLK_SRC_CTR1		},
	{ &_ctr1_16,	SIO4_ASYNC_USC_BRG1_CLK_SRC_CTR1,		&_brg1_off,	SIO4_ASYNC_USC_RX_CLK_SRC_CTR1		},
	{ &_ctr1_8,		SIO4_ASYNC_USC_BRG1_CLK_SRC_CTR1,		&_brg1_off,	SIO4_ASYNC_USC_RX_CLK_SRC_CTR1		},
	{ &_ctr1_4,		SIO4_ASYNC_USC_BRG1_CLK_SRC_CTR1,		&_brg1_off,	SIO4_ASYNC_USC_RX_CLK_SRC_CTR1		},

	// RxC -> CTR1 -> BRG1 -> RxClk
	{ &_ctr1_32,	SIO4_ASYNC_USC_BRG1_CLK_SRC_CTR1,		&_brg1_on,	SIO4_ASYNC_USC_RX_CLK_SRC_BRG1		},
	{ &_ctr1_16,	SIO4_ASYNC_USC_BRG1_CLK_SRC_CTR1,		&_brg1_on,	SIO4_ASYNC_USC_RX_CLK_SRC_BRG1		},
	{ &_ctr1_8,		SIO4_ASYNC_USC_BRG1_CLK_SRC_CTR1,		&_brg1_on,	SIO4_ASYNC_USC_RX_CLK_SRC_BRG1		},
	{ &_ctr1_4,		SIO4_ASYNC_USC_BRG1_CLK_SRC_CTR1,		&_brg1_on,	SIO4_ASYNC_USC_RX_CLK_SRC_BRG1		}
};

static const rx_div_t	_rx_div_list[]	=
{
	// divisor	tx_clk_rate
	{ 64,		SIO4_ASYNC_USC_RX_CLK_RATE_64X	},
	{ 32,		SIO4_ASYNC_USC_RX_CLK_RATE_32X	},
	{ 16,		SIO4_ASYNC_USC_RX_CLK_RATE_16X	}
};

static const tx_t	_tx_list[]	=
{
	// ctr0			brg0_clk_src							brg0		tx_clk_src

	// RxC -> TxClk
	{ &_ctr0_off,	SIO4_ASYNC_USC_BRG0_CLK_SRC_CTR0,		&_brg0_off,	SIO4_ASYNC_USC_TX_CLK_SRC_RXC_PIN	},

	// RxC -> BRG0 -> TxClk
	{ &_ctr0_off,	SIO4_ASYNC_USC_BRG0_CLK_SRC_RXC_PIN,	&_brg0_on,	SIO4_ASYNC_USC_TX_CLK_SRC_BRG0		},

	// RxC -> CTR0 -> TxClk
	{ &_ctr0_32,	SIO4_ASYNC_USC_BRG0_CLK_SRC_CTR0,		&_brg0_off,	SIO4_ASYNC_USC_TX_CLK_SRC_CTR0		},
	{ &_ctr0_16,	SIO4_ASYNC_USC_BRG0_CLK_SRC_CTR0,		&_brg0_off,	SIO4_ASYNC_USC_TX_CLK_SRC_CTR0		},
	{ &_ctr0_8,		SIO4_ASYNC_USC_BRG0_CLK_SRC_CTR0,		&_brg0_off,	SIO4_ASYNC_USC_TX_CLK_SRC_CTR0		},
	{ &_ctr0_4,		SIO4_ASYNC_USC_BRG0_CLK_SRC_CTR0,		&_brg0_off,	SIO4_ASYNC_USC_TX_CLK_SRC_CTR0		},

	// RxC -> CTR0 -> BRG0 -> TxClk
	{ &_ctr0_32,	SIO4_ASYNC_USC_BRG0_CLK_SRC_CTR0,		&_brg0_on,	SIO4_ASYNC_USC_TX_CLK_SRC_BRG0		},
	{ &_ctr0_16,	SIO4_ASYNC_USC_BRG0_CLK_SRC_CTR0,		&_brg0_on,	SIO4_ASYNC_USC_TX_CLK_SRC_BRG0		},
	{ &_ctr0_8,		SIO4_ASYNC_USC_BRG0_CLK_SRC_CTR0,		&_brg0_on,	SIO4_ASYNC_USC_TX_CLK_SRC_BRG0		},
	{ &_ctr0_4,		SIO4_ASYNC_USC_BRG0_CLK_SRC_CTR0,		&_brg0_on,	SIO4_ASYNC_USC_TX_CLK_SRC_BRG0		}
};

static const tx_div_t	_tx_div_list[]	=
{
	{ 64,	SIO4_ASYNC_USC_TX_CLK_RATE_64X	},
	{ 32,	SIO4_ASYNC_USC_TX_CLK_RATE_32X	},
	{ 16,	SIO4_ASYNC_USC_TX_CLK_RATE_16X	}
};


//*****************************************************************************
static void _tx_baud_compute(sio4_async_t* async)
{
	s32		bit_rate;
	s32		brg0		= 0;
	float	delta;
	float	delta_min	= 20000000L;
	int		div_no;
	int		init		= 0;
	float	prog;
	float	rate;
	int		tx_no;

	if (async->tx.bit_rate < 1)
		bit_rate	= 1;
	else
		bit_rate	= async->tx.bit_rate;

	if (async->osc.prog)
		prog	= (float) async->osc.prog;
	else
		prog	= (float) 20000000L;

	for (div_no = 0; div_no < ARRAY_ELEMENTS(_tx_div_list); div_no++)
	{
		for (tx_no = 0; tx_no < ARRAY_ELEMENTS(_tx_list); tx_no++)
		{
			rate	= prog
					/ _tx_div_list[div_no].divisor
					/ _tx_list[tx_no].ctr0->divisor;

			if (_tx_list[tx_no].brg0->divisor)
			{
				brg0	= (s32) (((float) rate / bit_rate) + 0.5);

				if (brg0 < 1)
					brg0	= 1;
				else if (brg0 > 0xFFFF)
					brg0	= 0xFFFF;

				rate	/= brg0;
			}

			delta	= rate - bit_rate;

			if (delta < 0)
				delta	= -delta;

			if ((init == 0) || (delta < delta_min))
			{
				init		= 1;
				delta_min	= delta;

				async->usc.ctr0.clk_src	= _tx_list[tx_no].ctr0->clk_src;
				async->usc.ctr0.rate	= _tx_list[tx_no].ctr0->rate;
				async->usc.brg0.clk_src	= _tx_list[tx_no].brg0_clk_src;
				async->usc.brg0.enable	= _tx_list[tx_no].brg0->enable;
				async->usc.brg0.divider	= brg0;
				async->usc.tx.clk_src	= _tx_list[tx_no].tx_clk_src;
				async->usc.tx.clk_rate	= _tx_div_list[div_no].tx_clk_rate;
				async->tx.bit_rate		= (s32) (rate + 0.5);
			}
		}
	}
}



//*****************************************************************************
static void _rx_baud_compute(sio4_async_t* async)
{
	s32		bit_rate;
	s32		brg1		= 0;
	float	delta;
	float	delta_min	= 0;
	int		div_no;
	int		init		= 0;
	float	prog;
	float	rate;
	int		rx_no;

	if (async->rx.bit_rate < 1)
		bit_rate	= 1;
	else
		bit_rate	= async->rx.bit_rate;

	if (async->osc.prog)
		prog	= (float) async->osc.prog;
	else
		prog	= (float) 20000000L;

	for (div_no = 0; div_no < ARRAY_ELEMENTS(_rx_div_list); div_no++)
	{
		for (rx_no = 0; rx_no < ARRAY_ELEMENTS(_rx_list); rx_no++)
		{
			rate	= prog
					/ _rx_div_list[div_no].divisor
					/ _rx_list[rx_no].ctr1->divisor;

			if (_rx_list[rx_no].brg1->divisor)
			{
				brg1	= (s32) (((float) rate / bit_rate) + 0.5);

				if (brg1 < 1)
					brg1	= 1;
				else if (brg1 > 0xFFFF)
					brg1	= 0xFFFF;

				rate	/= brg1;
			}

			delta	= rate - bit_rate;

			if (delta < 0)
				delta	= -delta;

			if ((init == 0) || (delta < delta_min))
			{
				init		= 1;
				delta_min	= delta;

				async->usc.ctr1.clk_src	= _rx_list[rx_no].ctr1->clk_src;
				async->usc.dpll.rate	= _rx_list[rx_no].ctr1->dpll_rate;
				async->usc.brg1.clk_src	= _rx_list[rx_no].brg1_clk_src;
				async->usc.brg1.enable	= _rx_list[rx_no].brg1->enable;
				async->usc.brg1.divider	= brg1;
				async->usc.rx.clk_src	= _rx_list[rx_no].rx_clk_src;
				async->usc.rx.clk_rate	= _rx_div_list[div_no].rx_clk_rate;
				async->rx.bit_rate		= (s32) (rate + 0.5);
			}
		}
	}
}



//*****************************************************************************
static const char* _sio4_async_init_t_validate(const sio4_async_init_t* init)
{
	const char*	psz	= NULL;

	for (;;)	// A convenience loop.
	{
		if (init == NULL)
		{
			psz	= "_sio4_async_init_t_validate(init)";
			break;
		}

		if ((init->rx_bit_rate < 1) || (init->rx_bit_rate > 1250000))
		{
			psz	= "init->rx_bit_rate";
			break;
		}

		if ((init->tx_bit_rate < 1) || (init->tx_bit_rate > 1250000))
		{
			psz	= "init->tx_bit_rate";
			break;
		}

		if ((init->osc_prog < 1) || (init->osc_prog > 20000000L))
		{
			psz	= "init->osc_prog";
			break;
		}

		break;
	}

	return(psz);
}



/******************************************************************************
*
*	Function:	sio4_async_init
*
*	Purpose:
*
*		Initialize the Asynchronous settings for the specified SIO4.
*
*	Arguments:
*
*		fd		The file descriptor for the SIO4 channel to access.
*
*		init	This provides preliminary settings upon which various device
*				clocking parameters are based.
*
*		async	The initialized settings are recorded here.
*
*		err		If a field value is invalid, this will name the field.
*
*	Returned:
*
*		0		All went well.
*		-errno	There was a problem.
*
******************************************************************************/

int sio4_async_init(
	int							fd,
	const sio4_async_init_t*	init,
	sio4_async_t*				async,
	const char**				err)
{
	device_t*	dev;
	const char*	psz		= NULL;
	int			ret;

	for (;;)	// A convenience loop.
	{
		// Make sure the library is initialized.
		ret	= sio4_async_lib_init();

		if (ret)
		{
			psz	= "sio4_async_lib_init";
			break;
		}

		// Validate the received arguments.

		if (init == NULL)
		{
			psz	= "sio4_async_init(,init,,)";
			ret	= -EINVAL;
			break;
		}

		if (async == NULL)
		{
			psz	= "sio4_async_init(,,async,)";
			ret	= -EINVAL;
			break;
		}

		// Gain access to the device.
		dev	= lib_fd_find_inc(fd);

		if (dev == NULL)
		{
			ret	= -ENODEV;
			psz	= "dev";
			break;
		}

		// Begin initialization.
		psz	= _sio4_async_init_t_validate(init);

		if (psz)
		{
			ret	= -EINVAL;
		}
		else
		{
			memset(async, 0, sizeof(sio4_async_t));
			psz	= sio4_async_t_access(dev, async, SIO4_ASYNC_ACTION_INIT);
		}

		if (psz)
		{
			ret	= -EINVAL;
		}
		else
		{
			psz	= sio4_async_t_access(dev, async, SIO4_ASYNC_ACTION_VERIFY);
			async->osc.prog		= init->osc_prog;
			async->tx.bit_rate	= init->tx_bit_rate;
			async->rx.bit_rate	= init->rx_bit_rate;
			_tx_baud_compute(async);
			_rx_baud_compute(async);
		}

		if (psz)
			ret	= -EINVAL;

		lib_dev_dec(dev);
		break;
	}

	if (err)
		err[0]	= psz;

	return(ret);
}


