// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/utils/util_monosync.c $
// $Rev: 53094 $
// $Date: 2023-06-13 10:21:23 -0500 (Tue, 13 Jun 2023) $

// SIO4: Utilities: source file

#include "main.h"

// Useless code for the validation script.
//	if (verbose
//	if (verbose



/******************************************************************************
*
*	Function:	_monosync_zilog_config
*
*	Purpose:
*
*		Configure the given channel for Monosync operation per the given
*		parameters.
*
*	Arguments:
*
*		fd		The handle to the channel of interest.
*
*		data	The desired monosync configuration.
*
*	Returned:
*
*		>= 0	The number of errors encounterred.
*
******************************************************************************/

static int _monosync_zilog_config(int fd, sio4_monosync_t* data)
{
	int						errs	= 0;
	zilog_rx_baud_config_t	rx_baud;
	zilog_tx_baud_config_t	tx_baud;
	u32						value;

	tx_baud.txc_rate	= data->tx.usc_txc_rate;
	tx_baud.baudrate	= data->tx.baud_want;
	zilog_tx_baud_compute(&tx_baud);
	data->tx.baud_comp	= tx_baud.baudrate_comp;

	rx_baud.rxc_rate	= data->rx.usc_rxc_rate;
	rx_baud.baudrate	= data->rx.baud_want;
	rx_baud.encoding	= data->rx.encoding;

	if (data->rx.dpll_use)
		zilog_dpll_baud_compute(&rx_baud);
	else
		zilog_rx_baud_compute(&rx_baud);

	data->rx.baud_comp	= rx_baud.baudrate_comp;

	/*
	*	Configure the GSC specific registers.
	*
	*	BCR			DON'T TOUCH!!!
	*	ICR			DON'T TOUCH!!!
	*	ISR			DON'T TOUCH!!!
	*
	*	FRR			Ignore at this time.
	*	FDR			Ignore at this time.
	*	SDR			Ignore at this time.
	*/

	/*	CCR			Handled by cable configuration.
	*	CSR			Handled by cable configuration.
	*/

	/*
	*	Configure the Zilog specific registers.
	*
	*	BCR			DON'T TOUCH!!!
	*	DCCR			DON'T TOUCH!!!
	*	ICR			DON'T TOUCH!!!
	*	IVR			DON'T TOUCH!!!
	*	MISR			DON'T TOUCH!!!
	*	RICR			DON'T TOUCH!!!
	*	SICR			DON'T TOUCH!!!
	*	TICR			DON'T TOUCH!!!
	*
	*	CDR (RDR & TDR)		Ignore at this time.
	*	RCCR			Ignore at this time.
	*	TMCR			Ignore at this time.
	*	TMDR			Ignore at this time.
	*	TCCR			Ignore at this time.
	*/

	/*
	*	CMR.TxMode		= Monosync
	*	CMR.15			= data->tx.crc_send_on_underrun
	*	CMR.13			= data->tx.preamble_send
	*	CMR.12			= data->tx.sync_char_size
	*	CMR.RxMode		= Monosync
	*	CMR.5			= data->rx.sync_char_act
	*	CMR.4			= data->rx.sync_char_size
	*/

	value	= 0;
	value	|= SIO4_USC_CMR_TXMODE_ENCODE(SIO4_USC_CMR_MODE_MONOSYNC);
	value	|= SIO4_USC_CMR_MST_TXCRC_ENCODE(data->tx.crc_send_on_underrun);
	value	|= SIO4_USC_CMR_MST_TXPRE_ENCODE(data->tx.preamble_send);
	value	|= SIO4_USC_CMR_MST_SSIZE_ENCODE(data->tx.sync_char_size);
	value	|= SIO4_USC_CMR_RXMODE_ENCODE(SIO4_USC_CMR_MODE_MONOSYNC);
	value	|= SIO4_USC_CMR_MSR_SCACT_ENCODE(data->rx.sync_char_act);
	value	|= SIO4_USC_CMR_MSR_SSIZE_ENCODE(data->rx.sync_char_size);
	errs	+= reg_write(fd, SIO4_USC_CMR, value);

	/*
	*	IOCR.CTSMode		= drive low, REQUIRED, CTS device error
	*	IOCR.DCDMode		= carrier detect
	*	IOCR.TxRMode		= DMA request, REQUIRED
	*	IOCR.RxRMode		= DMA request, REQUIRED
	*	IOCR.TxDMode		= transmit data, REQUIRED
	*	IOCR.TxCMode		= input, REQUIRED
	*	IOCR.RxCMode		= input, REQUIRED
	*/

	value	= 0;
	value	|= SIO4_USC_IOCR_CTSMODE_ENCODE(SIO4_USC_IOCR_CTSMODE_LOW);
	value	|= SIO4_USC_IOCR_DCDMODE_ENCODE(SIO4_USC_IOCR_DCDMODE_DCD);
	value	|= SIO4_USC_IOCR_TXRMODE_ENCODE(SIO4_USC_IOCR_REQMODE_DMA);
	value	|= SIO4_USC_IOCR_RXRMODE_ENCODE(SIO4_USC_IOCR_REQMODE_DMA);
	value	|= SIO4_USC_IOCR_TXDMODE_ENCODE(SIO4_USC_IOCR_TXDMODE_OUTPUT);
	value	|= SIO4_USC_IOCR_TXCMODE_ENCODE(SIO4_USC_IOCR_TXCMODE_INPUT);
	value	|= SIO4_USC_IOCR_RXCMODE_ENCODE(SIO4_USC_IOCR_RXCMODE_INPUT);
	errs	+= reg_write(fd, SIO4_USC_IOCR, value);

	/*
	*	TC0R				= tx_baud.tc0r_value
	*/

	errs	+= reg_write(fd, SIO4_USC_TC0R, tx_baud.tc0r_value);

	/*
	*	TC1R				= rx_baud.tc1r_value
	*/

	errs	+= reg_write(fd, SIO4_USC_TC1R, rx_baud.tc1r_value);

	/*
	*	TCLR				= data->tx.frame_size_max
	*/

	errs	+= USC_TCLR_WRITE(fd, data->tx.frame_size_max);

	errs	+= reg_read(fd, -1, 0, SIO4_USC_CCAR, &value);
	value	&= ~SIO4_USC_CCAR_CMD_ENCODE(-1);
	value	|= SIO4_USC_CCAR_CMD_ENCODE(SIO4_USC_CCAR_CMD_LOAD_TCC);
	errs	+= reg_write(fd, SIO4_USC_CCAR, value);

	/*
	*	RCLR				= data->rx.frame_size_max
	*/

	errs	+= USC_RCLR_WRITE(fd, data->rx.frame_size_max);
	errs	+= reg_read(fd, -1, 0, SIO4_USC_CCAR, &value);
	value	&= ~SIO4_USC_CCAR_CMD_ENCODE(-1);
	value	|= SIO4_USC_CCAR_CMD_ENCODE(SIO4_USC_CCAR_CMD_LOAD_RCC);
	errs	+= reg_write(fd, SIO4_USC_CCAR, value);

	/*
	*	CMCR.CTR1Src		= rx_baud.ctr1_src
	*	CMCR.CTR0Src		= tx_baud.ctr0_src
	*	CMCR.BRG1Src		= rx_baud.brg1_src
	*	CMCR.BRG0Src		= tx_baud.brg0_src
	*	CMCR.DPLLSrc		= rx_baud.dpll_src
	*	CMCR.TxCLKSrc		= tx_baud.txclk_src
	*	CMCR.RxCLKSrc		= rx_baud.rxclk_src
	*/

	value	= 0;
	value	|= SIO4_USC_CMCR_CTR1SRC_ENCODE(rx_baud.ctr1_src);
	value	|= SIO4_USC_CMCR_CTR0SRC_ENCODE(tx_baud.ctr0_src);
	value	|= SIO4_USC_CMCR_BRG1SRC_ENCODE(rx_baud.brg1_src);
	value	|= SIO4_USC_CMCR_BRG0SRC_ENCODE(tx_baud.brg0_src);
	value	|= SIO4_USC_CMCR_DPLLSRC_ENCODE(rx_baud.dpll_src);
	value	|= SIO4_USC_CMCR_TXCLKSRC_ENCODE(tx_baud.txclk_src);
	value	|= SIO4_USC_CMCR_RXCLKSRC_ENCODE(rx_baud.rxclk_src);
	errs	+= reg_write(fd, SIO4_USC_CMCR, value);

	/*
	*	HCR.CTR0Div			= tx_baud.ctr0_div
	*	HCR.CTR1DSel		= rx_baud.ctr1_div
	*	HCR.CVOK			= ignore
	*	HCR.DPLLDiv			= rx_baud.dpll_div
	*	HCR.DPLLMode		= rx_baud.dpll_mode
	*	HCR.TxAMode			= Tx DMA Acknowledge, REQUIRED
	*	HCR.BRG1S			= continuous
	*	HCR.BRG1E			= rx_baud.brg1_enable
	*	HCR.RxAMode			= Rx DMA Acknowledge, REQUIRED
	*	HCR.BRG0S			= continuous
	*	HCR.BRG0E			= tx_baud.brg0_enable
	*/

	value	= 0;
	value	|= SIO4_USC_HCR_CTR0DIV_ENCODE(tx_baud.ctr0_div);
	value	|= SIO4_USC_HCR_CTR1DSEL_ENCODE(rx_baud.ctr1_div_sel);
	value	|= SIO4_USC_HCR_DPLLDIV_ENCODE(rx_baud.dpll_div);
	value	|= SIO4_USC_HCR_DPLLMODE_ENCODE(rx_baud.dpll_mode);
	value	|= SIO4_USC_HCR_TXAMODE_ENCODE(SIO4_USC_HCR_AMODE_DMA_ACK);
	value	|= SIO4_USC_HCR_BRG1S_ENCODE(SIO4_USC_HCR_BRGS_CONTINUOUS);
	value	|= SIO4_USC_HCR_BRG1E_ENCODE(rx_baud.brg1_enable);
	value	|= SIO4_USC_HCR_RXAMODE_ENCODE(SIO4_USC_HCR_AMODE_DMA_ACK);
	value	|= SIO4_USC_HCR_BRG0E_ENCODE(tx_baud.brg0_enable);
	errs	+= reg_write(fd, SIO4_USC_HCR, value);

	/*
	*	CCR.TxCtrlBlk		= data->tx_control_block_size
	*	CCR.Wait4TxTrig		= ignore
	*	CCR.FlagPreamble	= ignore
	*	CCR.TxPreL			= data->tx.preamble_size
	*	CCR.TxPrePat		= data->tx.preamble_pattern
	*	CCR.RxStatBlk		= data->rx.status_block_size
	*	CCR.Wait4RxTrig		= ignore
	*/

	value	= 0;
	value	|= SIO4_USC_CCR_TXCTRLBLK_ENCODE(data->tx.control_block_size);
	value	|= SIO4_USC_CCR_TXPREL_ENCODE(data->tx.preamble_size);
	value	|= SIO4_USC_CCR_TXPREPAT_ENCODE(data->tx.preamble_pattern);
	value	|= SIO4_USC_CCR_RXSTATBLK_ENCODE(data->rx.status_block_size);
	errs	+= reg_write(fd, SIO4_USC_CCR, value);

	/*
	*	TCSR.TCmd			= 0 (none, ignore)
	*	TCSR.UnderWait		= 0 (ignore)
	*	TCSR.TxIdle			= data->tx.idle_output
	*	TCSR.PreSent		= 0 (ignore)
	*	TCSR.IdleSent		= 0 (ignore)
	*	TCSR.AbortSent		= 0 (ignore)
	*	TCSR.EOF/EOM Sent	= 0 (ignore)
	*	TCSR.CRCSent		= 0 (ignore)
	*	TCSR.AllSent		= 0 (ignore)
	*	TCSR.TxUnder		= 0 (ignore)
	*	TCSR.TxEmpty		= 0 (ignore)
	*/

	value	= 0;
	value	|= SIO4_USC_TCSR_TXIDLE_ENCODE(data->tx.idle_output);
	errs	+= reg_write(fd, SIO4_USC_TCSR, value);

	/*
	*	CCAR.RTcmd			= no command
	*	CCAR.RTReset		= release
	*	CCAR.RTMode			= unchanged
	*	CCAR.ChanLoad		= ignore
	*	CCAR.BW				= ignore
	*	CCAR.RegAddr		= ignore
	*	CCAR.UL				= ignore
	*/

	value	= 0;
	value	|= SIO4_USC_CCAR_CMD_ENCODE(SIO4_USC_CCAR_CMD_NONE);
	value	|= SIO4_USC_CCAR_RTRESET_ENCODE(SIO4_USC_CCAR_RTRESET_NO);
	errs	+= reg_write(fd, SIO4_USC_CCAR, value);

	/*
	*	CCSR.DPLLEdge		= data->rx.dpll_sync_edge
	*	CCSR.all others		= 0 (ignore)
	*/

	value	= 0;
	value	|= SIO4_USC_CCSR_DPLLEDGE_ENCODE(data->rx.dpll_sync_edge);
	errs	+= reg_write(fd, SIO4_USC_CCSR, value);

	/*
	*	TSR					= data->tx.sync_value
	*/

	errs	+= reg_write(fd, SIO4_USC_TSR, data->tx.sync_value);

	/*
	*	RSR					= data->rx.sync_value
	*/

	value	= SIO4_USC_RSR_MONOSYNC_ENCODE(data->rx.sync_value);
	errs	+= reg_write(fd, SIO4_USC_RSR, value);

	/*
	*	RMR.RxDecode		= data->rx.encoding
	*	RMR.RxCRCType		= data->rx.crc_type
	*	RMR.RxCRCStart		= data->rx.crc_start_value
	*	RMR.RxCRCEnab		= data->rx.crc_enable
	*	RMR.QAbort			= ignore
	*	RMR.RxParType		= data->rx.parity_type
	*	RMR.RxParEnab		= data->rx.parity_enable
	*	RMR.RxLength		= data->rx.char_size
	*	RMR.RxEnab			= data->rx.enable
	*/

	value	= data->rx.enable
			? SIO4_USC_ENAB_ON_NOW
			: SIO4_USC_ENAB_OFF_NOW;
	value	= SIO4_USC_RMR_RXENAB_ENCODE(value);
	value	|= SIO4_USC_RMR_RXDECODE_ENCODE(data->rx.encoding);
	value	|= SIO4_USC_RMR_RXCRCTYPE_ENCODE(data->rx.crc_type);
	value	|= SIO4_USC_RMR_RXCRCSTART_ENCODE(data->rx.crc_start_value);
	value	|= SIO4_USC_RMR_RXCRCENAB_ENCODE(data->rx.crc_enable);
	value	|= SIO4_USC_RMR_RXPARTYPE_ENCODE(data->rx.parity_type);
	value	|= SIO4_USC_RMR_RXPARENAB_ENCODE(data->rx.parity_enable);
	value	|= SIO4_USC_RMR_RXLENGTH_ENCODE(data->rx.char_size);
	errs	+= reg_write(fd, SIO4_USC_RMR, value);

	/*
	*	RCSR.RCmd			= Enter Hunt Mode
	*	RCSR.RxResidue		= ignore
	*	RCSR.ShortF/CVType	= ignore
	*	RCSR.ExitedHunt		= ignore
	*	RCSR.IdleRcvd		= ignore
	*	RCSR.Break/Abort	= ignore
	*	RCSR.RxBound		= ignore
	*	RCSR.CRCE/FE		= ignore
	*	RCSR.Abort/PE		= ignore
	*	RCSR.RxOver			= ignore
	*	RCSR.RxAvail		= ignore
	*/

	value	= 0;
	value	|= SIO4_USC_RCSR_RCMD_ENCODE(SIO4_USC_RCSR_RCMD_HUNT);
	errs	+= reg_write(fd, SIO4_USC_RCSR, value);

	/*
	*	TMR.TxEncode		= data->tx.encoding
	*	TMR.TxCRCType		= data->tx.crc_type
	*	TMR.TxCRCStart		= data->tx.crc_start_value
	*	TMR.TxCRCEnab		= data->tx.crc_enable
	*	TMR.TxCRCatEnd		= data->tx.crc_send_at_eof
	*	TMR.TxParType		= data->tx.parity_type
	*	TMR.TxParEnab		= data->tx.parity_enable
	*	TMR.TxLength		= data->tx.char_size
	*	TMR.TxEnab			= data->tx.enable
	*/

	value	= data->tx.enable
		? SIO4_USC_ENAB_ON_NOW
		: SIO4_USC_ENAB_OFF_NOW;
	value	= SIO4_USC_TMR_TXENAB_ENCODE(value);
	value	|= SIO4_USC_TMR_TXENCODE_ENCODE(data->tx.encoding);
	value	|= SIO4_USC_TMR_TXCRCTYPE_ENCODE(data->tx.crc_type);
	value	|= SIO4_USC_TMR_TXCRCSTART_ENCODE(data->tx.crc_start_value);
	value	|= SIO4_USC_TMR_TXCRCENAB_ENCODE(data->tx.crc_enable);
	value	|= SIO4_USC_TMR_TXCRCCATEND_ENCODE(data->tx.crc_send_at_eof);
	value	|= SIO4_USC_TMR_TXPARTYPE_ENCODE(data->tx.parity_type);
	value	|= SIO4_USC_TMR_TXPARENAB_ENCODE(data->tx.parity_enable);
	value	|= SIO4_USC_TMR_TXLENGTH_ENCODE(data->tx.char_size);
	errs	+= reg_write(fd, SIO4_USC_TMR, value);

	return(errs);
}



/******************************************************************************
*
*	Function:	sio4_monosync_config
*
*	Purpose:
*
*		Configure the given channel for monosync operation per the given
*		parameters.
*
*	Arguments:
*
*		fd		The handle to the channel of interest.
*
*		data	The desired Monosync configuration.
*
*	Returned:
*
*		>= 0	The number of errors encounterred.
*
******************************************************************************/

int sio4_monosync_config(int fd, sio4_monosync_t* data)
{
	int		cable;
	int		enable;
	int		errs	= 0;
	int		mode;
	s32		scd;
	long	want;
	int		xceiver;

	// Initialize the channel.
	errs	+= channel_init(fd, 0);

	// Apply the I/O settings.
	errs	+= feature_test__bcr_scd(fd, &scd, 0);

	if (	(data->rx.io_mode_force) ||
		(data->rx.io_mode != SIO4_IO_MODE_DMDMA))
	{
		mode	= data->rx.io_mode;
	}
	else
	{
		mode	= SIO4_IO_MODE_PIO;
	}

	errs	+= rx_io_mode_config(fd, mode, NULL, 0);

	if (	(data->tx.io_mode_force) ||
		(data->tx.io_mode != SIO4_IO_MODE_DMDMA))
	{
		mode	= data->tx.io_mode;
	}
	else
	{
		mode	= SIO4_IO_MODE_PIO;
	}

	errs	+= tx_io_mode_config(fd, mode, NULL, 0);
	errs	+= timeout_set_read(fd, data->rx.timeout, 0);
	errs	+= timeout_set_write(fd, data->tx.timeout, 0);

	// Initialize the FIFOs.
	errs	+= rx_fifo_full_cfg_chan(fd, data->rx.fifo_full_chan, NULL, 0);
	errs	+= rx_fifo_full_cfg_glb(fd, data->rx.fifo_full_glb, NULL, 0);
	errs	+= rx_fifo_ae_config(fd, data->rx.ae, NULL, 0);
	errs	+= rx_fifo_af_config(fd, data->rx.af, NULL, 0);
	errs	+= tx_fifo_ae_config(fd, data->tx.ae, NULL, 0);
	errs	+= tx_fifo_af_config(fd, data->tx.af, NULL, 0);

	// Disable the cable for now.
	errs	+= cable_config(fd, SIO4_CABLE_CONFIG_DISABLE, NULL, 0);

	// Initialize the oscillator.
	want	= data->tx.enable ? data->tx.baud_want : data->rx.baud_want;
	data->osc.ref_comp	= data->osc.ref_default;
	errs	+= osc_ref_compute(fd, &data->osc.ref_comp, want, 0);
	errs	+= osc_setup(fd, data->osc.ref_comp, &data->osc.ref_got, 0);

	if (data->rx.usc_rxc_config == SIO4_RXC_USC_CONFIG_IN_PRG_CLK)
		data->rx.usc_rxc_rate	= data->osc.ref_comp;

	if (data->tx.usc_txc_config == SIO4_TXC_USC_CONFIG_IN_PRG_CLK)
		data->tx.usc_txc_rate	= data->osc.ref_comp;

	// Configure the Zilog.
	errs	+= _monosync_zilog_config(fd, data);
	errs	+= rxc_usc_config(fd, data->rx.usc_rxc_config, NULL, 0);
	errs	+= txc_usc_config(fd, data->tx.usc_txc_config, NULL, 0);
	errs	+= zilog_mode_config(fd, !data->loopback.enable, 0);

	// Configure the cable interface. THIS MUST COME AFTER UART SETUP!
	errs	+= rx_enable(fd, 0);
	errs	+= cts_cable_config(fd, data->cable.cts, NULL, 0);
	errs	+= dcd_cable_config(fd, data->cable.dcd, NULL, 0);
	errs	+= tx_cable_data_config(fd, data->tx.cable_data, NULL, 0);
	errs	+= tx_cable_clock_config(fd, data->tx.cable_clock, NULL, 0);
	enable	= data->loopback.enable ? 0 : data->cable.dcedte.enable;
	errs	+= dcedte_config(	fd,
								enable,
								data->cable.dcedte.dce,
								data->loopback.enable,
								0);
	cable	= data->loopback.enable
		? SIO4_CABLE_CONFIG_DISABLE
		: data->cable.legacy.config;
	errs	+= cable_config(fd, cable, NULL, 0);
	xceiver	= data->loopback.enable
		? SIO4_MP_PROT_DISABLE
		: data->cable.xceiver;
	errs	+= mp_config(fd, xceiver, NULL, 0);
	errs	+= rx_enable(fd, 1);

	// Make sure the FIFOs are empty.
	errs	+= fifo_reset(fd, TX_FIFO, 0);
	errs	+= channel_command_send(fd, TX_FIFO_PURGE_CMD, 0);
	os_sleep_ms(10);
	errs	+= channel_command_send(fd, RX_FIFO_PURGE_CMD, 0);
	errs	+= fifo_reset(fd, RX_FIFO, 0);

	// Clear the USC's Tx Data interrupt caused by the Tx Purge.
	errs	+= reg_mod(fd, SIO4_USC_DCCR, 0x44, 0xC4);

	return(errs);
}



/******************************************************************************
*
*	Function:	monosync_t_init
*
*	Purpose:
*
*		initialize the given structure.
*
*	Arguments:
*
*		data	The structure to initialize.
*
*	Returned:
*
*		None.
*
******************************************************************************/

void sio4_monosync_t_init(sio4_monosync_t* data)
{
	memset(data, 0, sizeof(sio4_monosync_t));

	data->cable.dcedte.enable		= 0;
	data->cable.dcedte.dce			= 0;
	data->cable.legacy.config		= SIO4_CABLE_CONFIG_TXUPR_RXLWR;
	data->cable.cts					= SIO4_CTS_CABLE_CONFIG_DISABLE;
	data->cable.dcd					= SIO4_DCD_CABLE_CONFIG_DISABLE;
	data->cable.xceiver				= SIO4_MP_PROT_RS_422_485;

	data->loopback.enable			= 0;

	data->osc.ref_comp				= 0;
	data->osc.ref_default			= SIO4_OSC_FREQ_REF_DEFAULT;
	data->osc.ref_got				= 0;

	data->rx.ae						= 16;
	data->rx.af						= 16;
	data->rx.baud_comp				= 0;
	data->rx.baud_want				= 115200L;
	data->rx.char_size				= SIO4_USC_CHAR_LENGTH_8;
	data->rx.crc_enable				= 0;
	data->rx.crc_start_value		= 0;
	data->rx.crc_type				= SIO4_USC_CRCTYPE_16;
	data->rx.dpll_sync_edge			= SIO4_USC_CCSR_DPLL_EDGE_UP_DN;
	data->rx.dpll_use				= 1;
	data->rx.enable					= 1;
	data->rx.encoding				= SIO4_USC_ENCODING_BIPH_MARK;
	data->rx.fifo_full_chan			= SIO4_RX_FIFO_FULL_CFG_CHAN_OVER;
	data->rx.fifo_full_glb			= SIO4_RX_FIFO_FULL_CFG_GLB_OVER;
	data->rx.frame_size_max			= 512;
	data->rx.io_mode				= SIO4_IO_MODE_DEFAULT;
	data->rx.io_mode_force			= 0;
	data->rx.parity_enable			= 0;
	data->rx.parity_type			= SIO4_USC_PARTYPE_EVEN;
	data->rx.status_block_size		= SIO4_USC_CCR_BLOCK_SIZE_NONE;
	data->rx.sync_char_act			= SIO4_USC_CMR_MSR_SCACT_TOSS;
	data->rx.sync_char_size			= SIO4_USC_CMR_MS_SSIZE_8;
	data->rx.sync_value				= 0x16;
	data->rx.timeout				= SIO4_TIMEOUT_DEFAULT;
	data->rx.usc_rxc_config			= SIO4_RXC_USC_CONFIG_IN_PRG_CLK;
	data->rx.usc_rxc_rate			= SIO4_OSC_FREQ_REF_DEFAULT;

	data->tx.ae						= 16;
	data->tx.af						= 16;
	data->tx.baud_comp				= 0;
	data->tx.baud_want				= 115200L;
	data->tx.cable_clock			= SIO4_TX_CABLE_CLOCK_CONFIG_USC_TC;
	data->tx.cable_data				= SIO4_TX_CABLE_DATA_CONFIG_USC_TXD;
	data->tx.char_size				= SIO4_USC_CHAR_LENGTH_8;
	data->tx.control_block_size		= SIO4_USC_CCR_BLOCK_SIZE_NONE;
	data->tx.crc_enable				= 0;
	data->tx.crc_send_at_eof		= 0;
	data->tx.crc_send_on_underrun	= 0;
	data->tx.crc_start_value		= 0;
	data->tx.crc_type				= SIO4_USC_CRCTYPE_16;
	data->tx.enable					= 1;
	data->tx.encoding				= SIO4_USC_ENCODING_BIPH_MARK;
	data->tx.frame_size_max			= 512;
	data->tx.idle_output			= SIO4_USC_TCSR_TXIDLE_DEFAULT;
	data->tx.io_mode				= SIO4_IO_MODE_DEFAULT;
	data->tx.io_mode_force			= 0;
	data->tx.parity_enable			= 0;
	data->tx.parity_type			= SIO4_USC_PARTYPE_EVEN;
	data->tx.preamble_pattern		= SIO4_USC_CCR_TXPREPAT_0;
	data->tx.preamble_send			= 0;
	data->tx.preamble_size			= SIO4_USC_CCR_TXPREL_16;
	data->tx.sync_char_size			= SIO4_USC_CMR_MS_SSIZE_8;
	data->tx.sync_value				= 0x16;
	data->tx.timeout				= SIO4_TIMEOUT_DEFAULT;
	data->tx.usc_txc_config			= SIO4_TXC_USC_CONFIG_IN_PRG_CLK;
	data->tx.usc_txc_rate			= SIO4_OSC_FREQ_REF_DEFAULT;
}



