// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/sync/samples/txrate/main.c $
// $Rev: 53087 $
// $Date: 2023-06-13 10:12:24 -0500 (Tue, 13 Jun 2023) $

// SIO4: SYNC Protocol Library: Sample Application: source file

#include "main.h"



// macros *********************************************************************

#define	FR_MAX	SIO4_FREQ_MAX
#define	FR_MIN	SIO4_FREQ_MIN



//*****************************************************************************
static int _parse_args(int argc, char** argv, args_t* args)
{
	const gsc_arg_item_t	list[]	=
	{
		//	type				var					arg			values				desc
		{	GSC_ARG_S32_FLAG,	&args->pd_max,		"-b11",		{ 11 },				"The firmware post divider is 11-bits wide (default)."		},
		{	GSC_ARG_S32_FLAG,	&args->pd_max,		"-b15",		{ 15 },				"The firmware post divider is 15-bits wide (default)."		},
		{	GSC_ARG_S32_MIN,	&args->index,		"-B",		{ 0 },				"The zero based index of the board to access."				},
		{	GSC_ARG_S32_FLAG,	&args->init,		"-I",		{ 1 },				"Initialize channels not programmed."						},
		{	GSC_ARG_S32_FLAG,	&args->reset,		"-I",		{ 0 },				NULL														},
		{	GSC_ARG_S32_FLAG,	&args->init,		"-R",		{ 0 },				"Reset channels not programmed (default)."					},
		{	GSC_ARG_S32_FLAG,	&args->reset,		"-R",		{ 1 },				NULL														},
		{	GSC_ARG_S32_FLAG,	&args->scan,		"-s",		{ 1 },				"Perform a sequential scan from -sb# to -se#."				},
		{	GSC_ARG_S32_RANGE,	&args->begin,		"-sb",		{ FR_MIN, FR_MAX },	"Begin a sequential scan at # Hz (default: 19,150)."		},
		{	GSC_ARG_S32_RANGE,	&args->end,			"-se",		{ FR_MIN, FR_MAX },	"End a sequential scan at # Hz (default: 19,250)."			},
		{	GSC_ARG_DEV_INDEX,	&args->bit_rate[0],	"chan1st",	{ 4 },				"Program board's 1st channel for # b/S (default: 19,200)."	},
		{	GSC_ARG_DEV_INDEX,	&args->bit_rate[1],	"chan2nd",	{ 4 },				"Program board's 2nd channel for # b/S (default: 19,200)."	},
		{	GSC_ARG_DEV_INDEX,	&args->bit_rate[2],	"chan3rd",	{ 4 },				"Program board's 3rd channel for # b/S (default: 19,200)."	},
		{	GSC_ARG_DEV_INDEX,	&args->bit_rate[3],	"chan4th",	{ 4 },				"Program board's 4th channel for # b/S (default: 19,200)."	}
	};

	const gsc_arg_set_t	set	=
	{
		/* name	*/	"txrate",
		/* desc	*/	"Transfer clock rate calculation test.",
		/* qty	*/	SIZEOF_ARRAY(list),
		/* list	*/	list
	};

	int	errs;
	s32	tmp;

	memset(args, 0, sizeof(args_t));
	args->begin			= 19150;
	args->bit_rate[0]	= 19200;
	args->bit_rate[1]	= 19200;
	args->bit_rate[2]	= 19200;
	args->bit_rate[3]	= 19200;
	args->end			= 19250;
	args->fref			= 20000000;
	args->pd_max		= 15;
	args->reset			= 1;

	errs	= gsc_args_parse(argc, argv, &set);

	if ((args->scan) && (args->begin > args->end))
	{
		gsc_label(NULL);
		printf("SWAPPING -sb# and -se#\n");
		tmp			= args->begin;
		args->begin	= args->end;
		args->end	= tmp;
	}

	return(errs);
}



//*****************************************************************************
static int _show_access_index(const args_t* args)
{
	int	errs;

	for (;;)	// A convenience loop.
	{
		gsc_label("Accessing Board");
		printf("%d", args->index);

		if (args->index < 0)
		{
			errs	= 1;
			printf(" FAIL <---  (Invalid board index.)\n");
			break;
		}

		errs	= 0;
		printf("\n");
		break;
	}

	return(errs);
}



//*****************************************************************************
static void _show_time_stamp(void)
{
	const char*	psz;
	struct tm*	stm;
	time_t		tt;

	time(&tt);
	stm	= localtime(&tt);
	psz	= (char*) asctime(stm);
	gsc_label("Time Stamp");
	printf("%s", psz);
}



/******************************************************************************
*
*	Function:	main
*
*	Purpose:
*
*		Control the overall flow of the application.
*
*	Arguments:
*
*		argc			The number of command line arguments.
*
*		argv			The list of command line arguments.
*
*	Returned:
*
*		EXIT_SUCCESS	No errors were encounterred.
*		EXIT_FAILURE	One or more problems were encounterred.
*
******************************************************************************/

int main(int argc, char** argv)
{
	args_t	args;
	int		errs;
	time_t	exec		= time(NULL);
	long	failures	= 0;
	long	hours;
	long	mins;
	time_t	now;
	long	passes		= 0;
	int		ret;
	long	secs;
	time_t	test;
	long	tests		= 0;

	for (;;)	// A convenience loop.
	{
		gsc_label_init(24);
		test	= time(NULL);
		errs	= _parse_args(argc, argv, &args);

		// Introduction

		os_id_host();
		_show_time_stamp();
		_show_access_index(&args);
		errs		+= id_driver();
		args.qty	= count_boards();

		if (args.qty <= 0)
			errs++;

		// Open access to device (here, only the first)

		errs	+= chan_open(args.index * 4 + 1, &args.fd[0], 1);

		if (errs == 0)
		{
			// Identify the device, then close access
			errs	+= sio4_id_device(args.fd[0], -1, 1);
			errs	+= chan_close(args.index * 4 + 1, args.fd[0], 1);
		}

		printf("\n");

		// Open access to devices (here, all four)

		if (errs == 0)
		{
			errs	+= channels_open(&args, args.fd);

			if ((args.fd[0] >= 0) ||
				(args.fd[1] >= 0) ||
				(args.fd[2] >= 0) ||
				(args.fd[3] >= 0))
			{
				// Device identification

				// Perform testing

				errs	+= perform_tests(&args);

				// Close device access

				errs	+= channels_close(&args);
			}
		}

		// End of processing.

		now	= time(NULL);
		tests++;

		if (errs)
		{
			failures++;
			printf(	"\nRESULTS: FAIL <---  (%d error%s)",
					errs,
					(errs == 1) ? "" : "s");
		}
		else
		{
			passes++;
			printf("\nRESULTS: PASS");
		}

		secs	= (long) (now - test);
		hours	= secs / 3600;
		secs	= secs % 3600;
		mins	= secs / 60;
		secs	= secs % 60;
		printf(" (duration %ld:%ld:%02ld)\n", hours, mins, secs);

		secs	= (long) (now - exec);
		hours	= secs / 3600;
		secs	= secs % 3600;
		mins	= secs / 60;
		secs	= secs % 60;
		printf(	"SUMMARY: tests %ld, pass %ld, fail %ld"
				" (duration %ld:%ld:%02ld)\n\n",
				tests,
				passes,
				failures,
				hours,
				mins,
				secs);
		break;
	}

	ret	= ((args.qty <= 0) || (errs) || (failures)) ? EXIT_FAILURE : EXIT_SUCCESS;
	return(ret);
}


