// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/sync/samples/syncc2c/main.c $
// $Rev: 53085 $
// $Date: 2023-06-13 10:11:43 -0500 (Tue, 13 Jun 2023) $

// SIO4: SYNC Protocol Library: Sample Application: source file

#include "main.h"



//*****************************************************************************
static int _parse_args(int argc, char** argv, args_t* args)
{
	const gsc_arg_item_t	list[]	=
	{
		//	type				var						arg		values				desc
		{	GSC_ARG_S32_RANGE,	&args->baud,			"-b",	{ 10, 10000000},	"Transfer data at # b/S (default: 100,000)."	},
		{	GSC_ARG_S32_FLAG,	&args->continuous,		"-c",	{ -1 },				"Continue testing until an error occurs."		},
		{	GSC_ARG_S32_FLAG,	&args->continuous,		"-C",	{ +1 },				"Continue testing even if errors occur."		},
		{	GSC_ARG_S32_MIN,	&args->minute_limit,	"-m",	{ 1 },				"Continue testing for up to # minutes."			},
		{	GSC_ARG_S32_MIN,	&args->test_limit,		"-n",	{ 1 },				"Continue testing for up to # iterations."		},
		{	GSC_ARG_S32_RANGE,	&args->seconds,			"-s",	{ 1, 3600 },		"Transfer data for # seconds (default: 30)."	},
		{	GSC_ARG_DEV_INDEX,	&args->index_tx,		"tx#",	{ 2 },				"The index of the transmit device to access."	},
		{	GSC_ARG_DEV_INDEX,	&args->index_rx,		"rx#",	{ 2 },				"The index of the receive device to access."	}
	};

	const gsc_arg_set_t	set	=
	{
		/* name	*/	"syncc2c",
		/* desc	*/	"Perform SYNChronous channel-to-channel data transfer.",
		/* qty	*/	SIZEOF_ARRAY(list),
		/* list	*/	list
	};

	int	errs;

	memset(args, 0, sizeof(args_t));
	args->baud		= 100000;
	args->index_rx	= 1;
	args->index_tx	= 1;
	args->seconds	= 30;

	errs	= gsc_args_parse(argc, argv, &set);

	return(errs);
}



//*****************************************************************************
static int _show_access_index(const args_t* args)
{
	int	errs;

	for (;;)	// A convenience loop.
	{
		gsc_label("Accessing Devices");
		printf("%d -> %d", args->index_tx, args->index_rx);

		if (args->index_tx <= 0)
		{
			errs	= 1;
			printf(" FAIL <---  (Invalid Tx device index.)\n");
			break;
		}

		if (args->index_rx <= 0)
		{
			errs	= 1;
			printf(" FAIL <---  (Invalid Rx device index.)\n");
			break;
		}

		errs	= 0;
		printf("\n");
		break;
	}

	return(errs);
}



//*****************************************************************************
static void _show_time_stamp(void)
{
	const char*	psz;
	struct tm*	stm;
	time_t		tt;

	time(&tt);
	stm	= localtime(&tt);
	psz	= (char*) asctime(stm);
	gsc_label("Time Stamp");
	printf("%s", psz);
}



/******************************************************************************
*
*	Function:	main
*
*	Purpose:
*
*		Control the overall flow of the application.
*
*	Arguments:
*
*		argc			The number of command line arguments.
*
*		argv			The list of command line arguments.
*
*	Returned:
*
*		EXIT_SUCCESS	No errors were encounterred.
*		EXIT_FAILURE	One or more problems were encounterred.
*
******************************************************************************/

int main(int argc, char** argv)
{
	args_t	args;
	int		errs;
	time_t	exec		= time(NULL);
	long	failures	= 0;
	long	hours;
	long	mins;
	time_t	now;
	long	passes		= 0;
	int		ret;
	long	secs;
	time_t	t_limit;
	time_t	test;
	long	tests		= 0;

	for (;;)	// A convenience loop.
	{
		gsc_label_init(28);
		test	= time(NULL);
		errs	= _parse_args(argc, argv, &args);

		// Introduction

		t_limit	= exec + ((time_t) args.minute_limit * 60);
		os_id_host();
		_show_time_stamp();
		_show_access_index(&args);
		errs		+= id_driver();
		args.qty	= count_boards();

		if (args.qty <= 0)
			errs++;

		// Open access to devices

		if (errs == 0)
			errs	= chan_open(args.index_tx, &args.fd_tx, 1);

		if (errs)
			;
		else if (args.index_tx != args.index_rx)
			errs	= chan_open(args.index_rx, &args.fd_rx, 1);
		else
			args.fd_rx	= args.fd_tx;

		if (errs)
			args.continuous	= 0;

		if (errs == 0)
		{
			// Device identification

			errs	= sio4_id_device(args.fd_tx, args.index_tx, 1);

			if (args.index_rx != args.index_tx)
				errs	+= sio4_id_device(args.fd_rx, args.index_rx, 1);

			// Perform testing

			errs	+= perform_tests(&args, &args.skip);
		}

		// Close device access

		if (args.fd_tx >= 0)
			errs	+= chan_close(args.index_tx, args.fd_tx, 1);

		if ((args.fd_rx >= 0) && (args.fd_rx != args.fd_tx))
			errs	+= chan_close(args.index_rx, args.fd_rx, 1);

		// End of processing.

		now	= time(NULL);
		tests++;

		if (errs)
		{
			failures++;
			printf(	"\nRESULTS: FAIL <---  (%d error%s)",
					errs,
					(errs == 1) ? "" : "s");
		}
		else if (args.skip)
		{
			printf("\nRESULTS: SKIPPED");
		}
		else
		{
			passes++;
			printf("\nRESULTS: PASS");
		}

		secs	= (long) (now - test);
		hours	= secs / 3600;
		secs	= secs % 3600;
		mins	= secs / 60;
		secs	= secs % 60;
		printf(" (duration %ld:%ld:%02ld)\n", hours, mins, secs);

		secs	= (long) (now - exec);
		hours	= secs / 3600;
		secs	= secs % 3600;
		mins	= secs / 60;
		secs	= secs % 60;
		printf(	"SUMMARY: tests %ld, pass %ld, fail %ld"
				" (duration %ld:%ld:%02ld)\n\n",
				tests,
				passes,
				failures,
				hours,
				mins,
				secs);

		if (args.skip)
			break;

		if (args.continuous == 0)
			break;

		if ((args.continuous < 0) && (errs))
			break;

		if ((args.minute_limit > 0) && (now >= t_limit))
			break;

		if ((args.test_limit > 0) && (tests >= args.test_limit))
			break;
	}

	ret	= ((args.qty <= 0) || (errs) || (failures)) ? EXIT_FAILURE : EXIT_SUCCESS;
	return(ret);
}


