// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/sync/lib/tx.c $
// $Rev: 48522 $
// $Date: 2020-11-23 18:13:34 -0600 (Mon, 23 Nov 2020) $

// SIO4: SYNC Protocol Library: source file

#include "main.h"



/******************************************************************************
*
*	Function:	sio4_sync_tx_get
*
*	Purpose:
*
*		Retrieve the current Tx specific SYNC configuration settings.
*
*	Arguments:
*
*		fd		The file rescriptor for the SIO4 to access.
*
*		tx		Record the settings here.
*
*	Returned:
*
*		0		All went well.
*		> 0		An appropriate error status.
*
******************************************************************************/

int sio4_sync_tx_get(int fd, sio4_sync_tx_t* tx)
{
	__u32	csr;
	__u32	psrcr	= 0;
	int		ret;
	__u32	tcr		= 0;

	ret	= sync_reg_read(fd, SIO4_GSC_CSR, &csr);

	if (ret == 0)
		ret	= sync_reg_read(fd, SIO4_GSC_PSRCR, &psrcr);

	if (ret == 0)
		ret	= sync_reg_read(fd, SIO4_GSC_TCR, &tcr);

	if (ret == 0)
	{
		tx->enable		= (csr & SIO4_SYNC_CSR_TX_ENABLE) ? 1 : 0;
		tx->auto_dis	= (csr & SIO4_SYNC_CSR_TX_AUTO_DIS) ? 1 : 0;
		tx->lsbf		= (csr & SIO4_SYNC_CSR_TX_LSB_FIRST) ? 1 : 0;
		tx->word_size	= tcr & 0xFFFF;
		tx->gap_size	= tcr >> 16;

		// Tx Clock
		tx->clock.idle	= (psrcr & SIO4_SYNC_PSRCR_TXC_IDLE) ? 1 : 0;

		if (psrcr & 0x4)
		{
			// The signal is configured as GPIO.
			tx->clock.clock	= 0;
			tx->clock.ext	= 0;
			tx->clock.high	= (psrcr & 1) ? 1 : 0;
		}
		else if (psrcr & 0x2)
		{
			// The external clock is being used.
			tx->clock.clock	= 1;
			tx->clock.ext	= 1;
			tx->clock.high	= (psrcr & 1) ? 0 : 1;
		}
		else
		{
			// The internal clock / 2 is being used.
			tx->clock.clock	= 1;
			tx->clock.ext	= 0;
			tx->clock.high	= (psrcr & 1) ? 0 : 1;
		}

		// Tx Envelope

		if (psrcr & 0x20)
		{
			// The signal is configured as GPIO.
			tx->env.env		= 0;
			tx->env.high	= (psrcr & 0x10) ? 1 : 0;
		}
		else
		{
			// The signal is configured as Tx Envelope.
			tx->env.env		= 1;
			tx->env.high	= (psrcr & 0x10) ? 0 : 1;
		}

		// Tx Data

		if (psrcr & 0x100)
		{
			// The signal is configured as GPIO.
			tx->data.data	= 0;
			tx->data.high	= (psrcr & 0x40) ? 1 : 0;
		}
		else
		{
			// The signal is configured as Tx Data.
			tx->data.data	= 1;
			tx->data.high	= (psrcr & 0x40) ? 0 : 1;
		}

		// Tx Aux Clock

		if (psrcr & 0x400)
		{
			// The signal is configured as GPIO.
			tx->aux_clock.enable	= 1;
			tx->aux_clock.clock		= 0;
			tx->aux_clock.high		= (psrcr & 0x200) ? 1 : 0;
		}
		else if (psrcr & 0x200)
		{
			// The signal is configured as a clock.
			tx->aux_clock.enable	= 1;
			tx->aux_clock.clock		= 1;
			tx->aux_clock.high		= 0;
		}
		else
		{
			// The signal is disabled.
			tx->aux_clock.enable	= 0;
			tx->aux_clock.clock		= 0;
			tx->aux_clock.high		= 0;
		}

		// Tx Spare

		if (psrcr & 0x1000)
		{
			// The signal is configured as GPIO.
			tx->spare.enable	= 1;
			tx->spare.high		= (psrcr & 0x800) ? 1 : 0;
		}
		else
		{
			// The signal is disable/tri-stated.
			tx->spare.enable	= 0;
			tx->spare.high		= 0;
		}
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	sio4_sync_tx_set
*
*	Purpose:
*
*		Update the current Tx specific SYNC configuration.
*
*	Arguments:
*
*		fd		The file rescriptor for the SIO4 to access.
*
*		tx		The setting to apply.
*
*	Returned:
*
*		0		All went well.
*		> 0		An appropriate error status.
*
******************************************************************************/

int sio4_sync_tx_set(int fd, const sio4_sync_tx_t* tx)
{
	__u32	mask;
	__u32	reg;
	int		ret;

	// Channel Pin Source Register
	reg		= 0;
	mask	= 0;

	// Tx Clock
	reg		|= tx->clock.idle ? SIO4_SYNC_PSRCR_TXC_IDLE : 0;
	mask	|= SIO4_SYNC_PSRCR_TXC_IDLE;

	if (tx->clock.clock == 0)
	{
		// Configure TxC as GPIO.
		reg		|= (0x6) | (tx->clock.high ? 1 : 0);
		mask	|= SIO4_SYNC_PSRCR_TXC;
	}
	else if (tx->clock.ext)
	{
		// Use the external clock.
		reg		|= (0x2) | (tx->clock.high ? 0 : 1);
		mask	|= SIO4_SYNC_PSRCR_TXC;
	}
	else
	{
		// Use the internal clock / 2.
		reg		|= (0x0) | (tx->clock.high ? 0 : 1);
		mask	|= SIO4_SYNC_PSRCR_TXC;
	}

	// Tx Envelope

	if (tx->env.env)
	{
		// Configure the signal as Tx Envelope.
		reg		|= (0x00) | (tx->env.high ? 0 : 0x10);
		mask	|= SIO4_SYNC_PSRCR_TXE;
	}
	else
	{
		// Configure the signal as GPIO.
		reg		|= (0x20) | (tx->env.high ? 0x10 : 0);
		mask	|= SIO4_SYNC_PSRCR_TXE;
	}

	// Tx Data

	if (tx->data.data)
	{
		// Configure the signal as Tx Data.
		reg		|= (0x000) | (tx->data.high ? 0 : 0x40);
		mask	|= SIO4_SYNC_PSRCR_TXD;
	}
	else
	{
		// Configure the signal as GPIO.
		reg		|= (0x180) | (tx->data.high ? 0x40 : 0);
		mask	|= SIO4_SYNC_PSRCR_TXD;
	}

	// Tx Aux Clock

	if (tx->aux_clock.enable == 0)
	{
		// Disable/tri-state the signal.
		reg		|= 0x000;
		mask	|= SIO4_SYNC_PSRCR_TXAC;
	}
	else if (tx->aux_clock.clock)
	{
		// Configure Tx Aux Clock as a clock.
		reg		|= 0x200;
		mask	|= SIO4_SYNC_PSRCR_TXAC;
	}
	else
	{
		// Configure the signal as GPIO.
		reg		|= (0x400) | (tx->aux_clock.high ? 0x200 : 0);
		mask	|= SIO4_SYNC_PSRCR_TXAC;
	}

	// Tx Spare

	if (tx->spare.enable == 0)
	{
		// Disable/tri-state the signal.
		mask	|= SIO4_SYNC_PSRCR_TXSP;
	}
	else
	{
		// Configure the signal as GPIO.
		reg		|= (0x1000) | (tx->spare.high ? 0x800 : 0);
		mask	|= SIO4_SYNC_PSRCR_TXSP;
	}

	// Apply the results.
	ret	= sync_reg_mod(fd, SIO4_GSC_PSRCR, reg, mask);

	// Tx Count Register
	reg	= tx->word_size | (((__u32) tx->gap_size) << 16);
	ret	= ret ? ret : sync_reg_write(fd, SIO4_GSC_TCR, reg);

	// Channel Control/Status Register
	reg		= 0;
	mask	= 0;

	reg		|= tx->auto_dis ? SIO4_SYNC_CSR_TX_AUTO_DIS : 0;
	mask	|= SIO4_SYNC_CSR_TX_AUTO_DIS;

	reg		|= tx->lsbf ? SIO4_SYNC_CSR_TX_LSB_FIRST : 0;
	mask	|= SIO4_SYNC_CSR_TX_LSB_FIRST;

	// Apply the results.
	ret	= ret ? ret : sync_reg_mod(fd, SIO4_GSC_CSR, reg, mask);

	// Apply the enable option last.
	reg		|= tx->enable ? SIO4_SYNC_CSR_TX_ENABLE : 0;
	mask	|= SIO4_SYNC_CSR_TX_ENABLE;
	ret		= ret ? ret : sync_reg_mod(fd, SIO4_GSC_CSR, reg, mask);

	return(ret);
}



