// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/sync/lib/sync.c $
// $Rev: 48522 $
// $Date: 2020-11-23 18:13:34 -0600 (Mon, 23 Nov 2020) $

// SIO4: SYNC Protocol Library: source file

#include "main.h"



/******************************************************************************
*
*	Function:	sio4_sync_get
*
*	Purpose:
*
*		Retrieve the current general SYNC configuration settings.
*
*	Arguments:
*
*		fd		The file rescriptor for the SIO4 to access.
*
*		sync	Record the settings here.
*
*	Returned:
*
*		0		All went well.
*		> 0		An appropriate error status.
*
******************************************************************************/

int sio4_sync_get(int fd, sio4_sync_t* sync)
{
	__u32	psrcr;
	int		ret;

	ret	= sync_reg_read(fd, SIO4_GSC_PSRCR, &psrcr);

	if (ret == 0)
	{
		// DCE/DTE
		if ((psrcr & SIO4_SYNC_PSRCR_DCE_DTE_EN) == 0)
		{
			// DCE and DTE are disabled.
			sync->dce_enable	= 0;
			sync->dte_enable	= 0;
		}
		else if (psrcr & SIO4_SYNC_PSRCR_DCE_EN)
		{
			// DCE Mode
			sync->dce_enable	= 1;
			sync->dte_enable	= 0;
		}
		else
		{
			// DTE Mode
			sync->dce_enable	= 0;
			sync->dte_enable	= 1;
		}

		// Loopback

		if (psrcr & SIO4_SYNC_PSRCR_LB_INT)
		{
			// Loopback: Internal
			sync->lb.enable		= 1;
			sync->lb.internal	= 1;
		}
		else if (psrcr & SIO4_SYNC_PSRCR_LB_EXT)
		{
			// Loopback: External
			sync->lb.enable		= 1;
			sync->lb.internal	= 0;
		}
		else
		{
			// Loopback: Disabled
			sync->lb.enable		= 0;
			sync->lb.internal	= 0;
		}
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	sio4_sync_set
*
*	Purpose:
*
*		Update the current general SYNC configuration.
*
*	Arguments:
*
*		fd		The file rescriptor for the SIO4 to access.
*
*		sync	The setting to apply.
*
*	Returned:
*
*		0		All went well.
*		> 0		An appropriate error status.
*
******************************************************************************/

int sio4_sync_set(int fd, const sio4_sync_t* sync)
{
	__u32	mask;
	__u32	reg;
	int		ret;

	// Pin Source Register
	reg		= 0;
	mask	= 0;

	// DCE/DTE

	if (sync->dce_enable)
	{
		// DCE Mode
		reg		|= SIO4_SYNC_PSRCR_DCE_EN | SIO4_SYNC_PSRCR_DCE_DTE_EN;
		mask	|= SIO4_SYNC_PSRCR_DCE_EN | SIO4_SYNC_PSRCR_DCE_DTE_EN;
	}
	else if (sync->dte_enable)
	{
		// DTE Mode
		reg		|= SIO4_SYNC_PSRCR_DCE_DTE_EN;
		mask	|= SIO4_SYNC_PSRCR_DCE_EN | SIO4_SYNC_PSRCR_DCE_DTE_EN;
	}
	else
	{
		// Both Disabled
		mask	|= SIO4_SYNC_PSRCR_DCE_EN | SIO4_SYNC_PSRCR_DCE_DTE_EN;
	}

	// Loopback

	if (sync->lb.enable == 0)
	{
		// Loopback: Disabled
		mask	|= SIO4_SYNC_PSRCR_LB_INT | SIO4_SYNC_PSRCR_LB_EXT;
	}
	else if (sync->lb.internal)
	{
		// Loopback: Internal
		reg		|= SIO4_SYNC_PSRCR_LB_INT;
		mask	|= SIO4_SYNC_PSRCR_LB_INT | SIO4_SYNC_PSRCR_LB_EXT;
	}
	else
	{
		// Loopback: External
		reg		|= SIO4_SYNC_PSRCR_LB_EXT;
		mask	|= SIO4_SYNC_PSRCR_LB_INT | SIO4_SYNC_PSRCR_LB_EXT;
	}

	// Apply the results.
	ret	= sync_reg_mod(fd, SIO4_GSC_PSRCR, reg, mask);
	return(ret);
}



