// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/sync/lib/rx.c $
// $Rev: 48522 $
// $Date: 2020-11-23 18:13:34 -0600 (Mon, 23 Nov 2020) $

// SIO4: SYNC Protocol Library: source file

#include "main.h"



/******************************************************************************
*
*	Function:	sio4_sync_rx_get
*
*	Purpose:
*
*		Retrieve the current Rx specific SYNC configuration settings.
*
*	Arguments:
*
*		fd		The file rescriptor for the SIO4 to access.
*
*		rx		Record the settings here.
*
*	Returned:
*
*		0		All went well.
*		> 0		An appropriate error status.
*
******************************************************************************/

int sio4_sync_rx_get(int fd, sio4_sync_rx_t* rx)
{
	__u32	csr;
	__u32	psrcr	= 0;
	__u32	rcr		= 0;
	int		ret;

	ret	= sync_reg_read(fd, SIO4_GSC_CSR, &csr);

	if (ret == 0)
		ret	= sync_reg_read(fd, SIO4_GSC_PSRCR, &psrcr);

	if (ret == 0)
		ret	= sync_reg_read(fd, SIO4_GSC_RCR, &rcr);

	if (ret == 0)
	{
		rx->enable		= (csr & SIO4_SYNC_CSR_RX_ENABLE) ? 1 : 0;
		rx->lsbf		= (csr & SIO4_SYNC_CSR_RX_LSB_FIRST) ? 1 : 0;
		rx->reset		= (csr & SIO4_SYNC_CSR_RX_CNT_ERR_STS) ? 1 : 0;
		rx->word_size	= rcr & 0xFFFF;

		// Rx Clock
		rx->clock.high	= (psrcr & SIO4_SYNC_PSRCR_RXC) ? 1 : 0;

		// Rx Envelope

		if (psrcr & 0x8000)
		{
			// The signal is disabled.
			rx->env.enable	= 0;
			rx->env.high	= 0;
		}
		else
		{
			// The signal is enabled.
			rx->env.enable	= 1;
			rx->env.high	= (psrcr & 0x4000) ? 0 : 1;
		}

		// Rx Data
		rx->data.high	= (psrcr & 0x10000) ? 0 : 1;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	sio4_sync_rx_set
*
*	Purpose:
*
*		Update the current Rx specific SYNC configuration.
*
*	Arguments:
*
*		fd		The file rescriptor for the SIO4 to access.
*
*		rx		The setting to apply.
*
*	Returned:
*
*		0		All went well.
*		> 0		An appropriate error status.
*
******************************************************************************/

int sio4_sync_rx_set(int fd, const sio4_sync_rx_t* rx)
{
	__u32	mask;
	__u32	reg;
	int		ret;

	// Channel Pin Source Register
	reg		= 0;
	mask	= 0;

	// Rx Clock
	reg		|= rx->clock.high ? SIO4_SYNC_PSRCR_RXC : 0;
	mask	|= SIO4_SYNC_PSRCR_RXC;

	// Rx Envelope

	if (rx->env.enable)
	{
		// Disable the signal.
		reg		|= (0x0000) | (rx->env.high ? 0 : 0x4000);
		mask	|= SIO4_SYNC_PSRCR_RXE;
	}
	else
	{
		// Disable the signal.
		reg		|= (0x8000);
		mask	|= SIO4_SYNC_PSRCR_RXE;
	}

	// Rx Data
	reg		|= rx->data.high ? 0 : 0x10000;
	mask	|= SIO4_SYNC_PSRCR_RXD;

	// Apply the results.
	ret	= sync_reg_mod(fd, SIO4_GSC_PSRCR, reg, mask);

	// Channel Control/Status Register
	reg		= 0;
	mask	= 0;

	reg		|= rx->enable ? SIO4_SYNC_CSR_RX_ENABLE : 0;
	mask	|= SIO4_SYNC_CSR_RX_ENABLE;

	reg		|= rx->lsbf ? SIO4_SYNC_CSR_RX_LSB_FIRST : 0;
	mask	|= SIO4_SYNC_CSR_RX_LSB_FIRST;

	reg		|= rx->reset ? SIO4_SYNC_CSR_RX_CNT_ERR_RST : 0;
	mask	|= SIO4_SYNC_CSR_RX_CNT_ERR_RST;

	// Apply the results.
	ret	= ret ? ret : sync_reg_mod(fd, SIO4_GSC_CSR, reg, mask);
	return(ret);
}



