// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/samples/sbtest/service.c $
// $Rev: 53081 $
// $Date: 2023-06-13 10:10:10 -0500 (Tue, 13 Jun 2023) $

// SIO4: Sample Application: source file

#include "main.h"



//*****************************************************************************
static int _ioctl_set_by_pointer(int fd, int index, int cmd, s32 arg)
{
	int	errs;
	int	i;
	s32	v	= arg;

	for (;;)	// A convenience loop.
	{
		// Perform the operation.
		i	= ioctl(fd, cmd, (void*) &v);

		if (i == 0)
		{
			// All went well.
			errs	= 0;
			break;
		}

		// There was an error.
		errs	= 1;
		printf("FAIL <--- (");

		if (index >= 0)
			printf("index %d, ", index);

		printf("ioctl(), errno %d)\n", errno);
		break;
	}

	return(errs);
}



/******************************************************************************
*
*	Function:	_ioctl_set_by_value
*
*	Purpose:
*
*		Issue the given IOCTL call.
*
*	Arguments:
*
*		fd		The handle for the device to access.
*
*		index	The table index to report, or negactive to ignore.
*
*		cmd		The IOCTL service to request.
*
*		arg		The argument to pass to the service.
*
*	Returned:
*
*		>= 0	The number of errors encounterred.
*
******************************************************************************/

static int _ioctl_set_by_value(int fd, int index, int cmd, unsigned long arg)
{
	int		errs;
	int		i;

	for (;;)	// A convenience loop.
	{
		// Perform the operation.
		i	= ioctl(fd, cmd, (void*) arg);

		if (i == 0)
		{
			// All went well.
			errs	= 0;
			break;
		}

		// There was an error.
		errs	= 1;
		printf("FAIL <--- (");

		if (index >= 0)
			printf("index %d, ", index);

		printf("ioctl(), errno %d)\n", errno);
		break;
	}

	return(errs);
}



/******************************************************************************
*
*	Function:	_reg_mod
*
*	Purpose:
*
*		Perform a read-modify-write on the specified register.
*
*	Arguments:
*
*		fd		The handle for the device to access.
*
*		index	The table index to report, or negactive to ignore.
*
*		reg		The register to access. If zero then this is ignored.
*
*		mask	The set of register bits of interest.
*
*		value	The value to apply/check for the bits of interest.
*
*	Returned:
*
*		>= 0	The number of errors encounterred.
*
******************************************************************************/

static int _reg_mod(int fd, int index, u32 reg, u32 mask, u32 value)
{
	int					errs;
	int					i;
	REGISTER_MOD_PARAMS	parm;

	for (;;)	// A convenience loop.
	{
		if (reg == 0)
		{
			// We're supposed to ignore this register.
			errs	= 0;
			break;
		}

		// Perform the operation.
		parm.u32RegisterNumber	= reg;
		parm.u32Value			= value;;
		parm.u32Mask			= mask;
		i	= ioctl(fd, SIO4_MOD_REGISTER, &parm);

		if (i == 0)
		{
			// All went well.
			errs	= 0;
			break;
		}

		// There was an error.
		errs	= 1;
		printf("FAIL <--- (");

		if (index >= 0)
			printf("index %d, ", index);

		printf("SIO4_MOD_REGISTER, errno %d)\n", errno);
		break;
	}

	return(errs);
}



/******************************************************************************
*
*	Function:	_reg_read
*
*	Purpose:
*
*		Read a register, but do nothing with the value.
*
*	Arguments:
*
*		fd		The handle for the device to access.
*
*		index	The table index to report, or negactive to ignore.
*
*		reg		The register to access. If zero then this is ignored.
*
*	Returned:
*
*		>= 0	The number of errors encounterred.
*
******************************************************************************/

static int _reg_read(int fd, int index, u32 reg)
{
	int					errs;
	int					i;
	REGISTER_MOD_PARAMS	parm;

	for (;;)	// A convenience loop.
	{
		if (reg == 0)
		{
			// We're supposed to ignore this register.
			errs	= 0;
			break;
		}

		// Read the register.
		parm.u32RegisterNumber	= reg;
		parm.u32Value			= 0;;
		i	= ioctl(fd, SIO4_READ_REGISTER, &parm);

		if (i == -1)
		{
			// There was a problem.
			errs	= 1;
			printf("FAIL <--- (%d. ", __LINE__);

			if (index >= 0)
				printf("index %d, ", index);

			printf("SIO4_READ_REGISTER, errno %d)\n", errno);
			break;
		}

		errs	= 0;
		break;
	}

	return(errs);
}



/******************************************************************************
*
*	Function:	_reg_show
*
*	Purpose:
*
*		Read a register and display its content.
*
*	Arguments:
*
*		fd		The handle for the device to access.
*
*		index	The table index to report, or negactive to ignore.
*
*		reg		The register to access. If zero then this is ignored.
*
*		arg		If non-NULL this is a pointer to a sting to print out.
*
*	Returned:
*
*		>= 0	The number of errors encounterred.
*
******************************************************************************/

static int _reg_show(int fd, int index, u32 reg, unsigned long arg)
{
	int				errs;
	int				i;
	REGISTER_PARAMS	parm;
	char*			ptr		= (void*) arg;

	for (;;)	// A convenience loop.
	{
		if (reg == 0)
		{
			// We're supposed to ignore this register.
			errs	= 0;
			break;
		}

		// Read the register.
		parm.u32RegisterNumber	= reg;
		parm.u32Value			= 0;;
		i	= ioctl(fd, SIO4_READ_REGISTER, &parm);

		if (i == -1)
		{
			// There was a problem.
			errs	= 1;
			printf("FAIL <--- (%d. ", __LINE__);

			if (index >= 0)
				printf("index %d, ", index);

			printf("SIO4_READ_REGISTER, errno %d)\n", errno);
			break;
		}
		else
		{
			errs	= 0;

			if (ptr)
				printf("(%s 0x%lX) ", ptr, (long) parm.u32Value);
			else
				printf("(0x%lX) ", (long) parm.u32Value);
		}

		break;
	}

	fflush(stdout);
	return(errs);
}



/******************************************************************************
*
*	Function:	_reg_test
*
*	Purpose:
*
*		Verify that the specified register bits have the specified values.
*
*	Arguments:
*
*		fd		The handle for the device to access.
*
*		index	The table index to report, or negactive to ignore.
*
*		reg		The register to access. If zero then this is ignored.
*
*		mask	The set of register bits of interest.
*
*		value	The value to apply/check for the bits of interest.
*
*	Returned:
*
*		>= 0	The number of errors encounterred.
*
******************************************************************************/

static int _reg_test(int fd, int index, u32 reg, u32 mask, u32 value)
{
	int				errs;
	int				i;
	REGISTER_PARAMS	parm;

	for (;;)	// A convenience loop.
	{
		if (reg == 0)
		{
			// We're supposed to ignore this register.
			errs	= 0;
			break;
		}

		if (value & ~mask)
		{
			// Excess value bits are set.
			errs	= 1;
			printf("FAIL <--- (%d. ", __LINE__);

			if (index >= 0)
				printf("index %d, ", index);

			printf(	"INTERNAL ERROR, value 0x%lX, mask 0x%lX)\n",
					(long) value,
					(long) mask);
			break;
		}

		// Read the register.
		parm.u32RegisterNumber	= reg;
		parm.u32Value			= 0;;
		i	= ioctl(fd, SIO4_READ_REGISTER, &parm);

		if (i == -1)
		{
			// There was a problem.
			errs	= 1;
			printf("FAIL <--- (%d. ", __LINE__);

			if (index >= 0)
				printf("index %d, ", index);

			printf("SIO4_READ_REGISTER, errno %d)\n", errno);
			break;
		}

		if ((parm.u32Value & mask) == (value & mask))
		{
			// The register bits of interest are as expected.
			errs	= 0;
			break;
		}

		// The register bits of interest are NOT as expected.
		errs	= 1;
		printf("FAIL <--- (%d. ", __LINE__);

		if (index >= 0)
			printf("index %d, ", index);

		printf(	"expected 0x%lX, got 0x%lX (0x%08lX))\n",
				(long) value,
				(long) parm.u32Value & mask,
				(long) parm.u32Value);

		break;
	}

	return(errs);
}



/******************************************************************************
*
*	Function:	_reg_write
*
*	Purpose:
*
*		Write a value to a register the specified number of time.
*
*	Arguments:
*
*		fd		The handle for the device to access.
*
*		index	The table index to report, or negactive to ignore.
*
*		reg		The register to access. If zero then this is ignored.
*
*		arg		The number of times to write the value.
*
*		value	The value to apply/check for the bits of interest.
*
*	Returned:
*
*		>= 0	The number of errors encounterred.
*
******************************************************************************/

static int _reg_write(int fd, int index, u32 reg, s32 arg, u32 value)
{
	int				errs	= 0;
	int				i;
	REGISTER_PARAMS	parm;

	if (reg != 0)
	{
		for (; arg > 0; arg--)
		{
			parm.u32RegisterNumber	= reg;
			parm.u32Value			= value;
			i	= ioctl(fd, SIO4_WRITE_REGISTER, &parm);

			if (i == -1)
			{
				// There was an error.
				errs	= 1;
				printf("FAIL <--- (");

				if (index >= 0)
					printf("index %d, ", index);

				printf("SIO4_MOD_REGISTER, errno %d)\n", errno);
			}
		}
	}

	return(errs);
}



/******************************************************************************
*
*	Function:	_service_perform
*
*	Purpose:
*
*		Perform the specified services on the given structure.
*
*	Arguments:
*
*		fd		The handle for the device to access.
*
*		index	A table index to report. If zero then this is ignored.
*
*		service	The service to perform.
*
*		item	The structure to process.
*
*	Returned:
*
*		>= 0	The number of errors encounterred.
*
******************************************************************************/

static int _service_perform(
	int						fd,
	int						index,
	service_t				service,
	const service_data_t*	item)
{
	int	errs;

	switch (service)
	{
		default:
		case SERVICE_END_LIST:

			errs	= 1;
			printf(	"FAIL <---  (%d. INTERNAL ERROR: ", __LINE__);

					if (index >= 0)
						printf("index %d, ", index);

			printf("service %d)\n", (int) service);
			break;

		case SERVICE_BY_PTR:

			errs	= _ioctl_set_by_pointer(fd, index, item->cmd, item->arg);
			break;

		case SERVICE_BY_VAL:

			errs	= _ioctl_set_by_value(fd, index, item->cmd, item->arg);
			break;

		case SERVICE_REG_MOD:

			errs	= _reg_mod(fd, index, item->reg, item->mask, item->value);
			break;

		case SERVICE_REG_READ:

			errs	= _reg_read(fd, index, item->reg);
			break;

		case SERVICE_REG_SHOW:

			errs	= _reg_show(fd, index, item->reg, item->arg);
			break;

		case SERVICE_REG_TEST:

			errs	= _reg_test(fd, index, item->reg, item->mask, item->value);
			break;

		case SERVICE_REG_WRITE:

			errs	= _reg_write(fd, index, item->reg, item->arg, item->value);
			break;

		case SERVICE_SLEEP:

			errs	= 0;
			os_sleep_ms(item->arg * 1000);
			break;

		case SERVICE_SLEEP_MS:

			errs	= 0;

			os_sleep_ms(item->arg);
			break;
	}

	return(errs);
}



/******************************************************************************
*
*	Function:	_service_item
*
*	Purpose:
*
*		Perform the specified services on the given structure.
*
*	Arguments:
*
*		fd		The handle for the device to access.
*
*		index	A table index to report. Ignore this if negative.
*
*		item	The structure to process.
*
*		srv1	The first primary service to perform.
*
*		srv2	The second primary service to perform.
*
*	Returned:
*
*		>= 0	The number of errors encounterred.
*
******************************************************************************/

static int _service_item(
	int						fd,
	int						index,
	const service_data_t*	item,
	service_t				srv1,
	service_t				srv2)
{
	int	errs;

	if ((item->service == SERVICE_BY_VAL) ||
		(item->service == SERVICE_BY_PTR))
	{
		errs	= _service_perform(fd, index, item->service, item);

		if (errs == 0)
			errs	= _service_perform(fd, index, srv2, item);
	}
	else
	{
		errs	= _service_perform(fd, index, item->service, item);
	}

	return(errs);
}



/******************************************************************************
*
*	Function:	_service_list
*
*	Purpose:
*
*		Go through the list and, for each element, perform the specified
*		services.
*
*	Arguments:
*
*		fd		The handle for the device to access.
*
*		index	A table index to report. Ignore this if negative.
*
*		list	The list to go through.
*
*	Returned:
*
*		>= 0	The number of errors encounterred.
*
******************************************************************************/

static int _service_list(
	int						fd,
	const service_data_t*	list,
	service_t				srv1,
	service_t				srv2)
{
	int	errs	= 0;
	int	i;

	for (i = 0; (errs == 0) && (list[i].service != SERVICE_END_LIST); i++)
		errs	= _service_item(fd, i, &list[i], srv1, srv2);

	return(errs);
}



/******************************************************************************
*
*	Function:	service_ioctl_set_reg_list
*
*	Purpose:
*
*		Go through the list and, for each element, perform the specified
*		services.
*
*	Arguments:
*
*		fd		The handle for the device to access.
*
*		list	The list to go through.
*
*	Returned:
*
*		>= 0	The number of errors encounterred.
*
******************************************************************************/

int service_ioctl_set_reg_list(int fd, const service_data_t* list)
{
	int	errs;

	errs	= _service_list(fd, list, (service_t) -1, SERVICE_REG_TEST);
	return(errs);
}


