// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/samples/irq/usc_tx_idle_sent.c $
// $Rev: 53079 $
// $Date: 2023-06-13 10:07:58 -0500 (Tue, 13 Jun 2023) $

// SIO4: Sample Application: source file

#include "main.h"



// variables ******************************************************************

static	int	_qty;



/******************************************************************************
*
*	Function:	_irq_reset
*
*	Purpose:
*
*		Clean things up after we're done.
*
*	Arguments:
*
*		fd		The file descriptor for the channel to access.
*
*	Returned:
*
*		>= 0	The number of errors seen.
*
******************************************************************************/

static int _irq_reset(int fd)
{
	int	errs	= 0;
	int	status;

	// Reset the device.
	status	= sio4_channel_reset(fd, 0);
	errs	+= status ? 1 : 0;

	// Turn off asynchronous interrupt notification.
	errs	+= usc_notify_disable(fd);

	// Disable the interrupt.
	errs	+= GSC_INT_ALL_DISABLE(fd);
	errs	+= USC_MASTER_INT_DISABLE(fd);
	errs	+= USC_TX_STATUS_INT_DISABLE(fd);
	errs	+= USC_TX_IDLE_SENT_INT_DISARM(fd);
	return(errs);
}



/******************************************************************************
*
*	Function:	_irq_setup
*
*	Purpose:
*
*		Set things up so generation of the interrupt will be detected.
*
*	Arguments:
*
*		fd		The file descriptor for the channel to access.
*
*	Returned:
*
*		>= 0	The number of errors seen.
*
******************************************************************************/

static int _irq_setup(int fd)
{
	sio4_monosync_t	data;
	int				errs	= 0;
	int				status;

	// Reset the device.
	status	= sio4_channel_reset(fd, 0);
	errs	+= status ? 1 : 0;

	sio4_monosync_t_init(&data);
	data.cable.legacy.config	= SIO4_CABLE_CONFIG_TXLWR_RXLWR;
	data.rx.enable				= 1;
	data.rx.dpll_use			= 1;
	data.rx.frame_size_max		= 0xFFFF;
	data.tx.enable				= 1;
	data.tx.frame_size_max		= 0xFFFF;
	data.tx.idle_output			= SIO4_USC_TCSR_TXIDLE_CONT_1;
	data.tx.preamble_send		= 1;
	errs	+= sio4_monosync_config(fd, &data);

	// Enable the interrupt.
	USC_RX_STATUS_INT_CLEAR(fd);
	errs	+= USC_TX_IDLE_SENT_INT_ARM(fd);
	errs	+= USC_TX_STATUS_INT_ENABLE(fd);
	errs	+= USC_MASTER_INT_ENABLE(fd);
	errs	+= GSC_LOCAL_INT_ENABLE(fd);

	// Setup asynchronous interrupt notification.
	errs	+= USC_TX_STATUS_INT_CLEAR(fd);
	errs	+= USC_TX_IDLE_SENT_STATUS_CLR(fd);
	errs	+= usc_notify_enable(fd, USC_IRQ_TX_IDLE_SENT, &_qty);

	return(errs);
}



/******************************************************************************
*
*	Function:	_irq_trigger
*
*	Purpose:
*
*		Trigger the interrupt.
*
*	Arguments:
*
*		fd		The file descriptor for the channel to access.
*
*	Returned:
*
*		>= 0	The number of errors seen.
*
******************************************************************************/

static int _irq_trigger(int fd)
{
	char	buf[64];
	int		bytes	= sizeof(buf);
	int		errs;
	size_t	qty;

	memset(buf, 'B', sizeof(buf));
	errs	= sio4_write(fd, buf, bytes, &qty);

	if (qty != bytes)
		errs	= 1;

	os_sleep_ms(1000);	// Wait for things to occur.

	// Issue the Tx Send Preamble Command.
	errs	+= reg_mod(fd, SIO4_USC_TCSR, 0xF000, 0xF000);

	wait_for_irq(&_qty);
	return(errs);
}



/******************************************************************************
*
*	Function:	usc_tx_idle_sent_test
*
*	Purpose:
*
*		Verify the operation of the USC xxx interrupt.
*
*	Arguments:
*
*		fd		The file descriptor for the channel to access.
*
*	Returned:
*
*		>= 0	The number of errors seen.
*
******************************************************************************/

int usc_tx_idle_sent_test(int fd)
{
	int	errs	= 0;

	gsc_label("Tx Idle Sent");
	errs	+= _irq_setup(fd);
	errs	+= _irq_trigger(fd);
	errs	+= (_qty > 0) ? 0 : 1;
	errs	+= _irq_reset(fd);
	printf("%s\n", errs ? "FAIL <---" : "PASS");
	return(errs ? 1 : 0);
}



