// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/samples/irq/usc_rx_break_abort.c $
// $Rev: 53079 $
// $Date: 2023-06-13 10:07:58 -0500 (Tue, 13 Jun 2023) $

// SIO4: Sample Application: source file

#include "main.h"



// variables ******************************************************************

static	int	_qty;



/******************************************************************************
*
*	Function:	_irq_reset
*
*	Purpose:
*
*		Clean things up after we're done.
*
*	Arguments:
*
*		fd		The file descriptor for the channel to access.
*
*	Returned:
*
*		>= 0	The number of errors seen.
*
******************************************************************************/

static int _irq_reset(int fd)
{
	int	errs	= 0;
	int	status;

	// Reset the device.
	status	= sio4_channel_reset(fd, 0);
	errs	+= status ? 1 : 0;

	// Turn off asynchronous interrupt notification.
	errs	+= usc_notify_disable(fd);

	// Disable the interrupt.
	errs	+= GSC_INT_ALL_DISABLE(fd);
	errs	+= USC_MASTER_INT_DISABLE(fd);
	errs	+= USC_RX_STATUS_INT_DISABLE(fd);
	errs	+= USC_RX_BRK_ABRT_INT_DISARM(fd);
	return(errs);
}



/******************************************************************************
*
*	Function:	_irq_setup
*
*	Purpose:
*
*		Set things up so generation of the interrupt will be detected.
*
*	Arguments:
*
*		fd		The file descriptor for the channel to access.
*
*	Returned:
*
*		>= 0	The number of errors seen.
*
******************************************************************************/

static int _irq_setup(int fd)
{
	sio4_async_t	data;
	int				errs	= 0;
	int				status;

	// Reset the device.
	status	= sio4_channel_reset(fd, 0);
	errs	+= status ? 1 : 0;

	sio4_async_t_init(&data);
	data.cable.legacy.config	= SIO4_CABLE_CONFIG_TXLWR_RXLWR;
	data.rx.enable				= 1;
	data.tx.enable				= 1;
	data.loopback.enable		= 1;
	data.loopback.internal		= 0;
	errs	+= sio4_async_config(fd, &data, 0, 0);

	// Enable the interrupt.
	errs	+= USC_RX_BRK_ABRT_INT_ARM(fd);
	errs	+= USC_RX_STATUS_INT_ENABLE(fd);
	errs	+= USC_MASTER_INT_ENABLE(fd);
	errs	+= GSC_LOCAL_INT_ENABLE(fd);

	// Setup asynchronous interrupt notification.
	errs	+= USC_RX_STATUS_INT_CLEAR(fd);
	errs	+= USC_RX_BRK_ABRT_STATUS_CLR(fd);
	errs	+= usc_notify_enable(fd, USC_IRQ_RX_BREAK_ABORT, &_qty);

	return(errs);
}



/******************************************************************************
*
*	Function:	_irq_trigger
*
*	Purpose:
*
*		Trigger the interrupt.
*
*	Arguments:
*
*		fd		The file descriptor for the channel to access.
*
*	Returned:
*
*		>= 0	The number of errors seen.
*
******************************************************************************/

static int _irq_trigger(int fd)
{
	int	errs	= 0;

	// We drive the signal low and high because we don't know what transcievers
	// are present or what signal level they actually drive on the line.

	// Drive the data line low for an extended period.
	errs	+= reg_mod(fd, SIO4_USC_IOCR, 0x0080, 0x00C0);
	os_sleep_ms(1000);

	// Drive the data line high for an extended period.
	errs	+= reg_mod(fd, SIO4_USC_IOCR, 0x00C0, 0x00C0);
	os_sleep_ms(1000);

	// Restore normal transmitter output.
	errs	+= reg_mod(fd, SIO4_USC_IOCR, 0x0000, 0x00C0);
	wait_for_irq(&_qty);
	return(errs);
}



/******************************************************************************
*
*	Function:	usc_rx_break_abort_test
*
*	Purpose:
*
*		Verify the operation of the USC Break/Abort interrupt.
*
*	Arguments:
*
*		fd		The file descriptor for the channel to access.
*
*	Returned:
*
*		>= 0	The number of errors seen.
*
******************************************************************************/

int usc_rx_break_abort_test(int fd)
{
	int	errs	= 0;

	gsc_label("Rx Break/Abort");
	errs	+= _irq_setup(fd);
	errs	+= _irq_trigger(fd);
	errs	+= (_qty > 0) ? 0 : 1;
	errs	+= _irq_reset(fd);
	printf("%s\n", errs ? "FAIL <---" : "PASS");
	return(errs ? 1 : 0);
}



