// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/samples/irq/usc_dpll_desync.c $
// $Rev: 53079 $
// $Date: 2023-06-13 10:07:58 -0500 (Tue, 13 Jun 2023) $

// SIO4: Sample Application: source file

#include "main.h"



// variables ******************************************************************

static	int	_qty;



/******************************************************************************
*
*	Function:	_irq_reset
*
*	Purpose:
*
*		Clean things up after we're done.
*
*	Arguments:
*
*		fd		The file descriptor for the channel to access.
*
*	Returned:
*
*		>= 0	The number of errors seen.
*
******************************************************************************/

static int _irq_reset(int fd)
{
	int	errs	= 0;
	int	status;

	// Reset the device.
	status	= sio4_channel_reset(fd, 0);
	errs	+= status ? 1 : 0;

	// Turn off asynchronous interrupt notification.
	errs	+= usc_notify_disable(fd);

	// Disable the interrupt.
	errs	+= GSC_INT_ALL_DISABLE(fd);
	errs	+= USC_MASTER_INT_DISABLE(fd);
	errs	+= USC_MISC_INT_DISABLE(fd);
	errs	+= USC_DPLL_DESYNC_INT_DISARM(fd);
	return(errs);
}



/******************************************************************************
*
*	Function:	_irq_setup
*
*	Purpose:
*
*		Set things up so generation of the interrupt will be detected.
*
*	Arguments:
*
*		fd		The file descriptor for the channel to access.
*
*	Returned:
*
*		>= 0	The number of errors seen.
*
******************************************************************************/

static int _irq_setup(int fd)
{
	sio4_async_t	async;
	int				errs	= 0;
	int				status;

	// Reset the device.
	status	= sio4_channel_reset(fd, 0);
	errs	+= status ? 1 : 0;

	// Configure the DPLL.
	sio4_async_t_init(&async);
	async.cable.legacy.config	= SIO4_CABLE_CONFIG_TXUPR_RXUPR;
	async.loopback.enable		= 1;
	async.loopback.internal		= 1;
	async.rx.enable				= 1;
	async.rx.encoding			= SIO4_USC_ENCODING_BIPH_MARK;
	async.tx.enable				= 0;
	async.tx.encoding			= SIO4_USC_ENCODING_BIPH_MARK;
	errs	+= sio4_async_config(fd, &async, 0, 0);

	status	+= sio4_dpll_set_clock_source(fd, TXC_PIN_CLOCK, 0);
	errs	+= status ? 1 : 0;
	status	= sio4_dpll_set_divisor(fd, DPLL_8X, 0);
	errs	+= status ? 1 : 0;
	status	= sio4_dpll_set_mode(fd, DPLL_BIPHASE_MARK_SPACE, 0);
	errs	+= status ? 1 : 0;

	// Enable the interrupt.
	errs	+= USC_DPLL_DESYNC_INT_ARM(fd);
	errs	+= USC_MISC_INT_ENABLE(fd);
	errs	+= USC_MASTER_INT_ENABLE(fd);
	errs	+= GSC_LOCAL_INT_ENABLE(fd);

	// Setup asynchronous interrupt notification.
	errs	+= USC_MISC_INT_CLEAR(fd);
	errs	+= USC_DPLL_DESYNC_STATUS_CLR(fd);
	errs	+= usc_notify_enable(fd, USC_IRQ_DPLL_DESYNC, &_qty);
	return(errs);
}



/******************************************************************************
*
*	Function:	_irq_trigger
*
*	Purpose:
*
*		Trigger the interrupt.
*
*	Arguments:
*
*		fd		The file descriptor for the channel to access.
*
*	Returned:
*
*		>= 0	The number of errors seen.
*
******************************************************************************/

static int _irq_trigger(int fd)
{
	int	errs	= 0;
	int	status;

	// Disable BRG0.
	status	= sio4_brg0_enable(fd, 0, 0);
	errs	+= status ? 1 : 0;
	wait_for_irq(&_qty);
	return(errs);
}



/******************************************************************************
*
*	Function:	usc_dpll_desync_test
*
*	Purpose:
*
*		Verify the operation of the USC DPLL DeSync interrupt.
*
*	Arguments:
*
*		fd		The file descriptor for the channel to access.
*
*	Returned:
*
*		>= 0	The number of errors seen.
*
******************************************************************************/

int usc_dpll_desync_test(int fd)
{
	int	errs	= 0;

	gsc_label("DPLL DeSync");
	errs	+= _irq_setup(fd);
	errs	+= _irq_trigger(fd);
	errs	+= (_qty > 0) ? 0 : 1;
	errs	+= _irq_reset(fd);
	printf("%s\n", errs ? "FAIL <---" : "PASS");
	return(errs ? 1 : 0);
}


