// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/samples/id/perform.c $
// $Rev: 56757 $
// $Date: 2026-01-07 08:27:16 -0600 (Wed, 07 Jan 2026) $

// SIO4: Sample Application: source file

#include "main.h"



// variables ******************************************************************

static	int	_query_list[100];



//*****************************************************************************
static int _show_eol(int errs)
{
	if (errs)
		printf("FAIL <---");

	printf("\n");
	return(errs);
}



//*****************************************************************************
static int _current_fw_type(const int* fd)
{
	char	buf[32];
	int		err;
	int		errs	= 0;
	s32		fw_type;
	int		i;
	s32		val[4]	= { 0, 0, 0, 0 };

	gsc_label("Current Firmware Type");

	for (i = 0; i <= 3; i++)
	{
		if (fd[i] < 0)
		{
			local_label_str("----");
			continue;
		}

		err		= sio4_fw_type_config(fd[i], -1, 0, SIO4_FW_TYPE_CONFIG_READ, &fw_type);

		if (err)
		{
			errs++;
			local_label_str("FAIL <---");
			continue;
		}

		switch (fw_type)
		{
			default:

				errs++;
				sprintf(buf, "%ld", (long) val[i]);
				local_label_str(buf);
				break;

			case SIO4_FW_TYPE_CONFIG_SYNC:		local_label_str("SYNC");	break;
			case SIO4_FW_TYPE_CONFIG_Z16C30:	local_label_str("Z16C30");	break;
		}
	}

	_show_eol(errs);
	return(errs);
}



//*****************************************************************************
static int _query_list_set(int option)
{
	int	errs	= 0;

	if (option > SIO4_FEATURE_LAST_INDEX)
	{
		errs++;
	}
	else
	{
		errs	= 0;
		_query_list[option]++;
	}

	return(errs);
}



//*****************************************************************************
static int _validate_same_s32(const int* fd, s32* val)
{
	int	any		= 0;
	int	errs	= 0;
	s32	good	= 0;
	int	i;

	for (i = 0; i <= 3; i++)
	{
		if (fd[i] < 0)
		{
			continue;
		}
		else if (any == 0)
		{
			any		= 1;
			good	= val[i];
		}
		else if (val[i] != good)
		{
			errs++;
		}
	}

	return(errs);
}



//*****************************************************************************
static int _validate_same_u32(const int* fd, u32* val)
{
	int	any		= 0;
	int	errs	= 0;
	u32	good	= 0;
	int	i;

	for (i = 0; i <= 3; i++)
	{
		if (fd[i] < 0)
		{
		}
		else if (any == 0)
		{
			any		= 1;
			good	= val[i];
		}
		else if (val[i] != good)
		{
			errs++;
		}
	}

	return(errs);
}



//*****************************************************************************
static int _show_s32_long(const int* fd, const char* name, s32* val, const char* format)
{
	char	buf[32];
	int		errs	= 0;
	int		i;

	gsc_label(name);

	for (i = 0; i <= 3; i++)
	{
		if (fd[i] < 0)
		{
			local_label_str("----");
			continue;
		}

		sprintf(buf, format, (long) val[i]);
		local_label_str(buf);
	}

	return(errs);
}



//*****************************************************************************
static int _query_read_s32(const int* fd, s32* val, s32 query)
{
	int	errs	= 0;
	int	i;
	int	ret;
	s32	v;

	for (i = 0; i <= 3; i++)
	{
		if (fd[i] >= 0)
		{
			v		= query;
			ret		= sio4_ioctl(fd[i], SIO4_FEATURE_TEST, &v);
			val[i]	= v;
			errs	+= (ret < 0) ? 1 : 0;
		}
	}

	return(errs);
}



//*****************************************************************************
static int _query_read_u32(const int* fd, u32* val, s32 query)
{
	int	errs	= 0;
	int	i;
	int	ret;
	u32	v;

	for (i = 0; i <= 3; i++)
	{
		if (fd[i] >= 0)
		{
			v		= query;
			ret		= sio4_ioctl(fd[i], SIO4_FEATURE_TEST, &v);
			val[i]	= v;
			errs	+= (ret < 0) ? 1 : 0;
		}
	}

	return(errs);
}



//*****************************************************************************
static int _query_show_s32(const int* fd, s32* val, s32 query, const char* name, const char* format)
{
	int	errs	= 0;
	int	i;
	int	ret;
	s32	v;

	for (i = 0; i <= 3; i++)
	{
		if (fd[i] >= 0)
		{
			v		= query;
			ret		= sio4_ioctl(fd[i], SIO4_FEATURE_TEST, &v);
			val[i]	= v;
			errs	+= (ret < 0) ? 1 : 0;
		}
	}

	errs	+= _show_s32_long(fd, name, val, format);
	return(errs);
}



//*****************************************************************************
static int _show_user_jumper_val(const int* fd)
{
	int	errs	= 0;
	s32	val[4];

	errs	+= _query_list_set(SIO4_FEATURE_USER_JUMPER_VAL);
	errs	+= _query_show_s32(fd, val, SIO4_FEATURE_USER_JUMPER_VAL, "User Jumper Value", "0x%lX");
	errs	+= _validate_same_s32(fd, val);
	_show_eol(errs);
	return(errs);
}



//*****************************************************************************
static int _query_bool_str(const int* fd, int query, const char* title, int same, const char* yes, const char* no)
{
	char	buf[32];
	int		errs	= 0;
	int		i;
	s32		val[4];

	gsc_label(title);
	errs	+= _query_list_set(query);
	errs	+= _query_read_s32(fd, val, query);

	if (same)
		errs	+= _validate_same_s32(fd, val);

	for (i = 0; i <= 3; i++)
	{
		if (fd[i] < 0)
		{
			local_label_str("----");
			continue;
		}

		switch (val[i])
		{
			default:

				errs++;
				sprintf(buf, "%ld", (long) val[i]);
				local_label_str(buf);
				break;

			case 0:	local_label_str(no);	break;
			case 1:	local_label_str(yes);	break;
		}
	}

	_show_eol(errs);
	return(errs);
}



//*****************************************************************************
static int _query_bool(const int* fd, int query, const char* title, int same)
{
	int	errs;

	errs	= _query_bool_str(fd, query, title, same, "Yes", "No");
	return(errs);
}



//*****************************************************************************
static int _query_hex_str(const int* fd, int query, const char* title, int same)
{
	char	buf[32];
	int		errs	= 0;
	int		i;
	s32		val[4];

	gsc_label(title);
	errs	+= _query_list_set(query);
	errs	+= _query_read_s32(fd, val, query);

	if (same)
		errs	+= _validate_same_s32(fd, val);

	for (i = 0; i <= 3; i++)
	{
		if (fd[i] < 0)
		{
			local_label_str("----------");
			continue;
		}

		sprintf(buf, "0x%08lX", (long) val[i]);
		local_label_str(buf);
	}

	_show_eol(errs);
	return(errs);
}



//*****************************************************************************
static int _query_hex(const int* fd, int query, const char* title, int same)
{
	int	errs;

	errs	= _query_hex_str(fd, query, title, same);
	return(errs);
}



//*****************************************************************************
static int _query_decimal(const int* fd, int query, const char* title, const char* suffix, int same)
{
	char	buf[32];
	int		errs	= 0;
	int		i;
	s32		val[4];

	gsc_label(title);
	errs	+= _query_list_set(query);
	errs	+= _query_read_s32(fd, val, query);

	if (same)
		errs	+= _validate_same_s32(fd, val);

	for (i = 0; i <= 3; i++)
	{
		if (fd[i] < 0)
		{
			local_label_str("----");
			continue;
		}

		sprintf(buf, "%ld", (long) val[i]);
		local_label_str(buf);
	}

	if (errs)
		_show_eol(errs);
	else
		printf("%s\n", suffix);

	return(errs);
}



//*****************************************************************************
static int _query_u32(const int* fd, int query, const char* title, const char* suffix, const char* format, int same)
{
	char	buf[32];
	int		errs	= 0;
	int		i;
	u32		val[4];

	gsc_label(title);
	errs	+= _query_list_set(query);
	errs	+= _query_read_u32(fd, val, query);

	if (same)
		errs	+= _validate_same_u32(fd, val);

	for (i = 0; i <= 3; i++)
	{
		if (fd[i] < 0)
		{
			local_label_str("----");
			continue;
		}

		sprintf(buf, format, (long) val[i]);
		local_label_str(buf);
	}

	if (errs)
		_show_eol(errs);
	else
		printf("%s\n", suffix);

	return(errs);
}



//*****************************************************************************
static int _board_reset(const int* fd)
{
	int	errs	= 0;
	int	i;
	int	tmp;

	gsc_label("Board Reset");

	for (i = 0; i <= 3; i++)
	{
		if (i)
		{
			local_label_str("SKIP");
			continue;
		}

		tmp	= sio4_ioctl(fd[i], SIO4_INIT_BOARD, NULL);

		if (tmp)
		{
			local_label_str("FAIL <---");
			errs++;
		}
		else
		{
			local_label_str("PASS");
		}
	}

	if (errs)
		printf("FAIL <---");

	printf("\n");
	return(errs);
}



//*****************************************************************************
static int _query_form_factor(const int* fd)
{
	int			errs	= 0;
	int			i;
	const char*	psz		= NULL;
	s32			val[4];

	gsc_label("Form Factor");
	errs	+= _query_list_set(SIO4_FEATURE_FORM_FACTOR);
	errs	+= _query_read_s32(fd, val, SIO4_FEATURE_FORM_FACTOR);
	errs	+= _validate_same_s32(fd, val);

	for (i = 0; i <= 3; i++)
	{
		if (fd[i] < 0)
		{
			local_label_str("----");
			continue;
		}

		errs	+= sio4_id_form_factor(fd[i], &psz, 0);
		local_label_str(psz);
	}

	_show_eol(errs);
	return(errs);
}



//*****************************************************************************
static int _query_sio4_type(const int* fd)
{
	int			errs	= 0;
	int			i;
	const char*	psz		= NULL;
	s32			val[4];

	gsc_label("SIO4 Type");
	errs	+= _query_list_set(SIO4_FEATURE_SIO4_TYPE);
	errs	+= _query_read_s32(fd, val, SIO4_FEATURE_SIO4_TYPE);
	errs	+= _validate_same_s32(fd, val);

	for (i = 0; i <= 3; i++)
	{
		if (fd[i] < 0)
		{
			local_label_str("----");
			continue;
		}

		errs	+= sio4_id_type(fd[i], &psz);
		local_label_str(psz);
	}

	_show_eol(errs);
	return(errs);
}



//*****************************************************************************
static int _query_fw_type(const int* fd)
{
	char	buf[32];
	int		errs	= 0;
	int		i;
	s32		val[4];

	gsc_label("Default Firmware Type");
	errs	+= _query_list_set(SIO4_FEATURE_FW_TYPE);
	errs	+= _query_read_s32(fd, val, SIO4_FEATURE_FW_TYPE);
	errs	+= _validate_same_s32(fd, val);

	for (i = 0; i <= 3; i++)
	{
		if (fd[i] < 0)
		{
			local_label_str("----");
			continue;
		}

		switch (val[i])
		{
			default:

				errs++;
				sprintf(buf, "%ld", (long) val[i]);
				local_label_str(buf);
				break;

			case SIO4_FW_TYPE_SYNC:		local_label_str("SYNC");	break;
			case SIO4_FW_TYPE_Z16C30:	local_label_str("Z16C30");	break;
		}
	}

	_show_eol(errs);
	return(errs);
}



//*****************************************************************************
static int _query_model_base(const int* fd)
{
	int			errs	= 0;
	int			i;
	const char*	psz		= NULL;
	s32			val[4];

	gsc_label("SIO4 Base Model");
	errs	+= _query_list_set(SIO4_FEATURE_MODEL_BASE);
	errs	+= _query_read_s32(fd, val, SIO4_FEATURE_MODEL_BASE);
	errs	+= _validate_same_s32(fd, val);

	for (i = 0; i <= 3; i++)
	{
		if (fd[i] < 0)
		{
			local_label_str("----");
			continue;
		}

		errs	+= sio4_id_model(fd[i], &psz);
		local_label_str(psz);
	}

	_show_eol(errs);
	return(errs);
}



//*****************************************************************************
static int _query_index_channel(const int* fd)
{
	int	errs	= 0;
	int	i;
	s32	val[4];

	errs	+= _query_list_set(SIO4_FEATURE_INDEX_CHANNEL);
	errs	+= _query_read_s32(fd, val, SIO4_FEATURE_INDEX_CHANNEL);
	errs	+= _show_s32_long(fd, "Channel Index", val, "%ld");

	for (i = 0; i <= 3; i++)
	{
		if (fd[i] < 0)
			continue;

		if (val[i] != i)
			errs++;
	}

	_show_eol(errs);
	return(errs);
}



//*****************************************************************************
static int _query_index_device(const int* fd)
{
	int	errs	= 0;
	s32	val[4];

	errs	+= _query_list_set(SIO4_FEATURE_INDEX_DEVICE);
	errs	+= _query_read_s32(fd, val, SIO4_FEATURE_INDEX_DEVICE);
	errs	+= _show_s32_long(fd, "Device Index", val, "%ld");
	_show_eol(errs);
	return(errs);
}



//*****************************************************************************
static int _query_index_subdevice(const int* fd)
{
	int	errs	= 0;
	int	i;
	int	ret;
	s32	v;
	s32	val[4];

	errs	+= _query_list_set(SIO4_FEATURE_INDEX_SUBDEVICE);
	errs	+= _query_read_s32(fd, val, SIO4_FEATURE_INDEX_SUBDEVICE);
	errs	+= _show_s32_long(fd, "Subdevice Index", val, "%ld");
	errs	+= _validate_same_s32(fd, val);

	for (i = 0; i <= 3; i++)
	{
		if (fd[i] < 0)
			continue;

		if ((val[i] != 0) && (val[i] != 1))
		{
			errs++;
		}
		else
		{
			v	= SIO4_FEATURE_DEVICE_QTY;
			ret	= sio4_ioctl(fd[i], SIO4_FEATURE_TEST, &v);

			if ((ret < 0) || (val[i] > (v - 1)))
				errs++;
		}
	}


	_show_eol(errs);
	return(errs);
}



//*****************************************************************************
static int _query_fifo_size(const int* fd, int query, const char* title, int same)
{
	char	buf[32];
	int		errs	= 0;
	s32		fifo;
	int		i;
	s32		qty;
	s32		val[4];

	gsc_label(title);
	errs	+= _query_list_set(query);
	errs	+= _query_read_s32(fd, val, query);
	errs	+= sio4_feature_test(fd[0], SIO4_FEATURE_DEVICE_QTY, &qty, 0);

	if (same)
		errs	+= _validate_same_s32(fd, val);

	for (i = 0; i <= 3; i++)
	{
		if (fd[i] < 0)
		{
			local_label_str("----");
			continue;
		}

		fifo	= val[i];

		if ((query == SIO4_FEATURE_FIFO_SIZE_TOTAL) && (qty > 1))
			fifo	*= qty;

		if (fifo % 1024)
			sprintf(buf, "%ld", (long) fifo);
		else
			sprintf(buf, "%ldK", (long) fifo / 1024);

		local_label_str(buf);
	}

	if (errs)
		_show_eol(errs);
	else
		printf("bytes\n");

	return(errs);
}



//*****************************************************************************
static int _query_osc_chip(const int* fd)
{
	char	buf[32];
	int		errs	= 0;
	int		i;
	s32		val[4];

	gsc_label("OSC Chip Configuration");
	errs	+= _query_list_set(SIO4_FEATURE_OSC_CHIP);
	errs	+= _query_read_s32(fd, val, SIO4_FEATURE_OSC_CHIP);
	errs	+= _validate_same_s32(fd, val);

	for (i = 0; i <= 3; i++)
	{
		if (fd[i] < 0)
		{
			local_label_str("----");
			continue;
		}

		switch (val[i])
		{
			default:

				errs++;
				sprintf(buf, "%ld", (long) val[i]);
				local_label_str(buf);
				break;

			case SIO4_OSC_CHIP_FIXED:		local_label_str("Fixed");			break;
			case SIO4_OSC_CHIP_IDC2053B:	local_label_str("IDC2053B (1)");	break;
			case SIO4_OSC_CHIP_IDC2053B_4:	local_label_str("IDC2053B (4)");	break;
			case SIO4_OSC_CHIP_CY22393:		local_label_str("CY22393 (1)");		break;
			case SIO4_OSC_CHIP_CY22393_2:	local_label_str("CY22393 (2)");		break;
		}
	}

	_show_eol(errs);
	return(errs);
}



//*****************************************************************************
static int _query_mp_chip(const int* fd)
{
	char	buf[32];
	int		errs	= 0;
	int		i;
	s32		val[4];

	gsc_label("MP Chip Configuration");
	errs	+= _query_list_set(SIO4_FEATURE_MP_CHIP);
	errs	+= _query_read_s32(fd, val, SIO4_FEATURE_MP_CHIP);
	errs	+= _validate_same_s32(fd, val);

	for (i = 0; i <= 3; i++)
	{
		if (fd[i] < 0)
		{
			local_label_str("----");
			continue;
		}

		switch (val[i])
		{
			default:

				errs++;
				sprintf(buf, "%ld", (long) val[i]);
				local_label_str(buf);
				break;

			case SIO4_MP_CHIP_FIXED:	local_label_str("Fixed");	break;
			case SIO4_MP_CHIP_SP508:	local_label_str("SP508");	break;
		}
	}

	_show_eol(errs);
	return(errs);
}



//*****************************************************************************
static int _query_count(const int* fd)
{
	char	buf[32];
	char	err[64]	= "";
	int		errs	= 0;
	int		i;
	int		j;
	int		ret;
	s32		v;
	s32		val[4];

	gsc_label("Query Count");
	errs	+= _query_list_set(SIO4_FEATURE_COUNT);

	if ((errs) && (err[0] == 0))
		sprintf(err, "%d. _query_list_set", __LINE__);

	errs	+= _query_read_s32(fd, val, SIO4_FEATURE_COUNT);

	if ((errs) && (err[0] == 0))
		sprintf(err, "%d. _query_read_s32", __LINE__);

	errs	+= _validate_same_s32(fd, val);

	if ((errs) && (err[0] == 0))
		sprintf(err, "%d. _validate_same_s32", __LINE__);

	for (i = 0; i <= 3; i++)
	{
		if (fd[i] < 0)
		{
			local_label_str("----");
			continue;
		}

		sprintf(buf, "%ld", (long) val[i]);
		local_label_str(buf);

		if (val[i] != (SIO4_FEATURE_LAST_INDEX + 1))
		{
			errs++;

			if (err[0] == 0)
			{
				sprintf(err,
						"%d. incorrect count, ch[%d], expected %ld",
						__LINE__,
						i,
						(long) SIO4_FEATURE_LAST_INDEX + 1);
			}
		}

		// Verify that each query passes.

		for (j = 0; j < val[i]; j++)
		{
			if (j == 3)		// No longer supported.
				continue;

			v	= j;
			ret	= sio4_ioctl(fd[i], SIO4_FEATURE_TEST, &v);

			if (ret < 0)
			{
				errs++;

				if (err[0] == 0)
				{
					sprintf(err,
							"%d. SIO4_FEATURE_TEST request[%d]",
							__LINE__,
							j);
				}
			}

			if ((j == SIO4_FEATURE_REG_PSRCR_BITS) && (v == -1))
				continue;

			if ((j == SIO4_FEATURE_REG_PSTSR_BITS) && (v == -1))
				continue;

			if (v == -1)
			{
				errs++;

				if (err[0] == 0)
				{
					sprintf(err,
							"%d. SIO4_FEATURE_TEST request[%d] results",
							__LINE__,
							j);
				}
			}
		}
	}

	if (errs)
		printf("FAIL <---");

	if ((errs) && (err[0]))
		printf("  (%s)", err);

	printf("\n");
	return(errs);
}



//*****************************************************************************
static int _query_list_validate(void)
{
	int	err;
	int	errs	= 0;
	int	i;

	// Missed entries.

	for (err = 0, i = 0; i <= SIO4_FEATURE_LAST_INDEX; i++)
	{
		if (i == 3)		// This one is no longer supported.
			continue;

		if (_query_list[i] == 0)
		{
			if (err == 0)
			{
				err++;
				gsc_label("Query Options Missed");
				printf("FAIL <---  (%d", i);
			}
			else
			{
				err++;
				printf(", %d", i);
			}
		}
	}

	if (err)
	{
		errs++;
		printf(")\n");
	}

	// Duplicate entries.

	for (err = 0, i = 0; i < SIO4_FEATURE_LAST_INDEX; i++)
	{
		if (_query_list[i] > 1)
		{
			if (err == 0)
			{
				err++;
				gsc_label("Query Options Duplicated");
				printf("FAIL <---  (%d", i);
			}
			else
			{
				err++;
				printf(", %d", i);
			}
		}
	}

	if (err)
	{
		errs++;
		printf(")\n");
	}

	return(errs);
}



//*****************************************************************************
static int _show_zilog_rev(const int* fd)
{
	char	buf[32];
	int		err;
	int		errs		= 0;
	s32		fw_type[4]	= { -1, -1, -1, -1 };
	int		i;
	u32		val[4]		= { (u32) -1, (u32) -1, (u32) -1, (u32) -1 };

	gsc_label("Zilog Z16C30 Rev");

	for (i = 0; i <= 3; i++)
	{
		if (fd[i] < 0)
		{
			local_label_str("----");
			continue;
		}

		err		= ioctl(fd[i], SIO4_FW_TYPE_CONFIG, &fw_type[i]);

		if (err)
		{
			local_label_str("FAIL <---");
			errs	+= err;
			continue;
		}

		if (fw_type[i] != SIO4_FW_TYPE_CONFIG_Z16C30)
		{
			local_label_str("----");
			continue;
		}

		err		= 0;
		err		+= sio4_reg_write(fd[i], SIO4_USC_TMCR, 0x1F);
		err		+= sio4_reg_read(fd[i], SIO4_USC_TMDR, &val[i]);
		errs	+= err;

		if (err)
		{
			local_label_str("FAIL <---");
			continue;
		}

		sprintf(buf, "0x%04lX", (long) val[i]);
		local_label_str(buf);

		if (val[i] != 0x4444)
			errs++;
	}

	errs	+= _validate_same_u32(fd, val);
	_show_eol(errs);
	return(errs);
}



//*****************************************************************************
static int _register_map_list_gsc(const int* fd)
{
	char					buf[32];
	int						ch;
	int						err;
	int						errs	= 0;
	int						i;
	const gsc_reg_def_t*	def;
	int						fd_tmp[4];
	u32						mult;
	s32						supp[4];
	u32						type;
	u32						val[4]	= { 0, 0, 0, 0 };

	printf("\n");

	for (i = 0;; i++)
	{
		err	= 0;
		def	= sio4_reg_get_def_index(i);

		if (def == NULL)
			break;

		type	= SIO4_TYPE_DECODE(def->reg);
		mult	= SIO4_CH_MULT_DECODE(def->reg);

		if (type != SIO4_REG_GSC)
			break;

		gsc_label(def->name);

		// Read the register values.

		for (ch = 0; ch <= 3; ch++)
		{
			if (fd[ch] >= 0)
				err	+= sio4_reg_read(fd[ch], def->reg, &val[ch]);
		}

		// Report the register values.
		fd_tmp[0]	= fd[0];
		fd_tmp[1]	= fd[1];
		fd_tmp[2]	= fd[2];
		fd_tmp[3]	= fd[3];

		for (ch = 0; ch <= 3; ch++)
		{
			if (fd[ch] < 0)
			{
				local_label_str("----");
				continue;
			}

			supp[ch]	= 1;

			if (def->ask_support)
			{
				supp[ch]	= (def->decode)(fd[ch], 1, 0, 0);

				if (supp[ch] == 0)
				{
					fd_tmp[ch]	= -1;	// don't compare value if unsupported
					local_label_str("NA");
				}
			}

			if (supp[ch])
			{
				sprintf(buf, "0x%08lX", (long) val[ch]);
				local_label_str(buf);
			}

			if (mult)
				fd_tmp[ch]	= -1;	// don't compare value if this type reg

			if (def->reg == SIO4_GSC_GPIOSR)
				fd_tmp[ch]	= -1;	// don't compare values of this register

			if (def->reg == SIO4_GSC_TSR)
				fd_tmp[ch]	= -1;	// don't compare values of this register
		}

		// Perform channel value comparisons.

		if (def->ask_support)
			err	+= _validate_same_s32(fd, supp);

		err	+= _validate_same_u32(fd_tmp, val);

		if (err)
			printf("FAIL <---\n");
		else
			printf("%s\n", def->desc);

		errs	+= err;
	}

	return(errs);
}



//*****************************************************************************
static int _register_map_list_usc(const int* fd)
{
	char					buf[32];
	int						ch;
	int						err;
	int						errs		= 0;
	s32						fw_type[4]	= { -1, -1, -1, -1 };
	int						i;
	const gsc_reg_def_t*	def;
	u32						type;
	u32						val[4]		= { 0, 0, 0, 0 };

	printf("\n");

	for (ch = 0; ch <= 3; ch++)
	{
		if (fd[ch] < 0)
			continue;

		errs	+= ioctl(fd[ch], SIO4_FW_TYPE_CONFIG, &fw_type[ch]);
	}

	for (i = 0;; i++)
	{
		err	= 0;
		def	= sio4_reg_get_def_index(i);

		if (def == NULL)
			break;

		type	= SIO4_TYPE_DECODE(def->reg);

		if (type == SIO4_REG_GSC)
			continue;

		gsc_label(def->name);

		// Read the register values.

		for (ch = 0; ch <= 3; ch++)
		{
			if (fd[ch] < 0)
				continue;

			if (fw_type[ch] == SIO4_FW_TYPE_CONFIG_Z16C30)
				err	+= sio4_reg_read(fd[ch], def->reg, &val[ch]);
		}

		// Report the register values.

		for (ch = 0; ch <= 3; ch++)
		{
			if ((fd[ch] < 0) || (fw_type[ch] != SIO4_FW_TYPE_CONFIG_Z16C30))
			{
				local_label_str("----");
				continue;
			}

			sprintf(buf, "0x%04lX", (long) val[ch]);
			local_label_str(buf);
		}

		// Perform channel value comparisons.

		if (err)
			printf("FAIL <---\n");
		else
			printf("%s\n", def->desc);

		errs	+= err;
	}

	return(errs);
}



//*****************************************************************************
static int _frr_detail(int fd)
{
	gsc_reg_def_t			list[2]	= { { NULL }, { NULL } };

	const gsc_reg_def_t*	ptr;

	int	errs	= 0;

	printf("\n");
	ptr	= sio4_reg_get_def_id(SIO4_GSC_FRR);

	if (ptr)
	{
		list[0]	= ptr[0];
		errs	= gsc_reg_list(fd, list, 1, reg_read);
	}
	else
	{
		errs++;
		printf(	"FAIL <---  (%d. sio4_reg_get_def_id, SIO4_GSC_FRR)\n",
				__LINE__);
	}

	return(errs);
}



//*****************************************************************************
static int _fr_detail(int fd)
{
	gsc_reg_def_t			list[2]	= { { NULL }, { NULL } };
	const gsc_reg_def_t*	ptr;

	int	errs	= 0;

	printf("\n");
	ptr	= sio4_reg_get_def_id(SIO4_GSC_FR);

	if (ptr)
	{
		list[0]	= ptr[0];
		errs	= gsc_reg_list(fd, list, 1, reg_read);
	}
	else
	{
		errs++;
		printf(	"FAIL <---  (%d. sio4_reg_get_def_id, SIO4_GSC_FR)\n",
				__LINE__);
	}

	return(errs);
}



//*****************************************************************************
int perform_tests(const args_t* args)
{
	int	errs	= 0;
	int	i;
	s32	fw_type;
	int	usc		= 0;

	// This has been added as a recovery option when some boards start
	// returning invalid values for the Zilog version number. The value
	// returned is usually 0x0000.
	errs	+= _board_reset(args->fd);

	// Identification =====================================
	gsc_label("Identification");
	printf("\n");
	memset(_query_list, 0, sizeof(_query_list));

	gsc_label_level_inc();
	errs	+= _query_form_factor		(args->fd);
	errs	+= _query_sio4_type			(args->fd);
	errs	+= _query_fw_type			(args->fd);
	errs	+= _current_fw_type			(args->fd);
	errs	+= _query_bool				(args->fd, SIO4_FEATURE_MODEL_SYNC,				"SYNC Model Device",				1	);
	errs	+= _query_bool				(args->fd, SIO4_FEATURE_MODEL_Z16C30,			"Z16C30 Model Device",				1	);
	errs	+= _query_model_base		(args->fd);
	errs	+= _query_decimal			(args->fd, SIO4_FEATURE_INDEX_BOARD,			"Board Index",			"",			1	);
	errs	+= _query_index_device		(args->fd);
	errs	+= _query_index_subdevice	(args->fd);
	errs	+= _query_index_channel		(args->fd);
	errs	+= _query_fifo_size			(args->fd, SIO4_FEATURE_FIFO_SIZE_TOTAL,		"Total FIFO Size",					1	);
	errs	+= _query_fifo_size			(args->fd, SIO4_FEATURE_FIFO_SIZE_RX,			"Rx FIFO Size",						0	);
	errs	+= _query_fifo_size			(args->fd, SIO4_FEATURE_FIFO_SIZE_TX,			"Tx FIFO Size",						0	);
	errs	+= _query_bool				(args->fd, SIO4_FEATURE_RX_FIFO_FULL_CFG,		"Rx FIFO Full Config",				1	);
	errs	+= _query_bool				(args->fd, SIO4_FEATURE_RX_FIFO_FULL_CFG_GLB,	"Rx FIFO Full Config Glb",			1	);
	errs	+= _query_bool				(args->fd, SIO4_FEATURE_RX_FIFO_OVERRUN,		"Rx FIFO Overrun",					1	);
	errs	+= _query_bool				(args->fd, SIO4_FEATURE_RX_FIFO_UNDERRUN,		"Rx FIFO Underrun",					1	);
	errs	+= _query_bool				(args->fd, SIO4_FEATURE_TX_FIFO_EMPTY_CFG,		"Tx FIFO Empty Config",				1	);
	errs	+= _query_bool				(args->fd, SIO4_FEATURE_TX_FIFO_OVERRUN,		"Tx FIFO Overrun",					1	);
	errs	+= _query_bool				(args->fd, SIO4_FEATURE_FIFO_SPACE_CFG,			"FIFO Space Config",				1	);
	errs	+= _query_decimal			(args->fd, SIO4_FEATURE_USER_JUMPER_QTY,		"User Jumpers",			"",			1	);
	errs	+= _query_bool_str			(args->fd, SIO4_FEATURE_USER_JUMPER_SENSE,		"User Jumper Sense", 1, "1 = On", "0 = On");
	errs	+= _show_user_jumper_val	(args->fd);

	errs	+= _query_decimal			(args->fd, SIO4_FEATURE_BUS_SPEED,				"PCI Bus Speed",		"MHz",		1	);
	errs	+= _query_decimal			(args->fd, SIO4_FEATURE_BUS_WIDTH,				"PCI Bus Width",		"bits",		1	);
	errs	+= _query_decimal			(args->fd, SIO4_FEATURE_DEVICE_QTY,				"Board Devices",		"",			1	);
	errs	+= _query_decimal			(args->fd, SIO4_FEATURE_CHANNEL_QTY,			"Board Channels",		"",			1	);

	errs	+= _query_decimal			(args->fd, SIO4_FEATURE_LED_CHANNEL,			"Channel LEDs",			"",			1	);
	errs	+= _query_decimal			(args->fd, SIO4_FEATURE_LED_MAIN,				"Main LEDs",			"",			1	);

	errs	+= _query_bool				(args->fd, SIO4_FEATURE_FW_TYPE_CONFIG,			"Firmware Type Config",				1	);

	errs	+= _query_osc_chip			(args->fd);
	errs	+= _query_bool				(args->fd, SIO4_FEATURE_OSC_PER_CHANNEL,		"Osc Per Channel",					1	);
	errs	+= _query_bool				(args->fd, SIO4_FEATURE_OSC_PROGRAM,			"Osc Programmable",					1	);
	errs	+= _query_decimal			(args->fd, SIO4_FEATURE_OSC_PD_MAX,				"Osc Max Post Divider",	"",			1	);
	errs	+= _query_bool				(args->fd, SIO4_FEATURE_OSC_MEASURE,			"Osc Measure",						1	);

	errs	+= _query_bool				(args->fd, SIO4_FEATURE_MP,						"MP (Multi-Protocol)",				1	);
	errs	+= _query_mp_chip			(args->fd);
	errs	+= _query_bool				(args->fd, SIO4_FEATURE_MP_PROGRAM,				"MP Programmable",					1	);
	errs	+= _query_hex				(args->fd, SIO4_FEATURE_MP_BITMAP,				"MP Protocol Bitmap",				1	);

	errs	+= _query_bool				(args->fd, SIO4_FEATURE_BOARD_RESET,			"Board Reset",						1	);
	errs	+= _query_bool				(args->fd, SIO4_FEATURE_RX_STATUS_WORD,			"Rx Status Word Enable",			1	);
	errs	+= _query_bool				(args->fd, SIO4_FEATURE_TIME_STAMP,				"Timestamp",						1	);

	errs	+= _query_bool_str			(args->fd, SIO4_FEATURE_IRQ_32,					"32 IRQ Bits", 1, "Yes (32)", "No (16)"	);
	errs	+= _query_bool				(args->fd, SIO4_FEATURE_LEGACY_CABLE,			"Legacy Cable API",					1	);
	errs	+= _query_bool				(args->fd, SIO4_FEATURE_DMDMA_SCD,				"DMDMA Sin Cyc Dis",				1	);
	errs	+= _query_count				(args->fd);

	errs	+= _query_bool				(args->fd, SIO4_FEATURE_REG_BSR,				"Reg: BSR",							1	);
	errs	+= _query_bool				(args->fd, SIO4_FEATURE_REG_CCR,				"Reg: CCR",							1	);
	errs	+= _query_bool				(args->fd, SIO4_FEATURE_REG_FCR,				"Reg: FCR",							1	);
	errs	+= _query_bool				(args->fd, SIO4_FEATURE_REG_FR,					"Reg: FR",							1	);
	errs	+= _query_bool				(args->fd, SIO4_FEATURE_REG_FSR,				"Reg: FSR",							1	);
	errs	+= _query_bool				(args->fd, SIO4_FEATURE_REG_FTR,				"Reg: FTR",							1	);
	errs	+= _query_bool				(args->fd, SIO4_FEATURE_REG_GPIOSR,				"Reg: GPIOSR",						1	);
	errs	+= _query_bool				(args->fd, SIO4_FEATURE_REG_IELR,				"Reg: IELR",						1	);
	errs	+= _query_bool				(args->fd, SIO4_FEATURE_REG_IHLR,				"Reg: IHLR",						1	);
	errs	+= _query_bool				(args->fd, SIO4_FEATURE_REG_IOCR,				"Reg: IOCR",						1	);
	errs	+= _query_bool				(args->fd, SIO4_FEATURE_REG_PCR,				"Reg: PCR",							1	);
	errs	+= _query_bool				(args->fd, SIO4_FEATURE_REG_POCSR,				"Reg: POCSR",						1	);
	errs	+= _query_bool				(args->fd, SIO4_FEATURE_REG_PORAR,				"Reg: PORAR",						1	);
	errs	+= _query_bool				(args->fd, SIO4_FEATURE_REG_PORDR,				"Reg: PORDR",						1	);
	errs	+= _query_bool				(args->fd, SIO4_FEATURE_REG_PORD2R,				"Reg: PORD2R",						1	);
	errs	+= _query_bool				(args->fd, SIO4_FEATURE_REG_PSRCR,				"Reg: PSRCR",						1	);
	errs	+= _query_u32				(args->fd, SIO4_FEATURE_REG_PSRCR_BITS,			"Reg: PSRCR Bits",	"", "0x%08lX",	1	);
	errs	+= _query_bool				(args->fd, SIO4_FEATURE_REG_PSTSR,				"Reg: PSTSR",						1	);
	errs	+= _query_u32				(args->fd, SIO4_FEATURE_REG_PSTSR_BITS,			"Reg: PSTSR Bits",	"", "0x%08lX",	1	);
	errs	+= _query_bool				(args->fd, SIO4_FEATURE_REG_RCR,				"Reg: RCR",							1	);
	errs	+= _query_bool				(args->fd, SIO4_FEATURE_REG_SBR,				"Reg: SBR",							1	);
	errs	+= _query_bool				(args->fd, SIO4_FEATURE_REG_TCR,				"Reg: TCR",							1	);
	errs	+= _query_bool				(args->fd, SIO4_FEATURE_REG_TSR,				"Reg: TSR",							1	);

	gsc_label_level_dec();

	errs	+= _query_list_validate();

	errs	+= _show_zilog_rev(args->fd);

	errs	+= _register_map_list_gsc(args->fd);
	fw_type	= -1;

	for (i = 0; i <= 3; i++)
	{
		if (args->fd[i] >= 0)
		{
			fw_type	= SIO4_FW_TYPE_CONFIG_READ;
			errs	+= ioctl(args->fd[i], SIO4_FW_TYPE_CONFIG, &fw_type);

			if (fw_type == SIO4_FW_TYPE_CONFIG_Z16C30)
				usc	= 1;

			break;
		}
	}

	if (usc)
		errs	+= _register_map_list_usc(args->fd);

	for (i = 0; i <= 3; i++)
	{
		if (args->fd[i] >= 0)
		{
			errs	+= _frr_detail(args->fd[i]);
			errs	+= _fr_detail(args->fd[i]);
			break;
		}
	}

	printf("\n");

	return(errs);
}


