// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/driver/util.c $
// $Rev: 48525 $
// $Date: 2020-11-23 18:29:37 -0600 (Mon, 23 Nov 2020) $

// SIO4: Device Driver: source file

#include "main.h"



/******************************************************************************
*
*	Function:	_parm_2x32_check
*
*	Purpose:
*
*		Locate the given parameter value in the given list.
*
*	Arguments:
*
*		parm	The parameter to match within the below list.
*
*		list	The data list that described the supported parameter values.
*
*	Returned:
*
*		< 0		The value wasn't given in the list.
*		>= 0	The index in the list where the value was found.
*
******************************************************************************/

static int _parm_2x32_check(u32 parm, const parm_2x32_t* list)
{
	int	i;
	int	index	= -1;

	for (i = 0; list[i].mask1; i++)
	{
		if (parm == list[i].parm)
		{
			index	= i;
			break;
		}
	}

	return(index);
}



/******************************************************************************
*
*	Function:	parm_2x32_get
*
*	Purpose:
*
*		Read two registers and determine which parameter value corresponds to
*		known field values.
*
*	Arguments:
*
*		chan	The structure for the channel being acessed.
*
*		vaddr1	A 32-bit register to access.
*
*		vaddr2	A 32-bit register to access.
*
*		list	The data list that describes the parameter values and the
*				corresponding field values.
*
*	Returned:
*
*		< 0		The value wasn't given in the list.
*		>= 0	The index in the list where the value was found.
*
******************************************************************************/

int parm_2x32_get(
	chan_data_t*		chan,
	VADDR_T				vaddr1,
	VADDR_T				vaddr2,
	const parm_2x32_t*	list)
{
	dev_data_t* dev		= chan->dev;
	int			i;
	int			index	= -1;
	u32			reg1;
	u32			reg2;
	int			shift1;
	int			shift2;
	u32			val1;
	u32			val2;

	reg1	= os_reg_mem_rx_u32(dev, vaddr1);
	reg2	= os_reg_mem_rx_u32(dev, vaddr2);

	for (i = 0; list[i].mask1; i++)
	{
		shift1	= list[i].shift1 * (chan->index);
		shift2	= list[i].shift2 * (chan->index);
		val1	= (reg1 >> shift1) & list[i].mask1;
		val2	= (reg2 >> shift2) & list[i].mask2;

		if ((val1 == list[i].value1) && (val2 == list[i].value2))
		{
			index	= i;
			break;
		}
	}

	return(index);
}



/******************************************************************************
*
*	Function:	parm_2x32_set
*
*	Purpose:
*
*		Apply a parameter setting to two register fields.
*
*	Arguments:
*
*		chan	The structure for the channel being accessed.
*
*		vaddr1	A 32-bit register to access.
*
*		vaddr2	A 32-bit register to access.
*
*		parm	The parameter to match within the below list.
*
*		list	The data list that described the value for the supported list
*				of parameter values.
*
*	Returned:
*
*		< 0		The value wasn't given in the list.
*		>= 0	The index in the list where the value was found.
*
******************************************************************************/

int parm_2x32_set(
	chan_data_t*		chan,
	VADDR_T				vaddr1,
	VADDR_T				vaddr2,
	u32					parm,
	const parm_2x32_t*	list)
{
	dev_data_t*	dev	= chan->dev;
	int			i;
	u32			mask;
	int			shift;
	u32			val;

	i	= _parm_2x32_check(parm, list);

	if (i >= 0)
	{
		shift	= list[i].shift1 * (chan->index);
		mask	= list[i].mask1 << shift;
		val	= list[i].value1 << shift;
		os_reg_mem_mx_u32(dev, vaddr1, val, mask);

		shift	= list[i].shift2 * (chan->index);
		mask	= list[i].mask2 << shift;
		val	= list[i].value2 << shift;
		os_reg_mem_mx_u32(dev, vaddr2, val, mask);
	}

	return(i);
}



/******************************************************************************
*
*	Function:	parm_32_check
*
*	Purpose:
*
*		Locate the given parameter value in the given list.
*
*	Arguments:
*
*		parm	The parameter to match within the below list.
*
*		list	The data list that described the supported parameter values.
*
*	Returned:
*
*		< 0		The value wasn't given in the list.
*		>= 0	The index in the list where the value was found.
*
******************************************************************************/

int parm_32_check(u32 parm, const parm_32_t* list)
{
	int	i;
	int	index	= -1;

	for (i = 0; list[i].mask; i++)
	{
		if (parm == list[i].parm)
		{
			index	= i;
			break;
		}
	}

	return(index);
}



/******************************************************************************
*
*	Function:	parm_32_get
*
*	Purpose:
*
*		Read a register and determin parameter value corresponds to a field
*		value in the register.
*
*	Arguments:
*
*		dev		The structure for the device to access.
*
*		vaddr	The 32-bit register to access.
*
*		list	The data list that describes the parameter values and the
*				corresponding field values.
*
*	Returned:
*
*		< 0		The value wasn't given in the list.
*		>= 0	The index in the list where the value was found.
*
******************************************************************************/

int parm_32_get(dev_data_t* dev, VADDR_T vaddr, const parm_32_t* list)
{
	int	i;
	int	index	= -1;
	u32	reg;
	u32	val;

	reg	= os_reg_mem_rx_u32(dev, vaddr);

	for (i = 0; list[i].mask; i++)
	{
		val	= reg & list[i].mask;

		if (val == list[i].val)
		{
			index	= i;
			break;
		}
	}

	return(index);
}



/******************************************************************************
*
*	Function:	parm_32_set
*
*	Purpose:
*
*		Apply a parameter setting to a register field.
*
*	Arguments:
*
*		dev		The structure for the device to access.
*
*		vaddr	The 32-bit register to access.
*
*		parm	The parameter to match within the below list.
*
*		list	The data list that described the value for the supported
*				list of parameter values.
*
*	Returned:
*
*		< 0		The value wasn't given in the list.
*		>= 0	The index in the list where the value was found.
*
******************************************************************************/

int parm_32_set(dev_data_t* dev, VADDR_T vaddr, u32 parm, const parm_32_t* list)
{
	int	i;

	i	= parm_32_check(parm, list);

	if (i >= 0)
		os_reg_mem_mx_u32(dev, vaddr, list[i].val, list[i].mask);

	return(i);
}



/******************************************************************************
*
*	Function:	parm_32_usc_check
*
*	Purpose:
*
*		Locate the given parameter value in the given list.
*
*	Arguments:
*
*		parm	The parameter to match within the below list.
*
*		list	The data list that described the supported parameter values.
*
*	Returned:
*
*		< 0		The value wasn't given in the list.
*		>= 0	The index in the list where the value was found.
*
******************************************************************************/

int parm_32_usc_check(u32 parm, const parm_32_usc_t* list)
{
	int	i;
	int	index	= -1;

	for (i = 0; list[i].mask1; i++)
	{
		if (parm == list[i].parm)
		{
			index	= i;
			break;
		}
	}

	return(index);
}



/******************************************************************************
*
*	Function:	parm_32_usc_get
*
*	Purpose:
*
*		Read a 32-bit GSc register and a USC register, then determine which
*		parameter value corresponds to known field values.
*
*	Arguments:
*
*		dev		The structure for the device to access.
*
*		vaddr1	A 32-bit register to access.
*
*		vaddr2	A USC register to access.
*
*		list	The data list that describes the parameter values and the
*				corresponding field values.
*
*	Returned:
*
*		< 0		The value wasn't given in the list.
*		>= 0	The index in the list where the value was found.
*
******************************************************************************/

int parm_32_usc_get(
	dev_data_t*				dev,
	VADDR_T					vaddr1,
	VADDR_T					vaddr2,
	const parm_32_usc_t*	list)
{
	int	i;
	int	index	= -1;
	u32	reg1;
	u32	reg2;
	u32	val1;
	u32	val2;

	reg1	= os_reg_mem_rx_u32(dev, vaddr1);
	reg2	= usc_reg_vaddr_read(dev, vaddr2);

	for (i = 0; list[i].mask1; i++)
	{
		val1	= reg1 & list[i].mask1;
		val2	= reg2 & list[i].mask2;

		if ((val1 == list[i].value1) && (val2 == list[i].value2))
		{
			index	= i;
			break;
		}
	}

	return(index);
}



/******************************************************************************
*
*	Function:	parm_32_usc_set
*
*	Purpose:
*
*		Apply a parameter setting to a GSC firware register and to a USC
*		register.
*
*	Arguments:
*
*		dev		The structure for the device to access.
*
*		vaddr1	A 32-bit register to access.
*
*		vaddr2	A 32-bit register to access.
*
*		parm	The parameter to match within the below list.
*
*		list	The data list that described the value for the supported list
*				of parameter values.
*
*	Returned:
*
*		< 0		The value wasn't given in the list.
*		>= 0	The index in the list where the value was found.
*
******************************************************************************/

int parm_32_usc_set(
	dev_data_t*				dev,
	VADDR_T					vaddr1,
	VADDR_T					vaddr2,
	u32						parm,
	const parm_32_usc_t*	list)
{
	int	i;

	i	= parm_32_usc_check(parm, list);

	if (i >= 0)
	{
		os_reg_mem_mx_u32(dev, vaddr1, list[i].value1, list[i].mask1);
		usc_reg_vaddr_mod(dev, vaddr2, list[i].value2, list[i].mask2);
	}

	return(i);
}



