// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/driver/osc_fixed.c $
// $Rev: 53095 $
// $Date: 2023-06-13 10:42:41 -0500 (Tue, 13 Jun 2023) $

// SIO4: Device Driver: source file

#include "main.h"



/******************************************************************************
*
*	Function:	osc_fixed_close
*
*	Purpose:
*
*		Perform any needed work as a device is being closed.
*
*	Arguments:
*
*		dev		The device structure to access.
*
*		index	The index of the channel to access.
*
*	Returned:
*
*		None.
*
******************************************************************************/

void osc_fixed_close(dev_data_t* dev, int index)
{
	sio4_osc_t	arg;

	osc_fixed_reset(dev, index, &arg);
}



/******************************************************************************
*
*	Function:	osc_fixed_info
*
*	Purpose:
*
*		Process an Oscillator Informatio request. We don't realy do anything
*		here as the oscillator frequency cannot be altered.
*
*	Arguments:
*
*		dev		The device structure to access.
*
*		index	The index of the channel to access.
*
*		arg		Data is exchanged with the caller here.
*
*	Returned:
*
*		0		Success.
*		< 0		An appropriate error status.
*
******************************************************************************/

int osc_fixed_info(dev_data_t* dev, int index, sio4_osc_t* arg)
{
	osc_fixed_t*	osc		= &dev->osc.fixed;
	int				ret;

	for (;;)	// A convenience loop.
	{
		arg->chip	= SIO4_OSC_CHIP_FIXED;

		if ((index < 0) || (index > 3))
		{
			ret	= -EINVAL;
			break;
		}

		ret	= os_sem_lock(&osc->sem);

		if (ret)
		{
			ret	= -ERESTARTSYS;
			break;
		}

		arg->freq_ref	= osc->freq_ref;
		arg->freq_want	= osc->chan[index].freq_want;
		arg->freq_got	= osc->chan[index].freq_got;
		os_sem_unlock(&osc->sem);
		break;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	osc_fixed_init
*
*	Purpose:
*
*		Process an Oscillator Initialization request. We don't realy do
*		anything here as the oscillator frequency cannot be altered.
*
*	Arguments:
*
*		dev		The device structure to access.
*
*		index	The index of the channel to access.
*
*		arg		Data is exchanged with the caller here.
*
*	Returned:
*
*		0		Success.
*		< 0		An appropriate error status.
*
******************************************************************************/

int osc_fixed_init(dev_data_t* dev, int index, sio4_osc_t* arg)
{
	osc_fixed_t*	osc		= &dev->osc.fixed;
	int				ret;

	for (;;)	// A convenience loop.
	{
		arg->chip	= SIO4_OSC_CHIP_FIXED;

		if ((index < 0) || (index > 3))
		{
			ret	= -EINVAL;
			break;
		}

		ret	= os_sem_lock(&osc->sem);

		if (ret)
		{
			ret	= -ERESTARTSYS;
			break;
		}

		osc->chan[index].freq_want	= osc->freq_ref;
		osc->chan[index].freq_got	= osc->freq_ref;
		arg->freq_ref			= osc->freq_ref;
		arg->freq_want			= osc->chan[index].freq_want;
		arg->freq_got			= osc->chan[index].freq_got;
		os_sem_unlock(&osc->sem);
		break;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	osc_fixed_measure
*
*	Purpose:
*
*		Process a Frquency Measurement request. We don't realy do anything here
*		as the oscillator frequency cannot be measured.
*
*	Arguments:
*
*		dev		The device structure to access.
*
*		index	The index of the channel to access.
*
*		arg		Data is exchanged with the caller here.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

int osc_fixed_measure(dev_data_t* dev, int index, sio4_osc_t* arg)
{
	osc_fixed_t*	osc		= &dev->osc.fixed;
	int				ret;

	for (;;)	// A convenience loop.
	{
		arg->chip	= SIO4_OSC_CHIP_FIXED;

		if ((index < 0) || (index > 3))
		{
			ret	= -EINVAL;
			break;
		}

		ret	= os_sem_lock(&osc->sem);

		if (ret)
		{
			ret	= -ERESTARTSYS;
			break;
		}

		arg->freq_ref	= osc->freq_ref;
		arg->freq_want	= osc->chan[index].freq_want;
		arg->freq_got	= -1;
		os_sem_unlock(&osc->sem);
		break;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	osc_fixed_open
*
*	Purpose:
*
*		Perform any needed work as a device is being opened.
*
*	Arguments:
*
*		dev		The device structure to access.
*
*		index	The index of the channel to access.
*
*	Returned:
*
*		None.
*
******************************************************************************/

void osc_fixed_open(dev_data_t* dev, int index)
{
	sio4_osc_t	arg;

	osc_fixed_init(dev, index, &arg);
}



/******************************************************************************
*
*	Function:	osc_fixed_program
*
*	Purpose:
*
*		Process an Oscillator Programming request. We don't realy do anything
*		here as the oscillator frequency cannot be altered.
*
*	Arguments:
*
*		dev		The device structure to access.
*
*		index	The index of the channel to access.
*
*		arg		Data is exchanged with the caller here.
*
*	Returned:
*
*		0		Success.
*		< 0		An appropriate error status.
*
******************************************************************************/

int osc_fixed_program(dev_data_t* dev, int index, sio4_osc_t* arg)
{
	osc_fixed_t*	osc		= &dev->osc.fixed;
	int				ret;

	for (;;)	// A convenience loop.
	{
		arg->chip	= SIO4_OSC_CHIP_FIXED;

		if (	(index < 0) ||
			(index > 3) ||
			(arg->freq_want < 0) ||
			(arg->freq_want > _20MHZ))
		{
			ret	= -EINVAL;
			break;
		}

		ret	= os_sem_lock(&osc->sem);

		if (ret)
		{
			ret	= -ERESTARTSYS;
			break;
		}

		osc->chan[index].freq_want	= arg->freq_want;
		osc->chan[index].freq_got	= osc->freq_ref;
		arg->freq_ref			= osc->freq_ref;
		arg->freq_got			= osc->chan[index].freq_got;
		os_sem_unlock(&osc->sem);
		break;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	osc_fixed_reference
*
*	Purpose:
*
*		Process a Reference Frquency request. We don't realy do anything here
*		as the oscillator frequency cannot be altered.
*
*	Arguments:
*
*		dev		The device structure to access.
*
*		index	The index of the channel to access.
*
*		arg		Data is exchanged with the caller here.
*
*	Returned:
*
*		>= 0	Success.
*		< 0		Try another configuration option.
*
******************************************************************************/

int osc_fixed_reference(dev_data_t* dev, int index, sio4_osc_t* arg)
{
	osc_fixed_t*	osc		= &dev->osc.fixed;
	int				ret;

	for (;;)	// A convenience loop.
	{
		arg->chip	= SIO4_OSC_CHIP_FIXED;

		if ((index < 0) || (index > 3))
		{
			ret	= -EINVAL;
			break;
		}

		if (	(arg->freq_ref != -1) &&
			((arg->freq_ref < _1MHZ) ||
			(arg->freq_ref > _20MHZ)))
		{
			ret	= -EINVAL;
			break;
		}

		ret	= os_sem_lock(&osc->sem);

		if (ret)
		{
			ret	= -ERESTARTSYS;
			break;
		}

		if (arg->freq_ref > 0)
			osc->freq_ref	= arg->freq_ref;

		arg->freq_ref	= osc->freq_ref;
		arg->freq_want	= osc->chan[index].freq_want;
		arg->freq_got	= osc->chan[index].freq_got;
		os_sem_unlock(&osc->sem);
		break;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	osc_fixed_reset
*
*	Purpose:
*
*		Process an Oscillator Reset request. We don't realy do anything here as
*		the oscillator frequency cannot be altered.
*
*	Arguments:
*
*		dev		The device structure to access.
*
*		index	The index of the channel to access.
*
*		arg		Data is exchanged with the caller here.
*
*	Returned:
*
*		0		Success.
*		< 0		An appropriate error status.
*
******************************************************************************/

int osc_fixed_reset(dev_data_t* dev, int index, sio4_osc_t* arg)
{
	osc_fixed_t*	osc		= &dev->osc.fixed;
	int				ret;

	for (;;)	// A convenience loop.
	{
		arg->chip	= SIO4_OSC_CHIP_FIXED;

		if ((index < 0) || (index > 3))
		{
			ret	= -EINVAL;
			break;
		}

		ret	= os_sem_lock(&osc->sem);

		if (ret)
		{
			ret	= -ERESTARTSYS;
			break;
		}

		osc->chan[index].freq_want	= 0;
		osc->chan[index].freq_got	= osc->freq_ref;
		arg->freq_ref				= osc->freq_ref;
		arg->freq_want				= osc->chan[index].freq_want;
		arg->freq_got				= osc->chan[index].freq_got;
		os_sem_unlock(&osc->sem);
		break;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	osc_fixed_startup
*
*	Purpose:
*
*		Process an Oscillator Startup request. We don't realy do anything here
*		as the oscillator frequency cannot be altered.
*
*	Arguments:
*
*		dev		The device structure to access.
*
*	Returned:
*
*		None.
*
******************************************************************************/

void osc_fixed_startup(dev_data_t* dev)
{
	osc_fixed_t*	osc	= &dev->osc.fixed;

	memset(osc, 0, sizeof(osc[0]));
	os_sem_create(&osc->sem);
	osc->freq_ref	= SIO4_OSC_FREQ_REF_DEFAULT;
}



/******************************************************************************
*
*	Function:	osc_fixed_test
*
*	Purpose:
*
*		Process an Oscillator Test request. We don't realy do anything here as
*		the oscillator frequency cannot be altered.
*
*	Arguments:
*
*		dev		The device structure to access.
*
*		index	The index of the channel to access.
*
*		arg		Data is exchanged with the caller here.
*
*	Returned:
*
*		0		Success.
*		< 0		An appropriate error status.
*
******************************************************************************/

int osc_fixed_test(dev_data_t* dev, int index, sio4_osc_t* arg)
{
	osc_fixed_t*	osc		= &dev->osc.fixed;
	int				ret;

	for (;;)	// A convenience loop.
	{
		arg->chip	= SIO4_OSC_CHIP_FIXED;

		if (	(index < 0) ||
			(index > 3) ||
			(arg->freq_want < 0) ||
			(arg->freq_want > _20MHZ))
		{
			ret	= -EINVAL;
			break;
		}

		ret	= os_sem_lock(&osc->sem);

		if (ret)
		{
			ret	= -ERESTARTSYS;
			break;
		}

		arg->freq_ref			= osc->freq_ref;
		arg->freq_got			= osc->freq_ref;
		os_sem_unlock(&osc->sem);
		break;
	}

	return(ret);
}



