// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/driver/osc.c $
// $Rev: 48525 $
// $Date: 2020-11-23 18:29:37 -0600 (Mon, 23 Nov 2020) $

// SIO4: Device Driver: source file

#include "main.h"



/******************************************************************************
*
*	Function:	osc_close
*
*	Purpose:
*
*		Perform cleanup after the channel is closed.
*
*	Arguments:
*
*		chan	The channel structure to access.
*
*	Returned:
*
*		None.
*
******************************************************************************/

void osc_close(chan_data_t* chan)
{
	dev_data_t*	dev	= chan->dev;

	switch (dev->cache.osc_chip)
	{
		default:
		case SIO4_OSC_CHIP_UNKNOWN:

			osc_unknown_close(dev, chan->index);
			break;

		case SIO4_OSC_CHIP_CY22393:
		case SIO4_OSC_CHIP_CY22393_2:

			osc_cy22393_sio4_close(dev, chan->index);
			break;

		case SIO4_OSC_CHIP_FIXED:

			osc_fixed_close(dev, chan->index);
			break;

		case SIO4_OSC_CHIP_IDC2053B:
		case SIO4_OSC_CHIP_IDC2053B_4:

			break;
	}
}



/******************************************************************************
*
*	Function:	osc_info_ioctl
*
*	Purpose:
*
*		Implement the Oscillator Information IOCTL service. Provide information
*		on channel's overall configuration.
*
*	Arguments:
*
*		chan	The structure for the channel to access.
*
*		arg		The argument passed by the application.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

int osc_info_ioctl(chan_data_t* chan, void* arg)
{
	dev_data_t*	dev		= chan->dev;
	sio4_osc_t*	parm	= arg;
	int			ret;

	switch (dev->cache.osc_chip)
	{
		default:
		case SIO4_OSC_CHIP_UNKNOWN:

			ret	= osc_unknown_info(dev, chan->index, parm);
			break;

		case SIO4_OSC_CHIP_CY22393:
		case SIO4_OSC_CHIP_CY22393_2:

			ret	= osc_cy22393_sio4_info(dev, chan->index, parm);
			break;

		case SIO4_OSC_CHIP_FIXED:

			ret	= osc_fixed_info(dev, chan->index, parm);
			break;

		case SIO4_OSC_CHIP_IDC2053B:
		case SIO4_OSC_CHIP_IDC2053B_4:

			parm->chip	= dev->cache.osc_chip;
			ret			= -EIO;
			break;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	osc_init_ioctl
*
*	Purpose:
*
*		Implement the Programmable Oscillator Initialize IOCTL service.
*		Initialize the given channel's clock source.
*
*	Arguments:
*
*		chan	The structure for the channel to access.
*
*		arg		The argument passed by the application.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

int osc_init_ioctl(chan_data_t* chan, void* arg)
{
	dev_data_t*	dev		= chan->dev;
	sio4_osc_t*	parm	= arg;
	int			ret;

	switch (dev->cache.osc_chip)
	{
		default:
		case SIO4_OSC_CHIP_UNKNOWN:

			ret	= osc_unknown_init(dev, chan->index, parm);
			break;

		case SIO4_OSC_CHIP_CY22393:
		case SIO4_OSC_CHIP_CY22393_2:

			ret	= osc_cy22393_sio4_init(dev, chan->index, parm);
			break;

		case SIO4_OSC_CHIP_FIXED:

			ret	= osc_fixed_init(dev, chan->index, parm);
			break;

		case SIO4_OSC_CHIP_IDC2053B:
		case SIO4_OSC_CHIP_IDC2053B_4:

			parm->chip	= dev->cache.osc_chip;
			ret		= -EIO;
			break;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	osc_measure_ioctl
*
*	Purpose:
*
*		Implement the Programmable Oscillator Frequency Measure IOCTL service.
*		Measure the channel's output frequency, if possible.
*
*	Arguments:
*
*		chan	The structure for the channel to access.
*
*		arg		The argument passed by the application.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

int osc_measure_ioctl(chan_data_t* chan, void* arg)
{
	dev_data_t*	dev		= chan->dev;
	sio4_osc_t*	parm	= arg;
	int			ret;

	switch (dev->cache.osc_chip)
	{
		default:
		case SIO4_OSC_CHIP_UNKNOWN:

			ret	= osc_unknown_measure(dev, chan->index, parm);
			break;

		case SIO4_OSC_CHIP_CY22393:
		case SIO4_OSC_CHIP_CY22393_2:

			ret	= osc_cy22393_sio4_measure(dev, chan->index, parm);
			break;

		case SIO4_OSC_CHIP_FIXED:

			ret	= osc_fixed_measure(dev, chan->index, parm);
			break;

		case SIO4_OSC_CHIP_IDC2053B:
		case SIO4_OSC_CHIP_IDC2053B_4:

			parm->chip	= dev->cache.osc_chip;
			ret			= -EIO;
			break;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	osc_open
*
*	Purpose:
*
*		Perform setup for a channel being opened.
*
*	Arguments:
*
*		chan	The channel structure to access.
*
*	Returned:
*
*		None.
*
******************************************************************************/

void osc_open(chan_data_t* chan)
{
	dev_data_t*	dev	= chan->dev;

	switch (dev->cache.osc_chip)
	{
		default:
		case SIO4_OSC_CHIP_UNKNOWN:

			osc_unknown_open(dev, chan->index);
			break;

		case SIO4_OSC_CHIP_CY22393:
		case SIO4_OSC_CHIP_CY22393_2:

			osc_cy22393_sio4_open(dev, chan->index);
			break;

		case SIO4_OSC_CHIP_FIXED:

			osc_fixed_open(dev, chan->index);
			break;

		case SIO4_OSC_CHIP_IDC2053B:
		case SIO4_OSC_CHIP_IDC2053B_4:

			break;
	}
}



/******************************************************************************
*
*	Function:	osc_program_ioctl
*
*	Purpose:
*
*		Implement the Programmable Oscillator Program IOCTL service. Program
*		the oscillator, if possible, for the requested frequency.
*
*	Arguments:
*
*		chan	The structure for the channel to access.
*
*		arg		The argument passed by the application.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

int osc_program_ioctl(chan_data_t* chan, void* arg)
{
	dev_data_t*	dev		= chan->dev;
	sio4_osc_t*	parm	= arg;
	int			ret;

	switch (dev->cache.osc_chip)
	{
		default:
		case SIO4_OSC_CHIP_UNKNOWN:

			ret	= osc_unknown_program(dev, chan->index, parm);
			break;

		case SIO4_OSC_CHIP_CY22393:
		case SIO4_OSC_CHIP_CY22393_2:

			ret	= osc_cy22393_sio4_program(dev, chan->index, parm);
			break;

		case SIO4_OSC_CHIP_FIXED:

			ret	= osc_fixed_program(dev, chan->index, parm);
			break;

		case SIO4_OSC_CHIP_IDC2053B:
		case SIO4_OSC_CHIP_IDC2053B_4:

			parm->chip	= dev->cache.osc_chip;
			ret			= -EIO;
			break;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	osc_reference_ioctl
*
*	Purpose:
*
*		Implement the Oscillator Reference IOCTL service. Tell the oscillator
*		code the channel's reference frequency. If the specified reference
*		frequency is zero, then we report the current setting.
*
*	Arguments:
*
*		chan	The structure for the channel to access.
*
*		arg		The argument passed by the application.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

int osc_reference_ioctl(chan_data_t* chan, void* arg)
{
	dev_data_t*	dev		= chan->dev;
	sio4_osc_t*	parm	= arg;
	int			ret;

	switch (dev->cache.osc_chip)
	{
		default:
		case SIO4_OSC_CHIP_UNKNOWN:

			ret	= osc_unknown_reference(dev, chan->index, parm);
			break;

		case SIO4_OSC_CHIP_CY22393:
		case SIO4_OSC_CHIP_CY22393_2:

			ret	= osc_cy22393_sio4_reference(dev, chan->index, parm);
			break;

		case SIO4_OSC_CHIP_FIXED:

			ret	= osc_fixed_reference(dev, chan->index, parm);
			break;

		case SIO4_OSC_CHIP_IDC2053B:
		case SIO4_OSC_CHIP_IDC2053B_4:

			parm->chip	= dev->cache.osc_chip;
			ret			= -EIO;
			break;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	osc_reset_ioctl
*
*	Purpose:
*
*		Implement the Programmable Oscillator Reset IOCTL service. Reset the
*		given channel's clock source.
*
*	Arguments:
*
*		chan	The structure for the channel to access.
*
*		arg		The argument passed by the application.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

int osc_reset_ioctl(chan_data_t* chan, void* arg)
{
	dev_data_t*	dev		= chan->dev;
	sio4_osc_t*	parm	= arg;
	int			ret;

	switch (dev->cache.osc_chip)
	{
		default:
		case SIO4_OSC_CHIP_UNKNOWN:

			ret	= osc_unknown_reset(dev, chan->index, parm);
			break;

		case SIO4_OSC_CHIP_CY22393:
		case SIO4_OSC_CHIP_CY22393_2:

			ret	= osc_cy22393_sio4_reset(dev, chan->index, parm);
			break;

		case SIO4_OSC_CHIP_FIXED:

			ret	= osc_fixed_reset(dev, chan->index, parm);
			break;

		case SIO4_OSC_CHIP_IDC2053B:
		case SIO4_OSC_CHIP_IDC2053B_4:

			parm->chip	= dev->cache.osc_chip;
			ret			= -EIO;
			break;
	}

	return(ret);
}



/******************************************************************************
*
*	Function:	osc_startup
*
*	Purpose:
*
*		Perform a one-time initialization of the oscillator.
*
*	Arguments:
*
*		dev		The device structure to access.
*
*	Returned:
*
*		None.
*
******************************************************************************/

void osc_startup(dev_data_t* dev)
{
	switch (dev->cache.osc_chip)
	{
		default:
		case SIO4_OSC_CHIP_UNKNOWN:

			osc_unknown_startup(dev);
			break;

		case SIO4_OSC_CHIP_CY22393:
		case SIO4_OSC_CHIP_CY22393_2:

			osc_cy22393_sio4_startup(dev);
			break;

		case SIO4_OSC_CHIP_FIXED:

			osc_fixed_startup(dev);
			break;

		case SIO4_OSC_CHIP_IDC2053B:
		case SIO4_OSC_CHIP_IDC2053B_4:

			break;
	}
}



/******************************************************************************
*
*	Function:	osc_test_ioctl
*
*	Purpose:
*
*		Implement the Programmable Oscillator Test IOCTL service. Perform a
*		check to what frequency we should expect for a given desired frequency.
*
*	Arguments:
*
*		chan	The structure for the channel to access.
*
*		arg		The argument passed by the application.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

int osc_test_ioctl(chan_data_t* chan, void* arg)
{
	dev_data_t*	dev		= chan->dev;
	sio4_osc_t*	parm	= arg;
	int			ret;

	switch (dev->cache.osc_chip)
	{
		default:
		case SIO4_OSC_CHIP_UNKNOWN:

			ret	= osc_unknown_test(dev, chan->index, parm);
			break;

		case SIO4_OSC_CHIP_CY22393:
		case SIO4_OSC_CHIP_CY22393_2:

			ret	= osc_cy22393_sio4_test(dev, chan->index, parm);
			break;

		case SIO4_OSC_CHIP_FIXED:

			ret	= osc_fixed_test(dev, chan->index, parm);
			break;

		case SIO4_OSC_CHIP_IDC2053B:
		case SIO4_OSC_CHIP_IDC2053B_4:

			parm->chip	= dev->cache.osc_chip;
			ret			= -EIO;
			break;
	}

	return(ret);
}



