// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/driver/open.c $
// $Rev: 53095 $
// $Date: 2023-06-13 10:42:41 -0500 (Tue, 13 Jun 2023) $

// SIO4: Device Driver: source file

#include "main.h"



/******************************************************************************
*
*	Function:	os_open
*
*	Purpose:
*
*		Implement the device open functionality.
*
*	Arguments:
*
*		inode	The inode structure used to access the device.
*
*		filp	The file pointer.
*
*	Returned:
*
*		0		All went well.
*		< 0		An appropriate error status.
*
******************************************************************************/

s32 os_open(struct inode* inode, struct file* filp)
{
	dev_data_t*		dev;
	chan_data_t*	chan;
	s32				fw_type;
	u8*				pu8Addr;
	int				ret;

	for (;;)	// A convenience loop.
	{
		ret		= dev_lock_inode(inode, 1, &chan);

		if (ret)
		{
			// We coundn't get exclusive access.
			break;
		}

		if (chan->in_use)
		{
			// The channel is already in use.
			os_sem_unlock(&chan->sem);
			ret	= -EBUSY;
			break;
		}

		ret	= os_module_count_inc();

		if (ret)
		{
			// We coundn't increase the usage count.
			os_sem_unlock(&chan->sem);
			break;
		}

		chan->in_use		= 1;
		filp->private_data	= chan;
		chan->irq.driver	= 0;
		chan->irq.notify	= 0;
		chan->irq.status	= 0;

		dev	= chan->dev;
		ret	= dev_io_open(chan, &chan->tx);
		ret	= ret ? ret : dev_io_open(chan, &chan->rx);
		ret	= ret ? ret : dev_irq_open(dev);

		if (ret)
		{
			// I/O and/or IRQ initialization failed.
			os_sem_unlock(&chan->sem);
			close_channel(chan);
			os_module_count_dec();
			break;
		}

		osc_open(chan);
		mp_open(chan);

		// Force immediate read retry.
		os_reg_mem_tx_u32(dev, dev->vaddr.plx_marbr_32, 0x03000000);

		// Cleanup the Z16C30 resources.
		ret	= fw_type_config_get(chan, &fw_type);

		if ((ret == 0) && (fw_type == SIO4_FW_TYPE_Z16C30))
		{
			// This is a Z16C30 device.

			// clear the interrupt control/status registers
			pu8Addr = (u8*) dev->vaddr.gsc_regs;

			// clear the USC interrupt control register
			os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_INT_CTRL_LOW_REG(chan)), 0);
			os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_INT_CTRL_HIGH_REG(chan)), 0);

			// clear the receive interrupt ctl/stat reg
			os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_RX_INTR_CTRL_LOW_REG(chan)), 0);
			os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_RX_CMD_STAT_LOW_REG(chan)), 0xff);

			// clear the transmit interrupt ctl/stat reg
			os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_TX_INTR_CTRL_LOW_REG(chan)), 0);
			os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_TX_CMD_STAT_LOW_REG(chan)), 0xff);

			// clear the status interrupt ctl, misc stat reg
			os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_STAT_INT_CTRL_LOW_REG(chan)), 0);
			os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_MISC_INT_STAT_LOW_REG(chan)), 0xff);
			os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_MISC_INT_STAT_HIGH_REG(chan)), 0xff);

			// clear the DCCR
			os_reg_mem_tx_u8(dev, (VADDR_T) (pu8Addr + ZILOG_DAISY_CH_CTRL_LOW_REG(chan)), 0xbf);
		}

		// We're done.
		os_sem_unlock(&chan->sem);
		break;
	}

	return(ret);
}



