// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/driver/mp_sp508.c $
// $Rev: 48525 $
// $Date: 2020-11-23 18:29:37 -0600 (Mon, 23 Nov 2020) $

// SIO4: Device Driver: source file

#include "main.h"



/******************************************************************************
*
*	Function:	mp_sp508_close
*
*	Purpose:
*
*		Perform any work needed as the device is being closed.
*
*	Arguments:
*
*		dev		The device structure to access.
*
*		index	The index of the channel to access.
*
*	Returned:
*
*		None.
*
******************************************************************************/

void mp_sp508_close(dev_data_t* dev, int index)
{
	sio4_mp_t	arg;

	mp_sp508_reset(dev, index, &arg);
}



/******************************************************************************
*
*	Function:	mp_sp508_config
*
*	Purpose:
*
*		Process an MP Protocol Configuration request.
*
*	Arguments:
*
*		dev		The device structure to access.
*
*		index	The index of the channel to access.
*
*		arg		Data is exchanged with the caller here.
*
*	Returned:
*
*		0		Success.
*		< 0		An appropriate error status.
*
******************************************************************************/

int mp_sp508_config(dev_data_t* dev, int index, sio4_mp_t* arg)
{
	static const sio4_mp_prot_t	list[16]	=
	{
		SIO4_MP_PROT_RS_422_485,		// 0 RS-422/RS-485
		SIO4_MP_PROT_RS_423,			// 1 RS-423
		SIO4_MP_PROT_RS_232,			// 2 RS-232
		SIO4_MP_PROT_INVALID,			// 3 Unused
		SIO4_MP_PROT_RS_530_M1,			// 4 RS-530 Mode 1
		SIO4_MP_PROT_RS_530_M2,			// 5 RS-530 Mode 2
		SIO4_MP_PROT_V35_M1,			// 6 V.35 Mode 1
		SIO4_MP_PROT_V35_M2,			// 7 V.35 Mode 2
		SIO4_MP_PROT_RS_422_423_MM1,	// 8 RS422/423 Mixed Mode 1
		SIO4_MP_PROT_RS_422_423_MM2,	// 9 RS422/423 Mixed Mode 2
		SIO4_MP_PROT_INVALID,			// A Unused
		SIO4_MP_PROT_INVALID,			// B Unused
		SIO4_MP_PROT_INVALID,			// C Unused
		SIO4_MP_PROT_INVALID,			// D Unused
		SIO4_MP_PROT_INVALID,			// E Unused
		SIO4_MP_PROT_DISABLE			// F Hi Z
	};

	chan_data_t*	chan	= &dev->channel[index];
	u32				mask	= 0x0F000000;
	u32				v		= 0;

	arg->chip	= SIO4_MP_CHIP_SP508;

	switch (arg->prot_want)
	{
		default:						mask	= 0xFFFFFFFF;	break;
		case SIO4_MP_PROT_READ:			mask	= 0;			break;

		// List only those protocols supported by this device.
		case SIO4_MP_PROT_RS_422_485:		v	= 0x00000000;	break;
		case SIO4_MP_PROT_RS_423:			v	= 0x01000000;	break;
		case SIO4_MP_PROT_RS_232:			v	= 0x02000000;	break;
		case SIO4_MP_PROT_RS_530_M1:		v	= 0x04000000;	break;
		case SIO4_MP_PROT_RS_530_M2:		v	= 0x05000000;	break;
		case SIO4_MP_PROT_V35_M1:			v	= 0x06000000;	break;
		case SIO4_MP_PROT_V35_M2:			v	= 0x07000000;	break;
		case SIO4_MP_PROT_RS_422_423_MM1:	v	= 0x08000000;	break;
		case SIO4_MP_PROT_RS_422_423_MM2:	v	= 0x09000000;	break;
		case SIO4_MP_PROT_DISABLE:			v	= 0x0F000000;	break;
	}

	if (mask == 0xFFFFFFFF)
	{
		arg->prot_got	= SIO4_MP_PROT_INVALID;
	}
	else
	{
		if (mask)
			os_reg_mem_mx_u32(dev, chan->vaddr.gsc_psrcr_32, v, mask);

		v		= os_reg_mem_rx_u32(dev, chan->vaddr.gsc_psrcr_32);
		v		= (v >> 24) & 0xF;
		arg->prot_got	= list[v];
	}

	return(0);
}



/******************************************************************************
*
*	Function:	mp_sp508_init
*
*	Purpose:
*
*		Process an MP Protocol Initialization request. We select the SIO4's
*		default here, which is RS-422/RS-485.
*
*	Arguments:
*
*		dev		The device structure to access.
*
*		index	The index of the channel to access.
*
*		arg		Data is exchanged with the caller here.
*
*	Returned:
*
*		0		Success.
*		< 0		An appropriate error status.
*
******************************************************************************/

int mp_sp508_init(dev_data_t* dev, int index, sio4_mp_t* arg)
{
	int	ret;

	arg->prot_want	= SIO4_MP_PROT_RS_422_485;
	ret		= mp_sp508_config(dev, index, arg);
	return(ret);
}



/******************************************************************************
*
*	Function:	mp_sp508_open
*
*	Purpose:
*
*		Perform any work needed as the device is being opened.
*
*	Arguments:
*
*		dev		The device structure to access.
*
*		index	The index of the channel to access.
*
*	Returned:
*
*		None.
*
******************************************************************************/

void mp_sp508_open(dev_data_t* dev, int index)
{
	sio4_mp_t	arg;

	mp_sp508_init(dev, index, &arg);
}



/******************************************************************************
*
*	Function:	mp_sp508_reset
*
*	Purpose:
*
*		Process an MP Protocol Reset request. We select the DISABLE option
*		here.
*
*	Arguments:
*
*		dev		The device structure to access.
*
*		index	The index of the channel to access.
*
*		arg		Data is exchanged with the caller here.
*
*	Returned:
*
*		0		Success.
*		< 0		An appropriate error status.
*
******************************************************************************/

int mp_sp508_reset(dev_data_t* dev, int index, sio4_mp_t* arg)
{
	int	ret;

	arg->prot_want	= SIO4_MP_PROT_DISABLE;
	ret		= mp_sp508_config(dev, index, arg);
	return(ret);
}



/******************************************************************************
*
*	Function:	mp_sp508_startup
*
*	Purpose:
*
*		Process an MP Startup request.
*
*	Arguments:
*
*		dev		The device structure to access.
*
*	Returned:
*
*		None.
*
******************************************************************************/

void mp_sp508_startup(dev_data_t* dev)
{
}



/******************************************************************************
*
*	Function:	mp_sp508_test
*
*	Purpose:
*
*		Process an MP Protocol Test request.
*
*	Arguments:
*
*		dev		The device structure to access.
*
*		index	The index of the channel to access.
*
*		arg		Data is exchanged with the caller here.
*
*	Returned:
*
*		0		Success.
*		< 0		An appropriate error status.
*
******************************************************************************/

int mp_sp508_test(dev_data_t* dev, int index, sio4_mp_t* arg)
{
	arg->chip	= SIO4_MP_CHIP_SP508;

	switch (arg->prot_want)
	{
		default:

			arg->prot_got	= SIO4_MP_PROT_INVALID;
			break;

		// List only those protocols supported by this device.
		case SIO4_MP_PROT_DISABLE:
		case SIO4_MP_PROT_RS_232:
		case SIO4_MP_PROT_RS_422_485:
		case SIO4_MP_PROT_RS_422_423_MM1:
		case SIO4_MP_PROT_RS_422_423_MM2:
		case SIO4_MP_PROT_RS_423:
		case SIO4_MP_PROT_RS_530_M1:
		case SIO4_MP_PROT_RS_530_M2:
		case SIO4_MP_PROT_V35_M1:
		case SIO4_MP_PROT_V35_M2:

			arg->prot_got	= arg->prot_want;
			break;
	}

	return(0);
}



