// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/driver/mp_fixed.c $
// $Rev: 48525 $
// $Date: 2020-11-23 18:29:37 -0600 (Mon, 23 Nov 2020) $

// SIO4: Device Driver: source file

#include "main.h"



/******************************************************************************
*
*	Function:	_get_protocol
*
*	Purpose:
*
*		Report the current protocol.
*
*	Arguments:
*
*		chan	The channel structure to access.
*
*		prot	The protocol is reported here.
*
*	Returned:
*
*		None.
*
******************************************************************************/

static void _get_protocol(chan_data_t* chan, u32* prot)
{
	dev_data_t*	dev		= chan->dev;
	u32			psrcr;

	psrcr		= os_reg_mem_rx_u32(dev, chan->vaddr.gsc_psrcr_32);
	prot[0]		= psrcr & SIO4_GSC_PSRCR_MP_PROT;

	switch (prot[0])
	{
		default:

			prot[0]	= (u32) SIO4_MP_PROT_UNKNOWN;
			break;

		case 0x00000000:

			prot[0]	= SIO4_MP_PROT_RS_422_485;
			break;

		case 0x02000000:

			prot[0]	= SIO4_MP_PROT_RS_232;
			break;
	}
}



/******************************************************************************
*
*	Function:	mp_fixed_close
*
*	Purpose:
*
*		Perform any work needed as the device is being closed.
*
*	Arguments:
*
*		dev		The device structure to access.
*
*		index	The index of the channel to access.
*
*	Returned:
*
*		None.
*
******************************************************************************/

void mp_fixed_close(dev_data_t* dev, int index)
{
	sio4_mp_t	arg;

	mp_fixed_reset(dev, index, &arg);
}



/******************************************************************************
*
*	Function:	mp_fixed_config
*
*	Purpose:
*
*		Process an MP Protocol Configuration request.
*
*	Arguments:
*
*		dev		The device structure to access.
*
*		index	The index of the channel to access.
*
*		arg		Data is exchanged with the caller here.
*
*	Returned:
*
*		0		Success.
*		< 0		An appropriate error status.
*
******************************************************************************/

int mp_fixed_config(dev_data_t* dev, int index, sio4_mp_t* arg)
{
	chan_data_t*	chan	= &dev->channel[index];
	u32				prot;

	arg->chip	= SIO4_MP_CHIP_FIXED;

	switch (arg->prot_want)
	{
		default:

			arg->prot_got	= SIO4_MP_PROT_INVALID;
			break;

		case SIO4_MP_PROT_READ:

		// List ALL protocols defined by the interface.
		case SIO4_MP_PROT_DISABLE:
		case SIO4_MP_PROT_RS_232:
		case SIO4_MP_PROT_RS_422_485:
		case SIO4_MP_PROT_RS_423:

			_get_protocol(chan, &prot);
			arg->prot_got	= prot;
			break;
	}

	return(0);
}



/******************************************************************************
*
*	Function:	mp_fixed_init
*
*	Purpose:
*
*		Process an MP Protocol Initialization request.
*
*	Arguments:
*
*		dev		The device structure to access.
*
*		index	The index of the channel to access.
*
*		arg		Data is exchanged with the caller here.
*
*	Returned:
*
*		0		Success.
*		< 0		An appropriate error status.
*
******************************************************************************/

int mp_fixed_init(dev_data_t* dev, int index, sio4_mp_t* arg)
{
	chan_data_t*	chan	= &dev->channel[index];
	u32				prot;

	_get_protocol(chan, &prot);
	arg->chip	= SIO4_MP_CHIP_FIXED;
	arg->prot_want	= prot;
	arg->prot_got	= prot;
	return(0);
}



/******************************************************************************
*
*	Function:	mp_fixed_open
*
*	Purpose:
*
*		Perform any work needed as the device is being opened.
*
*	Arguments:
*
*		dev		The device structure to access.
*
*		index	The index of the channel to access.
*
*	Returned:
*
*		None.
*
******************************************************************************/

void mp_fixed_open(dev_data_t* dev, int index)
{
	sio4_mp_t	arg;

	mp_fixed_init(dev, index, &arg);
}



/******************************************************************************
*
*	Function:	mp_fixed_reset
*
*	Purpose:
*
*		Process an MP Protocol Reset request.
*
*	Arguments:
*
*		dev		The device structure to access.
*
*		index	The index of the channel to access.
*
*		arg		Data is exchanged with the caller here.
*
*	Returned:
*
*		0		Success.
*		< 0		An appropriate error status.
*
******************************************************************************/

int mp_fixed_reset(dev_data_t* dev, int index, sio4_mp_t* arg)
{
	chan_data_t*	chan	= &dev->channel[index];
	u32				prot;

	_get_protocol(chan, &prot);
	arg->chip	= SIO4_MP_CHIP_FIXED;
	arg->prot_want	= prot;
	arg->prot_got	= prot;
	return(0);
}



/******************************************************************************
*
*	Function:	mp_fixed_startup
*
*	Purpose:
*
*		Process an MP Startup request.
*
*	Arguments:
*
*		dev		The device structure to access.
*
*	Returned:
*
*		None.
*
******************************************************************************/

void mp_fixed_startup(dev_data_t* dev)
{
}



/******************************************************************************
*
*	Function:	mp_fixed_test
*
*	Purpose:
*
*		Process an MP Protocol Test request.
*
*	Arguments:
*
*		dev		The device structure to access.
*
*		index	The index of the channel to access.
*
*		arg		Data is exchanged with the caller here.
*
*	Returned:
*
*		0		Success.
*		< 0		An appropriate error status.
*
******************************************************************************/

int mp_fixed_test(dev_data_t* dev, int index, sio4_mp_t* arg)
{
	chan_data_t*	chan	= &dev->channel[index];
	u32				prot;

	arg->chip	= SIO4_MP_CHIP_FIXED;

	switch (arg->prot_want)
	{
		default:

			arg->prot_got	= SIO4_MP_PROT_INVALID;
			break;

		// List ALL protocols defined by the interface.
		case SIO4_MP_PROT_DISABLE:
		case SIO4_MP_PROT_RS_232:
		case SIO4_MP_PROT_RS_422_485:
		case SIO4_MP_PROT_RS_423:

			_get_protocol(chan, &prot);

			if (arg->prot_want == prot)
				arg->prot_got	= prot;
			else
				arg->prot_got	= SIO4_MP_PROT_INVALID;

			break;
	}

	return(0);
}



