// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/driver/main.h $
// $Rev: 53098 $
// $Date: 2023-06-13 10:47:01 -0500 (Tue, 13 Jun 2023) $

// SIO4: Device Driver: header file

#ifndef	__MAIN_H__
#define	__MAIN_H__

#include "sio4.h"

#define	DEV_BAR_SHOW						0
#define	DEV_PCI_ID_SHOW						0
#define	DEV_SUPPORTS_IRQ
#define	DEV_SUPPORTS_PROC_ID_STR
#define	DEV_SUPPORTS_READ
#define	DEV_SUPPORTS_WRITE
#define	GSC_ALT_DEV_GET(a)					(a)->dev
#define	GSC_ALT_STRUCT_GET(i,d)				(d)->channel[(i)]
#define	GSC_ALT_STRUCT_T					chan_data_t
#define	GSC_DEVS_PER_BOARD					4
typedef struct _chan_data_t					chan_data_t;
typedef struct _dev_data_t					dev_data_t;

#include "gsc_main.h"



// macros *********************************************************************

#define	DEV_VERSION							"1.59"
// 1.59	Moved C++ support to sio4_main.h.
//		Updated comments for consistency.
//		Updated firmware type configuration capability detection code.
//		Reduced instances where the firmware type is changed.
//		The firmware type is reset to the board's default when the driver loaded.
//		The firmware type is reset to the board's default when the board is reset.
//		IOCTL services affected by the firmware type use the current firmware type selection.
//		Multiple instances where the firmware type is used have been updated.
//		The Zilog chips are reset only when present.
//		LINTed the sources.
//		Updated for improved support of supported kernel versions.
//		White space cleanup for consistency.
//		Updated the Zilog reset code and removed unused functions.
// 1.58	Reorganized register encodings to comply with the common code format.
// 1.57	White space cleanup.
//		Updated initial file content for consistency.
//		PIO I/O code: fixed timeout bug, improved efficiency
//		Updated per changes to common code.
//		Updated interrupt support for clarity.
//		Deleted unused sources.
//		Renamed some macros and source files for consistency.
// 1.56	The FW Type Config. feature detection logic now resets the Zilog chips.
// 1.55	Updated to use the newer common driver sources.
//		Removed Linux specific content from non-Linux specific source files.
//		White space cleanup.
//		Now using a spinlock rather than enabling and disabling interrupts.
//		Made PCI registers accessible from inside an ISR.
//		Updated to support the 4.0 kernel.
// 1.54	Modified to use newer PLX register header files.
//		Changed from using driver's gsc_kernel_*.h to os_kernel_*.h.
//		Modified to use semaphore library from os_sem.c.
// 1.53	Updated logic for the Feature Type Register detection.
//		Added support for the Firmware Type Configuration option.
//		Updating user jumper data for the PCIE-SIO4BX2.
//		Expanded the content of /proc/sio4.
//		Updated /proc support for newer kernels.
//		Removed unneccessary DELAYs after various register accesses.
//		Additional LINTing.
//		Reduced #include list in driver interface header.
// 1.52	Added explicit MP support for the PCIE-SIO4BX-SYNC.
//		Modified to use the version number from gsc_common/linux/driver.
// 1.51	Corrected a bug in the ISR's asynchronous notification code.
//		All unused GSC ICR register bits are written as zero.
//		IELR and IHLR are accessed only if they are supported.
//		The PSTSR bits feature has been updated for the PCI-SIO4B-SYNC.
// 1.50	Linted the sources.
//		Corrected jumper sense for the PCIe4-SIO8BX2.
//		Updated the protocol support list for the SP508.
//		Corrected the computed bus width for SIO4B and earlier devices.
//		Fixed the Rx FIFO Full Config Global feature check for earlier devices.
//		Fixed the FIFO Size and FIFO Count registers check for earlier devices.
//		Corrected an initialization error (changed open to gsc_open, ...).
//		Corrected an I/O buffer allocation error.
//		Corrected a module count error for failed open() requests.
//		Removed 64-bit compile warnings.
//		Corrected MP service errors: included support for additional modes.
// 1.49	Cleanup of white space and comments.
//		Updated device index numbering.
//		Added macro SIO4_BASE_NAME.
// 1.48	Many, many changes. Refer to the release file for details.
// 1.47	Corrected the polarity of the Rx FIFO Full Config settings.
// 1.46	Renamed SIO4_RX_FIFO_FULL_CONFIG to SIO4_RX_FIFO_FULL_CFG_GLB.
//		Renamed SIO4_FEATURE_BCR_RX_FFC to SIO4_FEATURE_BCR_RX_FFC_GLB.
//		Added the SIO4_RX_FIFO_FULL_CFG_CHAN service.
//		Added the SIO4_FEATURE_CSR_RX_FFC_CHAN feature option.
// 1.45	Added a call to pci_set_master().
//		Corrected jiffies roll over bug in io.c and io_pio.c.
// 1.44	Modified the sources to be compatible with C++.
// 1.43	Updated for the 3.x kernel.
//		Updated code for detection of various device features.
//		Fixed bugs in the SIO4_RX_FIFO_SIZE and SIO4_TX_FIFO_SIZE services.
//		SIO4_RX_FIFO_SIZE and SIO4_TX_FIFO_SIZE return 0 if size is unknown.
//		Added several new Feature Test options.
// 1.42	Updated to use the gsc_kernel_2_[642].[ch] files.
//		Removed compile warnings for Fedora 15.
// 1.41	Fixed a bug in the SIO4_RX_FIFO_FULL_CONFIG implementation.
// 1.40	Corrected a DMA Abort bug.
// 1.39	Fixed "baud rate = 0" bug in the SIO4_INIT_CHANNEL IOCTL service.
// 1.38	No longer call remove_proc_entry() as part of proc_start().
// 1.37	Cleared the module use count macros so count is now accurate.
//		Implemented SIO4_OSC_CHIP_CY22393_2 support.
//		Removed all FIFO Type componants.
//		Fixed a bug in the SIO4_CLEAR_DPLL_STATUS service.
// 1.36	Updated the I/O buffer allocation for 64-bit systems with 4G+ RAM.
//		Fixed an SMP make bug in reg.c.
// 1.35	This is an official release of the previous 1.34.BETA release.
// 1.34 and 1.34.BETA
//		Modified jiffies upper limit for 64-bit builds (now use ULONG_MAX).
//		Serialized all register accesses for SMP operation.
// 1.33	Cached the FIFO Size Register value rather than read it during read
//		and write calls.
// 1.32	Corrected osc prog differences between 32-bit and 64-bit hosts.
// 1.31	BARx resource access requests are now based on the region sizes.
// 1.30	Added support for the SIO4BXR devices.
// 1.29	Fixed a bug with usage count maintenance.
// 1.28	Fixed an oscillator programming bug and a module count bug.
// 1.27	Fixed an INTCSR access bug in the DMA code.
// 1.26	Updated to kernel 2.6.27.
// 1.25	Corrected an error in the firmware's max oscillator post divider.
// 1.24	Updated the oscillator programming code and added sync/txrate.
// 1.23	Updated the driver to 2.6.25 (Red Hat Fedora 9).
// 1.22	Made changes to the oscillator programming code.
//		Measure and reporgram with the CY22393 as needed.
//		Made a few more minor 2.6 porting changes.
// 1.21	Additional porting and updates for newer kernels.
// 1.20	Ported to 64-bit version of Linux.
// 1.19	More updating/fixing of the osc programming code.
//		Correct STOP BITS fied definition in z1630.h.
// 1.18	Removed FIFO Internal/External interface referance items.
//		Fixed a bug in SMP support of the interrupt handler.
// 1.17	Fixed a bug in the CY22393 code for frequencies under 1M.
// 1.16	Added the External FIFOs feature.
// 1.15	Added support for the Single Cycle Disable feature.
// 1.14	More modularization work, plus some housekeeping updates.
// 1.13	Ported to the 2.6 kernel.
// 1.12	Added IOCTL SIO4_CABLE_CONFIG with value macros.
//		Added IOCTL SIO4_DCD_CABLE_CONFIG with value macros.
//		Updated the SIO4_CTS_CABLE_CONFIG value macros.
//		Added IOCTL SIO4_TX_CABLE_DATA_CONFIG with value macros.
// 1.11	Fixed a write bug where error status wasn't returned when
//		copy_from_user() failed.
//		Added SMP support (added a spinlock where appropriate).
//		Made SIO4_RX_CABLE_CLOCK_CONFIG defunct: It isn't in firmware.
//		More firmware ISR bits are checked in the int handler.
// 1.10	Fixed a bug in the ISR where RX & Tx tasks weren't resumed.
// 1.09	Added transceiver support to the device init and channel reset code.
// 1.08	Fixed a bug in the SIO4_INIT_CHANNEL code I created 1.07.
// 1.07	Changed the PIO register write from long to byte.
// 1.06	Added I/O Mode selection ro read()/write().
//		Deleted DMA IOCTL services.
//		Added Multi-Protocol transceiver support w/ IOCTL and data types.
//		Added programmable oscillator support w/ IOCTLs and data types.
//		Added SIO4_MOD_REGISTER and REGISTER_MOD_PARAMS.
//		Added SIO4_READ_REGISTER_RAW.
//		Limited SIO4_WRITE_REGISTER to bits the channel can alter.
//		Fixed an SIO4_TX_FIFO_TYPE bug.

#define	DMA_PIO_THRESHOLD					35	// Use PIO if count is this or less.
#define	SIO4_NAME							"sio4"
#define	DEV_NAME							SIO4_NAME
#define	DEV_MODEL							"SIO4"
#define	DEV_SUPPORTS_PROC_ID_STR

// DMA macros.
#define	DMA_CSR_ENABLE						0x01
#define	DMA_CSR_START						0x02
#define	DMA_CSR_ABORT						0x04
#define	DMA_CSR_CLEAR						0x08
#define	DMA_CSR_DONE						0x10



// These are the masks for the Clock Control Register.
#define	EN_TX_UPPER_SHIFT					0
#define	EN_TX_LOWER_SHIFT					1
#define	EN_RX_UPPER_SHIFT					2
#define	EN_RX_LOWER_SHIFT					3

// Clock Control Register
#define	_GSC_CCR_TCCE						0x3	// Tx Clock Control
#define	_GSC_CCR_TCCE_L						0x2	// Lower
#define	_GSC_CCR_TCCE_U						0x1	// Upper
#define	_GSC_CCR_TCCE_UL					0x3	// Upper & Lower
#define	_GSC_CCR_RCCE						0xC	// Rx Clock Control
#define	_GSC_CCR_RCCE_L						0x8	// Lower
#define	_GSC_CCR_RCCE_U						0x4	// Upper

// This is for receive registers in zilog usc

#define	ZILOG_RX_MODE_LOW_REG(chan)			(34 + ((chan->index + 1) * 0x100))
#define	ZILOG_RX_MODE_HIGH_REG(chan)		(35 + ((chan->index + 1) * 0x100))
#define	ZILOG_RX_CMD_STAT_LOW_REG(chan)		(36 + ((chan->index + 1) * 0x100))
//#define	ZILOG_RX_CMD_STAT_HIGH_REG(chan)	(37 + ((chan->index + 1) * 0x100))
#define	ZILOG_RX_INTR_CTRL_LOW_REG(chan)	(38 + ((chan->index + 1) * 0x100))
//#define	ZILOG_RX_INTR_CTRL_HIGH_REG(chan)	(39 + ((chan->index + 1) * 0x100))
#define	ZILOG_RX_SYNC_LOW_REG(chan)			(40 + ((chan->index + 1) * 0x100))
//#define	ZILOG_RX_SYNC_HIGH_REG(chan)		(41 + ((chan->index + 1) * 0x100))
//#define	ZILOG_RX_CNT_LIMIT_LOW_REG(chan)	(42 + ((chan->index + 1) * 0x100))
//#define	ZILOG_RX_CNT_LIMIT_HIGH_REG(chan)	(43 + ((chan->index + 1) * 0x100))
//#define	ZILOG_RX_CHAR_CNT_LOW_REG(chan)		(44 + ((chan->index + 1) * 0x100))
//#define	ZILOG_RX_CHAR_CNT_HIGH_REG(chan)	(45 + ((chan->index + 1) * 0x100))
#define	ZILOG_TIME_CONST0_LOW_REG(chan)		(46 + ((chan->index + 1) * 0x100))
#define	ZILOG_TIME_CONST0_HIGH_REG(chan)	(47 + ((chan->index + 1) * 0x100))

// This is for transmit registers in zilog usc

#define	ZILOG_TX_MODE_LOW_REG(chan)			(50 + ((chan->index + 1) * 0x100))
#define	ZILOG_TX_MODE_HIGH_REG(chan)		(51 + ((chan->index + 1) * 0x100))
#define	ZILOG_TX_CMD_STAT_LOW_REG(chan)		(52 + ((chan->index + 1) * 0x100))
#define	ZILOG_TX_CMD_STAT_HIGH_REG(chan)	(53 + ((chan->index + 1) * 0x100))
#define	ZILOG_TX_INTR_CTRL_LOW_REG(chan)	(54 + ((chan->index + 1) * 0x100))
//#define	ZILOG_TX_INTR_CTRL_HIGH_REG(chan)	(55 + ((chan->index + 1) * 0x100))
//#define	ZILOG_TX_SYNC_LOW_REG(chan)		(56 + ((chan->index + 1) * 0x100))
//#define	ZILOG_TX_SYNC_HIGH_REG(chan)		(57 + ((chan->index + 1) * 0x100))
//#define	ZILOG_TX_CNT_LIMIT_LOW_REG(chan)	(58 + ((chan->index + 1) * 0x100))
//#define	ZILOG_TX_CNT_LIMIT_HIGH_REG(chan)	(59 + ((chan->index + 1) * 0x100))
//#define	ZILOG_TX_CHAR_CNT_LOW_REG(chan)	(60 + ((chan->index + 1) * 0x100))
//#define	ZILOG_TX_CHAR_CNT_HIGH_REG(chan)	(61 + ((chan->index + 1) * 0x100))
//#define	ZILOG_TIME_CONST1_LOW_REG(chan)	(62 + ((chan->index + 1) * 0x100))
//#define	ZILOG_TIME_CONST1_HIGH_REG(chan)	(63 + ((chan->index + 1) * 0x100))

// Board Control Registers in zilog
// Note: primary and secondary reserved registers are not used

#define	ZILOG_CMD_ADD_HIGH_REG(chan)		(1 + ((chan->index + 1) * 0x100))
#define	ZILOG_CHAN_MODE_LOW_REG(chan)		(2 + ((chan->index + 1) * 0x100))
#define	ZILOG_CHAN_MODE_HIGH_REG(chan)		(3 + ((chan->index + 1) * 0x100))
//#define	ZILOG_CMD_STATUS_LOW_REG(chan)	(4 + ((chan->index + 1) * 0x100))
#define	ZILOG_CMD_STATUS_HIGH_REG(chan)		(5 + ((chan->index + 1) * 0x100))
#define	ZILOG_CHAN_CTRL_LOW_REG(chan)		(6 + ((chan->index + 1) * 0x100))
#define	ZILOG_CHAN_CTRL_HIGH_REG(chan)		(7 + ((chan->index + 1) * 0x100))
//#define	ZILOG_TEST_MODE_DATA_LOW_REG(chan)(12 + ((chan->index + 1) * 0x100))
//#define	ZILOG_TEST_MODE_DATA_HIGH_REG(chan)	(13 + ((chan->index + 1) * 0x100))
//#define	ZILOG_TEST_MODE_CTRL_LOW_REG(chan)(14 + ((chan->index + 1) * 0x100))
//#define	ZILOG_TEST_MODE_CTRL_HIGH_REG(chan)	(15 + ((chan->index + 1) * 0x100))
#define	ZILOG_CLOCK_MODE_CTRL_LOW_REG(chan)	(16 + ((chan->index + 1) * 0x100))
#define	ZILOG_CLOCK_MODE_CTRL_HI_REG(chan)	(17 + ((chan->index + 1) * 0x100))
#define	ZILOG_HWC_LOW_REG(chan)				(18 + ((chan->index + 1) * 0x100))
#define	ZILOG_HWC_HIGH_REG(chan)			(19 + ((chan->index + 1) * 0x100))
//#define	ZILOG_INT_VEC_LOW_REG(chan)		(20 + ((chan->index + 1) * 0x100))
//#define	ZILOG_INT_VEC_HIGH_REG(chan)		(21 + ((chan->index + 1) * 0x100))
//#define	ZILOG_IO_CTRL_LOW_REG(chan)		(22 + ((chan->index + 1) * 0x100))
#define	ZILOG_IO_CTRL_HIGH_REG(chan)		(23 + ((chan->index + 1) * 0x100))
#define	ZILOG_INT_CTRL_LOW_REG(chan)		(24 + ((chan->index + 1) * 0x100))
#define	ZILOG_INT_CTRL_HIGH_REG(chan)		(25 + ((chan->index + 1) * 0x100))
#define	ZILOG_DAISY_CH_CTRL_LOW_REG(chan)	(26 + ((chan->index + 1) * 0x100))
//#define	ZILOG_DAISY_CH_CTRL_HIGH_REG(chan)(27 + ((chan->index + 1) * 0x100))
#define	ZILOG_MISC_INT_STAT_LOW_REG(chan)	(28 + ((chan->index + 1) * 0x100))
#define	ZILOG_MISC_INT_STAT_HIGH_REG(chan)	(29 + ((chan->index + 1) * 0x100))
#define	ZILOG_STAT_INT_CTRL_LOW_REG(chan)	(30 + ((chan->index + 1) * 0x100))
//#define	ZILOG_STAT_INT_CTRL_HIGH_REG(chan)(31 + ((chan->index + 1) * 0x100))



#if	(LINUX_VERSION_CODE >= KERNEL_VERSION(4,0,0)) && \
	(LINUX_VERSION_CODE <  KERNEL_VERSION(5,0,0))

#define	os_bar_pci_address(d,b)		pci_resource_start(((d)->pci.pd),(b))
#define	f_dentry					f_path.dentry

#endif

#if	(LINUX_VERSION_CODE >= KERNEL_VERSION(3,0,0)) && \
	(LINUX_VERSION_CODE <  KERNEL_VERSION(4,0,0))

#define	os_bar_pci_address(d,b)		pci_resource_start(((d)->pci.pd),(b))

#endif

#if	(LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)) && \
	(LINUX_VERSION_CODE <  KERNEL_VERSION(2,7,0))

#define	os_bar_pci_address(d,b)		pci_resource_start(((d)->pci.pd),(b))

#endif

#if	(LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)) && \
	(LINUX_VERSION_CODE <  KERNEL_VERSION(2,5,0))

#define	os_bar_pci_address(d,b)		pci_resource_start(((d)->pci.pd),(b))

#endif

#if	(LINUX_VERSION_CODE >= KERNEL_VERSION(2,2,0)) && \
	(LINUX_VERSION_CODE <  KERNEL_VERSION(2,3,0))

#define	os_bar_pci_address(d,b)		(((d)->pci.pd->base_address[(int)(b)]) & ~0xFL)

#endif



// data types *****************************************************************

struct _dev_data_t;	// forward decleration

typedef struct
{
	int					in_use;
	u32					int_enable;
	u32					dir_mask;	// For DMDMA in BCR.
	u32					dir_shift;	// For DMDMA in BCR.
	WAIT_QUEUE_HEAD_T*	queue;		// Wait for interrupts.

#if	(LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0))
	int					condition;	// Wait for interrupts.
#endif

	struct
	{
		VADDR_T			csr_8;		// DMA X Command/Status Register
		VADDR_T			dpr_32;		// DMA X Descriptor Pointer Register
		VADDR_T			ladr_32;	// DMA X Local Bus Address Register
		VADDR_T			mode_32;	// DMA X Mode Register
		VADDR_T			padr_32;	// DMA X PCI Bus Address Register
		VADDR_T			siz_32;		// DMA X Size Register
	} vaddr;
} dev_dma_t;

struct _dev_io_t
{
	os_sem_t			sem;		// control process access
	int					abort;		// Abort the I/O operation.
	u32					dmdma_dir;	// DMDMA Direction for BCR.
	os_mem_t			mem;
	s32					mode;		// PIO, DMA or DMDMA
	s32					timeout;	// In seconds.
	WAIT_QUEUE_HEAD_T	queue;		// Wait for interrupts.

#if	(LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0))
	int					condition;	// Wait for interrupts.
#endif
};

struct _chan_data_t
{
	os_sem_t			sem;		// control process access
	struct _dev_data_t*	dev;
	int					in_use;		// Already in use (opened)?
	int					index;		// 0 - 3
	int					dev_index;	// same as minor number
	dev_io_t			rx;
	dev_io_t			tx;

	struct
	{
		u8						driver;	// The driver is using these.
		u8						notify;	// Notify the user of these.
		u8						status;	// Report this to the user.
		struct fasync_struct*	fasync;
	} irq;

	struct
	{
		u32		bsr_rx_fifo;			// Rx FIFO Present mask
		u32		bsr_tx_fifo;			// Tx FIFO Present mask

		s32		fifo_size_rx;			// Rx FIFO Size
		s32		fifo_size_tx;			// Tx FIFO Size

		u32		gsc_ccr_mask;			// Clock Control bits
		u32		gsc_ielr_defalt;		// Interrupt Edge/Level Reg
		u32		gsc_ihlr_defalt;		// Interrupt Hi/Low Register
		u32		gsc_irq_mask;			// Interrupt bits
		int		gsc_irq_shift;			// Shift IRQs by this amount.
		u32		gsc_fdr_offset;			// Offset of FIFO Data Reg.
		u32		gsc_fsr_32;				// FIFO Size Register

		s32		mp;						// Multi-Protocol Xcvr fw?
		s32		mp_bitmap;				// Corresponds to SP508 protocols (i.e. 1 << x)
		s32		mp_program;				// Can we program changes?

		s32		reg_ccr;				// Clock Control Register
		s32		reg_gpiosr;				// GPIO Source Registers
		s32		reg_psrcr;				// Pin Source Registers
		u32		reg_psrcr_bits;			// Bitmap of PSRCR supported bits
		s32		reg_pstsr;				// Pin Status Registers
		u32		reg_pstsr_bits;			// Which PSTSR bits are supported.
		s32		reg_rcr;				// Rx Counter Registers
		s32		reg_sbr;				// Sync Byte Registers
		s32		reg_tcr;				// Tx Counter Registers
		s32		rx_fifo_full_cfg_chan;	// Rx FIFO Full Config - channel
		s32		rx_fifo_overrun;		// CSR D16: Rx FIFO Overrun?
		s32		rx_status_word;			// CSR D3: Rx Status Word Enable?

		s32		tx_fifo_empty_cfg;		// CSR D18/D26: Tx Disable on FIFO Empty?

	} cache;

	struct
	{
		VADDR_T	gsc_csr_32;		// Command/Status Register
		VADDR_T	gsc_fcr_32;		// FIFO Count Register
		VADDR_T	gsc_fdr_8;		// FIFO Data Register
		VADDR_T	gsc_fsr_32;		// FIFO Size Register
		VADDR_T	gsc_psrcr_32;	// Pin Source Register
		VADDR_T	gsc_rar_32;		// Rx Almost Register
		VADDR_T	gsc_sbr_32;		// Sync Byte Register
		VADDR_T	gsc_tar_32;		// Tx Almost Register

		VADDR_T	usc_regs;		// Base of USC registers.
		VADDR_T	usc_bcr;		// Bus Configuration Reg
		VADDR_T	usc_ccar;		// Chan Command/Address Reg
		VADDR_T	usc_hcr;		// Hardware Config Reg
		VADDR_T	usc_iocr;		// Input/Output Control Reg
	} vaddr;

};



#include "mp.h"
#include "mp_fixed.h"
#include "mp_sp508.h"
#include "mp_unknown.h"

#include "osc.h"
#include "osc_cy22393.h"
#include "osc_cy22393_sio4.h"
#include "osc_fixed.h"
#include "osc_unknown.h"



struct _dev_data_t
{
	os_pci_t		pci;			// The kernel PCI device descriptor.
	os_spinlock_t	spinlock;		// Control ISR access.
	gsc_dev_type_t	board_type;		// Corresponds to basic device type.
	const char*		model;			// Base model number in upper case.
	int				board_index;	// Index of the device being accessed.
	char			model_buf[32];
	char			proc_id_str[32];

	chan_data_t		channel[4];

	struct
	{
		os_sem_t	sem;	// control process access
		dev_dma_t	channel[2];
	} dma;

	struct
	{
		os_sem_t	sem;	// control process access
		int			users;	// Channels open.
		os_irq_t	os;		// For OS level control.
#ifdef __SMP__
		// This is needed because the SIO4 has multiple channels
		// accessing the same device. On single channel devices spin
		// locks are not required bacause Linux makes sure that only
		// a single instance of the ISR runs at a time.
		spinlock_t	spinlock;
#endif
	} irq;

	struct
	{
		u32					gsc_fr_32;			// Features Register
		u32					gsc_frr_32;			// Firmware Revision Register

		// Features

		s32					board_reset;		// Board Reset in BCR
		s32					bus_speed;			// 0, 33, 66
		s32					bus_width;			// 0, 32, 64

		s32					channel_qty;		// Number of channels on the device.

		s32					device_qty;			// Number of SIO4's on the device.
		s32					dmdma_scd;			// DMDMA Single Cycle Disable

		s32					fifo_size_total;	// Total for all channels.
		s32					fifo_space_cfg;		// FIFO Space Config
		sio4_form_factor_t	form_factor;		// PCI, PMC, ...
		sio4_fw_type_t		fw_type;			// Zilog, SYNC, ...
		s32					fw_type_config;		// Yes or No

		s32					index_subdevice;	// SIO4: 0, SIO8: 0, 1
		s32					irq_32;				// 32 interrupts?
		u32					irq_mask;			// 0xFFFFFFFF or 0x0000FFFF

		s32					led_channel;		// Channel specific LEDs
		s32					led_main;			// Common LEDs
		int					legacy_cable;		// Legacy Cable Interface?

		sio4_model_t		model;				// SIO4_MODEL_SIOXXX_XXX
		s32					model_sync;			// Yes or No
		s32					model_z16c30;		// Yes or No
		sio4_mp_chip_t		mp_chip;			// Which MP chip is present?

		sio4_osc_chip_t		osc_chip;			// The oscillator chip model.
		s32					osc_measure;		// Can we measure the clock?
		u8					osc_pd_max;			// Oscillator maximum FW post divider value.
		s32					osc_per_chan;		// Is there an osc per chan?
		s32					osc_program;		// Can we program the clock?

		s32					reg_bsr;			// Board Status Register
		s32					reg_fcr;			// FIFO Count Registers
		s32					reg_fr;				// Features Register
		s32					reg_fsr;			// FIFO Size Registers
		s32					reg_ftr;			// Firmware Type Register
		s32					reg_ielr;			// Interrupt Edge/Level Register
		u32					reg_ielr_32;		// Value from Int Edge/Level Register
		s32					reg_ihlr;			// Interrupt High/Low Register
		s32					reg_iocr;			// I/O Control Register
		s32					reg_pcr;			// Programmable Clock Register?
		s32					reg_pocsr;			// Programmable Oscillator Control/Status Register
		s32					reg_porar;			// Programmable Oscillator RAM Address Register
		s32					reg_pordr;			// Programmable Oscillator RAM Data Register
		s32					reg_pord2r;			// Programmable Oscillator RAM Data 2 Register
		s32					reg_tsr;			// Time Stamp Registers
		s32					rx_fifo_full_cfg_glb;	// Rx FIFO Full Config - GLOBAL!
		s32					rx_fifo_underrun;	// CSR D18: Rx FIFO Underrun?

		sio4_type_t			sio4_type;			// SIO4/A/B/BX...

		s32					time_stamp;			//
		s32					tx_fifo_overrun;	// CSR D17: Tx FIFO Overrun?

		s32					user_jumper_qty;	// >= 0, 0 = none
		s32					user_jumper_sense;	// Jumper is on if bit is XXX.
		s32					user_jumper_val;	// jumper value from the BSR.

	} cache;

	union
	{
		// The value of features.osc_chip controls selection.
		osc_fixed_t			fixed;
		osc_cy22393_sio4_t	cy22393;
	} osc;

	struct
	{
		VADDR_T	gsc_regs;		// Base of GSC registers.
		VADDR_T	gsc_bcr_32;		// Board Control Reg
		VADDR_T	gsc_bsr_32;		// Board Status Reg
		VADDR_T	gsc_ccr_32;		// Clock Control Register
		VADDR_T	gsc_fr_32;		// Features Register
		VADDR_T	gsc_frr_32;		// FirmWare Register
		VADDR_T	gsc_ftr_32;		// Firmware Type Register
		VADDR_T	gsc_icr_32;		// Interrupt Control Register
		VADDR_T	gsc_ielr_32;	// Interrupt Edge/Level Reg
		VADDR_T	gsc_ihlr_32;	// Interrupt Hi/Low Register
		VADDR_T	gsc_isr_32;		// Interrupt Status Register
		VADDR_T	gsc_pocsr_32;	// Prog Osc Ctrl/Status Reg
		VADDR_T	gsc_porar_32;	// Prog Osc RAM Adrs Reg
		VADDR_T	gsc_pordr_32;	// Prog Osc RAM Data Reg
		VADDR_T	gsc_pord2r_32;	// Prog Osc RAM Data 2 Reg

		VADDR_T	plx_regs;		// Base of PLX register.
		VADDR_T	plx_dmaarb_32;	// DMA Mode Arbitration Reg
		VADDR_T	plx_dmathr_32;	// DMA Threshold Register
		VADDR_T	plx_intcsr_32;	// Interrupt Ctrl/Stat Reg
		VADDR_T	plx_marbr_32;	// Mode/Arbitration Reg
	} vaddr;
};

typedef struct
{
	s32	parm;
	u32	mask;	// Zero terminates a list.
	u32	val;
} parm_32_t;

typedef struct
{
	s32	parm;
	u32	mask1;	// Zero terminates a list.
	u32	value1;
	int	shift1;	// Shift this number of bits per channel.
	u32	mask2;	// Zero terminates a list.
	u32	value2;
	int	shift2;	// Shift this number of bits per channel.
} parm_2x32_t;

typedef struct
{
	s32	parm;
	u32	mask1;	// Zero terminates a list.
	u32	value1;
	u16	mask2;	// Zero terminates a list.
	u16	value2;
} parm_32_usc_t;



// prototypes *****************************************************************

s32		board_init_ioctl(chan_data_t* chan, void* arg);
s32		board_jumpers_ioctl(chan_data_t* chan, s32* arg);

s32		cable_config_ioctl(chan_data_t* chan, void* arg);
int		channel_features_init(chan_data_t* chan);
s32		channel_reset_ioctl(chan_data_t* chan, void* arg);
void	close_channel(chan_data_t* chan);
s32		cts_cable_config_ioctl(chan_data_t* chan, void* arg);

s32		dcd_cable_config_ioctl(chan_data_t* chan, void* arg);
void	dev_io_close(dev_io_t* io);
int		dev_io_open(chan_data_t* chan, dev_io_t* io);
void	dev_irq_close(dev_data_t* dev);
int		dev_irq_open(dev_data_t* dev);
int		dev_lock_filp(struct file* filp, chan_data_t** chan);
int		dev_lock_inode(struct inode* inode, int open, chan_data_t** chan);
void	device_reset(dev_data_t* dev);
s32		device_reset_ioctl(chan_data_t* chan, void* arg);

s32		feature_test_ioctl(chan_data_t* chan, void* arg);
s32		fifo_count(chan_data_t* chan, int shift);
int		fw_type_config_get(chan_data_t* chan, s32* get);		// sio4ioctl.h
int		fw_type_config_ioctl(chan_data_t* chan, void* arg);

s32		gsc_fasync(int inode, struct file* filp, int mode);

int		io_abort(dev_io_t* io);
void	io_dev_data_t_init(dev_data_t* dev, int index, int tx);
int		io_mode_config(dev_io_t* io, s32* value);
s32		int_notify_ioctl(chan_data_t* chan, void* arg);
s32		int_status_ioctl(chan_data_t* chan, void* arg);
int		irq_intcsr_mod(dev_data_t* dev, u32 value, u32 mask);

int		parm_32_check(u32 parm, const parm_32_t* list);
int		parm_32_get(dev_data_t* dev, VADDR_T vaddr, const parm_32_t* list);
int		parm_32_set(dev_data_t* dev, VADDR_T vaddr, u32 parm, const parm_32_t* list);
int		parm_32_usc_check(u32 parm, const parm_32_usc_t* list);

void	reg_mem32_mod_serial(dev_data_t* dev, VADDR_T vaddr, u32 value, u32 mask);
s32		reg_mod_ioctl(chan_data_t* chan, void* arg);
s32		reg_read_ioctl(chan_data_t* chan, void* arg);
s32		reg_read_raw_ioctl(chan_data_t* chan, void* arg);
s32		reg_write_ioctl(chan_data_t* chan, void* arg);
s32		rx_cable_config_ioctl(chan_data_t* chan, void* arg);
s32		rx_fifo_ae_config_ioctl(chan_data_t* chan, void* arg);
s32		rx_fifo_af_config_ioctl(chan_data_t* chan, void* arg);
s32		rx_fifo_count_ioctl(chan_data_t* chan, void* arg);
s32		rx_fifo_full_cfg_chan_ioctl(chan_data_t* chan, void* arg);
s32		rx_fifo_full_cfg_glb_ioctl(chan_data_t* chan, void* arg);
s32		rx_fifo_size_ioctl(chan_data_t* chan, void* arg);
s32		rx_fifo_status_ioctl(chan_data_t* chan, void* arg);
s32		rx_io_abort_ioctl(chan_data_t* chan, void* arg);
s32		rx_io_mode_config_ioctl(chan_data_t* chan, void* arg);
s32		rxc_usc_config_ioctl(chan_data_t* chan, void* arg);

s32		tx_cable_clock_config_ioctl(chan_data_t* chan, void* arg);
s32		tx_cable_config_ioctl(chan_data_t* chan, void* arg);
s32		tx_cable_data_config_ioctl(chan_data_t* chan, void* arg);
s32		tx_fifo_ae_config_ioctl(chan_data_t* chan, void* arg);
s32		tx_fifo_af_config_ioctl(chan_data_t* chan, void* arg);
s32		tx_fifo_count_ioctl(chan_data_t* chan, void* arg);
s32		tx_fifo_size_ioctl(chan_data_t* chan, void* arg);
s32		tx_fifo_status_ioctl(chan_data_t* chan, void* arg);
s32		tx_io_abort_ioctl(chan_data_t* chan, void* arg);
s32		tx_io_mode_config_ioctl(chan_data_t* chan, void* arg);
s32		txc_usc_config_ioctl(chan_data_t* chan, void* arg);

void	usc_reg_vaddr_mod(dev_data_t* dev, VADDR_T vaddr, u16 value, u16 mask);
u16		usc_reg_vaddr_read(dev_data_t* dev, VADDR_T vaddr);
void	usc_reg_vaddr_write(dev_data_t* dev, VADDR_T vaddr, u16 value);
s32		usc_reset_ioctl(chan_data_t* chan, void* arg);

s32		zilog_reset_ioctl(chan_data_t* chan, void* arg);

ssize_t	dma_perform(			dev_data_t*				dev,
								chan_data_t*			chan,
								int						to_dev,
								os_time_tick_t			st_end,
								int						dmdma,
								const void*				buff,
								ssize_t					count);

ssize_t	dma_read_available(		dev_data_t*				dev,
								chan_data_t*			chan,
								ssize_t					count);

ssize_t	dma_read_work(			dev_data_t*				dev,
								chan_data_t*			chan,
								char*					buff,
								ssize_t					count,
								os_time_tick_t			st_end);

ssize_t	dma_write_available(	dev_data_t*				dev,
								chan_data_t*			chan,
								ssize_t					count);

ssize_t	dma_write_work(			dev_data_t*				dev,
								chan_data_t*			chan,
								const char*				buff,
								ssize_t					count,
								os_time_tick_t			st_end);

ssize_t	dmdma_read_available(	dev_data_t*				dev,
								chan_data_t*			chan,
								ssize_t					count);

ssize_t	dmdma_read_work(		dev_data_t*				dev,
								chan_data_t*			chan,
								char*					buff,
								ssize_t					count,
								os_time_tick_t			st_end);

ssize_t	dmdma_write_available(	dev_data_t*				dev,
								chan_data_t*			chan,
								ssize_t					count);

ssize_t	dmdma_write_work(		dev_data_t*				dev,
								chan_data_t*			chan,
								const char*				buff,
								ssize_t					count,
								os_time_tick_t			st_end);

s32		fifo_almost_config(		chan_data_t*			chan,
								s32						value,
								VADDR_T					vaddr,
								int						shift,
								u32						reset);

int		parm_2x32_get(			chan_data_t*			chan,
								VADDR_T					vaddr1,
								VADDR_T					vaddr2,
								const parm_2x32_t*		list);

int		parm_2x32_set(			chan_data_t*			chan,
								VADDR_T					vaddr1,
								VADDR_T					vaddr2,
								u32						parm,
								const parm_2x32_t*		list);

int		parm_32_usc_get(		dev_data_t*				dev,
								VADDR_T					vaddr1,
								VADDR_T					vaddr2,
								const parm_32_usc_t*	list);

int		parm_32_usc_set(		dev_data_t*				dev,
								VADDR_T					vaddr1,
								VADDR_T					vaddr2,
								u32						parm,
								const parm_32_usc_t*	list);

ssize_t	pio_read_available(		dev_data_t*				dev,
								chan_data_t*			chan,
								ssize_t					count);

ssize_t	pio_read_work(			dev_data_t*				dev,
								chan_data_t*			chan,
								char*					buff,
								ssize_t					count,
								os_time_tick_t			st_end);

ssize_t	pio_write_available(	dev_data_t*				dev,
								chan_data_t*			chan,
								ssize_t					count);

ssize_t	pio_write_work(			dev_data_t*				dev,
								chan_data_t*			chan,
								const char*				buff,
								ssize_t					count,
								os_time_tick_t			st_end);

int	SIO4ClearDPLLStatus(chan_data_t* pstrDev, void* arg);
int	SIO4EnableBRG0(chan_data_t* pstrDev, void* arg);
int	SIO4EnableBRG1(chan_data_t* pstrDev, void* arg);
int	SIO4InitChannel(chan_data_t* pstrChannelDev, void* arg);
int	SIO4ResetFifo(chan_data_t* pstrChannelDev, void* arg);
int	SIO4SelectDPLLResync(chan_data_t* pstrDev, void* arg);
int	SIO4SendChanCmd(chan_data_t* pstrChannelDev, void* arg);
int	SIO4SetBRG0Mode(chan_data_t* pstrDev, void* arg);
int	SIO4SetBRG0Source(chan_data_t* pstrChannelDev, void* arg);
int	SIO4SetBRG1Mode(chan_data_t* pstrDev, void* arg);
int	SIO4SetBRG1Source(chan_data_t* pstrChannelDev, void* arg);
int	SIO4SetCTR0Source(chan_data_t* pstrChannelDev, void* arg);
int	SIO4SetCTR1Source(chan_data_t* pstrChannelDev, void* arg);
int	SIO4SetDPLLDivisor(chan_data_t* pstrDev, void* arg);
int	SIO4SetDPLLMode(chan_data_t* pstrDev, void* arg);
int	SIO4SetDPLLSource(chan_data_t* pstrChannelDev, void* arg);
int	SIO4SetReadTimeout(chan_data_t* pstrDev, void* arg);
int	SIO4SetReceiveASYNCProtocol(chan_data_t* pstrDev, void* arg);
int	SIO4SetReceiveHDLCProtocol(chan_data_t* pstrDev, void* arg);
int	SIO4SetReceiveISOCHRProtocol(chan_data_t* pstrDev, void* arg);
int	SIO4SetRxClockSource(chan_data_t* pstrChannelDev, void* arg);
int	SIO4SetSyncByte(chan_data_t* pstrDev, void* arg);
int	SIO4SetTransmitASYNCProtocol(chan_data_t* pstrDev, void* arg);
int	SIO4SetTransmitISOCHRProtocol(chan_data_t* pstrDev, void* arg);
int	SIO4SetTransmitHDLCProtocol(chan_data_t* pstrDev, void* arg);
int	SIO4SetTransmitHDLCSDLCLoopProtocol(chan_data_t* pstrDev, void* arg);
int	SIO4SetTxClockSource(chan_data_t* pstrChannelDev, void* arg);
int	SIO4SetUSCDMAOptions(chan_data_t* pstrDev, void* arg);
int	SIO4SetWriteTimeout(chan_data_t* pstrDev, void* arg);



#endif
