// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/driver/io_dma.c $
// $Rev: 53095 $
// $Date: 2023-06-13 10:42:41 -0500 (Tue, 13 Jun 2023) $

// SIO4: Device Driver: source file

#include "main.h"



/******************************************************************************
*
*	Function:	_rx_avail_csr
*
*	Purpose:
*
*		See how much DMA data is available using the Control/Status Register.
*
*	Arguments:
*
*		chan	The channel data structure.
*
*		bytes	The number of bytes desired.
*
*	Returned:
*
*		>= 0	This is the number of bytes to read.
*
******************************************************************************/

static ssize_t _rx_avail_csr(chan_data_t* chan, ssize_t bytes)
{
	dev_data_t*	dev		= chan->dev;
	u32			csr;
	ssize_t		qty;

	csr	= os_reg_mem_rx_u32(dev, chan->vaddr.gsc_csr_32);

	if (csr & SIO4_GSC_CSR_RFE)
		qty	= bytes;
	else
		qty	= 0;

	return(qty);
}



/******************************************************************************
*
*	Function:	_rx_avail_csr_fsr
*
*	Purpose:
*
*		See how much DMA data is available using the Control/Status Register
*		and the FIFO Size Register.
*
*	Arguments:
*
*		chan	The channel data structure.
*
*		bytes	The number of bytes desired.
*
*	Returned:
*
*		>= 0	This is the number of bytes to read.
*
******************************************************************************/

static ssize_t _rx_avail_csr_fsr(chan_data_t* chan, ssize_t bytes)
{
	dev_data_t*	dev		= chan->dev;
	u32			csr;
	ssize_t		qty;

	csr	= os_reg_mem_rx_u32(dev, chan->vaddr.gsc_csr_32);

	if ((csr & SIO4_GSC_CSR_RFE) == 0)
		qty	= 0;
	else if (bytes > (ssize_t) chan->cache.fifo_size_rx)
		qty	= chan->cache.fifo_size_rx;
	else
		qty	= bytes;

	return(qty);
}



/******************************************************************************
*
*	Function:	_rx_avail_fcr
*
*	Purpose:
*
*		See how much DMA data is available using the FIFO Count Register.
*
*	Arguments:
*
*		chan	The channel data structure.
*
*	Returned:
*
*		>= 0	This is the number of bytes to read.
*
******************************************************************************/

static ssize_t _rx_avail_fcr(chan_data_t* chan)
{
	dev_data_t*	dev		= chan->dev;
	u32			fcr;
	ssize_t		qty;

	fcr	= os_reg_mem_rx_u32(dev, chan->vaddr.gsc_fcr_32);
	qty	= fcr >> 16;
	return(qty);
}



/******************************************************************************
*
*	Function:	_tx_avail_csr
*
*	Purpose:
*
*		See how much DMA space is available using the Control/Status Register.
*
*	Arguments:
*
*		chan	The channel data structure.
*
*		bytes	The number of bytes desired.
*
*	Returned:
*
*		>= 0	This is the number of bytes to write.
*
******************************************************************************/

static ssize_t _tx_avail_csr(chan_data_t* chan, ssize_t bytes)
{
	dev_data_t*	dev		= chan->dev;
	u32			csr;
	ssize_t		qty;

	csr	= os_reg_mem_rx_u32(dev, chan->vaddr.gsc_csr_32);

	if (csr & SIO4_GSC_CSR_TFF)
		qty	= bytes;
	else
		qty	= 0;

	return(qty);
}



/******************************************************************************
*
*	Function:	_tx_avail_csr_fsr
*
*	Purpose:
*
*		See how much DMA space is available using the Control/Status Register
*		and the FIFO Size Register.
*
*	Arguments:
*
*		chan	The channel data structure.
*
*		bytes	The number of bytes desired.
*
*	Returned:
*
*		>= 0	This is the number of bytes to write.
*
******************************************************************************/

static ssize_t _tx_avail_csr_fsr(chan_data_t* chan, ssize_t bytes)
{
	dev_data_t* dev		= chan->dev;
	u32			csr;
	ssize_t		qty;

	csr	= os_reg_mem_rx_u32(dev, chan->vaddr.gsc_csr_32);

	if ((csr & SIO4_GSC_CSR_TFF) == 0)
		qty	= 0;
	else if (bytes > (ssize_t) chan->cache.fifo_size_tx)
		qty	= chan->cache.fifo_size_tx;
	else
		qty	= bytes;

	return(qty);
}



/******************************************************************************
*
*	Function:	_tx_avail_fcr_fsr
*
*	Purpose:
*
*		See how much DMA space is available using the FIFO Count Register and
*		the FIFO Size Register.
*
*	Arguments:
*
*		chan	The channel data structure.
*
*	Returned:
*
*		>= 0	This is the number of bytes to write.
*
******************************************************************************/

static ssize_t _tx_avail_fcr_fsr(chan_data_t* chan)
{
	dev_data_t*	dev		= chan->dev;
	u32			count;
	u32			fcr;
	ssize_t		qty;

	fcr	= os_reg_mem_rx_u32(dev, chan->vaddr.gsc_fcr_32);
	count	= fcr & 0xFFFF;
	qty	= chan->cache.fifo_size_tx - count;
	return(qty);
}



/******************************************************************************
*
*	Function:	dma_read_available
*
*	Purpose:
*
*		Determine the number of bytes that can be read for a DMA mode read
*		operation.
*
*	Arguments:
*
*		dev		The device data structure.
*
*		chan	The channel data structure.
*
*		bytes	The number of bytes desired.
*
*	Returned:
*
*		>= 0	This is the number of bytes to read.
*
******************************************************************************/

ssize_t dma_read_available(dev_data_t* dev, chan_data_t* chan, ssize_t bytes)
{
	ssize_t	qty;

	if (dev->cache.reg_fcr)
		qty	= _rx_avail_fcr(chan);
	else if (dev->cache.reg_fsr)
		qty	= _rx_avail_csr_fsr(chan, bytes);
	else
		qty	= _rx_avail_csr(chan, bytes);

	if (qty > bytes)
		qty	= bytes;

	if (qty > DMA_PIO_THRESHOLD)
		qty	&= ~0x3L;	// Force it to a 32-bit boundary.

	return(qty);
}



/******************************************************************************
*
*	Function:	dma_read_work
*
*	Purpose:
*
*		Implement the data reading portion of the DMA read operation.
*
*	Arguments:
*
*		dev		The device data structure.
*
*		chan	The channel data structure.
*
*		buff	The samples read go here.
*
*		count	The number of bytes to read.
*
*		st_end	End at this point in time (in system timer ticks). This is
*				zero if we're to end as soon at no data is available.
*
*	Returned:
*
*		>= 0	This is the number of samples read.
*
******************************************************************************/

ssize_t dma_read_work(
	dev_data_t*		dev,
	chan_data_t*	chan,
	char*			buff,
	ssize_t			count,
	os_time_tick_t	st_end)
{
	ssize_t	qty;

	if (count <= DMA_PIO_THRESHOLD)
	{
		qty	= pio_read_work(dev, chan, buff, count, st_end);
	}
	else
	{
		qty	= dma_perform(	dev,
							chan,
							0,	// read
							st_end,
							0,	// NOT DMDMA
							buff,
							count);
	}

	return(qty);
}



/******************************************************************************
*
*	Function:	dma_write_available
*
*	Purpose:
*
*		Determine the number of bytes that can be written for a DMA mode write
*		operation.
*
*	Arguments:
*
*		dev		The device data structure.
*
*		samples	The number of samples desired.
*
*	Returned:
*
*		>= 0	This is the number of bytes to write.
*
******************************************************************************/

ssize_t dma_write_available(dev_data_t* dev, chan_data_t* chan, ssize_t bytes)
{
	ssize_t	qty;

	if (dev->cache.reg_fsr == 0)
		qty	= _tx_avail_csr(chan, bytes);
	else if (dev->cache.reg_fcr)
		qty	= _tx_avail_fcr_fsr(chan);
	else
		qty	= _tx_avail_csr_fsr(chan, bytes);

	if (qty > bytes)
		qty	= bytes;

	if (qty > DMA_PIO_THRESHOLD)
		qty	&= ~0x3L;	// Force it to a 32-bit boundary.

	return(qty);
}



/******************************************************************************
*
*	Function:	dma_write_work
*
*	Purpose:
*
*		Implement the data writing portion of the PIO write operation.
*
*	Arguments:
*
*		dev		The device data structure.
*
*		chan	The channel data structure.
*
*		buff	The samples read go here.
*
*		count	The number of bytes to write.
*
*		st_end	End at this point in time (in system timer ticks). This is
*				zero if we're to end as soon at no data is available.
*
*	Returned:
*
*		>= 0	This is the number of samples written.
*
******************************************************************************/

ssize_t dma_write_work(
	dev_data_t*		dev,
	chan_data_t*	chan,
	const char*		buff,
	ssize_t			count,
	os_time_tick_t	st_end)
{
	ssize_t	qty;

	if (count <= DMA_PIO_THRESHOLD)
	{
		qty	= pio_write_work(dev, chan, buff, count, st_end);
	}
	else
	{
		qty	= dma_perform(	dev,
							chan,
							1,	// write
							st_end,
							0,	// NOT DMDMA
							buff,
							count);
	}

	return(qty);
}



