// $URL: http://subversion:8080/svn/gsc/trunk/drivers/LINUX/SIO4%20and%20SIO8/SIO4_Linux_1.x.x_GSC_DN/driver/dma.c $
// $Rev: 53095 $
// $Date: 2023-06-13 10:42:41 -0500 (Tue, 13 Jun 2023) $

// SIO4: Device Driver: source file

#include "main.h"



// macros *********************************************************************

#define	DMA_DPR_END_OF_CHAIN				0x00000002
#define	DMA_DPR_TERMINAL_COUNT_IRQ			0x00000004
#define	DMA_DPR_HOST_TO_BOARD				0x00000000
#define	DMA_DPR_BOARD_TO_HOST				0x00000008

#define	DMA_MODE_SIZE_8_BITS				0x00000000
#define	DMA_MODE_INPUT_ENABLE				0x00000040
#define	DMA_MODE_BURSTING_LOCAL				0x00000100
#define	DMA_MODE_INTERRUPT_WHEN_DONE		0x00000400
#define	DMA_MODE_LOCAL_ADRESS_CONSTANT		0x00000800
#define	DMA_MODE_STANDARD_DMA				0x00000000
#define	DMA_MODE_DEMAND_MODE_DMA			0x00001000
#define	DMA_MODE_PCI_INTERRUPT_ENABLE		0x00020000



/******************************************************************************
*
*	Function:	_dma_channel_get
*
*	Purpose:
*
*		Get a PLX DMA channel for the specified operation.
*
*	Arguments:
*
*		dev		The device data structure.
*
*	Returned:
*
*		< 0		The request failed.
*		0		Use channel zero.
*		1		Use channel one.
*
******************************************************************************/

static int _dma_channel_get(dev_data_t* dev)
{
	int	status;

	for (;;)	// Use a loop for convenience.
	{
		status	= os_sem_lock(&dev->dma.sem);

		if (status)
		{
			status	= -1;
			break;
		}

		if (dev->dma.channel[0].in_use == 0)
		{
			dev->dma.channel[0].in_use	= 1;
			status	= 0;
			os_sem_unlock(&dev->dma.sem);
			break;
		}

		if (dev->dma.channel[1].in_use == 0)
		{
			dev->dma.channel[1].in_use	= 1;
			status	= 1;
			os_sem_unlock(&dev->dma.sem);
			break;
		}

		status	= -1;
		os_sem_unlock(&dev->dma.sem);
		break;
	}

	return(status);
}



/******************************************************************************
*
*	Function:	dma_perform
*
*	Purpose:
*
*		Perform a DMA transfer per the given arguments.
*
*		NOTES:
*
*		With interrupts it is possible for the requested interrupt to occure
*		before the current task goes to sleep. This is possible, for example,
*		because the DMA transfer could complete before the task gets to sleep.
*		So, to make sure things work in these cases, we take manual steps to
*		insure that the current task can be awoken by the interrupt before the
*		task actually sleeps and before the interrupt is enabled.
*
*	Arguments:
*
*		dev		The device data structure.
*
*		chan	The channel structure in use.
*
*		to_dev	transfer data to the device? Else from it.
*
*		st_end	The amount of time left in our timeout period. This is in
*				system timer ticks.
*
*		dmdma	Use Demand Mode DMA? Else DMA.
*
*		buff	The data transfer buffer.
*
*		count	The number of bytes to transfer.
*
*	Returned:
*
*		>= 0	The number of bytes transferred.
*		< 0		There was an error.
*
******************************************************************************/

ssize_t	dma_perform(
	dev_data_t*		dev,
	chan_data_t*	chan,
	int				to_dev,
	os_time_tick_t	st_end,
	int				dmdma,
	const void*		buff,
	ssize_t			count)
{
	unsigned int		dir;
	dev_dma_t*			dma;
	int					index;
	dev_io_t*			io;
	u32					mask;
	u32					offset		= chan->cache.gsc_fdr_offset;
	ssize_t				qty;
	unsigned long		timeout;
	unsigned int		type;
	u32					ul;
	WAIT_QUEUE_ENTRY_T	wait;

	for (;;)	// Use a loop for convenience.
	{
		index	= _dma_channel_get(dev);

		if (index < 0)
		{
			// We couldn't get a DMA channel.
			qty	= 0;
			break;
		}

		dma	= &dev->dma.channel[index];

		if (dmdma)
		{
			io		= to_dev ? &chan->tx : &chan->rx;
			ul		= io->dmdma_dir << dma->dir_shift;
			mask	= dma->dir_mask;

			if (dev->cache.dmdma_scd)
			{
				ul		|= 0x88;
				mask	|= 0x88;
			}

			os_reg_mem_mx_u32(dev, dev->vaddr.gsc_bcr_32, ul, mask);
		}

		// Enable the interrupt.
		irq_intcsr_mod(dev, dma->int_enable, dma->int_enable);

		// DMAMODEx
		type	= dmdma
				? DMA_MODE_DEMAND_MODE_DMA
				: DMA_MODE_STANDARD_DMA;
		ul		= type
				| DMA_MODE_SIZE_8_BITS
				| DMA_MODE_INPUT_ENABLE
				| DMA_MODE_BURSTING_LOCAL
				| DMA_MODE_INTERRUPT_WHEN_DONE
				| DMA_MODE_LOCAL_ADRESS_CONSTANT
				| DMA_MODE_PCI_INTERRUPT_ENABLE;
		os_reg_mem_tx_u32(dev, dma->vaddr.mode_32, ul);

		// DMAPADRx, The DMA buffer is in 32-bit addressible memory.
		ul	= (u32) virt_to_bus((void*) buff);
		os_reg_mem_tx_u32(dev, dma->vaddr.padr_32, ul);

		// DMALADRx
		os_reg_mem_tx_u32(dev, dma->vaddr.ladr_32, offset);

		// DMASIZx
		os_reg_mem_tx_u32(dev, dma->vaddr.siz_32, (u32) count);

		// DMADPRx
		dir	= to_dev
			? DMA_DPR_HOST_TO_BOARD
			: DMA_DPR_BOARD_TO_HOST;
		ul	= dir
			| DMA_DPR_END_OF_CHAIN
			| DMA_DPR_TERMINAL_COUNT_IRQ;
		os_reg_mem_tx_u32(dev, dma->vaddr.dpr_32, ul);

		// DMAARB
		os_reg_mem_tx_u32(dev, dev->vaddr.plx_dmaarb_32, 0);

		// DMATR
		os_reg_mem_tx_u32(dev, dev->vaddr.plx_dmathr_32, 0);

		// Do the magic!

		dma->queue	= to_dev ? &chan->tx.queue : &chan->rx.queue;
		memset(&wait, 0, sizeof(wait));
		WAIT_QUEUE_ENTRY_INIT(&wait, current);	// NOTES
		SET_CURRENT_STATE(TASK_INTERRUPTIBLE);	// NOTES
		add_wait_queue(dma->queue, &wait);		// NOTES
		ul	= os_reg_mem_rx_u8(dev, dma->vaddr.csr_8);
		ul	|= DMA_CSR_ENABLE;
		os_reg_mem_tx_u8(dev, dma->vaddr.csr_8, ul);		// Enable
		ul	|= DMA_CSR_START;
		os_reg_mem_tx_u8(dev, dma->vaddr.csr_8, ul);		// Start

		// Concurrent added this as it prevented a timeout
		udelay(10);

		timeout = st_end ? (st_end - os_time_tick_get()) : (unsigned long) os_time_tick_rate();

		// Concurrent added these mods as testing under a debug 4.4.126 gave a
		// might_sleep warning. 4.1.37 gave no warning, 4.4.126 gave a warning.

		SET_CURRENT_STATE(TASK_RUNNING);		// NOTES
		EVENT_WAIT_IRQ_TO(dma->queue, dma->condition, timeout);
		SET_CURRENT_STATE(TASK_INTERRUPTIBLE);	// NOTES
		remove_wait_queue(dma->queue, &wait);	// NOTES
		SET_CURRENT_STATE(TASK_RUNNING);		// NOTES

		// Clean up.

		// Disable the DMA interrupt.
		irq_intcsr_mod(dev, 0, dma->int_enable);

		// Disable the DONE interrupt.
		os_reg_mem_mx_u32(dev, dma->vaddr.mode_32, 0, DMA_MODE_INTERRUPT_WHEN_DONE);

		// Check the results.
		ul	= os_reg_mem_rx_u8(dev, dma->vaddr.csr_8);

		if (ul & DMA_CSR_DONE)
		{
			qty	= count;
		}
		else
		{
			// If it isn't done then the operation timed out or was aborted.
			// Unfortunately the PLX PCI9080 doesn't tell us how many bytes
			// were transferred. So, instead of reporting the number of
			// transferred bytes, or zero, we return an error status.

			// Force DMA termination.
			ul	&= ~DMA_CSR_ENABLE;
			os_reg_mem_tx_u8(dev, dma->vaddr.csr_8, ul);
			ul	|= DMA_CSR_ABORT;
			os_reg_mem_tx_u8(dev, dma->vaddr.csr_8, ul);
			os_time_us_delay(10);		// Wait for completion.
			qty	= -ETIMEDOUT;

			// Let someone else use the DMA channel.
		}

		// Clear the DMA.
		os_reg_mem_tx_u8(dev, dma->vaddr.csr_8, 0);
		os_reg_mem_tx_u8(dev, dma->vaddr.csr_8, DMA_CSR_CLEAR);

		// Let someone else use the DMA channel.
		dma->queue	= NULL;
		dma->in_use	= 0;
		break;
	}

	return(qty);
}



